/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: STRTOK.C $
   $Locker:  $	$Name:  $	$State: Exp $

	char_t *strtok(char_t *str, char_t *delim)

	Tokenize a string. Tokens are delimited by any combination
	of characters of the character set delim.

	A call with str != NULL will always return str. This call initiates
	a sequence of calls with str == NULL. On return the first character
	of the string that is a member of delim has been overwritten with the
	NUL character to terminate the current token. The position of the
	firstcharacter behind this one that is no member of delim will be
	saved to be used for the next call of strtok with str == NULL.

	strtok() is not reentrant and a call with str != NULL will overwrite
	the saved position of any previous call of strtok.

	When strtok found no further token, NULL is returned. Further calls
	with str == NULL may be made savely, thus, they will return NULL, too.

	Input:
		delim != NULL && delim != ""

	Return:
		str if str != NULL
		NULL if no further token has been found
		pointer to the next toke, otherwise

	Note:
		<none>

	Conforms to:
		ANSI

	See also:

	Target compilers:
		Any C compiler

	Origin:
		1997/10/10 Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.2  1998/02/08 04:55:59  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <_string.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: STRTOK.C 1.2 1998/02/08 04:55:59 ska Exp $";
#endif

_CLibFunc char_t ___PTR *strtok(char_t ___PTR *str
 , char_t const ___PTR * const delim)
{	static char_t ___PTR *nxtToken = 0;
	REG char_t ___PTR *p;

	assert(delim != NULL);

	if(!str && (str = nxtToken) == NULL)
		return NULL;	/* no further token */

	/* str := start of this token
		now search for the start of the next token */
	if((p = strpbrk(str, delim)) == NULL)	/* end of this token */
		/* no further delimiter -> no further token */
		nxtToken = NULL;
	else {
		*p = NUL;			/* mark the end of this token */
		/* search for the end of the delimiters.
			If none -> nxttoken := NULL */
		nxtToken = _strspnp(p + 1, delim);
	}

	return str;
}
