   /* ---------- DIRFNS.C - directory-related functions ---- */
/* 

   Copyright (c) 1996 M. M. Peritsky.  All rights reserved.
   This program is part of the Free-DOS Project.
   See COPY.TXT or XCOPY.TXT for more details.

*/
   /* ---------- include files ----------------------------- */

#include <stdio.h>
#include <string.h>
#include <direct.h>
#include <dos.h>
#include <ctype.h>

#include "copy.h"
#include "strfns.h"
#include "dirfns.h"

   /* ---------- declaration of symbolic constants --------- */

#define TRUE 1
#define FALSE 0

#define MAX_FILE_EXT 13

   /* ---------- declaration of macros --------------------- */

#define STRCAT2(x, y) strcat3(x, y, "")

#define CWD_IS_ROOT (!cwd_parent_exists)
#define DWD_IS_ROOT (!dwd_parent_exists)
#define CWD_IS_NOT_ROOT cwd_parent_exists
#define DWD_IS_NOT_ROOT dwd_parent_exists

#define FNSPLITX(x) fnsplit(x, tf.drive, tf.dir, tf.file, tf.ext)
#define FNMERGEX(x) fnmerge(x, tf.drive, tf.dir, tf.file, tf.ext)
#define FNSPLIT FNSPLITX(fn_path)
#define FNMERGE FNMERGEX(fn_path)

   /* ---------- declaration of global variables ----------- */

   extern char cwd[MAXPATH];
   extern char cwd_parent[MAXPATH];
   extern int cwd_parent_exists;

   extern char *dwd;
   extern int dwd_parent_exists;

   extern int s_switch;

   extern struct ffblk filedata;

   /* ---------- start fullpath function ------------------- */

char *fullpath(char *q)
{
   int i;

   char *p;

   char fn_path[MAXPATH];
   struct fn_parts {
      char drive[MAXDRIVE];
      char dir[MAXDIR];
      char file[MAXFILE];
      char ext[MAXEXT];
   } tf;

   char *devices[] = { "AUX", "COM1", "COM2", "COM3", "COM4",
      "CON", "LPT1", "LPT2", "LPT3", "NUL", "PRN", NULL };

   for (i = 0; devices[i] != NULL; i++) {
      if (strcmpi(devices[i], q) == 0) {
         return strdup1(devices[i]);
      }
   }

   if (strcmp(q, ".") == 0) {
      return CWD_IS_ROOT ? STRCAT2(cwd, "*.*")
                         : STRCAT2(cwd, "\\*.*");
   }

   if (strcmp(q, "..") == 0) {
      if (cwd_parent_exists) {
         return STRCAT2(cwd_parent, "*.*");
      }
      else {
         printf("Specification .. is invalid, root does not have parent.\n");
         main_exit(1);
      }
   }

   if (strncmp(q, "..\\", 3) == 0) {
      if (cwd_parent_exists) {
         p = STRCAT2(cwd_parent, (q+3));
      }
      else {
         printf("Specification .. is invalid, root does not have parent.\n");
         main_exit(1);
      }
   }
   else {
      if (strncmp(q, ".\\", 2) == 0) {
         p = STRCAT2(cwd, (CWD_IS_ROOT ? (q+2) : (q+1)));
      }
      else {
         p = strdup1(q);
      }
   }

   i = FNSPLITX(p);

   if ((i & DRIVE) != 0) update_dwd(tf.drive);               

   /* The mask-and-shift operations in the next statement cause 
      the bits in the result to have the following meanings:
      0x01 -- wildcards are present
      0x02 -- filename and/or extension are present
      0x04 -- directory is present
      0x08 -- drive is present
   */

   switch( ((i & 28) >> 1) | (i & 3) ) {
      case 2: /* example: a.b */
         strcpy(fn_path, cwd);
         if (CWD_IS_NOT_ROOT) strcat(fn_path,"\\");
         strcat(fn_path, p);
         if (is_dir(fn_path)) strcat(fn_path,"\\*.*");
         break;
      case 3: /* example: *.* */
         strcpy(fn_path, cwd);
         if (CWD_IS_NOT_ROOT) strcat(fn_path,"\\");
         strcat(fn_path, p);
         break;
      case 4: /* example: d\ or \d\ */
         if (*(tf.dir) != '\\') {
            strcpy(fn_path, cwd);
            if (CWD_IS_NOT_ROOT) strcat(fn_path,"\\");
         }
         else {
            strncpy(fn_path, cwd, 2);
            fn_path[2] = '\0';
         }
         strcat(fn_path, p);
         strcat(fn_path,"*.*");
         break;
      case 6: /* example: d\a.b or \d\a.b */
         if (*(tf.dir) != '\\') {
            strcpy(fn_path, cwd);
            if (CWD_IS_NOT_ROOT) strcat(fn_path,"\\");
         }
         else {
            strncpy(fn_path, cwd, 2);
            fn_path[2] = '\0';
         }
         strcat(fn_path, p);
         if (is_dir(fn_path)) strcat(fn_path,"\\*.*");
         break;
      case 7: /* example: d\*.* or \d\*.* */
         if (*(tf.dir) != '\\') {
            strcpy(fn_path, cwd);
            if (CWD_IS_NOT_ROOT) strcat(fn_path,"\\");
         }
         else {
            strncpy(fn_path, cwd, 2);
            fn_path[2] = '\0';
         }
         strcat(fn_path, p);
         break;
      case 8: /* example: e: */
         strcpy(fn_path, dwd);
         if (DWD_IS_NOT_ROOT) strcat(fn_path,"\\");
         strcat(fn_path, "*.*");
         break;
      case 10: /* example: e:a.b */
         strcpy(fn_path, dwd);
         if (DWD_IS_NOT_ROOT) strcat(fn_path,"\\");
         strcat(fn_path, tf.file);
         strcat(fn_path, tf.ext);
         if (is_dir(fn_path)) strcat(fn_path,"\\*.*");
         break;
      case 11: /* example: e:*.* */
         strcpy(fn_path, dwd);
         if (DWD_IS_NOT_ROOT) strcat(fn_path,"\\");
         strcat(fn_path, tf.file);
         strcat(fn_path, tf.ext);
         break;
      case 12: /* example: e:d\ or e:\d\ */
         if (*(tf.dir) == '\\') strcpy(fn_path, p);
         else {
            strcpy(fn_path, dwd);
            if (DWD_IS_NOT_ROOT) strcat(fn_path,"\\");
            strcat(fn_path, tf.dir);
         }
         strcat(fn_path, "*.*");
         break;
      case 14: /* example: e:d\a.b or e:\d\a.b */
         if (*(tf.dir) == '\\') strcpy(fn_path, p);
         else {
            strcpy(fn_path, dwd);
            if (DWD_IS_NOT_ROOT) strcat(fn_path,"\\");
            strcat(fn_path, tf.dir);
            strcat(fn_path, tf.file);
            strcat(fn_path, tf.ext);
         }
         if (is_dir(fn_path)) strcat(fn_path,"\\*.*");
         break;
      case 15: /* example: e:d\*.* or e:\d\*.* */
         if (*(tf.dir) == '\\') strcpy(fn_path, p);
         else {
            strcpy(fn_path, dwd);
            if (DWD_IS_NOT_ROOT) strcat(fn_path,"\\");
            strcat(fn_path, tf.dir);
            strcat(fn_path, tf.file);
            strcat(fn_path, tf.ext);
         }
         break;
      default: /* error */
         printf("Specification %s is not valid.\n", p);
         main_exit(1);
         break;
   }            
   free(p);
   return strdup1(fn_path);
}

   /* ---------- end fullpath function --------------------- */

   /* ---------- start getcurdir1 function ----------------- */

char *getcurdir1(char *p)
{
   char q[MAXDIR];

   if (getcurdir(toupper(*p)-'A'+1, q) == 0) {
      return strcat3(p, "\\", q);
   }
   else {
      printf("Unable to determine current directory of %s\n", p);
      main_exit(1);
   }
}

   /* ---------- end getcurdir1 function ------------------- */

   /* ---------- start update_dwd function ----------------- */

void update_dwd(char *p)
{
   if (toupper(*p) != toupper(*dwd)) {
      free(dwd);
      dwd = getcurdir1(p);
      dwd_parent_exists = (strlen(dwd) > 3);
   }
}

   /* ---------- end update_dwd function ------------------- */

   /* ---------- start is_dir function --------------------- */

int is_dir(char *p)
{
   int i = FALSE;
   struct ffblk fd;

   if (findfirst(p, &fd, (FA_DIREC | FA_SYSTEM | FA_HIDDEN)) == 0) {
      if ( (fd.ff_attrib & FA_DIREC) != 0 ) {
         i = TRUE;
      }
   }
   return i;
}

   /* ---------- end is_dir function ----------------------- */

   /* ---------- start fnsplit2 function ------------------- */

int fnsplit2(char *x, char **drive_dir, char **file_ext)
{
   char fn_path[MAXPATH];
   struct fn_parts {
      char drive[MAXDRIVE];
      char dir[MAXDIR];
      char file[MAXFILE];
      char ext[MAXEXT];
   } tf;

   int i;
   i = FNSPLITX(x);

   *drive_dir = STRCAT2(tf.drive, tf.dir);
   *file_ext = STRCAT2(tf.file, tf.ext);

   return i;
}

   /* ---------- end fnsplit2 function --------------------- */

   /* ---------- start newname function -------------------- */

char *newname(char *f, char *p)
{

   /* This function creates a new string containing the filename
      implied by the input filename and pattern. Example: input
      file "ABC.TXT", pattern "*.DOC", result "ABC.DOC".
   */

   char q[MAX_FILE_EXT];

   int i;

   for (i = 0; i < MAX_FILE_EXT; i++) q[i] = '\0';

   i = 0;
   do {
      switch (*p) {
         case '?':
            switch (*f) {
               case '.':
                  /* break intentionally omitted */
               case '\0':
                  p++;
                  break;
               default:
                  q[i++] = *(f++);
                  p++;
                  break;
            }
            break;
         case '*':
            switch (*f) {
               case '.':
                  /* break intentionally omitted */
               case '\0':
                  p++;
                  break;
               default:
                  do {
                     q[i++] = *(f++);
                  } while ((*f != '.')
                        && (*f != '\0')
                        && (i < MAX_FILE_EXT));
                  do {
                     p++;
                  } while ((*p != '.')
                        && (*p != '\0'));
                  break;
            }
            break;
         case '.':
            switch (*f) {
               case '.':
                  f++;
                  /* break intentionally omitted */
               case '\0':
                  q[i++] = '.';
                  p++;
                  break;
               default:
                  do {
                     f++;
                  } while ((*f != '.')
                        && (*f != '\0'));
                  break;
            }
            break;
         case '\0':
            break;
         default:
            switch (*f) {
               case '.':
                  /* break intentionally omitted */
               case '\0':
                  break;
               default:
                  f++;
                  break;
            }
            q[i++] = *(p++);
            break;
      }
   } while (((*p) != '\0') && (i < MAX_FILE_EXT));

   return strdup1(q);
}

   /* ---------- end newname function ---------------------- */

   /* ---------- start file_exists function ---------------- */

int file_exists(char *p)
{
   struct ffblk fd;

   return (findfirst(p, &fd, FA_SYSTEM | FA_HIDDEN) == 0);
}

   /* ---------- end file_exists function ------------------ */
