/*
 *
 * DeleteBut(DB) v.1.2a Deletes files in current dir that were not specified
 * Copyright (C) 1998 Robert Lake
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 * Robert Lake can be contacted at
 * email: rlake@cs.mun.ca
 * 
 * Synopsis:
 *         delbut [/aq] file.ext [...]
 *         delbut [/?|h] (* for help *)
 *
 * Rob Lake: July 11, 1998 - Fixed up code
 *                         - compile under tiny now
 *                         - Got rid of quiet and all ints in db struct,
 *                              now uses a flag along with defines
 *
 * Rob Lake: July 6, 1998 - added debugging possibilities
 *                        - added Out of memory error messages
 *                        - compiled under small model, was tiny
 *
 * Rob Lake: June 23, 1998 - included io.h in delbut.h
 *                         - let cmdline use /a/q instead of /aq
 *
 * June 15, 1998 - rewritten to C (Rob Lake)
 * 
 */
#include "delbut.h"

#pragma option -C

/* Prints information about program and exits */
void
help_exit(void)
{
	printf("DB: %s, By: %s\n"\
		"Usage: db [/aq] file1...filen\n"\
		"\t/a\tDelete files with read only,"\
		" hidden, and system attributes.\n"\
		"\t/q\tDo not print file names when deleting.\n"\
		"\t/\?\tDisplay help screen.\n", DB_VERSION, DB_AUTHOR);
	exit(0);
}

void
MemoryError(void)
{
        fprintf(stderr, "Out of memory.\n");
        exit(12);
}

/* initializes the deletebut structure */
DeleteBut Init(DeleteBut *db)
{
	DeleteBut *start;
	db->flags = 0;
	db->List = 0;
	db->Start = 0;
	db->Last = 0;

	db->List = (NoDeleteList *)malloc(sizeof(NoDeleteList));
	if(db->List == NULL) {
		db->flags |= FLG_ERR;
		return *db;
	}
	db->List->file = (char *)malloc(128);
	if(db->List == NULL) {
		db->flags |= FLG_ERR;
		return *db;
	}

	strcpy(db->List->file, ".");
	db->List->another_list = NULL;
	if(db->Last)
		db->Last->another_list = db->List;
	else
		db->Start = db->List;
	db->Last->another_list = db->List;
	db->Last = db->List;

	db->List = db->List->another_list;

	db->List = (NoDeleteList *)malloc(sizeof(NoDeleteList));
	if(db->List == NULL) {
		db->flags |= FLG_ERR;
		return *db;
	}
	db->List->file = (char *)malloc(128);
	if(db->List == NULL) {
		db->flags |= FLG_ERR;
		return *db;
	}
	strcpy(db->List->file, "..");
	db->List->another_list = NULL;
	if(db->Last)
		db->Last->another_list = db->List;
	else
		db->Start = db->List;
	db->Last->another_list = db->List;
	db->Last = db->List;

	return *db;
}

/* deletes the file by calling the real unlink */
int
Unlink(char *file, DeleteBut *db)
{
	_chmod(file, SETATTRIB, NORMAL);
	if((db->flags & FLG_QUIET) == 0)
#ifdef DEBUG
;
#endif
		printf("Removing: %s\n", file);
#ifndef DEBUG
	return(!unlink(file) ? 1 : ERROR);
#else
	return 0;
#endif
}

/*
 * Fills the no delete list, soon to implement BST for faster access,
 * or proabably a hash function for super-fast access
 */
DeleteBut FillList(int argc, char *files[], DeleteBut *db)
{
	int done, index, attrib;
	char file[MAXPATH];
	struct ffblk f;

	if(argc == 1)
		help_exit();

	attrib = 0;
	if(db->flags & FLG_ALL)
		attrib = (SYSTEM | HIDDEN | RDONLY);
	done = 0;
	index = 1;

	db->List = db->List->another_list;

	do {
		fnsplit(files[index], db->drive, db->subdir, NULL, NULL);
		sprintf(db->path, "%s%s", db->drive, db->subdir);
		done = findfirst(files[index], &f, attrib);
		while(!done) {
			sprintf(file, "%s%s", db->path, f.ff_name);
#ifdef DEBUG
			printf("Found: %s\n", file);
#endif
			db->List = (NoDeleteList *)malloc(sizeof(NoDeleteList));
			if(db->List == NULL) {
				db->flags |= FLG_ERR;
				return *db;
			}

			db->List->file = (char *)malloc(128);
			if(db->List == NULL) {
				db->flags |= FLG_ERR;
				return *db;
			}

			strcpy(db->List->file, file);
			db->List->another_list = NULL;
			if(db->Last)
				db->Last->another_list = db->List;
			else
				db->Start = db->List;
			db->Last->another_list = db->List;
			db->Last = db->List;

			done = findnext(&f);
		}
	} while(index++ != argc);
	return *db;
}

/* Removes the files */
void RemoveAllBut(DeleteBut *db)
{
	/* We don't want to try to delete directories */
	unsigned mode = NORMAL;
	int done, okay_to_del;
	char file[MAXPATH];
	struct ffblk f;

	if(db->flags & FLG_ALL)
		mode = (HIDDEN | SYSTEM | RDONLY);

	done = findfirst("*.*", &f, mode);

	while(!done) {
		sprintf(file, "%s%s", db->path, f.ff_name);
		db->List = db->Start;
		do {
			if(strcmp(db->List->file, file) == 0) {
				okay_to_del = 0;
				break;
			}
			okay_to_del = 1;
			db->List = db->List->another_list;
		} while(db->List);

		if((_chmod(file, GETATTRIB) & (SYSTEM | HIDDEN | RDONLY) &&
			(db->flags & FLG_ALL) == 0))
			okay_to_del = 0;
		if(okay_to_del)
			if(Unlink(file, db) == ERROR) {
				perror(file);
				exit(9);
			}
		done = findnext(&f);
	}
}

/* Gets the commands that may have been sent to the program */
DeleteBut GetCmds(int argc, char *argv[], DeleteBut *db)
{
	if(argv[1][0] == '/') {
		if(argc < 3)
			help_exit();
		while(*argv[1]++ != '\0') {
			switch(tolower(*argv[1])) {
				case '\?':
				case 'h':
					help_exit();
					break;
				case 'a':
					db->flags |= FLG_ALL;
					break;
				case 'q':
					db->flags |= FLG_QUIET;
					break;
				case '/':
					break;
			}
		}
	}
	return *db;
}

int main(int argc, char *argv[])
{
	DeleteBut Db;

	if(argc == 1)
		help_exit();

	Db = Init(&Db);
	if(Db.flags & FLG_ERR)
                MemoryError();

	Db = GetCmds(argc, argv, &Db);

	Db = FillList(argc, argv, &Db);
	if(Db.flags & FLG_ERR)
                MemoryError();

	RemoveAllBut(&Db);
	return 1;
}