/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: _INTR.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int _intr(int intnr, struct REGPACK far*rp)

	Performs Intel 80x86 software interrupt #intnr.
	The initial register values are read from *rp, upon exit of the
	interrupt the registers are saved back into *rp.

	The interrupt may change all real mode registers (AX, BX, CX, DX, DI,
	SI, DS, ES, CS, SS, flags) and the topmost four (4) words of the stack.

	The interrupt is called with the lobyte of flags only.

	Input:
		intnr & 0xFF: Number of interrupt to invoke
		special meaning of intnr & 0xFF00:
			== 0x0000: none
			== 0xFF00: none
		
		rp must be != NULL

	Return:
		rp->r_flags

	Note:
		<none>

	Conforms to:
		<none>

	See also:
		intr(), struct REGPACK, intdos(), intdosx()
		_callDOS()

	Target compilers:
		Borland C (using inline assembly)

	Origin:
		1997/10/10 Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.2  1998/01/29 07:09:32  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <dos.h>
#include <memory.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: _INTR.C 1.2 1998/01/29 07:09:32 ska Exp $";
#endif

static const byte varCode[] = {	/* code that must be manipulated in order */
	/* to perform intr() */
	/* Note: While this code is active, CS: == SS:, before INT, BP is start of
		code */
'\x8f', '\x46', '\x12',				// POP Word Ptr [bp+OFF_ROFF]
'\x8f', '\x46', '\x14',				// POP Word Ptr [bp+OFF_RSEG]
'\x5d',								// POP BP
'\xCD', '\0',						// INT xx			; perform interrupt
'\x2e', '\x8e', '\x16', '\0', '\0',	// MOV SS, CS:[aaaa]; restore SS:SP
'\xbc', '\0', '\0',					// MOV SP, xxxx
'\xea', '\0', '\0', '\0', '\0',		// JMP FAR xxxx:xxxx; return to _intr()
'\0', '\0',							// DW ?				; SS
'\0', '\0', '\0', '\0'				// DD ?				; absolute start addr

#define OFF_INT_NR 8
#define OFF_REF_LABEL1 12
#define OFF_SP 15
#define OFF_RET_OFF 18
#define OFF_RET_SEG 20
#define OFF_LABEL1 22
#define OFF_SS 22
#define OFF_START 24

};

_CLibFunc int _intr(int intnr, struct REGPACK far * const rp)
{	byte code[sizeof(varCode)];		/* buffer for variant code */

	assert(rp != NULL);

/* Copy and initialize the variant code in the stack segment.
	That keeps intr() reentrant and the code segment unmodified. */

	memcpy(code, varCode, sizeof(varCode));
	bRef(code[OFF_INT_NR]) = intnr;			/* interrupt number */

	/* internal references */
	wRef(code[OFF_REF_LABEL1]) = FP_OFF(&code[OFF_LABEL1]);
	dRef(code[OFF_START]) = dPtr(code);

	asm {
		pushf							 /*save interesting registers*/
		push bp							 /*save stack frame */
		 /* SI & DI will be saved by C entry/exit code */
				/* we need the stack frame accessable via SS:SP */
		lds bx, rp						/* pointer to registers */
		lea bp, code					/* start of variant code */

		mov Word Ptr bp[OFF_SS], ss		 /*save ss:sp*/
		mov Word Ptr bp[OFF_SP], sp
		sub sp, 8						 /* may be clobbered by INT */


		 /*set registers*/
		mov ah, ds:[bx+18]				 /* flags */
		sahf      						 /*only loword of flags are valid*/
		mov ax, ds:[bx]
		mov cx, ds:[bx+4]
		mov dx, ds:[bx+6]
		mov si, ds:[bx+10]
		mov di, ds:[bx+12]
		mov es, ds:[bx+16]

		push Word Ptr ds:[bx+8]			/* BP, is restored in variant code */
		push Word Ptr ds:[bx+2]			/* BX */
		mov ds, ds:[bx+14]				/*  set DS & BX */
		pop bx

		/*call far [bp+OFF_START]*/		/*invoke variant code*/
		/* The inline assembly seems to ignore this instruction without
			error or warning */
		db 0ffh, 05eh, OFF_START

		push ds							/* will be clobbered otherwise */
		push bp							/* save registers that */
		push bx

		mov bp, sp
		mov bp, [bp+6]					/* restore old stack frame */

		lds bx, rp						/* pointer to registers */

		mov Word Ptr ds:[bx], ax
		pop Word Ptr ds:[bx+2]			/*  BX */
		mov Word Ptr ds:[bx+4], cx
		mov Word Ptr ds:[bx+6], dx
		pop Word Ptr ds:[bx+8]			/*  BP */
		mov Word Ptr ds:[bx+10], si
		mov Word Ptr ds:[bx+12], di
		pop Word Ptr ds:[bx+14]			/*  DS */
		mov Word Ptr ds:[bx+16], es
		pushf
		pop ax							/* AX := flags (return value) */
		mov Word Ptr ds:[bx+18], ax		/*  flags */

		pop bp							/*  restore saved registers */
		popf
	}
	return _AX;		// Shut up "Function should return a value" warning
}
