/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: FULLPATH.C $
   $Locker:  $	$Name:  $	$State: Exp $

   	char *fullpath(char *buffer, char *path, int buflen)

	Generate the fully-qualfied path into buffer.
	Supports UNC paths.
   	If buffer != NULL, buflen tells its size.
   	If buffer == NULL, a buffer of buflen bytes is malloc'ed.
   	If buflen == 0, the buffer is assumed to be large enough.

	Input:
		path != NULL
		buffer may be NULL

	Return:
		== NULL: malloc() failed, drive access failed
		!= NULL: buffer or malloc()'ed buffer

	Note:
		<none>

	Conforms to:
		<none>

	See also:

	Target compilers:
		Any C compiler

	Origin:
		1997/10/10 Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.2  1997/10/27 05:43:49  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <alloc.h>
#include <ctype.h>
#include <dir.h>
#include <dos.h>
#include <limits.h>
#include <string.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: FULLPATH.C 1.2 1997/10/27 05:43:49 ska Exp $";
#endif

_CLibFunc char *fullpath(char * const Xbuffer, const char *path, int Xbuflen)
{
	int drive;
	int buflen;
	char *p;	/* points to place, where to start a new path component */
	char *buffer;	/* buffer to be work over */
	char *lowbound;	/* boundary never to undergo */

	if((buflen = Xbuflen) < 0
	 || (buffer = Xbuffer) == NULL
	  && (buffer = malloc(buflen? buflen: (buflen = PATH_MAX))) == NULL)
		return NULL;

	if(*path == '\\' && path[1] == '\\') {	/* assume an UNC path */
		/* two shares must be present! */
		if(*(p = (char *)&path[2]) == '\\' || *p == '/')
			goto errRet;
		while(*p && *p != '\\' && *p != '/') ++p;
		if(!*p) goto errRet;			/* at least two shares */
		while(*p && *p != '\\' && *p != '/') ++p;
		if(buflen && buflen <= (drive = p - path))
			goto errRet;
		memcpy(buffer, path, drive);
		path = p;
		lowbound = p = buffer + drive;
	}
	else {
		lowbound = buffer + 2;
		if(path[1] == ':') {
			if(!isalpha(*path))
				goto errRet;		/* invalid file spec */
			drive = toupper(path[0]) - 'A';
			path += 2;
		}
		else
			drive = getdisk();		/* current drive */

		if(*path == '\\' || *path == '/') {
			/* fully-qualified path needn't getcwd */
			*buffer = drive + 'A';
			buffer[1] = ':';
			*(p = &buffer[2]) = '\0';
		}
		else if(!getdcwd(drive + 1, buffer, buflen))
			/* get working directory failed */
			goto errRet;
		else if((p = strchr(buffer, '\0'))[-1] == '\\')
			--p;
	}

	/* Now copy all remaining components from path to buffer
		Remove '.' path components
		Process '..' components */

	do {
		/* first chop path delimiter characters */
		while(*path == '/' || *path == '\\') ++path;
		/* second break, if path has been used up */
		if(!*path) break;
		/* third identify '.' and '..' */
		if(*path == '.') switch(path[1]) {
			case '/': case '\\': case '\0':
				++path;		/* skip '.' */
				continue;
			case '.':		/* probably: '..' */
				switch(path[2]) {
				case '/': case '\\': case '\0':
					path += 2;		/* skip '..' */
					/* chop last path component */
					while(--p > lowbound && *p != '/' && *p != '\\');
					continue;
				}
				break;
		}
		/* forth copy this path component */
		*p++ = '\\';
		while(*path && *path != '/' && *path != '\\')
			if(buflen && p >= buffer + buflen)
				goto errRet;		/* buffer overflow */
			else *p++ = *path++;
	} while(1);

	if(p == lowbound)		/* root drive */
		*p++ = '\\';
	*p = '\0';

	return (!Xbuffer && !Xbuflen)? strrealloc(buffer): buffer;

errRet:
	if(!Xbuffer)
		free(buffer);

	return NULL;
}
