/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: OPENDIR.C $
   $Locker:  $	$Name:  $	$State: Exp $

	DIR *opendir(char_t *dirname)

	Open a directory for reading. This is a portable version of
	findfirst()/findnext(). More than one directory can be opened
	simultaneous even the same directory.

	There are no seekdir() and telldir() function. In a multitasking
	environment these function would cause more problems than
	advantages, though.

	Input:
		dirname != NULL

	Return:
		NULL: open failed
		else: directory pointer to be passed to other functions

	Note:
		Each DIR* item is allocated dynamically and is destroyed
		with closedir().

	Conforms to:
		POSIX

	See also:
		readdir, closedir, rewinddir

	Target compilers:
		Any C compiler

	Origin:
		Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.2  1998/01/29 07:10:15  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <alloc.h>
#include <dirent.h>
#include <string.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: OPENDIR.C 1.2 1998/01/29 07:10:15 ska Exp $";
#endif

_CLibFunc DIR *
opendir(char_t const * const dirname)
{	REG DIR *nd;
	char cwd[PATH_MAX + 2048];
	/* make sure that cwd can hold:
		1) a path component
		2) the string terminator
		3) the drive spec
		4) the "root" backslash
		5) some room while constructing the fully-qualified path
		6) search pattern
	*/

	/* It is essentinal to also expand shortnames into longnames
		under Win95, because the shortname might change during the
		directory is open. */
	if(!_fexpand(cwd, dirname, sizeof(cwd) - sizeof(_DIR_PATTERN)))
		return NULL;

	if(!cwd[3])		/* root */
		strcpy(&cwd[2], _DIR_PATTERN);
	else strcat(cwd, _DIR_PATTERN);		/* append the search pattern */

	/* construct DIR */
	if((nd = (DIR *)malloc(sizeof(DIR) + strlen(cwd))) == NULL)
		return NULL;

	strcpy(nd->dd_dir, cwd);	/* preserve the directory name */
#ifdef LFN_WIN95
	nd->dd_dta.ff_lfn.ff_status = _NOLFN;	/* for findstop() */
#endif

	rewinddir(nd);				/* check if the directory exists */
								/* & perform the first search */
	if(nd->dd_stat) {		/* failed */
	  free(nd);
	  return NULL;
	}

	return nd;
}
