/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: FCLOSE.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int fclose(FILE *fp)

	Close a stream.

	All pending data is flushed and the associated file descriptor
	is closed.

	If a dynamically allocated buffer is associated with the stream,
	the buffer is freed. If the FILE structure itself was allocated,
	it is freed, too.

	A closed handle is initialized that way that further access to this
	stream will always fail, though, because most FILE structures are
	deallocated and possibly overwritten soon, the caller cannot rely on
	that feature.

	Input:
		fp != NULL; a valid FILE pointer

	Return:
		0: on success
		else: on failure

	Note:
		<none>

	Conforms to:
		ANSI

	See also:
		fopen, close, fflush

	Target compilers:
		Any C compiler

	Origin:
		1997/11/03 Robert de Bath (see CONTRIB\STDIO2.ZIP)

	Revised by:
		1997/11/23 Steffen Kaiser (ska)

	File Revision:    Revision 1.1  1997/11/26 04:03:45  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include "stdio.h"
#include <alloc.h>			/* free() */
#include <errno.h>
#include <io.h>				/* close() */

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: FCLOSE.C 1.1 1997/11/26 04:03:45 ska Exp $";
#endif

_CLibFunc int
fclose(FILE *fp)
{
   int   rv = 0;

   if (fp == NULL) {
      errno = EINVAL;
      return EOF;
   }
   if (fflush(fp))
      return EOF;

   if (close(fp->fd))
      rv = EOF;
   fp->fd = -1;

   if (fp->mode & __MODE_FREEBUF) {
      free(fp->bufstart);
      fp->mode &= ~__MODE_FREEBUF;
      fp->bufstart = fp->bufend = 0;
   }

   if (fp->mode & __MODE_FREEFIL) {
      FILE *prev = 0, *ptr;
      fp->mode = 0;

      for (ptr = __IO_list; ptr && ptr != fp; ptr = ptr->next)
		 ;
      if (ptr == fp) {
		 if (prev == 0)
			__IO_list = fp->next;
		 else
			prev->next = fp->next;
      }
      free(fp);
   }
   else
      fp->mode = 0;

   return rv;
}
