/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: FFLUSH.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int fflush(FILE *fp)

	Flush the stream.

	Any pending data is transfered to the system. This does not mean
	that the data arrives at its destination, because the system
	might buffer the data, too. To ensure that the system commits
	its buffers, too, perform "commit(fileno(fp))" or "sync()" to
	commit all system buffers.

	If fp == NULL, fflush() behaves as fflushall().

	Input:
		fp == NULL or a valid FILE pointer

	Return:
		0: all OK
		else: an error occured

	Note:
		<none>

	Conforms to:
		ANSI, except fflush(NULL)

	See also:
		fclose, fflushall, fopen

	Target compilers:
		Any C compiler

	Origin:
		1997/11/03 Robert de Bath (see CONTRIB\STDIO2.ZIP)

	Revised by:
		1997/11/23 Steffen Kaiser (ska)

	File Revision:    Revision 1.1  1997/11/26 04:03:45  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include "stdio.h"
#include <errno.h>
#include <io.h>				/* write(), lseek() */

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: FFLUSH.C 1.1 1997/11/26 04:03:45 ska Exp $";
#endif

_CLibFunc int
fflush(FILE *fp)
{
	int   len, cc, rv=0;
	char * bstart;

	if (fp == NULL) {		/* On NULL flush the lot. */
		rv = fflushall();
		return rv == EOF? EOF: 0;
	}

	/* If there's output data pending */
	if (fp->mode & __MODE_WRITING) {
		if((len = fp->bufpos - fp->bufstart) != 0) {
			bstart = fp->bufstart;
			/*
			* The loop is so we don't get upset by signals or partial writes.
			*/
			do {
				cc = write(fp->fd, bstart, len);
				if( cc > 0 ) {
				   bstart+=cc; len-=cc;
				}
			} while ( len>0 && (cc>0 || (cc == -1 && errno == EINTR)));
			/*
			* If we get here with len!=0 there was an error, exactly what to
			* do about it is another matter ...
			*
			* I'll just clear the buffer.
			*/
			if (len) {
				fp->mode |= __MODE_ERR;
				rv = EOF;
			}
		}
	}
	/* If there's data in the buffer sychronise the file positions */
	else if (fp->mode & __MODE_READING) {
		/* Humm, I think this means sync the file like fpurge() ... */
		/* Anyway the user isn't supposed to call this function when reading */

		len = fp->bufread - fp->bufpos;	/* Bytes buffered but unread */
		/* If it's a file, make it good */
		if (len > 0 && lseek(fp->fd, (long)-len, SEEK_CUR) == _SEEK_FAIL) {
		/* Hummm - Not certain here, I don't think this is reported */
			/*
			* fp->mode |= __MODE_ERR; return EOF;
			*/
		/* I agree here, because there are reported un-seekable streams;
			to deny access in this case won't do much good.
			Seekable streams would in most cases re-throw that error
			condition with the next read or write access - 1997/11/14 ska*/
		}
	}

	/* All done, no problem */
	fp->mode &= (~(__MODE_READING|__MODE_WRITING|__MODE_EOF|__MODE_UNGOT));
	fp->bufread = fp->bufwrite = fp->bufpos = fp->bufstart;
	return rv;
}
