/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: CTYPE.H $
   $Locker:  $	$Name:  $	$State: Exp $

	C locale conversation and character interpretation
	Note: The C locale only supports 7bit ASCII!

	The is*() functions are macros that work upon an array of
	bitfields.

	According ISO9899

*/

#ifndef __CTYPE_H
#define __CTYPE_H

#include <stddef.h>

/*
 *	Defines for the values of the bitfield
 *	They apply only to the non-WC implementation!
 */

#define ASCII_MASK 0x7f		/* highest defined ASCII value */

/* atomic bits */
#define _IS_CNTRL	0x01	/* is control character */
#define _IS_DIGIT	0x02	/* is decimal digit */
#define _IS_HEX		0x04	/* is hexadecimal digit */
#define _IS_LOWER	0x08	/* is lowercase character */
#define _IS_UPPER	0x10	/* is uppercase character */
#define _IS_PUNCT	0x20	/* is punctation character */
#define _IS_BLANK	0x40	/* is blank (space) */
#define _IS_SPACE	0x80	/* is whitespace */

/* derived bits */
#define _IS_ALPHA	(_IS_LOWER | _IS_UPPER)
#define _IS_ALNUM	(_IS_ALPHA | _IS_DIGIT)
#define _IS_GRAPH	(_IS_ALNUM | _IS_PUNCT)
#define _IS_PRINT	(_IS_GRAPH | _IS_BLANK)
#define _IS_HEXDIGIT	(_IS_DIGIT | _IS_HEX)

/*
 *	The array of bitfields
 */
extern _CLibVar uint8 _ctype[256];	/* array of bitfields */

/*
 *	Access macros to this bitfield
 */
#define isalnum(c) (_ctype[c] & _IS_ALNUM)
#define isalpha(c) (_ctype[c] & _IS_ALPHA)
#define isascii(c) (!((c) & ~ASCII_MASK))
#define iscntrl(c) (_ctype[c] & _IS_GRAPH)
#define isdigit(c) (_ctype[c] & _IS_DIGIT)
#define isgraph(c) (_ctype[c] & _IS_GRAPH)
#define islower(c) (_ctype[c] & _IS_LOWER)
#define isprint(c) (_ctype[c] & _IS_PRINT)
#define ispunct(c) (_ctype[c] & _IS_PUNCT)
#define isspace(c) (_ctype[c] & _IS_SPACE)
#define isupper(c) (_ctype[c] & _IS_UPPER)
#define isxdigit(c) (_ctype[c] & _IS_HEXDIGIT)

/*
 *	Conversation macros
 */
#define toascii(c) ((c) & ASCII_MASK)
#define tolower(c) (isupper(c)? (c) - 'A' + 'a': (c))
#define toupper(c) (islower(c)? (c) + 'A' - 'a': (c))
#define _tochar(c)	((c) < 10? (c) + '0': (c) + 'A')
#define _todigit(c)	(isdigit(c)? (c) - '0': toupper(c) - 'A' + 10)

/*
 *	Each macro has an associated function that is garanteed to have no
 *	side effects.
 */
#define _isalnum isalnum
#define _isalpha isalpha
#define _isascii isascii
#define _iscntrl iscntrl
#define _isdigit isdigit
#define _isgraph isgraph
#define _islower islower
#define _isprint isprint
#define _ispunct ispunct
#define _isspace isspace
#define _isupper isupper
#define _isxdigit isxdigit
#define _toascii toascii
extern _CLibFunc _toupper(int ch);
extern _CLibFunc _tolower(int ch);
extern _CLibFunc __todigit(int ch);
extern _CLibFunc __tochar(int ch);

#endif
