/*
// Program:  Format
// Written By:  Brian E. Reifsnyder
// Version:  0.5
// Copyright:  1999 under the terms of the GNU GPL
*/

#define NAME "Format"
#define VERSION "0.5"

/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <bios.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>

#include "bootcode.h"
/*
/////////////////////////////////////////////////////////////////////////////
//  DEFINES
/////////////////////////////////////////////////////////////////////////////
*/

#define TRUE 1
#define FALSE 0

//#define NULL 0

#define PERCENTAGE 1

#define PRIMARY 0
#define EXTENDED 1

#define HARD 0
#define FLOPPY 1

#define FAT12 0
#define FAT16 1
#define FAT32 2

#define WHOLE_TRACK -5
/*
/////////////////////////////////////////////////////////////////////////////
//  GLOBAL VARIABLES
/////////////////////////////////////////////////////////////////////////////
*/

/* Physical Drive Parameters */
int physical_drive;
int logical_drive;

unsigned long logical_sector_offset;

long total_cylinders;
long total_heads;
long total_sectors;
long floppy_drive_type;
int drive_type;
int media_descriptor;

/* Logical drive information */
long sectors_per_fat;
long sectors_in_root_dir;
int type_of_fat;
unsigned long number_of_logical_sectors_on_drive;
long number_of_root_directory_entries;

/* Flags ////////////////*/
/* Switches */
int v=FALSE;
int q=FALSE;
int u=FALSE;
int f=FALSE;
int b=FALSE;
int s=FALSE;
int t=FALSE;
int n=FALSE;
int one=FALSE;
int four=FALSE;
int eight=FALSE;

/* User entered information */
char entered_label[12];
unsigned long entered_size;
unsigned long entered_tracks;
unsigned long entered_sectors;

/* Extracted cylinder & sector from partition table */
unsigned long g_cylinder;
unsigned long g_sector;

/* Partition Information */

int numeric_partition_type[24];

unsigned long starting_cylinder[24];
unsigned long starting_head[24];
unsigned long starting_sector[24];

unsigned long ending_cylinder[24];
unsigned long ending_head[24];
unsigned long ending_sector[24];

int relative_sectors_1[24];
int relative_sectors_2[24];
int relative_sectors_3[24];
int relative_sectors_4[24];

int partition_size_1[24];
int partition_size_2[24];
int partition_size_3[24];
int partition_size_4[24];

unsigned long partition_size[24];

int number_of_partitions;

/* "Logical Drive to Format" translation information */
int physical_drive_number;

unsigned long partition_starting_cylinder;
unsigned long partition_starting_head;
unsigned long partition_starting_sector;

unsigned long partition_ending_cylinder;
unsigned long partition_ending_head;
unsigned long partition_ending_sector;

unsigned long drive_maximum_cylinders;
unsigned long drive_maximum_heads;
unsigned long drive_maximum_sectors;

unsigned long total_logical_sectors;
unsigned long total_physical_sectors;
int number_of_sectors_per_cluster;

int partition_on_hard_disk;

/* Translated values from "void Convert_Logical_To_Physical" */
unsigned long translated_head;
unsigned long translated_cylinder;
unsigned long translated_sector;

/* Misc variables */
int bad_boot_sector=FALSE;
int unconditional_format_flag=FALSE;
int bad_sector_flag=FALSE;
int display_disk_access=TRUE;

unsigned long integer1;
unsigned long integer2;
unsigned long integer3;
unsigned long integer4;

int optimize;

/* Buffers */
unsigned char *sector_buffer;
unsigned char *big_sector_buffer;
unsigned char *mirror_map;
unsigned long *bad_sector_map;

/* Floppy Disk Specific Variables */
unsigned int floppy_root_directory_entries;
unsigned int floppy_sectors_per_fat;

/*
/////////////////////////////////////////////////////////////////////////////
//  PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/
unsigned long Decimal_Number(unsigned long hex1, unsigned long hex2, unsigned long hex3, unsigned long hex4);

void Allocate_Buffers();
void Clear_Sector_Buffer();
void Convert_Huge_To_Integers(unsigned long number);
void Convert_Logical_To_Physical(unsigned long sector);
void Create_File_System();
void Display_Disk_Access_Location();
void Display_Help_Screen();
void Display_Invalid_Combination();
void Deallocate_Buffers();
void Extract_Cylinder_and_Sector(unsigned long hex1, unsigned long hex2);
void Get_Drive_Parameters();
void Get_Physical_Floppy_Drive_Parameters();
void Get_Physical_Hard_Drive_Parameters();
void Read_Boot_Sector();
void Read_Partition_Table();
void Read_Physical_Sector(int drive, int head, long cyl, int sector);
void Read_Sector(unsigned long sector);
void Reset_Drive();
void Save_File_System();
void Unconditional_Format();
void Write_Physical_Sector(int drive, int head, long cyl, int sector);
void Write_Sector(unsigned long sector);

/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

/* Allocate Buffers */
void Allocate_Buffers()
{
  long index;

  bad_sector_map=(unsigned long *)malloc(4097);
  mirror_map=(unsigned char *)malloc(5121);
  sector_buffer=(unsigned char *)malloc(512);

  if( (bad_sector_map==NULL) || (mirror_map==NULL) || (sector_buffer==NULL) )
    {
    printf("\nBuffer Allocation Failure...Program Terminated.\n");
    Deallocate_Buffers();
    exit(1);
    }

  /* Clear bad_sector_map */
  index=0;
  do
    {
    bad_sector_map[index]=0x00;
    index++;
    }while(index<=1023);
}

/* Clear Sector Buffer */
void Clear_Sector_Buffer()
{
  long loop=0;

  do
    {
    sector_buffer[loop]=0;

    loop++;
    }while(loop<512);
}

/* Convert a logical sector to a physical drive location */
void Convert_Logical_To_Physical(unsigned long sector)
{
  unsigned long remaining;

  if(drive_type==HARD) sector=sector+logical_sector_offset;

  translated_cylinder=sector/(total_sectors*(total_heads+1));
  remaining=sector % (total_sectors*(total_heads+1));
  translated_head=remaining/(total_sectors);
  translated_sector=remaining % (total_sectors);
  translated_sector++;
}

/* Convert Hexidecimal Number to a Decimal Number */
unsigned long Decimal_Number(unsigned long hex1, unsigned long hex2, unsigned long hex3, unsigned long hex4)
{
  return((hex1) + (hex2*256) + (hex3*65536) + (hex4*16777216));
}

/* Convert Huge number into 4 LMB integer values */
void Convert_Huge_To_Integers(unsigned long number)
{
  unsigned long temp=number;
  unsigned long mask=0x000000ff;

  integer1=temp & mask;

  temp=number;
  mask=0x0000ff00;
  integer2=(temp & mask) >> 8;

  temp=number;
  mask=0x00ff0000;
  integer3=(temp & mask) >> 16;

  temp=number;
  mask=0xff000000;
  integer4=(temp & mask) >> 24;
}

/* Create File System */
void Create_File_System()
{
  unsigned long sectors_per_track;
  unsigned long sectors_per_fat;
  unsigned long root_directory_entries;
  unsigned long total_sectors_in_volume;
  unsigned long directory_ending_sector;

  unsigned long number_of_clusters_on_disk;

  unsigned long number_of_sectors_to_zero;

  unsigned int byte1;
  unsigned int byte2;

  unsigned long fat_table_offset_counter;
  unsigned long possible_cluster;

  unsigned long last_cluster=0xffff;

  unsigned long cluster;
  unsigned long pointer;
  unsigned long index;
  unsigned long sub_index;
  unsigned long start_sector;
  unsigned long fat_table_1_start_sector;
  unsigned long fat_table_2_start_sector;
  unsigned long fat_table_offset_sector;
  unsigned long remainder;

  int fat12_table[14000];

  long flag=FALSE;

  unsigned long even;

  printf("\n\nCreating file system...\n");

  Reset_Drive();

  Clear_Sector_Buffer();

  total_sectors_in_volume=total_logical_sectors;

  /* Add jump instructions */
  sector_buffer[0x00]=0xeb;
  sector_buffer[0x01]=0x3c;
  sector_buffer[0x02]=0x90;

  /* Add name and/or version # (FreeDOS)*/
  sector_buffer[0x03]=0x46;
  sector_buffer[0x04]=0x72;
  sector_buffer[0x05]=0x65;
  sector_buffer[0x06]=0x65;
  sector_buffer[0x07]=0x44;
  sector_buffer[0x08]=0x4f;
  sector_buffer[0x09]=0x53;

  /* Add bytes per sector (512)*/
  sector_buffer[0x0b]=0x00;
  sector_buffer[0x0c]=0x02;

  /* Add sectors per cluster */
  if(drive_type==FLOPPY)
    {
    /* 1 for a floppy drive */
    sector_buffer[0x0d]=0x01;
    number_of_clusters_on_disk=total_logical_sectors;
    }
  else
    {
    /* if a hard disk use number_of_sectors_per_cluster value */
    sector_buffer[0x0d]=number_of_sectors_per_cluster;
    number_of_clusters_on_disk=total_sectors_in_volume/number_of_sectors_per_cluster;
    }

  /* Add number of reserved sectors */
  sector_buffer[0x0e]=0x01;
  sector_buffer[0x0f]=0x00;

  /* Add number of FATs (2)*/
  sector_buffer[0x10]=0x02;

  /* Add number of entries in the root directory */
  if(drive_type==FLOPPY)
    {
    sector_buffer[0x11]=floppy_root_directory_entries;
    sector_buffer[0x12]=0x00;
    root_directory_entries=floppy_root_directory_entries;
    }
  else
    {
    /* Make it 512 for a hard disk */
    sector_buffer[0x11]=0x00;
    sector_buffer[0x12]=0x02;
    root_directory_entries=512;
    }

  /* Add number of sectors in volume for a floppy disk */
  /* or a FAT12 hard disk partition.                   */

  asm{
    mov ax,WORD PTR total_sectors_in_volume
    mov BYTE PTR byte2,ah
    mov BYTE PTR byte1,al
    }

  sector_buffer[0x13]=byte1;
  sector_buffer[0x14]=byte2;

  if(number_of_clusters_on_disk>4085)
    {
    sector_buffer[0x13]=0x00;
    sector_buffer[0x14]=0x00;
    }

  /* Add Media descriptor */
  sector_buffer[0x15]=media_descriptor;

  /* Add number of sectors per FAT */
  if(drive_type==FLOPPY)
    {
    sector_buffer[0x16]=floppy_sectors_per_fat;
    sector_buffer[0x17]=0x00;
    sectors_per_fat=floppy_sectors_per_fat;
    }
  else
    {
    sectors_per_fat=((total_sectors_in_volume/number_of_sectors_per_cluster)/256)+1;

    asm{
      mov ax,WORD PTR sectors_per_fat
      mov BYTE PTR byte2,ah
      mov BYTE PTR byte1,al
      }

    sector_buffer[0x16]=byte1;
    sector_buffer[0x17]=byte2;
    }

  /* Add number of sectors per track */
  sectors_per_track=total_sectors;

  asm{
    mov ax,WORD PTR sectors_per_track;
    mov BYTE PTR byte2,ah;
    mov BYTE PTR byte1,al;
    }

  sector_buffer[0x18]=byte1;
  sector_buffer[0x19]=byte2;

  /* Add number of read/write heads */
  sector_buffer[0x1a]=(total_heads+1);
  sector_buffer[0x1b]=0x00;

  /* Add special number of hidden sectors if the drive is a hard disk */
  if(drive_type==HARD)
    {
    sector_buffer[0x1c]=relative_sectors_1[partition_on_hard_disk];
    sector_buffer[0x1d]=relative_sectors_2[partition_on_hard_disk];
    sector_buffer[0x1e]=relative_sectors_3[partition_on_hard_disk];
    sector_buffer[0x1f]=relative_sectors_4[partition_on_hard_disk];
    }

  /* Add big total number of sectors if the drive is a FAT16 hard disk */
  if(number_of_clusters_on_disk>=4086)
    {
    sector_buffer[0x20]=partition_size_1[partition_on_hard_disk];
    sector_buffer[0x21]=partition_size_2[partition_on_hard_disk];
    sector_buffer[0x22]=partition_size_3[partition_on_hard_disk];
    sector_buffer[0x23]=partition_size_4[partition_on_hard_disk];
    }

  /* Add physical drive number */
  sector_buffer[0x24]=physical_drive;

  /* Add extended boot record signature */
  sector_buffer[0x26]=0x29;

  /* Add volume label, if applicable */
  index=0x2b;

  do
    {
    sector_buffer[index]=0x20;
    index++;
    }while(index<=(0x2b+11));

  index=0x2b;

  if(v==TRUE)
    {
    flag=TRUE;
    do
      {
      if(entered_label[index-0x2b]==0) flag=FALSE;

      if(flag==TRUE) sector_buffer[index]=entered_label[index-0x2b];

      index++;
      }while(index<=(0x2b+11));
    }

  /* Add file system id */

  /* 'FAT1' */
  sector_buffer[0x36]=0x46;
  sector_buffer[0x37]=0x41;
  sector_buffer[0x38]=0x54;
  sector_buffer[0x39]=0x31;


  if(number_of_clusters_on_disk<=4085)
    {
    /* set the id to 'FAT12   ' */
    sector_buffer[0x3a]=0x32;
    }

  if(number_of_clusters_on_disk>=4086)
    {
    /* set the id to 'FAT16   ' */
    sector_buffer[0x3a]=0x36;
    }

  /* spaces in file system id */
  sector_buffer[0x3b]=0x20;
  sector_buffer[0x3c]=0x20;
  sector_buffer[0x3d]=0x20;


  /* Add bootcode to sector buffer (130 bytes) */
  index=0x3e;
  do
    {
    sector_buffer[index]=boot_code[index-0x3e];
    index++;
    }while(index<=(0x3e+130));

  /* Add "executable sector marker" (0x55 0xaa) */
  sector_buffer[0x1fe]=0x55;
  sector_buffer[0x1ff]=0xaa;

  /* Write sector */
  Write_Sector(0);

  /* Clear space for FAT tables & root directory */

  number_of_sectors_to_zero=(sectors_per_fat*2)+(root_directory_entries/16);

  Clear_Sector_Buffer();

  start_sector=1;
  index=start_sector;

  do
    {
    Write_Sector(index);

    index++;
    }while(index<(number_of_sectors_to_zero+start_sector));

  directory_ending_sector=number_of_sectors_to_zero+start_sector;

  /* Write FAT Headers */

  Clear_Sector_Buffer();

  sector_buffer[0x00]=media_descriptor;
  sector_buffer[0x01]=0xff;
  sector_buffer[0x02]=0xff;

  if(number_of_clusters_on_disk>=4086) sector_buffer[0x03]=0xff;

  /* Do FAT table 1 */

  start_sector=1;
  fat_table_1_start_sector=start_sector;

  Write_Sector(start_sector);

  /* Do FAT table 2 */
  start_sector=start_sector+sectors_per_fat;
  fat_table_2_start_sector=start_sector;

  /* Compute offset of data area from the beginning of the disk */
  //data_area_offset_sector=(fat_table_2_start_sector+sectors_per_fat)+(root_directory_entries/16);

  Write_Sector(start_sector);

  /* Write system files, if applicable */
  if(s==TRUE)
    {
    char sys[7] = {'s','y','s',' ','x',':','\0'};

    sys[4]=logical_drive+65;

    printf("\n\nWriting system files...\n");
    system(sys);
    }

  /* Add volume label to root directory, if applicable */
  if(v==TRUE)
    {
    printf("\nWriting volume label...\n");
    Clear_Sector_Buffer();

    start_sector=start_sector+sectors_per_fat;
    Read_Sector(start_sector);
    pointer=0;

    /* Find an empty directory entry */
    do
       {
      if(sector_buffer[pointer*32]==0x00)
	{
	/* Add the label to the entry */
	flag=FALSE;
	index=0;
	do
	  {
	  if(entered_label[index]==0) flag=TRUE;

	  if(flag==FALSE) sector_buffer[index+(pointer*32)]=entered_label[index];
	  else sector_buffer[index+(pointer*32)]=0x20;

	  index++;
	  }while(index<11);

	/* Set the archive and volume bits */
	sector_buffer[(pointer*32)+11]=0x28;
	pointer=1000;
	}

      pointer++;
      }while(pointer<999);

    Write_Sector(start_sector);
    }

  /* If bad sectors are recorded then the FAT tables need marked. */

  /* FAT16 has 256 entries per sector */
  /* FAT12 has 340.6666 entries per sector (Yuck!) */
  if(bad_sector_flag==TRUE)
    {
    index=0;
    printf("\n\nRecording bad clusters in FAT tables...\n");
    display_disk_access=FALSE;

    do
      {
      /* Make sure that no bad clusters are in the file system locations */
      /* and adjust the bad_sector_map for the file system areas.        */
      if(bad_sector_map[index]<directory_ending_sector)
	{
	printf("\n\nA bad cluster is located in file system areas...program terminated.\n\n");
	Deallocate_Buffers();
	exit(4);
	}
      else
	{
	bad_sector_map[index]=bad_sector_map[index]-directory_ending_sector;
	}

      cluster=(bad_sector_map[index])/number_of_sectors_per_cluster;

      if(last_cluster!=cluster)
	{
	printf("%u ",(cluster));
	last_cluster=cluster;
	}

      if(type_of_fat==FAT12)
	{
	/* Read FAT table into buffer. */

	pointer=0;
	do
	  {
	  Read_Sector(pointer+fat_table_1_start_sector);

	  sub_index=0;
	  do
	    {
	    fat12_table[((pointer*512)+sub_index)]=sector_buffer[sub_index];
	    sub_index++;
	    }while(sub_index<512);

	  pointer++;
	  }while(pointer<sectors_per_fat);

	/* Find the location of the cluster to mark as bad */

	fat_table_offset_counter=0;
	possible_cluster=0;
	do
	  {
	  fat_table_offset_counter=fat_table_offset_counter+3;
	  possible_cluster=possible_cluster+2;
	  }while(cluster>=possible_cluster);

	/* Is cluster even? */
	even=!(cluster & 0x00000001);

	if(even==TRUE)
	  {
	  /* cluster is even */
	  if(fat12_table[fat_table_offset_counter-3]!=0xf7)
	    {
	    fat12_table[fat_table_offset_counter-3]=0xf7;
	    fat12_table[(fat_table_offset_counter-2)]=fat12_table[(fat_table_offset_counter-2)] + 0x0f;
	    }
	  }
	else
	  {
	  /* cluster is odd */
	  if(fat12_table[(fat_table_offset_counter-1)]!=0xff)
	    {
	    fat12_table[(fat_table_offset_counter-1)]=0xff;
	    fat12_table[(fat_table_offset_counter-2)]=fat12_table[(fat_table_offset_counter-2)] + 0x70;
	    }
	  }

	/* Write the FAT tables to disk */
	pointer=0;
	do
	  {
	  sub_index=0;
	  do
	    {
	    sector_buffer[sub_index]=fat12_table[((pointer*512)+sub_index)];
	    sub_index++;
	    }while(sub_index<512);

	  Write_Sector(pointer+fat_table_1_start_sector);
	  Write_Sector(pointer+fat_table_2_start_sector);

	  pointer++;
	  }while(pointer<sectors_per_fat);
	}

      if(type_of_fat==FAT16)
	{
	fat_table_offset_sector=cluster/256;
	remainder=(cluster % 256);

	Read_Sector(fat_table_offset_sector+fat_table_1_start_sector);
	sector_buffer[(remainder*2)]=0xf7;
	sector_buffer[(remainder*2)+1]=0xff;

	Write_Sector(fat_table_offset_sector+fat_table_1_start_sector);
	Write_Sector(fat_table_offset_sector+fat_table_2_start_sector);
	}

      if(type_of_fat==FAT32)
	{
	/* FAT32 support is not available at this time */
	}

      index++;
      }while(bad_sector_map[index]!=0);
    }
  display_disk_access=TRUE;

  printf("\n\nFormat operation complete.\n\n");
}

/* De-allocate the buffers */
void Deallocate_Buffers()
{
  free(mirror_map);
  free(bad_sector_map);
  free(sector_buffer);
}

/* Display the cylinder and head of the drive being accessed */
void Display_Disk_Access_Location()
{
  if(display_disk_access==TRUE)
    {
    printf("Cylinder:  %5i  ",translated_cylinder);
    printf("Head:  %2i      \n",translated_head);

    /* Re-position cursor back to the beginning of the line */
    asm{
      /* Get current video display mode */
      mov ah,0x0f
      int 0x10

      /* Get cursor position */
      mov ah,0x03
      int 0x10

      /* Set cursor position to beginning of line */
      mov ah,0x02
      sub dh,1
      int 0x10
      }
    }
}

/* Extract Cylinder & Sector */
void Extract_Cylinder_and_Sector(unsigned long hex1, unsigned long hex2)
{
  unsigned long cylinder_and_sector = ( (256*hex2) + hex1 );

  g_sector = cylinder_and_sector % 64;

  g_cylinder = ( ( (cylinder_and_sector*4) & 768) + (cylinder_and_sector /256) );
}

/* Help Routine */
void Display_Help_Screen()
{
  printf("\n%6s Version %s\n",NAME,VERSION);
  printf("Written By:  Brian E. Reifsnyder\n");
  printf("Copyright 1999 under the terms of the GNU GPL.\n\n");
  printf("Syntax:\n\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/F:size] [/B | /S]\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/T:tracks /N:sectors] [/B | /S]\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/1] [/4] [/B | /S]\n");
  printf("FORMAT drive: [/Q] [/U] [/1] [/4] [/8] [/B | /S]\n\n");
  printf(" /V:label   Specifies a volume label for the disk.\n");
  printf(" /Q         Quick formats the disk.  The disk can be UNFORMATed.\n");
  printf(" /U         Unconditionally formats the disk.  The disk cannot be UNFORMATted\n");
  printf(" /F:size    Specifies the size of the floppy disk to format.  Valid sizes are:\n");
  printf("              160, 180, 320, 360, 720, 1200, 1440, or 2880.\n");
  printf(" /B         Kept for compatibility, formally reserved space for the boot files.\n");
  printf(" /S         Copies the operating system files to make the disk bootable.\n");
  printf(" /T:tracks  Specifies the number of tracks on a floppy disk.\n");
  printf(" /N:sectors Specifies the number of sectors on a floppy disk.\n");
  printf(" /1         Formats a single side of a floppy disk.\n");
  printf(" /4         Formats a 360K floppy disk in a 1.2 MB floppy drive.\n");
  printf(" /8         Formats a 5.25\" disk with 8 sectors per track.\n");
}

void Display_Invalid_Combination()
{
  printf("\nInvalid combination of options...please consult documentation.\n");
  printf("Operation Terminated.\n");
  Deallocate_Buffers();
  exit(4);
}

/* Get Drive Parameters */

/* The physical drive parameters are necessary to allow this program  */
/* to bypass the DOS interrupts.                                      */
void Get_Drive_Parameters()
{
  int drives_found_flag;
  long hard_disk_installed[8];
  int increment_heads=FALSE;
  int logical_drive_found=FALSE;
  int partition_number_of_logical_drive;
  int pointer=0;
  int possible_logical_drive;
  int possible_physical_hard_disk;
  int sub_pointer=0;
  int result;

  /* Brief partition table variables */
  long partition_type_table[8][26];
  long maximum_partition_assigned[8];

  if(drive_type==FLOPPY)
    {
    /* Set the parameters for a floppy drive */
    physical_drive=logical_drive;

    Get_Physical_Floppy_Drive_Parameters();

    partition_starting_cylinder=0;
    partition_starting_head=0;
    partition_starting_sector=1;

    partition_ending_cylinder=total_cylinders;
    partition_ending_head=total_heads;
    partition_ending_sector=total_sectors;

    drive_maximum_cylinders=total_cylinders;

    if(one==FALSE) drive_maximum_heads=total_heads;
    else drive_maximum_heads=0;

    drive_maximum_sectors=total_sectors;

    total_logical_sectors=(total_cylinders+1)*(total_heads+1)*total_sectors;
    total_physical_sectors=total_logical_sectors;
    number_of_sectors_per_cluster=1;

    type_of_fat=FAT12;

    /* Set parameters specific to the floppy drive type */
    if(floppy_drive_type==0x01)        /* 5.25", 360K */
      {
      media_descriptor=0xfd;
      floppy_root_directory_entries=112;
      floppy_sectors_per_fat=2;
      }

    if(floppy_drive_type==0x02)        /* 5.25", 1.2MB */
      {
      media_descriptor=0xf9;
      floppy_root_directory_entries=224;
      floppy_sectors_per_fat=7;
      }

    if(floppy_drive_type==0x03)        /* 3.5", 720K */
      {
      media_descriptor=0xf9;
      floppy_root_directory_entries=112;
      floppy_sectors_per_fat=3;
      }

    if(floppy_drive_type==0x04)        /* 3.5", 1.44MB */
      {
      media_descriptor=0xf0;
      floppy_root_directory_entries=224;
      floppy_sectors_per_fat=9;
      }
    }
  else
    {
    /* Set the parameters for a hard disk */

    /* Get availability of physical hard drives and */
    /* assemble a brief partition table.            */

    /* Initialize partition_type_table */
    pointer=0;
    do
      {
      partition_type_table[0][pointer]=0;
      partition_type_table[1][pointer]=0;
      partition_type_table[2][pointer]=0;
      partition_type_table[3][pointer]=0;
      partition_type_table[4][pointer]=0;
      partition_type_table[5][pointer]=0;
      partition_type_table[6][pointer]=0;
      partition_type_table[7][pointer]=0;

      pointer++;
      }while(pointer<=26);

    /* Initialize maximum_partition_assigned */
    maximum_partition_assigned[0]=0;
    maximum_partition_assigned[1]=0;
    maximum_partition_assigned[2]=0;
    maximum_partition_assigned[3]=0;
    maximum_partition_assigned[4]=0;
    maximum_partition_assigned[5]=0;
    maximum_partition_assigned[6]=0;
    maximum_partition_assigned[7]=0;

    pointer=128;
    drives_found_flag=FALSE;

    do
      {
      result=biosdisk(2, pointer, 0, 0, 1, 1, sector_buffer);

      if(result==0)
	{
	drives_found_flag=TRUE;
	hard_disk_installed[pointer-128]=TRUE;

	physical_drive=pointer;
	Read_Partition_Table();

	sub_pointer=0;
	do
	  {
	  partition_type_table[pointer-128][sub_pointer]=numeric_partition_type[sub_pointer];
	  sub_pointer++;
	  }while(sub_pointer<=number_of_partitions);
	maximum_partition_assigned[pointer-128]=number_of_partitions;
	}
      else hard_disk_installed[pointer-128]=FALSE;

      pointer++;
      }while(pointer<136);

    /* If there aren't any physical hard drives available, exit program */

    if(drives_found_flag==FALSE)
      {
      printf("\nNo hard disks found...operation terminated.\n");
      Deallocate_Buffers();
      exit(4);
      }

    /* Determine on which physical hard drive the logical drive is */
    /* and on which partition it is located.                       */

    /* Check on primary partitions */
    possible_logical_drive=1;
    possible_physical_hard_disk=128;

    do
      {
      if(hard_disk_installed[possible_physical_hard_disk-128]==TRUE)
	{
	pointer=0;

	do
	  {
	  if( (partition_type_table[possible_physical_hard_disk-128][pointer]==0x01) || (partition_type_table[possible_physical_hard_disk-128][pointer]==0x04) || (partition_type_table[possible_physical_hard_disk-128][pointer]==0x06) )
	    {
	    possible_logical_drive++;

	    if(logical_drive==possible_logical_drive)
	      {
	      physical_drive=possible_physical_hard_disk;
	      partition_number_of_logical_drive=pointer;
	      logical_drive_found=TRUE;
	      }
	    }
	  pointer++;
	  }while(pointer<4);
	}
      possible_physical_hard_disk++;
      }while(possible_physical_hard_disk<136);

    /* Check on extended partitions */
    possible_physical_hard_disk=128;

    if(logical_drive_found==FALSE)
      {
      do
	{
	if( (hard_disk_installed[possible_physical_hard_disk-128]==TRUE) && (maximum_partition_assigned[possible_physical_hard_disk-128]>=4) )
	  {
	  pointer=4;

	  do
	    {
	    if( (partition_type_table[possible_physical_hard_disk-128][pointer]==0x01) || (partition_type_table[possible_physical_hard_disk-128][pointer]==0x04) || (partition_type_table[possible_physical_hard_disk-128][pointer]==0x06) )
	      {
	      possible_logical_drive++;

	      if(logical_drive==possible_logical_drive)
		{
		physical_drive=possible_physical_hard_disk;
		partition_number_of_logical_drive=pointer;
		increment_heads=TRUE;
		logical_drive_found=TRUE;
		}
	      }
	    pointer++;
	    }while(pointer<=maximum_partition_assigned[possible_physical_hard_disk-128]);
	  }
	possible_physical_hard_disk++;
	}while(possible_physical_hard_disk<136);
      }

    /* Get a full partition table listing for that drive */

    if(logical_drive_found==TRUE)
      {
      Get_Physical_Hard_Drive_Parameters();
      Read_Partition_Table();
      }
    else
      {
      printf("\nDrive letter entered is out of range...operation terminated.\n");
      Deallocate_Buffers();
      exit(4);
      }

    /* Load physical partition information */

    partition_starting_cylinder=starting_cylinder[partition_number_of_logical_drive];
    partition_starting_head=starting_head[partition_number_of_logical_drive];
    partition_starting_sector=starting_sector[partition_number_of_logical_drive];

    partition_ending_cylinder=ending_cylinder[partition_number_of_logical_drive];
    partition_ending_head=ending_head[partition_number_of_logical_drive];
    partition_ending_sector=ending_sector[partition_number_of_logical_drive];

    drive_maximum_cylinders=ending_cylinder[partition_number_of_logical_drive];
    drive_maximum_heads=total_heads;
    drive_maximum_sectors=total_sectors;

    total_logical_sectors=(((partition_ending_cylinder-partition_starting_cylinder)+1)*(total_heads+1)*total_sectors)-total_sectors;
    total_physical_sectors=total_logical_sectors;

    logical_sector_offset=((partition_starting_cylinder)*(total_heads+1)*(total_sectors));
    if(logical_sector_offset==0) logical_sector_offset=(partition_starting_head)*(total_sectors);
    if(increment_heads==TRUE) logical_sector_offset=logical_sector_offset+total_sectors;

    type_of_fat=FAT16;

    /* Compute cluster size */
    unsigned long size=(total_logical_sectors*512)/1000000;

    /* Sectors per cluster conversion table */
    if (size<=2048) number_of_sectors_per_cluster=64;
    if (size<=1024) number_of_sectors_per_cluster=32;
    if (size<=512) number_of_sectors_per_cluster=16;
    if (size<=256) number_of_sectors_per_cluster=8;
    if (size<=128) number_of_sectors_per_cluster=4;

    if (size<=16)
      {
      number_of_sectors_per_cluster=8;
      type_of_fat=FAT12;
      }

    /* Set media descriptor */
    media_descriptor=0xf8;
    }
}

/* Get Physical Floppy Drive Parameters */
void Get_Physical_Floppy_Drive_Parameters()
{
  asm{
    mov ah, 0x08
    mov dl, BYTE PTR physical_drive
    int 0x013

    mov BYTE PTR total_sectors, cl
    mov BYTE PTR total_cylinders, ch
    mov BYTE PTR total_heads, dh
    mov BYTE PTR floppy_drive_type, bl
    }
}

/* Get Physical Hard Drive Parameters */
void Get_Physical_Hard_Drive_Parameters()
{
  asm{
    mov ah, 0x08
    mov dl, BYTE PTR physical_drive
    int 0x13

    mov bl,cl
    and bl,00111111B

    mov BYTE PTR total_sectors, bl

    mov bl,cl
    mov cl,ch
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1

    mov ch,bl

    mov WORD PTR total_cylinders, cx
    mov BYTE PTR total_heads, dh
    }
}

/* Read Boot Sector /////////////////// */
void Read_Boot_Sector()
{
  long fat_16_total_sectors;
  long fat_12_total_sectors;

  printf("\n\nReading boot sector...\n");
  Read_Sector(0);

  sectors_per_fat = sector_buffer[22];
  fat_16_total_sectors=Decimal_Number(sector_buffer[32],sector_buffer[33],sector_buffer[34],sector_buffer[35]);
  fat_12_total_sectors=Decimal_Number(sector_buffer[19],sector_buffer[20],0,0);
  number_of_root_directory_entries=Decimal_Number(sector_buffer[17],sector_buffer[18],0,0);

  if(fat_16_total_sectors==0)
    {
    type_of_fat=FAT12;
    number_of_logical_sectors_on_drive=fat_12_total_sectors;
    }
  else
    {
    type_of_fat=FAT16;
    number_of_logical_sectors_on_drive=fat_16_total_sectors;
    }
  if( (fat_12_total_sectors>0) && (fat_16_total_sectors>0) ) bad_boot_sector=TRUE;
  if(sector_buffer[16]!=2) bad_boot_sector=TRUE;
}

/* Read Partition Table */
void Read_Partition_Table()
{
  long index=0x1be;

  int exiting_primary=TRUE;
  int extended=FALSE;
  int partition_designation=PRIMARY;
  int pointer=0;
  int record_extended_info_flag=FALSE;

  int done_flag=FALSE;

  unsigned long extended_cylinder;
  unsigned long extended_head;
  unsigned long extended_sector;

  Read_Physical_Sector(physical_drive,0,0,1);

  do{
    if(pointer==4) partition_designation=EXTENDED;

    if((pointer>=4) && (extended==TRUE))
      {
      Read_Physical_Sector(physical_drive,extended_head,extended_cylinder,extended_sector);
      extended=FALSE;
      index=0x1be;

      if(exiting_primary==FALSE)
	{
	pointer--;
	}
      else
	{
	exiting_primary=FALSE;
	}
      }

    /* Determine Partition Type */
    numeric_partition_type[pointer]=sector_buffer[index+4];

    if(sector_buffer[index+4]==0x00)
      {
      if(partition_designation==EXTENDED)
	{
	number_of_partitions=pointer;
	done_flag=TRUE;
	}
      }
    if(sector_buffer[index+4]==0x05)
      {
      extended=TRUE;
      record_extended_info_flag=TRUE;
      }

    starting_head[pointer] = sector_buffer[index+1];
    ending_head[pointer] = sector_buffer[index+5];

    partition_size[pointer]=Decimal_Number(sector_buffer[index+12],sector_buffer[index+13],sector_buffer[index+14],sector_buffer[index+15])/2000;

    Extract_Cylinder_and_Sector(sector_buffer[index+2],sector_buffer[index+3]);

    starting_cylinder[pointer]=g_cylinder;
    starting_sector[pointer]=g_sector;

    if((extended==TRUE) && (record_extended_info_flag==TRUE))
      {
      extended_cylinder=starting_cylinder[pointer];
      extended_head=starting_head[pointer];
      extended_sector=starting_sector[pointer];
      record_extended_info_flag=FALSE;
      }

    Extract_Cylinder_and_Sector(sector_buffer[index+6],sector_buffer[index+7]);

    ending_cylinder[pointer]=g_cylinder;
    ending_sector[pointer]=g_sector;

    partition_size_1[pointer]=sector_buffer[index+12];
    partition_size_2[pointer]=sector_buffer[index+13];
    partition_size_3[pointer]=sector_buffer[index+14];
    partition_size_4[pointer]=sector_buffer[index+15];

    relative_sectors_1[pointer]=sector_buffer[index+8];
    relative_sectors_2[pointer]=sector_buffer[index+9];
    relative_sectors_3[pointer]=sector_buffer[index+10];
    relative_sectors_4[pointer]=sector_buffer[index+11];

    pointer++;
    number_of_partitions=pointer-1;

    if((extended==FALSE) && (pointer==4))
      {
      number_of_partitions=4;
      done_flag=TRUE;
      }

    index=index + 16;
    } while(done_flag==FALSE);
}

/* Read Physical Sector */
void Read_Physical_Sector(int drive, int head, long cyl, int sector)
{
  int result;

  result=biosdisk(2, drive, head, cyl, sector, 1, sector_buffer);

    if (result!=0)
      {
      printf("\nRead error...operation terminated.\n");
      Deallocate_Buffers();
      exit(4);
      }
}

/* Read Sector From Disk */
void Read_Sector(unsigned long sector)
{
  Convert_Logical_To_Physical(sector);

  Display_Disk_Access_Location();

  Read_Physical_Sector(physical_drive,translated_head,translated_cylinder,translated_sector);
}

/* Reset drive controller */
void Reset_Drive()
{
  asm{
    mov ah,0
    mov dl,BYTE PTR physical_drive
    int 0x13

    mov ah,0
    mov dl,BYTE PTR physical_drive
    int 0x13
    }
}

/* Save the old file system for possible recovery with unformat */
void Save_File_System()
{
  int end_of_root_directory_flag;

  long loop=512;
  unsigned long pointer=0;

  unsigned long offset_from_end=5;

  unsigned long destination_sector;
  unsigned long source_sector;

  unsigned long mirror_beginning;
  unsigned long mirror_map_beginning;
  unsigned long mirror_map_size;
  unsigned long mirror_size;
  unsigned long number_of_bytes_in_mirror_map;

  unsigned long beginning_of_fat;
  unsigned long beginning_of_root_directory;

  unsigned long end_of_fat;
  unsigned long end_of_root_directory;

  unsigned int sector_flag[] = {
  'A','M','S','E','S','L','I','F','V','A','S',
  'R','O','R','I','M','E','S','A','E','P'};

  unsigned int mirror_map_header[] = {
  ':','\\','M','I','R','R','O','R','.','F','I','L'};

  if(drive_type==HARD) offset_from_end=20;

  /* Get the boot sector, compute the FAT size, compute the root dir size,*/
  /* and get the end of the logical drive. */
  Read_Boot_Sector();

  /* If the boot sector is not any good, don't save the file system. */
  if(bad_boot_sector==FALSE)
    {
    printf("\n\nSaving UNFORMAT information...\n");

    /* Clear the last 200 sectors of a hard disk or the last 10 sectors of */
    /* a floppy disk to ensure that UNFORMAT does not get confused.        */
    if(drive_type==FLOPPY) pointer=number_of_logical_sectors_on_drive-10;
    else pointer=number_of_logical_sectors_on_drive-200;

    Clear_Sector_Buffer();
    do
      {
      Write_Sector(pointer);
      pointer++;
      }while(pointer<number_of_logical_sectors_on_drive);

    /* Compute the beginning sector of the mirror map and the size of */
    /* the mirror image.     */
    mirror_size=1+sectors_per_fat+66;

    mirror_map_size=(mirror_size/64)+2;

    mirror_beginning=(number_of_logical_sectors_on_drive-mirror_size)-offset_from_end;
    mirror_map_beginning=mirror_beginning-mirror_map_size;

    /* Compute the locations of the first FAT and the root directory */
    beginning_of_fat=1;
    end_of_fat=beginning_of_fat+sectors_per_fat-1;

    beginning_of_root_directory=(beginning_of_fat+sectors_per_fat-1)*2;
    end_of_root_directory=beginning_of_root_directory+(number_of_root_directory_entries/32);

    /* Write the mirror map pointer to the last sectors of the logical drive. */
    Clear_Sector_Buffer();

    Convert_Huge_To_Integers(mirror_map_beginning);

    sector_buffer[0]=integer1;
    sector_buffer[1]=integer2;
    sector_buffer[2]=integer3;
    sector_buffer[3]=integer4;

    pointer=4;

    do                                           /* Add pointer sector flag */
      {
      sector_buffer[pointer]=sector_flag[pointer-4];
      pointer++;
      }while(pointer<=24);

    Write_Sector(number_of_logical_sectors_on_drive-offset_from_end);

    /* Create the mirror map and copy the file system to the mirror.  */
    Clear_Sector_Buffer();

    pointer=0;

    do                                           /* Clear mirror_map buffer */
      {
      mirror_map[pointer]=0;
      pointer++;
      }while(pointer<=5120);

    mirror_map[0]=logical_drive+65;

    pointer=1;

    do                                           /* Add mirror map header */
      {
      mirror_map[pointer]=mirror_map_header[pointer-1];
      pointer++;
      }while(pointer<=12);

					       /* Main mirror map creation */
					       /* and copying loop.        */
    pointer=84;
    source_sector=0;
    destination_sector=mirror_beginning;

    end_of_root_directory_flag=FALSE;
    number_of_bytes_in_mirror_map=0;

    do
      {

      if( (source_sector!=0) && (source_sector<beginning_of_fat) ) source_sector=beginning_of_fat;


      if( (source_sector>end_of_fat) && (source_sector<beginning_of_root_directory) ) source_sector=beginning_of_root_directory;

      /* Copy mirror image one sector at a time */
      Read_Sector(source_sector);

      Write_Sector(destination_sector);

      /* Enter mapping information into mirror map buffer */

      Convert_Huge_To_Integers(source_sector);

      mirror_map[pointer+0]=integer1;
      mirror_map[pointer+1]=integer2;
      mirror_map[pointer+2]=integer3;
      mirror_map[pointer+3]=integer4;

      Convert_Huge_To_Integers(destination_sector);

      mirror_map[pointer+4]=integer1;
      mirror_map[pointer+5]=integer2;
      mirror_map[pointer+6]=integer3;
      mirror_map[pointer+7]=integer4;

      source_sector++;
      destination_sector++;
      pointer=pointer+8;
      number_of_bytes_in_mirror_map=pointer;

      if(source_sector>end_of_root_directory) end_of_root_directory_flag=TRUE;

      }while(end_of_root_directory_flag==FALSE);

    /* Write trailer in mirror map */

    mirror_map[pointer+0]=0;
    mirror_map[pointer+1]=0;
    mirror_map[pointer+2]=0;
    mirror_map[pointer+3]=0;

    /* Write the mirror map onto the disk.   */

    pointer=0;
    destination_sector=mirror_map_beginning;

    do
      {
      loop=0;

      do                                         /* Load the sector buffer */
	{
	sector_buffer[loop]=mirror_map[pointer+loop];

	loop++;
	}while(loop<512);

      Write_Sector(destination_sector);          /* Write the mirror map   */
						 /* sector.                */
      destination_sector++;
      pointer=pointer+512;
      }while(pointer < number_of_bytes_in_mirror_map);

    }
  else
    {
    printf("\n\nDrive appears unformatted, UNFORMAT information not saved.");
    }
}

/* Unconditional Format */
void Unconditional_Format()
{
  long bad_sector_map_index=0;
  unsigned long sector_counter=0;
  unsigned long index=0;

  unsigned long track_change_sector;

  int track_change=TRUE;

  long current_cylinder=-1;
  long current_head=-1;

  int result;

  long verify_index;

  /* Allocate space for big_sector_buffer */
  big_sector_buffer=(unsigned char *)malloc(512*total_sectors+1);
  if(big_sector_buffer==NULL)
    {
    printf("\nBuffer Allocation Failure(big_sector_buffer)...Program Terminated.\n");
    Deallocate_Buffers();
    exit(1);
    }

  /* Drive specific initialization code */
  Reset_Drive();

  /* Fill Sector Buffer */
  index=0;
  do
    {
    sector_buffer[index]=0xf6;
    index++;
    }while(index<512);

  /* Fill big_sector_buffer */
  index=0;
  do
    {
    big_sector_buffer[index]=0xf6;
    index++;
    }while(index<(512*total_sectors));

  /* Format each sector */
  printf("\n\nFormatting...\n");

  do
    {
    /* Custom track formatting routine with bad sector tracking:  */
    /* (This was created to both optimize the formatting routine  */
    /* and keep track of bad sectors.)                            */

    Convert_Logical_To_Physical(sector_counter);

    if( (current_cylinder!=translated_cylinder) || (current_head!=translated_head) )
      {
      track_change=TRUE;
      Display_Disk_Access_Location();
      }

    if(track_change==TRUE)
      {
      track_change=FALSE;
      current_cylinder=translated_cylinder;
      current_head=translated_head;
      track_change_sector=sector_counter;

      /* Write to track. */
      result=biosdisk(3, physical_drive, translated_head, translated_cylinder, 1, total_sectors, big_sector_buffer);

      /* Verify that the track is correct. */
      result=biosdisk(4, physical_drive, translated_head, translated_cylinder, 1, total_sectors, big_sector_buffer);

      /* If the track is not correct, determine which sector(s) is/are bad. */
      if(result!=0)
	{
	int individual_sector=1;

	do
	  {
	  result=biosdisk(4, physical_drive, translated_head, translated_cylinder, individual_sector, 1, sector_buffer);

	  if(result!=0)
	    {
	    bad_sector_map[bad_sector_map_index]=track_change_sector+individual_sector;
	    bad_sector_map_index++;

	    if(bad_sector_map_index>1023)
	      {
	      printf("\n\nNumber of bad sectors on disk has exceeded buffer size...\nOperation Terminated.\n");
	      Deallocate_Buffers();
	      exit(4);
	      }
	    }

	  individual_sector++;
	  }while(individual_sector<=total_sectors);
	}
      }

    sector_counter++;
    }while(sector_counter<(total_logical_sectors));

  /* If there are bad sectors recorded set the bad_sector_flag to alert the */
  /* Create_File_System() function that the FAT tables need modified.       */
  if(bad_sector_map[0]>0) bad_sector_flag=TRUE;
  free(big_sector_buffer);
}

/* Write Physical Sector */
void Write_Physical_Sector(int drive, int head, long cyl, int sector)
{
  int result;

  result=biosdisk(3, drive, head, cyl, sector, 1, sector_buffer);

  if (result!=0)
    {
    printf("\nDisk write error...operation terminated.\n");
    Deallocate_Buffers();
    exit(4);
    }
}

/* Write Sector To Disk */

void Write_Sector(unsigned long sector)
{
  Convert_Logical_To_Physical(sector);

  Display_Disk_Access_Location();

  Write_Physical_Sector(physical_drive,translated_head,translated_cylinder,translated_sector);
}

/*
/////////////////////////////////////////////////////////////////////////////
//  MAIN ROUTINE
/////////////////////////////////////////////////////////////////////////////
*/
void main(int argc, char *argv[])
{
  int loop=0;
  int override_automatic_media_checking;

  Allocate_Buffers();

  /* if FORMAT is typed without any options or with /? */
  if( (argc==1) || (argv[1][1]=='?') )
    {
    Display_Help_Screen();
    exit(1);
    }

  /*if FORMAT is typed with a drive letter */
  if(argc>=2)
    {
    int index=3;

    char compare_character[2];
    char drive_letter[2];

    logical_drive=argv[1] [0];

    if(logical_drive>=97) logical_drive = logical_drive - 97;
    if(logical_drive>=65) logical_drive = logical_drive - 65;

    if( (logical_drive<0) || (logical_drive> 25) )
      {
      printf("\nDrive letter is out of range...Operation Terminated.\n");
      Deallocate_Buffers();
      exit(4);
      }

    drive_letter[0] = logical_drive+65;
    drive_letter[1] = 58;

    /* Default to a drive type of floppy unless... */
    drive_type=FLOPPY;

    /* ...the drive is a hard drive. */
    if(logical_drive>1)
      {
      drive_type=HARD;
      }

    /* Logical drives are designated as follows:              */
    /* drive   letter   descriptions                          */
    /*    0       A     First floppy drive                    */
    /*    1       B     Second floppy drive                   */
    /*    2       C     Primary partition of first hard disk  */
    /*    3       D     Primary partition of second hard disk */
    /*                   or first extended partition of first */
    /*                   hard disk.                           */
    /*    etc.    etc.  etc.                                  */

    /* When the physical drive # is established for hard disks */
    /* the physical hard disk # will become drive              */

    /*place valid drive checking code here*/



    /*valid drive checking code placed above this line*/


    if(argc>=3)
      {
      index=3;
      do
	{
	compare_character[0]=argv[index-1][1];

	/* Determine which switches were entered */
	if( ('v'==compare_character[0]) || ('V'==compare_character[0]) )
	  {
	  v=TRUE;

	  loop=0;
	  do
	    {
	    entered_label[loop]=argv[index-1][loop+3];
	    if(entered_label[loop]>96) entered_label[loop]=entered_label[loop]-32;

	    loop++;
	    }while(loop<11);
	  }

	if( ('q'==compare_character[0]) || ('Q'==compare_character[0]) )
	  {
	  q=TRUE;
	  }

	if( ('u'==compare_character[0]) || ('U'==compare_character[0]) )
	  {
	  u=TRUE;
	  }

	if( ('f'==compare_character[0]) || ('F'==compare_character[0]) )
	  {
	  f=TRUE;
	  if( ('1'==argv[index-1][3]) && ('6'==argv[index-1][4]) ) entered_size=160;
	  if( ('1'==argv[index-1][3]) && ('8'==argv[index-1][4]) ) entered_size=180;
	  if( ('3'==argv[index-1][3]) && ('2'==argv[index-1][4]) ) entered_size=320;
	  if( ('3'==argv[index-1][3]) && ('6'==argv[index-1][4]) ) entered_size=360;
	  if( ('7'==argv[index-1][3]) ) entered_size=720;
	  if( ('1'==argv[index-1][3]) && ('2'==argv[index-1][4]) ) entered_size=1200;
	  if( ('1'==argv[index-1][3]) && ('2'==argv[index-1][5]) ) entered_size=1200;
	  if( ('1'==argv[index-1][3]) && ('4'==argv[index-1][4]) ) entered_size=1440;
	  if( ('1'==argv[index-1][3]) && ('4'==argv[index-1][5]) ) entered_size=1440;
	  if( ('2'==argv[index-1][3]) ) entered_size=2880;
	  }

	if( ('b'==compare_character[0]) || ('B'==compare_character[0]) )
	  {
	  b=TRUE;
	  }

	if( ('s'==compare_character[0]) || ('S'==compare_character[0]) )
	  {
	  s=TRUE;
	  }

	if( ('t'==compare_character[0]) || ('T'==compare_character[0]) )
	  {
	  t=TRUE;
	  if('4'==argv[index-1][3]) entered_tracks=40;
	  if('8'==argv[index-1][3]) entered_tracks=80;
	  }

	if( ('n'==compare_character[0]) || ('N'==compare_character[0]) )
	  {
	  n=TRUE;
	  if('8'==argv[index-1][3]) entered_sectors=8;
	  if('9'==argv[index-1][3]) entered_sectors=9;
	  if( ('1'==argv[index-1][3]) && ('5'==argv[index-1][4]) ) entered_sectors=15;
	  if( ('1'==argv[index-1][3]) && ('8'==argv[index-1][4]) ) entered_sectors=18;
	  if('3'==argv[index-1][3]) entered_sectors=36;
	  }

	if('1'==compare_character[0])
	  {
	  one=TRUE;
	  }

	if('4'==compare_character[0])
	  {
	  four=TRUE;
	  }

	if('8'==compare_character[0])
	  {
	  eight=TRUE;
	  }

	index++;
	}while(index<=argc);
      }

    /* Ensure that valid switch combinations were entered */
    if( (b==TRUE) && (s==TRUE) ) Display_Invalid_Combination();
    if( (v==TRUE) && (eight==TRUE) ) Display_Invalid_Combination();
    if( ( (one==TRUE) || (four==TRUE) ) && ( (f==TRUE) || (t==TRUE) || (n==TRUE) ) ) Display_Invalid_Combination();
    if( ( (t==TRUE) && (n!=TRUE) ) || ( (n==TRUE) && (t!=TRUE) ) ) Display_Invalid_Combination();
    if( (f==TRUE) && ( (t==TRUE) || (n==TRUE) ) )Display_Invalid_Combination();
    if( ( (one==TRUE) || (four==TRUE) ) && (eight==TRUE) )Display_Invalid_Combination();
    if( ( (four==TRUE) || (eight==TRUE) ) && (one==TRUE) )Display_Invalid_Combination();
    if( ( (eight==TRUE) || (one==TRUE) ) && (four==TRUE) )Display_Invalid_Combination();

    /* Determine if the user wants to override the automatic media checking.*/
    /* (Did the user specify the number of tracks, number of sectors, size  */
    /* of the floppy disk, etc?                                             */
    override_automatic_media_checking=FALSE;
    if( (f==TRUE) || (t==TRUE) || (n==TRUE) || (one==TRUE) || (four==TRUE) || (eight==TRUE) ) override_automatic_media_checking=TRUE;

    /* Get drive parameters */
    Get_Drive_Parameters();

    /* Did the user specify the number of tracks, number of sectors, size   */
    /* of the disk, etc. and the disk is a hard disk?  If so, then abort.   */
    if( ( (f==TRUE) || (t==TRUE) || (n==TRUE) || (one==TRUE) || (four==TRUE) || (eight==TRUE) ) && (drive_type==HARD) ) Display_Invalid_Combination();

    /* Change drive parameters if the users gave formatting specifics */
    if(override_automatic_media_checking==TRUE)
      {
      u=TRUE;
      q=FALSE;

      /* if /f was entered */
      if(f==TRUE)
	{
	}

      /* if /t was entered */
      if(t==TRUE)
	{
	}

      /* if /n was entered */
      if(n==TRUE)
	{
	}

      /* if /1 was entered */
      if(one==TRUE)
	{
	if(floppy_drive_type==0x01) media_descriptor=0xfc;
	if(floppy_drive_type==0x02)
	  {
	  if(four==TRUE) media_descriptor=0xfc;
	  else Display_Invalid_Combination();
	  }
	if(floppy_drive_type==0x03) Display_Invalid_Combination();
	if(floppy_drive_type==0x04) Display_Invalid_Combination();
	}

      /* if /4 was entered */
      if(four==TRUE)
	{
	if(floppy_drive_type==0x02)
	  {
	  if(one==FALSE) media_descriptor=0xfd;
	  }
	else Display_Invalid_Combination();
	}

      /* if /8 was entered */
      if(eight==TRUE)
	{
	}
      }

    /* Format Drive */
    if( (u==TRUE) && (q==FALSE) )
      {
      /* Unconditional Format */
      Unconditional_Format();
      Create_File_System();
      Deallocate_Buffers();
      exit(0);
      }

    if( (u==FALSE) && (q==TRUE) )
      {
      /* Quick Format */
      Save_File_System();
      Create_File_System();
      Deallocate_Buffers();
      exit(0);
      }

    if( (u==TRUE) && (q==TRUE) )
      {
      /* Quick Unconditional format */
      Create_File_System();
      Deallocate_Buffers();
      exit(0);
      }

    if( (u==FALSE) && (q==FALSE) )
      {
      /* Safe Format */
      Save_File_System();
      Create_File_System();
      Deallocate_Buffers();
      exit(0);
      }
    }

  Deallocate_Buffers();
}
