/*
// Program:  Free FDISK
// Written By:  Brian E. Reifsnyder
// Version:  0.95
// Copyright:  1999 under the terms of the GNU GPL
*/

/* Phil Brutsche - November 20, 1998
   Fixed a number of bugs:
   * In several places, I found the use of gets (bad in itself...) in
     conjunction with uninitialized pointers.  Specifically, this:

	  char *input_buffer;

	  long partition_size;

	     Clear_Screen();
	     printf("\n\n\nEnter Primary Partition Size: ");
	     gets(input_buffer);

     That's a no-no; you don't know where input_buffer points to - it could
     point to, say, 0x00000010, and royally screw up your interrupt table
     when you use gets.  Plus, `gets` doesn't check the length of the string
     you are reading - automatic buffer overflow.  You may get your input,
     but it'll royally screw up anything that it physically next to the
     buffer in memory - maybe even part of your program!

   * Created some preprocessor directives to simplify the help screens.
*/



#define NAME "Free FDISK"
#define VERSION "0.95"
#define SIZE_OF_MBR 445
			 /* ***** This is the number of bytes for the */
			 /* ***** boot code in the partition table.   */
			 /* ***** If the boot code in the partition   */
			 /* ***** table changes, this WILL need       */
			 /* ***** changed.  (Start the count at 0.)   */
			 /* ***** The MBR code is in the file         */
			 /* ***** bootcode.h                          */

/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <bios.h>
#include <conio.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "bootcode.h"

/*
/////////////////////////////////////////////////////////////////////////////
//  DEFINES
/////////////////////////////////////////////////////////////////////////////
*/

#define READ 2

#define TRUE 1
#define FALSE 0

#define PRIMARY 1
#define EXTENDED 2
#define LOGICAL 3
#define SPECIAL 4

#define LAST 99

#define PERCENTAGE 1

#define STANDARD 0
#define TECHNICAL 1

#define INTERNAL 0
#define EXTERNAL 1

#define COLOR 1
#define MONOCHROME 0

/*
/////////////////////////////////////////////////////////////////////////////
//  GLOBAL VARIABLES
/////////////////////////////////////////////////////////////////////////////
*/

/* Buffers */
unsigned char *partition_lookup_table_buffer;

unsigned char sector_buffer[512];

unsigned long g_cylinder;
unsigned long g_sector;

/* Hard Drive Parameters */
unsigned long total_cylinders;
unsigned long total_heads;
unsigned long total_sectors;

/* Ending Mapping Variables */
unsigned long computed_ending_cylinder;
unsigned long computed_ending_head;
unsigned long computed_ending_sector;
unsigned long computed_partition_size;

/* Partition Table Information */
int extended_partition_exists=FALSE;
int primary_partition_exists=FALSE;
int logical_drives_exist=FALSE;

int active_flag[4];

unsigned char partition_type[24] [80];

int numeric_partition_type[24];

unsigned long starting_cylinder[24];
unsigned long starting_head[24];
unsigned long starting_sector[24];

unsigned long ending_cylinder[24];
unsigned long ending_head[24];
unsigned long ending_sector[24];

unsigned long true_partition_size[24];
long partition_size[24];
int number_of_partitions;
unsigned long hard_disk_space_used;
unsigned long whole_extended_partition_size;
unsigned long extended_partition_space_used;

unsigned long extended_partition_cylinder[24];
unsigned long extended_partition_head[24];
unsigned long extended_partition_sector[24];

/* Integers converted from long numbers */
int integer1;
int integer2;

/* Misc. Flags */
int automatic=FALSE;

int FAT32=FALSE;
int int13ext=FALSE;

int partition_lookup_table=INTERNAL;
int sectors_have_changed=FALSE;
int menu_in_use=FALSE;
int error_encountered=FALSE;
int show_primary_partition_table=TRUE;
int show_extended_partition_table=TRUE;
int video=COLOR;

/* Misc. */
int total_number_of_hard_drives;
int available_primary_partitions;

/*
/////////////////////////////////////////////////////////////////////////////
//  PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/

char Input();

int External_Lookup_File_Exists();
int Delete_P_Or_E_Partition(int drive,int partition_type);
int More_Than_One_Hard_Disk();

unsigned long Combine_Cylinder_and_Sector(unsigned long cylinder, unsigned long sector);

unsigned long Decimal_Number(unsigned long hex1, unsigned long hex2, unsigned long hex3, unsigned long hex4);

void Allocate_Memory_For_Buffers();
void Ask_User_About_FAT32_Support();
void Automatically_Partition_Hard_Drive(int drive);
void Calculate_Partition_Ending_Cylinder(long start_cylinder,long size);
void Check_For_INT13_Extensions();
void Clear_Screen();
void Clear_Sector_Buffer();
void Clear_Partition_Table(int drive);
void Convert_Long_To_Integer(long number);
void Create_Alternate_MBR(int drive);
void Create_MBR(int drive);
void Create_New_Partition(int drive,int create_partition_type,unsigned long size,int percentage,int special_flag,int special_partition_type);
void DeAllocate_Memory_For_Buffers();
void Delete_Partition(int drive, long partition_number);
void Determine_Color_Video_Support();
void Display_Help_Screen();
void Display_Line();
void Display_Partition_Table(int drive,int tech_flag);
void Dump_Partition_Information();
void Extract_Cylinder_and_Sector(unsigned long hex1, unsigned long hex2);
void Get_Hard_Drive_Parameters(int drive);
void Initialize_Variables();
void List_Available_Drives();
void Load_External_Lookup_Table();
void Menu_Routine();
void Modify_Partition_Type(int drive,int partition_number,int type_number);
void Pause();
void Position_Cursor(int row,int column);
void Read_Partition_Table(int drive,int view_type);
void Read_Sector(int drive, int head, long cyl, int sector);
void Remove_MBR(int drive);
void Save_MBR(int drive);
void Scan_External_Lookup_Table(int indicated_partition_type,int partition,int view_type);
void Toggle_Active_Partition(int drive, long partition_number, int toggle);
void Write_Sector(int drive, int head, long cyl, int sector);

/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

/* Allocate far memory for larger buffers */
void Allocate_Memory_For_Buffers()
{
  partition_lookup_table_buffer=(unsigned char *)malloc((80*256)+1);

  if(partition_lookup_table_buffer==NULL)
    {
    printf("\nBuffer Allocation Failure...Program Terminated.\n");
    exit(1);
    }
}

/* Ask user if they want to use large disk support */
/* But, first check for int0x13 extensions         */
void Ask_User_About_FAT32_Support()
{
  /* First check for int 0x13 support */
  int input=0;

  /* Ask user if they want large disk support (FAT32) */
  Clear_Screen();
  Position_Cursor(17,12);
  printf("Do you want to use large disk support? (Y/N) ");
  input=Input();

  if( (input=='Y') || (input=='y') )
    {
    FAT32=TRUE;
    }
  else
    {
    FAT32=FALSE;
    }
}

/* Automatically partition the selected hard drive */
void Automatically_Partition_Hard_Drive(int drive)
{
  int index=0;
  int diagnostic_partition_exists=FALSE;
  int diagnostic_partition_number;

  unsigned long available_extended_partition_space;
  unsigned long total_hard_disk_size;
  unsigned long hard_disk_space_available;

  Read_Partition_Table(drive,TECHNICAL);
  /* Ensure that no primary, extended, or non-dos (except diagnostic) */
  /* partitions exist.  If a diagnostic partition exists, set the     */
  /* variable diagnostic_partition_exists=TRUE.                       */
  do
    {
    if((0x06==numeric_partition_type[index]) || (0x04==numeric_partition_type[index]) || (0x01==numeric_partition_type[index]) || (0x0b==numeric_partition_type[index]) || (0x0c==numeric_partition_type[index]) || (0x0e==numeric_partition_type[index]) )
      {
      printf("\nPrimary DOS partition has been found...Operation Terminated.\n");
      exit(1);
      }

    if( (0x05==numeric_partition_type[index]) || (0x0f==numeric_partition_type[index]) )
      {
      printf("\nExtended DOS partition has been found...Operatin Terminated.\n");
      exit(1);
      }

    if(18==numeric_partition_type[index])
      {
      diagnostic_partition_exists=TRUE;
      diagnostic_partition_number=index;
      }

    if( ( (numeric_partition_type[index]>0) && (numeric_partition_type[index]<18) ) || (numeric_partition_type[index]>18) )
      {
      printf("\nNon-DOS partition has been found...Operation Terminated.\n");
      exit(1);
      }

    index++;
    }while(index<4);

  /* Arrange partition table to allow for proper partitioning if the system */
  /* has a Compaq Diagnostic Partition and the partition is not the first   */
  /* in the primary partition table.                                        */

  if((diagnostic_partition_exists==TRUE) && (diagnostic_partition_number>0))
    {
    index=0;

    do
      {
      sector_buffer[(0x1be+index)]=sector_buffer[(0x1be+(diagnostic_partition_number*16)+index)];
      sector_buffer[(0x1be+(diagnostic_partition_number*16)+index)]=0;
      index++;
      }while(index<16);

    Write_Sector(drive,0,0,1);
    }

  /* Create primary partition */
  Create_New_Partition(drive,PRIMARY,100,PERCENTAGE,NULL,NULL);

  if(automatic!=PRIMARY)
    {
    /* Check for free space */
    Read_Partition_Table(drive,TECHNICAL);
    Get_Hard_Drive_Parameters(drive);

    total_hard_disk_size = (total_cylinders+1) * (total_heads+1) * (total_sectors);
    hard_disk_space_available=total_hard_disk_size-hard_disk_space_used;

    /* If there is more than 20 megabytes of available space,        */
    /* create the extended partition and fill it with logical drives */
    if(hard_disk_space_available>41943)
      {
      Create_New_Partition(drive,EXTENDED,100,PERCENTAGE,NULL,NULL);
      Read_Partition_Table(drive,TECHNICAL);

      available_extended_partition_space=whole_extended_partition_size-extended_partition_space_used;

      /* Create logical drives in the extended partition as long as more */
      /* than 10MB are available */
      do
	{
	Create_New_Partition(drive,LOGICAL,100,PERCENTAGE,NULL,NULL);

	Read_Partition_Table(drive,TECHNICAL);

	available_extended_partition_space=whole_extended_partition_size-extended_partition_space_used;
	}while(available_extended_partition_space>20971);
      }
    }

  /* Set the primary partition active */
  if(diagnostic_partition_exists==FALSE)
    {
    Toggle_Active_Partition(drive,1,1);
    }
  else
    {
    Toggle_Active_Partition(drive,2,1);
    }
}

/* Calculate the end of a Partition */
/* Replaces void Compute_Partition_End */
void Calculate_Partition_Ending_Cylinder(long start_cylinder,long size)
{
  long cylinder_size=(total_heads+1)*(total_sectors);

  computed_partition_size=0;
  computed_ending_cylinder=start_cylinder;

  do
    {
    computed_ending_cylinder++;
    computed_partition_size=computed_partition_size+cylinder_size;
    }while(computed_partition_size<size);

  computed_ending_cylinder--;
  computed_partition_size=computed_partition_size-cylinder_size;
}

void Check_For_INT13_Extensions()
{
  int carry=99;

  asm{
    mov ah,0x41
    mov bx,0x55aa
    mov dl,0x80
    int 0x13
    jnc carry_flag_not_set    /* Jump if the carry flag is clear  */
    }                         /* If the carry flag is clear, then */
			      /* the extensions exist.            */
  carry=1;
  int13ext=FALSE;

  carry_flag_not_set:
  if(carry==99) int13ext=TRUE;
}

/* Clear Screen */
void Clear_Screen()
{
  asm{
    mov ah,0
    mov al,3
    int 0x10
    }
}

/* Clear Sector Buffer */
void Clear_Sector_Buffer()
{
  int index=0;

  do
    {
    sector_buffer[index]=0;
    index ++;
    } while(index<512);
}

/* Clear Partition Table */
void Clear_Partition_Table(int drive)
{
  Clear_Sector_Buffer();
  Write_Sector(drive, 0, 0, 1);
}

/* Combine Cylinder and Sector Values */
unsigned long Combine_Cylinder_and_Sector(unsigned long cylinder, unsigned long sector)
{
  long value = 0;

  asm{
    mov ax,WORD PTR cylinder
    mov bx,WORD PTR sector

    mov dl,ah
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1

    mov dh,al

    add dx,bx

    mov WORD PTR value,dx
    }

  return(value);
}

/* Convert Long number to 2 integers */
void Convert_Long_To_Integer(long number)
{
  integer1=0;
  integer2=0;

  asm{
    mov ax,WORD PTR number

    mov BYTE PTR integer1, al
    mov BYTE PTR integer2, ah
    }
}

/* Create Alternate Master Boot Code */
void Create_Alternate_MBR(int drive)
{
  long index=0;

  FILE *file_pointer;

  Read_Sector(drive,0,0,1);

  /* Clear old MBR, if any */
  do
    {
    sector_buffer[index]=0x00;
    index++;
    }while(index<0x1be);
  index=0;

  file_pointer=fopen("boot.mbr","rb");

  if(!file_pointer)
    {
    printf("\nThe\"BOOT.MBR\" file has not been found...Operation Terminated.\n");
    exit(1);
    }

  do
    {
    sector_buffer[index]=fgetc(file_pointer);
    index++;
    }while(index<0x1be);

  fclose(file_pointer);

  Write_Sector(drive,0,0,1);
}

/* Create Master Boot Code */
void Create_MBR(int drive)
{
  long number_of_bytes=SIZE_OF_MBR;

  long loop=0;

  Read_Sector(drive,0,0,1);

  do
    {
    sector_buffer[loop]=boot_code[loop];

    loop++;
    }while(loop<=number_of_bytes);

  Write_Sector(drive,0,0,1);
}

/* Create New Partition */
void Create_New_Partition(int drive,int create_partition_type,unsigned long size,int percentage,int special_flag,int special_partition_type)
{
  int index;
  int next_available_partition;
  int partition_created=FALSE;

  unsigned long temp;

  unsigned long hard_disk_space_available;

  unsigned long extended_partition_size;
  unsigned long extended_partition_size_high;
  unsigned long extended_partition_size_low;
  unsigned long available_extended_partition_space;

  unsigned long relative_sectors;
  unsigned long relative_sectors_high;
  unsigned long relative_sectors_low;

  unsigned long total_hard_disk_size;

  long partition_table_pointer;

  long cp_starting_cylinder;
  long cp_starting_head;
  long cp_starting_sector;

  unsigned long partition_size_high;
  unsigned long partition_size_low;

  Read_Partition_Table(drive,TECHNICAL);
  Get_Hard_Drive_Parameters(drive);

  total_hard_disk_size = (total_cylinders+1) * (total_heads+1) * (total_sectors);
  hard_disk_space_available=total_hard_disk_size-hard_disk_space_used;

  /* Compute the size of the partition */
  available_extended_partition_space=whole_extended_partition_size-extended_partition_space_used;

  if(percentage==PERCENTAGE)
    {
    if(create_partition_type==PRIMARY)
      {
      if( (total_hard_disk_size>4194304) && (FAT32==FALSE) )
	{
	size=(4194304*size)/100;
	}
      else
	{
	size=(hard_disk_space_available*size)/100;
	}
      }
    if(create_partition_type==EXTENDED)
      {
      size=(hard_disk_space_available*size)/100;
      }
    if(create_partition_type==LOGICAL)
      {
      if( (available_extended_partition_space>4194304) && (FAT32==FALSE) )
	{
	size=(4194304*size)/100;
	}
      else
	{
	size=(available_extended_partition_space*size)/100;
	}
      }
    }
  else
    {
    size=size*2097;
    }

  /* Adjust partition size to a legal partition size, if necessary */
  if(size>total_hard_disk_size) size = total_hard_disk_size;
  if((create_partition_type==PRIMARY) && (size>4194304) && (FAT32==FALSE) ) size=4194304;
  if((create_partition_type==EXTENDED) && (size>(hard_disk_space_available))) size=hard_disk_space_available;
  if((create_partition_type==LOGICAL) && (size>4194304) && (FAT32==FALSE) ) size=4194304;
  if((create_partition_type==LOGICAL) && (size>whole_extended_partition_size)) size=whole_extended_partition_size;
  if((create_partition_type==LOGICAL) && (size>available_extended_partition_space)) size=available_extended_partition_space;
  if(size<4000)
    {
    if(menu_in_use==TRUE) Clear_Screen();

    printf("\nPartition size to be created is too small...Operation Terminated\n");

    if(menu_in_use==TRUE)
      {
      error_encountered=TRUE;
      Pause();
      Clear_Screen();
      }
    else
      {
      exit(1);
      }
    }

  if(((create_partition_type==PRIMARY) || (create_partition_type==EXTENDED)) && error_encountered==FALSE)
    {
    index=0;

    do
      {
      if( (extended_partition_exists==TRUE) && (create_partition_type==EXTENDED) )
	{
	if(menu_in_use==TRUE) Clear_Screen();

	printf("\nExtended partition already exists...operation terminated.\n");

	if(menu_in_use==TRUE)
	  {
	  error_encountered=TRUE;
	  Pause();
	  Clear_Screen();
	  }
	else
	  {
	  exit(1);
	  }
	}
      /* Find unused primary partition */
      if(0==numeric_partition_type[index])
	{
	/* Unused primary partition found! */


	/* If this is the first partition */
	if(0==index)
	  {
	  cp_starting_cylinder=0;
	  cp_starting_head=1;
	  cp_starting_sector=1;
	  Calculate_Partition_Ending_Cylinder(cp_starting_cylinder,size);
	  }

	if(0!=index)
	  {
	  cp_starting_cylinder=(ending_cylinder[index-1]+1);
	  cp_starting_head=0;
	  cp_starting_sector=1;
	  Calculate_Partition_Ending_Cylinder(cp_starting_cylinder,size);
	  }

	if(special_flag!=SPECIAL)
	  {
	  if(create_partition_type==PRIMARY)
	    {
	    if(size>0xffff) numeric_partition_type[index]=0x06;
	    if(size<=0xffff) numeric_partition_type[index]=0x04;
	    if(size<=0x7fff) numeric_partition_type[index]=0x01;
	    if( (FAT32==TRUE) && (size>1048576) )numeric_partition_type[index]=0x0b;
	    }
	  else
	    {
	    numeric_partition_type[index]=0x05;
	    }
	  }
	else
	  {
	  numeric_partition_type[index]=special_partition_type;
	  }

	/* Make computations & write partition */
	Read_Sector(drive,0,0,1);

	partition_table_pointer=0x1be+(index*16);

	temp=Combine_Cylinder_and_Sector(cp_starting_cylinder,cp_starting_sector);

	Convert_Long_To_Integer(temp);

	sector_buffer[partition_table_pointer+0x02]=integer1;
	sector_buffer[partition_table_pointer+0x03]=integer2;

	sector_buffer[partition_table_pointer+0x01]=cp_starting_head;

	sector_buffer[partition_table_pointer+0x04]=numeric_partition_type[index];

	temp=Combine_Cylinder_and_Sector(computed_ending_cylinder-1,total_sectors);

	Convert_Long_To_Integer(temp);

	sector_buffer[partition_table_pointer+0x06]=integer1;
	sector_buffer[partition_table_pointer+0x07]=integer2;

	sector_buffer[partition_table_pointer+0x05]=total_heads;

	/* Adjust size to fit inside partition */
	if(size> ((total_heads+1)*total_sectors))
	  {
	  size=((computed_ending_cylinder)-cp_starting_cylinder)*( (total_heads+1)*(total_sectors) );
	  }

	if(create_partition_type==PRIMARY) size=size-total_sectors;

	partition_size_high = size >> 16;
	temp=partition_size_high << 16;
	if(size > 0xffff)
	  {
	  partition_size_low = size - temp;
	  }
	else
	  {
	  partition_size_low = size;
	  }

	Convert_Long_To_Integer(partition_size_low);

	sector_buffer[partition_table_pointer+0x0c]=integer1;
	sector_buffer[partition_table_pointer+0x0d]=integer2;

	Convert_Long_To_Integer(partition_size_high);

	sector_buffer[partition_table_pointer+0x0e]=integer1;
	sector_buffer[partition_table_pointer+0x0f]=integer2;

	/* Set the relative sector field in the partition table. */
	if(create_partition_type==PRIMARY)
	  {
	  sector_buffer[partition_table_pointer+0x08]=total_sectors;
	  }
	else
	  {
	  int secondary_index=0;
	  int primary_partition_number=-1;

	  /* Which partition is the primary partition? */
	  do
	    {
	    if(0x01==numeric_partition_type[secondary_index]) primary_partition_number=secondary_index;
	    if(0x04==numeric_partition_type[secondary_index]) primary_partition_number=secondary_index;
	    if(0x06==numeric_partition_type[secondary_index]) primary_partition_number=secondary_index;

	    secondary_index++;
	    }while(primary_partition_number==-1);

	  relative_sectors=(cp_starting_cylinder-starting_cylinder[primary_partition_number])*( (total_heads+1) * (total_sectors) );

	  relative_sectors_high = relative_sectors >> 16;
	  temp=relative_sectors_high << 16;

	  if(relative_sectors > 0xffff)
	    {
	    relative_sectors_low = relative_sectors - temp;
	    }
	  else
	    {
	    relative_sectors_low = relative_sectors;
	    }

	  Convert_Long_To_Integer(relative_sectors_low);

	  sector_buffer[partition_table_pointer+0x08]=integer1;
	  sector_buffer[partition_table_pointer+0x09]=integer2;

	  Convert_Long_To_Integer(relative_sectors_high);

	  sector_buffer[partition_table_pointer+0x0a]=integer1;
	  sector_buffer[partition_table_pointer+0x0b]=integer2;
	  }

	sector_buffer[0x1fe]=0x55;
	sector_buffer[0x1ff]=0xaa;

	Write_Sector(drive,0,0,1);

	/* Clear boot sector of newly created partition */
	Clear_Sector_Buffer();
	Write_Sector(drive,cp_starting_head,cp_starting_cylinder,1);

	partition_created=TRUE;
	}
      index++;
      } while((index<4) && (partition_created==FALSE));
    }

  if(((create_partition_type==LOGICAL) && (extended_partition_exists==TRUE)) && error_encountered==FALSE)
    {

    /* If this is not the first logical drive,   */
    /* create the extended entry in the previous */
    /* logical drive table.                      */
    if(number_of_partitions>=5)
      {
      Read_Sector(drive,extended_partition_head[number_of_partitions-4],extended_partition_cylinder[number_of_partitions-4],extended_partition_sector[number_of_partitions-4]);

      partition_table_pointer=0x1be+16;

      /* Set the 2nd partition entry in this table to extended */
      sector_buffer[partition_table_pointer+0x04]=0x05;

      Extract_Cylinder_and_Sector(sector_buffer[(0x1be+0x06)],sector_buffer[(0x1be+0x07)]);

      /* Save the location of the next partition */
      extended_partition_cylinder[number_of_partitions-3]=g_cylinder+1;
      extended_partition_sector[number_of_partitions-3]=1;
      extended_partition_head[number_of_partitions-3]=0;

      if(extended_partition_cylinder[number_of_partitions-3]>total_cylinders)
	{
	if(menu_in_use==TRUE) Clear_Screen();

	printf("\nPartition cannot be created past end of drive...Operation Terminated.\n");

	if(menu_in_use==TRUE)
	  {
	  error_encountered=TRUE;
	  Pause();
	  Clear_Screen();
	  }
	else
	  {
	  exit(1);
	  }
	}

      /* Add the logical drive pointer information to the sector buffer */
      sector_buffer[partition_table_pointer+0x01]=0;

      temp=Combine_Cylinder_and_Sector((extended_partition_cylinder[number_of_partitions-3]),1);

      Convert_Long_To_Integer(temp);

      sector_buffer[partition_table_pointer+0x02]=integer1;
      sector_buffer[partition_table_pointer+0x03]=integer2;

      /* Compute the end of the partition */
      Calculate_Partition_Ending_Cylinder((extended_partition_cylinder[number_of_partitions-3]),size);

      sector_buffer[partition_table_pointer+0x05]=total_heads;

      temp=Combine_Cylinder_and_Sector(computed_ending_cylinder-1,total_sectors);

      Convert_Long_To_Integer(temp);

      sector_buffer[partition_table_pointer+0x06]=integer1;
      sector_buffer[partition_table_pointer+0x07]=integer2;

      /* Compute the "relative sectors" */
      relative_sectors=(extended_partition_cylinder[number_of_partitions-3]-starting_cylinder[4])*((total_heads+1)*(total_sectors));

      relative_sectors_high = relative_sectors >> 16;
      temp=relative_sectors_high << 16;

      if(relative_sectors > 0xffff)
	{
	relative_sectors_low = relative_sectors - temp;
	}
      else
	{
	relative_sectors_low = relative_sectors;
	}

      Convert_Long_To_Integer(relative_sectors_low);

      sector_buffer[partition_table_pointer+0x08]=integer1;
      sector_buffer[partition_table_pointer+0x09]=integer2;

      Convert_Long_To_Integer(relative_sectors_high);

      sector_buffer[partition_table_pointer+0x0a]=integer1;
      sector_buffer[partition_table_pointer+0x0b]=integer2;

      /* Compute the partition size */
      extended_partition_size=((computed_ending_cylinder)-extended_partition_cylinder[number_of_partitions-3])*( (total_heads+1)*(total_sectors)  );

      extended_partition_size_high = extended_partition_size >> 16;
      temp=extended_partition_size_high << 16;

      if(extended_partition_size > 0xffff)
	{
	extended_partition_size_low = extended_partition_size - temp;
	}
      else
	{
	extended_partition_size_low = extended_partition_size;
	}

      Convert_Long_To_Integer(extended_partition_size_low);

      sector_buffer[partition_table_pointer+0x0c]=integer1;
      sector_buffer[partition_table_pointer+0x0d]=integer2;

      Convert_Long_To_Integer(extended_partition_size_high);

      sector_buffer[partition_table_pointer+0x0e]=integer1;
      sector_buffer[partition_table_pointer+0x0f]=integer2;

      /* Write Sector */
      if(error_encountered==FALSE) Write_Sector(drive,extended_partition_head[number_of_partitions-4],extended_partition_cylinder[number_of_partitions-4],extended_partition_sector[number_of_partitions-4]);
      }

    /* Get the next available partition to use */
    next_available_partition=number_of_partitions-3;

    /* Compute the beginning of the partition */
    cp_starting_cylinder=extended_partition_cylinder[next_available_partition];

    /* Compute the end of the partition */
    Calculate_Partition_Ending_Cylinder(cp_starting_cylinder,size);

    Clear_Sector_Buffer();

    partition_table_pointer=0x1be;

    /* Create partition table */
    temp=Combine_Cylinder_and_Sector(cp_starting_cylinder,1);

    Convert_Long_To_Integer(temp);

    sector_buffer[partition_table_pointer+0x02]=integer1;
    sector_buffer[partition_table_pointer+0x03]=integer2;

    sector_buffer[partition_table_pointer+0x01]=1;

    if(special_flag!=SPECIAL)
      {
      if(size>0xffff) sector_buffer[partition_table_pointer+0x04]=0x06;
      if(size<=0xffff) sector_buffer[partition_table_pointer+0x04]=0x04;
      if(size<=4096) sector_buffer[partition_table_pointer+0x04]=0x01;
      if( (FAT32==TRUE) && (size>1048576) )sector_buffer[partition_table_pointer+0x04]=0x0b;
      }
    else
      {
      sector_buffer[partition_table_pointer+0x04]=special_partition_type;
      }

    temp=Combine_Cylinder_and_Sector((computed_ending_cylinder-1),total_sectors);

    Convert_Long_To_Integer(temp);

    sector_buffer[partition_table_pointer+0x06]=integer1;
    sector_buffer[partition_table_pointer+0x07]=integer2;

    sector_buffer[partition_table_pointer+0x05]=total_heads;

    /* Adjust size to fit inside partition */
    size=(((computed_ending_cylinder)-cp_starting_cylinder)*( (total_heads+1) * (total_sectors) )) - total_sectors;

    partition_size_high = size >> 16;
    temp=partition_size_high << 16;
    if(size > 0xffff)
      {
      partition_size_low = size - temp;
      }
    else
      {
      partition_size_low = size;
      }

    Convert_Long_To_Integer(partition_size_low);

    sector_buffer[partition_table_pointer+0x08]=total_sectors;

    sector_buffer[partition_table_pointer+0x0c]=integer1;
    sector_buffer[partition_table_pointer+0x0d]=integer2;

    Convert_Long_To_Integer(partition_size_high);

    sector_buffer[partition_table_pointer+0x0e]=integer1;
    sector_buffer[partition_table_pointer+0x0f]=integer2;

    sector_buffer[0x1fe]=0x55;
    sector_buffer[0x1ff]=0xaa;

    if(error_encountered==FALSE) Write_Sector(drive,0,cp_starting_cylinder,1);

    /* Clear "boot" sector of new partition */
    Clear_Sector_Buffer();
    if(error_encountered==FALSE) Write_Sector(drive,1,cp_starting_cylinder,1);
    }

  if((create_partition_type==LOGICAL) && (extended_partition_exists==FALSE) && error_encountered==FALSE)
    {
    if(menu_in_use==TRUE) Clear_Screen;

    printf("\nExtended partition non-existant...operation terminated.\n");

    if(menu_in_use==TRUE)
      {
      error_encountered=TRUE;
      Pause();
      Clear_Screen();
      }
    else
      {
      exit(1);
      }
    }
}

/* Convert Hexidecimal Number to a Decimal Number */
unsigned long Decimal_Number(unsigned long hex1, unsigned long hex2, unsigned long hex3, unsigned long hex4)
{
  return((hex1) + (hex2*256) + (hex3*65536) + (hex4*16777216));
}

/* De-Allocate Memory For Buffers */
void DeAllocate_Memory_For_Buffers()
{
  free(partition_lookup_table_buffer);
}

/* Delete Primary Partition or Extended Partition Table */

/* Returns TRUE if successful, FALSE if not successful */

int Delete_P_Or_E_Partition(int drive,int partition_type)
{
  long index=0;
  int partition_number_to_delete;
  int partition_found=FALSE;
  int return_code=FALSE;

  /* Get primary partition table */
  Read_Partition_Table(drive,TECHNICAL);

  /* Find partition */
  do
    {
    if(partition_type==PRIMARY)
      {
      if((0x06==numeric_partition_type[index]) || (0x04==numeric_partition_type[index]) || (0x01==numeric_partition_type[index]) || (0x0b==numeric_partition_type[index]) || (0x0c==numeric_partition_type[index]) || (0x0e==numeric_partition_type[index]) )
	{
	partition_number_to_delete=index;
	partition_found=TRUE;
	}
      }
    if(partition_type==EXTENDED)
      {
      if( (0x05==numeric_partition_type[index]) || (0x0f==numeric_partition_type[index]) )
	{
	partition_number_to_delete=index;
	partition_found=TRUE;
	}
      }

    index++;
    }while((index<4) && (partition_found==FALSE));


  /* Delete partition--if it has been found */
  if(partition_found==TRUE)
    {
    Delete_Partition(drive,(partition_number_to_delete+1));
    return_code=TRUE;
    }

  return(return_code);
}

/* Delete Partition */
void Delete_Partition(int drive,long partition_number)
{
  int loop=0;

  long offset;

  if(partition_number==LAST)
    {
    Read_Partition_Table(drive,TECHNICAL);

    if(number_of_partitions==5)
      {
      Clear_Sector_Buffer();
      Write_Sector(drive,extended_partition_head[1],extended_partition_cylinder[1],extended_partition_sector[1]);
      }
    else
      {
      Read_Sector(drive,extended_partition_head[number_of_partitions-5],extended_partition_cylinder[number_of_partitions-5],extended_partition_sector[number_of_partitions-5]);

      offset=(0x1be+16);

      do
	{
	sector_buffer[offset+loop]=0x00;
	loop++;
	}while(loop < 16);

      Write_Sector(drive,extended_partition_head[number_of_partitions-5],extended_partition_cylinder[number_of_partitions-5],extended_partition_sector[number_of_partitions-5]);
      }
    }
  else
    {
    Read_Sector(drive,0,0,1);

    offset=((partition_number*16)-16)+0x1be;

    do
      {
      sector_buffer[offset+loop]=0x00;
      loop++;
      }while(loop < 16);

    Write_Sector(drive,0,0,1);
    }
}

/* Determine if the video display will support boldfacing text */
void Determine_Color_Video_Support()
{
  asm{
    mov bx,es:0x10
    and bx,0x30
    cmp bx,0x30
    jne color
    }

  video=MONOCHROME;
  goto type_determined;

  color:
  video=COLOR;

  type_determined:

  if(video==COLOR)
    {
    textcolor(15);
    }
  else
    {
    textcolor(7);
    }
}

/* Display Help Screens */
void Display_Help_Screen()
{
  printf("\n\n%10s                                                         Version %s\n", NAME, VERSION);
  printf("Written By:  Brian E. Reifsnyder\n\n");
  printf("\n");
  printf("Syntax:\n\n");
  printf("FDISK\n");
  printf("FDISK [/MONO]\n");
  printf("FDISK [/A [drive#] ]\n");
  printf("FDISK [/N drive# partition_type size [/P] [/S type#] ]\n");
  printf("FDISK [/D drive# {partition# || /L || /P || /E} ]\n");
  printf("FDISK [/I [drive# [/tech] ]\n");
  printf("FDISK [/L]\n");
  printf("FDISK [/MBR [drive#] ]\n");
  printf("FDISK [/RMBR [drive#] ]\n");
  printf("FDISK [/SMBR [drive#] ]\n");
  printf("FDISK [/AMBR [drive#] ]\n");
  printf("FDISK [/C drive#]\n");
  printf("FDISK [/T drive# partition# active_toggle]\n");
  printf("FDISK [/M drive# primarypartition# newtype#]\n");
  printf("FDISK [/DUMP]\n");
  Pause();
  Clear_Screen();
  printf("  /MONO   Runs FDISK in Monochrome mode.  (For users with monochrome monitors.)\n");
  printf("  /A      Automatically partitions drive#.  If no drive# is specified, the\n");
  printf("            program will automatically partition the first drive.  If a primary\n");
  printf("            and/or extended partition exists this function will abort.\n");
  printf("  /N      Creates a new partition based upon the drive# (1-4), type (P, E,\n");
  printf("            or L), and size (in megabytes).  If the /P switch is added,\n");
  printf("            the size is a percentage of the total drive space.  If the /S\n");
  printf("            switch is used, then a type# partition is created.\n");
  printf("  /D      Deletes a partition based upon the drive# (1-4) and the partition\n");
  printf("            number, the last logical partition (/L), the primary partition (/P)\n");
  printf("            or the extended partition (/E).\n");
  printf("  /I      Displays the partition table for drive# (1-4).  If the \"/tech\"\n");
  printf("            switch is used, it also shows the starting and ending information\n");
  printf("            for each partition.\n");
  Pause();
  Clear_Screen();
  printf("  /L      Lists the available hard disks and their sizes.\n");
  printf("  /MBR    Writes the Master Boot Record to drive#.\n");
  printf("  /RMBR   Removes the Master Boot Record from drive#.\n");
  printf("  /SMBR   Saves the current Master Boot program to the \"BOOT.MBR\" file.\n");
  printf("  /AMBR   Writes the Master Boot program in \"BOOT.MBR\" to drive#.\n");
  printf("  /C      Clears the partition sector.  (Use with extreme caution!)\n");
  printf("  /T      Toggles the partition# as active or not active (1 or 0).\n");
  printf("  /M      Modifies the partition type of primarypartition# to\n");
  printf("            newtype#.\n");
  printf("  /DUMP   Dumps technical partition information on all installed hard disks\n");
  printf("            to the screen.  (The output can optionally be redirected with the\n");
  printf("            standard redirection commands. (i.e. \"FDISK /DUMP > filename.ext\") )");
  Pause();
  Clear_Screen();
  printf("\nCopyright 1999 under the terms of the GNU General Public License.\n");
  printf("\nThis program comes as-is and without warranty of any kind.  The author of this\n");
  printf("software assumes no responsibility pertaining to the use or mis-use of this\n");
  printf("software.  By using this software, the operator is understood to be agreeing to\n");
  printf("the terms of the above.\n");
}

/* Display Partition Table */
void Display_Partition_Table(int drive,int tech_flag)
{
  char table[9];
  char view[9];

  int partition_number=0;
  unsigned long total_hard_disk_size;

  /* Set the lookup table type for display */
  if(partition_lookup_table==EXTERNAL) strcpy(table,"External");
  else strcpy(table,"Internal");

  /* Set the view type variable */
  if(tech_flag==STANDARD)
    {
    strcpy(view,"Standard ");
    }
  else
    {
    strcpy(view,"Technical");
    }


  /* Get partition table */
  Read_Partition_Table(drive,tech_flag);

  /* Display partition table */
  if(show_primary_partition_table==TRUE)
    {
    if( (menu_in_use==TRUE) && ((numeric_partition_type[0]+numeric_partition_type[1]+numeric_partition_type[2]+numeric_partition_type[3])==0) )
      {
      /* do nothing */
      }
    else
      {
      printf("\nPartition:  Primary              | View:  %9s | Lookup Table: %9s |\n",view,table);
      printf("-------------------------------------------------------------------------------\n");

      if(tech_flag==TECHNICAL)
	{
	printf(" #  A   Type             Starting Location   Ending Location     Partition Size\n");
	printf("                         Cyl   Head  Sect    Cyl   Head  Sect     in Megabytes\n\n");
	}
      else
	{
	printf(" #  A   Type                                                     Partition Size\n");
	printf("                                                                  in Megabytes\n\n");
	}
      }

    do
      {
      if( (menu_in_use==TRUE) && (numeric_partition_type[partition_number]==0) )
	{
	/* Do not print anything */
	}
      else
	{
	/* Print partition number */
	cprintf(" %d",(partition_number+1));

	/* Is partition a boot partition? */
	if(active_flag[partition_number]==TRUE)
	  {
	  printf("  *   ");
	  }
	else
	  {
	  printf("      ");
	  }

	if(tech_flag==TECHNICAL)
	  {
	  /* Display Partition Type */
	  printf(partition_type[partition_number]);

	  /* Display Partition Mappings */
	  if(0!=numeric_partition_type[partition_number])
	    {
	    printf("    %4d",starting_cylinder[partition_number]);
	    printf("  %4d",starting_head[partition_number]);
	    printf("  %4d",starting_sector[partition_number]);

	    printf("    %4d",ending_cylinder[partition_number]);
	    printf("  %4d",ending_head[partition_number]);
	    printf("  %4d",ending_sector[partition_number]);

	    printf("       %6d",partition_size[partition_number]);
	    }
	  }
	else
	  {
	  if(partition_lookup_table==INTERNAL)
	    {
	    printf(partition_type[partition_number]);
	    printf("                                        ");
	    }
	  else
	    {
	    printf(partition_type[partition_number]);
	    printf("   ");
	    }

	  /* If partition is used, print the size */
	  if(0!=numeric_partition_type[partition_number])
	    {
	    printf("       %6d",partition_size[partition_number]);
	    }
	  }

	/* Carriage Return */
	printf("\n");
	}

      partition_number++;
      } while(partition_number<4);
    }

  if( (menu_in_use==TRUE) && ((numeric_partition_type[0]+numeric_partition_type[1]+numeric_partition_type[2]+numeric_partition_type[3])==0) )
    {
    /* Do nothing */
    }
  else
    {
    if(show_primary_partition_table==TRUE) printf("-------------------------------------------------------------------------------\n");
    }

  if((number_of_partitions>4) && (show_extended_partition_table==TRUE))
    {
    partition_number=4;

    printf("Partition:  Extended\n");
    printf("-------------------------------------------------------------------------------\n");

    if(tech_flag==TECHNICAL)
      {
      printf(" #      Type             Starting Location   Ending Location     Partition Size\n");
      printf("                         Cyl   Head  Sect    Cyl   Head  Sect     in Megabytes\n\n");
      }
    else
      {
      printf(" #      Type                                                     Partition Size\n");
      printf("                                                                  in Megabytes\n\n");
      }

    do{

      /* Print partition number */
      cprintf(" %d",(partition_number+1));
      printf("      ");

      if(tech_flag==TECHNICAL)
	{
	/* Display Partition Type */
	printf(partition_type[partition_number]);

	/* Display Partition Mappings */
	if(0!=strcmp(partition_type[partition_number],"Unused       "))
	  {
	  printf("    %4d",starting_cylinder[partition_number]);
	  printf("  %4d",starting_head[partition_number]);
	  printf("  %4d",starting_sector[partition_number]);

	  printf("    %4d",ending_cylinder[partition_number]);
	  printf("  %4d",ending_head[partition_number]);
	  printf("  %4d",ending_sector[partition_number]);

	  printf("       %6d",partition_size[partition_number]);
	  }
	}
      else
	{
	if(partition_lookup_table==INTERNAL)
	  {
	  printf(partition_type[partition_number]);
	  printf("                                        ");
	  }
	else
	  {
	  printf(partition_type[partition_number]);
	  printf("   ");
	  }

	/* If the partition is used, print the size */
	if(0!=numeric_partition_type[partition_number])
	  {
	  printf("       %6d",partition_size[partition_number]);
	  }
	}

      /* Carriage Return */
      printf("\n");

      partition_number++;
      }while(partition_number<number_of_partitions);
    printf("-------------------------------------------------------------------------------\n");
    }

  if( (menu_in_use==TRUE) && ((numeric_partition_type[0]+numeric_partition_type[1]+numeric_partition_type[2]+numeric_partition_type[3])==0) )
    {
    Position_Cursor(0,21);
    cprintf("No partitions defined");
    }
  else
    {
    Get_Hard_Drive_Parameters(drive);
    total_hard_disk_size=(total_cylinders+1)*(total_heads+1)*total_sectors;
    total_hard_disk_size=total_hard_disk_size/2000;
    printf("Total hard disk size:  %6d MB",total_hard_disk_size);
    }
}

/* Dump the partition tables from all drives to screen */
void Dump_Partition_Information()
{
  unsigned char temp_sector_buffer[512];

  int drive=128;
  int loop=128;
  int result;

  int maximum_hard_drive=128;

  /* Find the maximum number of hard drives. */
  do
    {
    result=biosdisk(2, loop, 0, 0, 1, 1, temp_sector_buffer);

    if (result==0)
      {
      maximum_hard_drive++;
      }

    loop++;
    }while(loop<=131);

  /* Print the partition tables for all the drives */
  char table[9];
  char view[9];

  int partition_number=0;
  unsigned long total_hard_disk_size;

  /* Set the lookup table type */
  partition_lookup_table=INTERNAL;

  do{
    /* Get partition table */
    Read_Partition_Table(drive,TECHNICAL);

    /* Get hard drive parameters */
    Get_Hard_Drive_Parameters(drive);

    /* Display hard disk number */
    printf("\nHard drive:  %d   ",drive-127);
    printf("(TC:%3d",total_cylinders);
    printf(" TH:%3d",total_heads);
    printf(" TS:%3d)",total_sectors);

    /* Display partition table */
    printf("\nPartition:  Primary\n");

    do
      {
      /* Print partition number */
      printf(" %d  ",(partition_number+1));

      /* Is partition a boot partition? */
      if(active_flag[partition_number]==TRUE)
	{
	printf(" Active ");
	}
      else
	{
	printf("        ");
	}

      /* Display Partition Type */
      printf(" %s ",partition_type[partition_number]);
      printf(" Numeric:%3d",numeric_partition_type[partition_number]);

      /* Display Partition Mappings */
      if(0!=numeric_partition_type[partition_number])
	{
	printf("\n             (SC:%3d",starting_cylinder[partition_number]);
	printf(" SH:%3d",starting_head[partition_number]);
	printf(" SS:%3d)",starting_sector[partition_number]);

	printf(" (EC:%3d",ending_cylinder[partition_number]);
	printf(" EH:%3d",ending_head[partition_number]);
	printf(" ES:%3d)",ending_sector[partition_number]);

	printf("   SIZE:%6u",partition_size[partition_number]);
	}

      /* Carriage Return */
      printf("\n");

      partition_number++;
      } while(partition_number<4);

    if(number_of_partitions>4)
      {
      partition_number=4;

      printf("Partition:  Extended\n");

      do{

	/* Print partition number */
	printf(" %d",(partition_number+1));

	/* Display Partition Mappings */
	if(0!=strcmp(partition_type[partition_number],"Unused       "))
	  {
	  printf("           %s ",partition_type[partition_number]);
	  printf(" Numeric:%3d ",numeric_partition_type[partition_number]);

	  printf("\n             (SC:%3d",starting_cylinder[partition_number]);
	  printf(" SH:%3d",starting_head[partition_number]);
	  printf(" SS:%3d)",starting_sector[partition_number]);

	  printf(" (EC:%3d",ending_cylinder[partition_number]);
	  printf(" EH:%3d",ending_head[partition_number]);
	  printf(" ES:%3d)",ending_sector[partition_number]);

	  printf("   SIZE:%6u",partition_size[partition_number]);
	  }

	/* Carriage Return */
	printf("\n");

	partition_number++;
	}while(partition_number<number_of_partitions);
      }

    total_hard_disk_size=(total_cylinders+1)*(total_heads+1)*total_sectors;
    total_hard_disk_size=total_hard_disk_size/2000;
    printf("Total hard drive size:  %d\n",total_hard_disk_size);

    drive++;
    partition_number=0;
    }while(drive<maximum_hard_drive);
}

/* Does "part.dat" exist? */
int External_Lookup_File_Exists()
{
  int result=NULL;

  FILE *file;

  file=fopen("part.dat","rt");

  if(!file)
    {
    result=FALSE;
    }
  else
    {
    result=TRUE;
    fclose(file);
    }

  return(result);
}

/* Extract Cylinder & Sector */
void Extract_Cylinder_and_Sector(unsigned long hex1, unsigned long hex2)
{
  unsigned long cylinder_and_sector = ( (256*hex2) + hex1 );

  g_sector = cylinder_and_sector % 64;

  g_cylinder = ( ( (cylinder_and_sector*4) & 768) + (cylinder_and_sector /256) );
}


/* Get Hard Drive Parameters */
void Get_Hard_Drive_Parameters(int drive)
{
  asm{
    mov ah, 0x08
    mov dl, BYTE PTR drive
    int 0x13

    mov bl,cl
    and bl,00111111B

    mov BYTE PTR total_sectors, bl

    mov bl,cl
    mov cl,ch
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1

    mov ch,bl

    mov WORD PTR total_cylinders, cx
    mov BYTE PTR total_heads, dh
    }
}

/* Initialize Variables */
void Initialize_Variables()
{
  int loop=0;

  do
    {
    active_flag[loop]=FALSE;

    loop++;
    }while(loop<=4);

  extended_partition_exists=FALSE;

  error_encountered=FALSE;
}

/* Get input from keyboard */
char Input()
{
  char input;

  asm{
    mov ah,7
    int 0x21
    mov BYTE PTR input,al
    }

  return(input);
}

/* List Available Hard Drives */
void List_Available_Drives()
{
  int loop=128;
  int result;
  unsigned long total_hard_disk_size;

  unsigned char temp_sector_buffer[512];

  printf("\n  Drive              Size\n");
  printf("  Number          (Megabytes)\n\n");
  do
    {
    result=biosdisk(2, loop, 0, 0, 1, 1, temp_sector_buffer);

    if (result==0)
      {
      Get_Hard_Drive_Parameters(loop);
      total_hard_disk_size=(total_cylinders+1)*(total_heads+1)*total_sectors;
      total_hard_disk_size=total_hard_disk_size/2000;

      if(menu_in_use==TRUE)
	{
	cprintf("    %d",(loop-127));
	printf("                %d",total_hard_disk_size);
	printf("\n");
	}
      else
	{
	printf("    %d                %d\n",(loop-127),total_hard_disk_size);
	}
      total_number_of_hard_drives=loop-127;
      }

    loop++;
    }while(loop<=170);
  printf("\n\n");
}

/* Load External Partition Type Lookup Table */
void Load_External_Lookup_Table()
{
  int index=0;
  int end_of_file_marker_encountered=FALSE;
  int offset=0;

  FILE *file;

  char line_buffer[256];

  file=fopen("part.dat","rt");

  if(!file)
    {
    printf("\nThe \"part.dat\" file is not found...Operation Terminated.\n");
    exit(1);
    }

  while(fgets(line_buffer,255,file) !=NULL)
    {
    if( (0!=strncmp(line_buffer,";",1)) && (0!=strncmp(line_buffer,"999",3)) && (end_of_file_marker_encountered==FALSE) )
      {
      offset=0;
      do
	{
	partition_lookup_table_buffer[ ((index*80)+offset) ]=line_buffer[offset];
	offset++;
	} while(offset<=79);

      index++;
      }
    if(0==strncmp(line_buffer,"999",3)) end_of_file_marker_encountered=TRUE;
    }
  fclose(file);
}

/* Menu Routine */
void Menu_Routine()
{
  int exit=FALSE;
  int exit_sub_menu=FALSE;
  int drive=128;
  int show_version_information=TRUE;

  long partition_number;

  int result;
  int stored_drive;
  int temp_sector_buffer[512];
  int more_than_one_hard_disk=FALSE;

  int partition_to_delete=NULL;

  char input;
  char input_buffer[10];
  char output_buffer[30];

  unsigned long total_hard_disk_size;
  unsigned long hard_disk_space_available;
  unsigned long extended_partition_space_available;

  menu_in_use=TRUE;

  if(TRUE==More_Than_One_Hard_Disk()) more_than_one_hard_disk=TRUE;

  do
    {
    Initialize_Variables();
    Clear_Screen();

    if(show_version_information==TRUE)
      {
      Position_Cursor(34,0);
      printf("Version %s\n",VERSION);
      Position_Cursor(28,1);
      printf("Fixed Disk Setup Program\n");
      Position_Cursor(22,2);
      printf("Copyright 1999 by Brian E. Reifsnyder\n");
      Position_Cursor(26,3);
      printf("under the terms of the GNU GPL\n\n");
      }

    Position_Cursor(33,5);
    cprintf("FDISK Options");
    printf("\n\n    Current fixed disk drive: ");
    cprintf("%d",(drive-127));
    printf("\n\n    Choose one of the following:\n\n");
    cprintf("    1.");
    printf(" Create DOS partition or Logical DOS Drive\n");
    cprintf("    2.");
    printf(" Set active partition\n");
    cprintf("    3.");
    printf(" Delete partition or Logical DOS Drive\n");
    cprintf("    4.");
    printf(" Display partition information\n");

    if(more_than_one_hard_disk==TRUE)
      {
      cprintf("    5.");
      printf(" Change current fixed disk drive\n");
      }
    else
      {
      printf("\n");
      }

    printf("\n\n");
    printf("    Enter choice: ");
    cprintf("[");

    Position_Cursor(4,24);
    printf("Press ");
    cprintf("Esc");
    printf(" to exit FDISK");
    Position_Cursor(20,18);
    cprintf("]");
    Position_Cursor(19,18);

    /* Get input from keyboard. */
    input=Input();

    /* Create DOS partition or Logical DOS Drive */
    if(input==49)
      {
      int partition_type_to_create;

      char partition_label[9];

      exit_sub_menu=FALSE;
      show_version_information=FALSE;

      do
	{
	Clear_Screen();
	Position_Cursor(20,5);
	cprintf("Create DOS Partition or Logical DOS Drive");
	printf("\n\n    Current fixed disk drive: ");
	cprintf("%d",(drive-127));
	printf("\n\n    Choose one of the following:\n\n");
	cprintf("    1.");
	printf(" Create Primary DOS Partition\n");
	cprintf("    2.");
	printf(" Create Extended DOS Partition\n");
	cprintf("    3.");
	printf(" Create Logical DOS Drive(s) in the Extended DOS Partition\n\n\n\n\n");
	printf("    Enter choice: ");
	cprintf("[");

	Position_Cursor(4,24);
	printf("Press ");
	cprintf("Esc");
	printf(" to return to FDISK Options");
	Position_Cursor(20,18);
	cprintf("]");
	Position_Cursor(19,18);

	input=Input();

	partition_type_to_create=EXTENDED;

	/* If applicable, set partition_type_to_create to PRIMARY */
	if(input==49)
	  {
	  input=50;
	  partition_type_to_create=PRIMARY;
	  strcpy(partition_label,"Primary");
	  }

	/* Create Primary or Extended DOS Partition */
	if(input==50)
	  {
	  Create_Extended_DOS_Partition:
	  int skip_partition_creation=FALSE;
	  int skip_to_end=FALSE;
	  int percentage_flag=NULL;
	  int true=1;

	  long partition_size;

	  if(partition_type_to_create==EXTENDED) strcpy(partition_label,"Extended");

	  Read_Partition_Table(drive,TECHNICAL);

	  Clear_Screen();
	  Position_Cursor(23,1);
	  cprintf("Create %s DOS Partition",partition_label);

	  if( (partition_type_to_create==PRIMARY) && (primary_partition_exists==TRUE) ) skip_partition_creation=TRUE;
	  if( (partition_type_to_create==EXTENDED) && (extended_partition_exists==TRUE) ) skip_partition_creation=TRUE;

	  if( (partition_type_to_create==EXTENDED) && (primary_partition_exists==FALSE) ) skip_partition_creation=TRUE;

	  if(skip_partition_creation==FALSE)
	    {
	    Get_Hard_Drive_Parameters(drive);
	    total_hard_disk_size=(total_cylinders+1)*(total_heads+1)*total_sectors;
	    total_hard_disk_size=total_hard_disk_size/2000;
	    hard_disk_space_available=total_hard_disk_size-(hard_disk_space_used/2000);

	    printf("\n\n  Current fixed disk drive: ");
	    cprintf("%d",(drive-127));

	    if(partition_type_to_create==PRIMARY)
	      {
	      printf("\n\n  Do you wish to use the maximum available size for a Primary DOS Partition\n");
	      printf("  and make the partition active (Y/N).....................? ");
	      input=Input();

	      if( (input=='Y') || (input=='y') )
		{
		printf("Y");
		automatic=PRIMARY;
		Automatically_Partition_Hard_Drive(drive);
		skip_to_end=TRUE;
		}
	      else
		{
		printf("N\n");
		}
	      }
	    else
	      {
	      printf("\n");
	      }

	    if(skip_to_end==FALSE)
	      {
	      printf("\n  Total space available for %s DOS partition:  ",partition_label);
	      cprintf("%d MB",hard_disk_space_available);
	      printf("\n\n  Enter the partition size as either the size (in megabytes) of the partition\n");
	      printf("  that you want to create or as a percentage of the total space available\n");
	      printf("  for the partition.\n");
	      printf("\n  Enter %s DOS Partition Size: ",partition_label);
	      fgets(input_buffer,10,stdin);

	      partition_size=atol(input_buffer);

	      /* trap bad input */
	      if(partition_size<=0)
		{
		partition_size=5;
		}

	      if( (partition_size>2048) && (partition_type_to_create==PRIMARY) && (FAT32==FALSE) )
		{
		printf("\n\n  Entered partition size is too large (> 2048)...Operation Terminated.\n");
		}
	      else
		{
		if(partition_size<=100)
		  {
		  printf("\n  Is the partition size entered a percentage of the total\n");
		  printf("  space available for the %s DOS partition (Y/N)?",partition_label);
		  fgets(input_buffer,10,stdin);

		  true=1;
		  true=stricmp(input_buffer,"Y\n");
		  if(0==true)
		    {
		    percentage_flag=PERCENTAGE;
		    }
		  else
		  /* trap bad input */
		    {
		    percentage_flag=NULL;
		    }
		  }

		Create_New_Partition(drive,partition_type_to_create,partition_size,percentage_flag,NULL,NULL);
		}

	      }
	      Clear_Screen();
	      Position_Cursor(0,21);

	      if(partition_type_to_create==PRIMARY) cprintf("Primary DOS Partition created, drive letters changed or added");
	      else cprintf("Extended DOS Partition created");
	    }
	  else
	    {
	    if( (partition_type_to_create==EXTENDED) && (primary_partition_exists==FALSE) )
	      {
	      Clear_Screen();
	      Position_Cursor(0,22);
	      cprintf("No Primary DOS Partition exists.  A Primary DOS Partition must be created");
	      Position_Cursor(0,23);
	      cprintf("before an Extended DOS Partition can be created...operation terminated.");
	      }
	    else
	      {
	      Clear_Screen();
	      Position_Cursor(0,22);
	      cprintf("%s DOS Partition already exists.",partition_label);
	      }
	    }

	  Position_Cursor(23,1);
	  cprintf("Create %s DOS Partition",partition_label);
	  printf("\n\nCurrent fixed disk drive: ");
	  cprintf("%d",(drive-127));
	  printf("\n");
	  show_extended_partition_table=FALSE;
	  Display_Partition_Table(drive,STANDARD);
	  show_extended_partition_table=TRUE;

	  if(partition_type_to_create==EXTENDED)
	    {
	    Position_Cursor(0,22);
	    printf("Do you want to create Logical DOS Drives in the Extended DOS Partition (Y/N)? ");
	    input=Input();

	    if( (input=='Y') || (input=='y') )
	      {
	      printf("Y");
	      input=51;
	      }
	    else
	      {
	      printf("N");
	      Position_Cursor(0,24);
	      Pause();
	      input=99;
	      }
	    }

	  if(partition_type_to_create==PRIMARY)
	    {
	    /* Check for free space */
	    Read_Partition_Table(drive,TECHNICAL);
	    Get_Hard_Drive_Parameters(drive);

	    total_hard_disk_size = (total_cylinders+1) * (total_heads+1) * (total_sectors);
	    hard_disk_space_available=total_hard_disk_size-hard_disk_space_used;

	    /* If there is more than 20 megabytes of available space,     */
	    /* ask to create the extended partition.                      */
	    if(hard_disk_space_available>41943)
	      {
	      Position_Cursor(0,22);
	      printf("Do you want to create an Extended partition for Logical DOS Drives (Y/N)? ");
	      input=Input();

	      if( (input=='Y') || (input=='y') )
		{
		printf("Y");
		input=49;
		partition_type_to_create=EXTENDED;
		goto Create_Extended_DOS_Partition;
		}
	      else
		{
		printf("N");
		Position_Cursor(0,24);
		Pause();
		input=99;
		}
	      }
	    else
	      {
	      Position_Cursor(0,24);
	      Pause();
	      input=99;
	      }
	    }
	  }

	/* Create Logical DOS Drive(s) in the Extended DOS Partition */
	if(input==51)
	  {
	  Create_L_D_D:

	  long partition_size;

	  int percentage_flag=NULL;
	  int true=1;

	  Clear_Screen();
	  Position_Cursor(12,1);
	  cprintf("Create Logical DOS Drive(s) in the Extended DOS Partition");

	  printf("\n\n  Current fixed disk drive: ");
	  cprintf("%d",(drive-127));

	  Get_Hard_Drive_Parameters(drive);
	  extended_partition_space_available=(whole_extended_partition_size-extended_partition_space_used)/2000;

	  printf("\n\n  Total space available for Logical DOS Drive:  ");
	  cprintf("%d MB\n",extended_partition_space_available);

	  printf("\n  Enter the partition size as either the size (in megabytes) of the partition\n");
	  printf("  that you want to create or as a percentage of the total space available\n");
	  printf("  for the partition.\n");

	  printf("\n  Enter Logical DOS Drive Size: ");
	  fgets(input_buffer,10,stdin);

	  partition_size=atol(input_buffer);

	  /* trap bad input */
	  if(partition_size<=0)
	    {
	    partition_size=5;
	    strcpy(input_buffer,"null");
	    }

	  if( (partition_size>2048) && (FAT32==FALSE) )
	    {
	    printf("\n\n  Entered partition size is too large (> 2048)...Operation Terminated.\n");
	    Pause();
	    }
	  else
	    {
	    if(partition_size<=100)
	      {
	      printf("\n  Is the partition size entered a percentage of the total\n");
	      printf("  space available for the Logical DOS Drive (Y/N)?");
	      fgets(input_buffer,10,stdin);

	      true=1;
	      true=stricmp(input_buffer,"Y\n");
	      if(0==true)
		{
		percentage_flag=PERCENTAGE;
		}
	      else
		{
		percentage_flag=NULL;
		strcpy(input_buffer,"null");
		}
	      }
	    Create_New_Partition(drive,LOGICAL,partition_size,percentage_flag,NULL,NULL);
	    }

	  Clear_Screen();
	  Position_Cursor(12,1);
	  cprintf("Create Logical DOS Drive(s) in the Extended DOS Partition");

	  printf("\n\nCurrent fixed disk drive: ");
	  cprintf("%d",(drive-127));
	  printf("\n\n");
	  show_primary_partition_table=FALSE;
	  Display_Partition_Table(drive,STANDARD);
	  show_primary_partition_table=TRUE;

	  Read_Partition_Table(drive,TECHNICAL);

	  extended_partition_space_available=whole_extended_partition_size-extended_partition_space_used;

	  /* Ask to create logical drives in the extended partition */
	  /* as long as more than 10MB are available */

	  if(extended_partition_space_available>20971)
	    {
	    Position_Cursor(0,22);
	    printf("Do you want to create another Logical DOS Drive (Y/N)? ");

	    input=Input();

	    if( (input=='Y') || (input=='y') )
	      {
	      goto Create_L_D_D;
	      }
	    else
	      {
	      printf("N");
	      Position_Cursor(0,24);
	      Pause();
	      input=99;
	      }
	    }
	  else
	    {
	    Position_Cursor(0,24);
	    Pause();
	    input=99;
	    }
	  }

	/* Return to Main Menu */
	if(input==27)
	  {
	  exit_sub_menu=TRUE;
	  }

	}while(exit_sub_menu==FALSE);

      input=99; /* change value to prevent any other options from being executed */
      }

    /* Set active partition */
    if(input==50)
      {
      show_version_information=FALSE;

      int set_active_partition_menu=TRUE;
      int do_not_clear_screen_flag=FALSE;

      do{
	Initialize_Variables();

	if(do_not_clear_screen_flag==FALSE)
	  {
	  Clear_Screen();
	  }
	else
	  {
	  do_not_clear_screen_flag=FALSE;
	  }

	Position_Cursor(30,1);
	cprintf("Set Active Partition");
	printf("\n\nFixed Disk Drive: ");
	cprintf("%d",drive-127);
	show_extended_partition_table=FALSE;
	Display_Partition_Table(drive,STANDARD);
	show_extended_partition_table=TRUE;
	Position_Cursor(0,18);
	printf("Enter the number of the partition you want to make active...................");
	cprintf("[");
	Position_Cursor(78,18);
	cprintf("]");
	Position_Cursor(0,24);
	printf("Press ");
	cprintf("Esc");
	printf(" to return to FDISK Options");
	Position_Cursor(77,18);

	/* Get input from keyboard. */
	input=Input();

	partition_number=input-48;

	if(input==27) set_active_partition_menu=FALSE;

	if((partition_number>=1) && (partition_number <=(available_primary_partitions+1)))
	  {
	  Toggle_Active_Partition(drive,partition_number,1);

	  if(partition_number!=1) Toggle_Active_Partition(drive,1,0);
	  if(partition_number!=2) Toggle_Active_Partition(drive,2,0);
	  if(partition_number!=3) Toggle_Active_Partition(drive,3,0);
	  if(partition_number!=4) Toggle_Active_Partition(drive,4,0);
	  }
	else
	  {
	  if(set_active_partition_menu==TRUE)
	    {
	    Position_Cursor(0,22);
	    cprintf("%i is not a choice. Please enter 1-",partition_number);
	    cprintf("%i.",(available_primary_partitions+1));
	    do_not_clear_screen_flag=TRUE;
	    }
	  }

	}while(set_active_partition_menu==TRUE);

      input=99; /* change value to prevent any other options from being executed */
      }

    /* Delete partition or Logical DOS Drive */
    if(input==51)
      {
      show_version_information=FALSE;
      exit_sub_menu=FALSE;

      do
	{
	Read_Partition_Table(drive,TECHNICAL);

	Clear_Screen();
	Position_Cursor(19,5);
	cprintf("Delete DOS Partition or Logical DOS Drive");
	printf("\n\n    Current fixed disk drive: ");
	cprintf("%d",(drive-127));
	printf("\n\n    Choose one of the following:\n\n");
	cprintf("    1.");
	printf(" Delete Primary DOS Partition\n");
	cprintf("    2.");
	printf(" Delete Extended DOS Partition\n");
	cprintf("    3.");
	printf(" Delete Logical DOS Drive(s) in the Extended DOS Partition\n");
	cprintf("    4.");
	printf(" Delete Non-DOS Partition\n\n\n\n");
	printf("    Enter choice: ");
	cprintf("[");

	Position_Cursor(4,24);
	printf("Press ");
	cprintf("Esc");
	printf(" to return to FDISK Options");
	Position_Cursor(20,18);
	cprintf("]");
	Position_Cursor(19,18);

	input=Input();

	/* Delete Primary DOS Partition */
	if(input==49)
	  {
	  if(primary_partition_exists==TRUE)
	    {
	    partition_to_delete=PRIMARY;
	    input=50;
	    }
	  else
	    {
	    Position_Cursor(4,22);
	    cprintf("No Primary DOS Partition to delete.");

	    Position_Cursor(4,24);
	    printf("                                    ");
	    Position_Cursor(4,24);
	    Pause();

	    input=99;
	    }
	  }

	/* Delete Primary or Extended DOS Partition */
	if(input==50)
	  {
	  char type[8];

	  if(partition_to_delete==NULL)
	     {
	     partition_to_delete=EXTENDED;
	     strcpy(type,"Extended");
	     }
	  else
	     {
	     strcpy(type,"Primary");
	     }

	  if( ( (partition_to_delete==EXTENDED) && (extended_partition_exists==TRUE) ) || (partition_to_delete==PRIMARY) )
	    {
	    Clear_Screen();
	    Position_Cursor(27,1);
	    cprintf("Delete %s DOS partition",type);
	    printf("\n");

	    printf("\nCurrent fixed disk drive: ");
	    cprintf("%d",(drive-127));
	    printf("\n");

	    show_extended_partition_table=FALSE;
	    Display_Partition_Table(drive,STANDARD);
	    show_extended_partition_table=TRUE;

	    Position_Cursor(0,24);
	    printf("Press ");
	    cprintf("Esc");
	    printf(" to return to FDISK Options");

	    if(video==COLOR)
	      {
	      textcolor(WHITE|BLINK);
	      }
	    else
	      {
	      textcolor(7);
	      }

	    Position_Cursor(0,19);
	    cprintf("WARNING!");

	    if(video==COLOR)
	      {
	      textcolor(15);
	      }
	    else
	      {
	      textcolor(7);
	      }

	    printf(" Data in the %s DOS Partition will be lost.",type);
	    printf("\nContinue with operation (");
	    cprintf("y");
	    printf("/");
	    cprintf("n");
	    printf(")? ");
	    input=Input();

	    if( (input==89) || (input==121) )
	      {
	      result=Delete_P_Or_E_Partition(drive,partition_to_delete);

	      Clear_Screen();
	      Position_Cursor(27,1);
	      cprintf("Delete %s DOS partition",type);

	      printf("\n\nCurrent fixed disk drive: ");
	      cprintf("%d",(drive-127));
	      printf("\n");

	      show_extended_partition_table=FALSE;
	      Display_Partition_Table(drive,STANDARD);
	      show_extended_partition_table=TRUE;

	      if(result==FALSE)
		{
		/* If the partition was not deleted */
		Position_Cursor(0,22);
		cprintf("Error attempting to delete partition.");
		Position_Cursor(0,24);
		Pause();
		}
	      else
		{
		/* If the partition was deleted */
		Position_Cursor(0,22);
		cprintf("Partition deleted successfully.");
		Position_Cursor(0,24);
		Pause();
		}
	      }
	    input=99;
	    }
	  else
	    {
	    Position_Cursor(4,22);
	    cprintf("No Extended DOS Partition to delete.");

	    Position_Cursor(4,24);
	    printf("                                    ");
	    Position_Cursor(4,24);
	    Pause();

	    input=99;
	    }
	  partition_to_delete=NULL;
	  }

	/* Delete Logical DOS Drive(s) in the Extended DOS Partition */
	if(input==51)
	  {
	  if(logical_drives_exist==TRUE)
	    {
	    Clear_Screen();
	    Position_Cursor(12,1);
	    cprintf("Delete Logical DOS Drive(s) in the Extended DOS Partition");
	    Position_Cursor(0,3);

	    printf("Current fixed disk drive: ");
	    cprintf("%d",(drive-127));
	    printf("\n\n");

	    show_primary_partition_table=FALSE;
	    Display_Partition_Table(drive,STANDARD);
	    show_primary_partition_table=TRUE;

	    Position_Cursor(0,24);
	    printf("Press ");
	    cprintf("Esc");
	    printf(" to return to FDISK Options");

	    if(video==COLOR)
	      {
	      textcolor(WHITE|BLINK);
	      }
	    else
	      {
	      textcolor(7);
	      }

	    Position_Cursor(0,19);
	    cprintf("WARNING!");

	    if(video==COLOR)
	      {
	      textcolor(15);
	      }
	    else
	      {
	      textcolor(7);
	      }

	    printf(" Data in the Last Logical Drive will be lost.\n");
	    printf("Continue with operation (");
	    cprintf("y");
	    printf("/");
	    cprintf("n");
	    printf(")? ");
	    input=Input();

	    if( (input==89) || (input==121) ) /* If the user entered "y" */
	      {
	      Read_Partition_Table(drive,TECHNICAL);

	      if(number_of_partitions<5)
		{
		Position_Cursor(0,22);
		printf("No logical partitions exist...Operation Terminated.");
		Position_Cursor(0,24);
		Pause();
		}
	      else
		{
		Delete_Partition(drive,LAST);

		Clear_Screen();

		Read_Partition_Table(drive,TECHNICAL);

		Position_Cursor(12,1);
		cprintf("Delete Logical DOS Drive(s) in the Extended DOS Partition");
		Position_Cursor(0,3);

		printf("Current fixed disk drive: ");
		cprintf("%d",(drive-127));
		printf("\n\n");

		if(logical_drives_exist==TRUE)
		  {
		  show_primary_partition_table=FALSE;
		  Display_Partition_Table(drive,STANDARD);
		  show_primary_partition_table=TRUE;
		  }
		else
		  {
		  Position_Cursor(0,21);
		  cprintf("No partitions defined");
		  }

		Position_Cursor(0,24);
		printf("Press ");
		cprintf("Esc");
		printf(" to return to FDISK Options");

		Position_Cursor(0,22);
		cprintf("Partition Deleted successfully.");
		Position_Cursor(0,24);
		printf("                                    ");
		Position_Cursor(0,24);
		Pause();
		}
	      }
	    input=99;
	    }
	  else
	    {
	    Position_Cursor(4,22);
	    cprintf("No Logical DOS drive(s) to delete.");

	    Position_Cursor(4,24);
	    printf("                                    ");
	    Position_Cursor(4,24);
	    Pause();

	    input=99;
	    }
	  }

	/* Delete Non-DOS Partition */
	if(input==52)
	  {
	  int partition_number;

	  if( (numeric_partition_type[0]+numeric_partition_type[1]+numeric_partition_type[2]+numeric_partition_type[3])>0)
	    {
	    Clear_Screen();
	    Position_Cursor(28,1);
	    cprintf("Delete Non-DOS Partition");

	    printf("\n\nCurrent Fixed disk drive: ");
	    cprintf("%d",(drive-127));
	    printf("\n");

	    show_extended_partition_table=FALSE;
	    Display_Partition_Table(drive,STANDARD);
	    show_extended_partition_table=TRUE;

	    if(video==COLOR)
	      {
	      textcolor(WHITE|BLINK);
	      }
	    else
	      {
	      textcolor(7);
	      }

	    Position_Cursor(0,17);
	    cprintf("Warning!");

	    if(video==COLOR)
	      {
	      textcolor(15);
	      }
	    else
	      {
	      textcolor(7);
	      }

	    printf(" You are about to DELETE a partition.  To exit, type \"0\" and\n");
	    printf("         press [ENTER].  Otherwise, enter the partition number to delete.\n");
	    printf("\nPartition number to delete: ");
	    fgets(input_buffer,10,stdin);

	    partition_number=atoi(input_buffer);

	    if((partition_number>=1) && (partition_number<=4))
	      {
	      Delete_Partition(drive,partition_number);

	      Position_Cursor(0,22);
	      cprintf("Partition deleted successfully.");
	      Position_Cursor(0,24);
	      Pause();
	      }
	    else
	      /* trap bad input */
	      {
	      Position_Cursor(0,22);
	      cprintf("Operation Aborted.");
	      Position_Cursor(0,24);
	      Pause();
	      partition_number=0;
	      strcpy(input_buffer,"null");
	      }
	    }
	  else
	    {
	    Position_Cursor(4,22);
	    cprintf("No partitions available to delete.");

	    Position_Cursor(4,24);
	    printf("                                    ");
	    Position_Cursor(4,24);
	    Pause();

	    input=99;
	    }
	  }

	/* Return to Main Menu */
	if(input==27)
	  {
	  exit_sub_menu=TRUE;
	  }

	}while(exit_sub_menu==FALSE);
      input=99; /* change value to prevent any other options from being executed */
      }

    /* Display partition information */
    if(input==52)
      {
      show_version_information=FALSE;
      Clear_Screen();
      Position_Cursor(26,1);
      cprintf("Display Partition Information");
      printf("\n\nCurrent fixed Disk Drive: ");
      cprintf("%d",drive-127);
      Display_Partition_Table(drive,STANDARD);
      Position_Cursor(0,24);
      Pause();
      }

    /* Change current fixed disk drive */
    if((input==53) && (more_than_one_hard_disk==TRUE))
      {
      show_version_information=FALSE;

      int possible_drive;
      int exit_change_drive=FALSE;

      stored_drive=drive-126;

      Clear_Screen();
      do
	{
	Position_Cursor(24,1);
	cprintf("Change Current Fixed Disk Drive");
	printf("\n");
	List_Available_Drives();
	Position_Cursor(4,20);
	printf("(1 MByte = 1048576 bytes)");
	Position_Cursor(4,21);
	printf("Enter Fixed Disk Drive Number (1-%i).......................",total_number_of_hard_drives);
	cprintf("[ ");
	Position_Cursor(4,24);
	printf("Press ");
	cprintf("0");
	printf(" to return to FDISK Options");
	if(total_number_of_hard_drives>=10)
	  {
	  Position_Cursor(65,21);
	  cprintf("]");
	  Position_Cursor(63,21);
	  }
	else
	  {
	  Position_Cursor(64,21);
	  cprintf("]");
	  Position_Cursor(63,21);
	  }

	/* Get input from keyboard. */
	fgets(input_buffer,3,stdin);

	possible_drive=atoi(input_buffer)+1;

	/*check to make sure the drive is a legitimate number*/
	if((possible_drive<2) || (possible_drive>(total_number_of_hard_drives+1)))
	  {
	  if(possible_drive==1)
	    {
	    exit_change_drive=TRUE;
	    }
	  else
	    {
	    Position_Cursor(4,23);
	    cprintf("%i is not a choice. Please enter 1-%i.",(possible_drive-1),total_number_of_hard_drives);
	    }

	  possible_drive=stored_drive;
	  }
	else
	  {
	  exit_change_drive=TRUE;
	  }

	/* Set working drive number */
	drive=possible_drive+126;
	}while(exit_change_drive==FALSE);
      }

    /* Exit */
    if(input==27) exit=TRUE;

    }while(exit==FALSE);

  Clear_Screen();
  if(sectors_have_changed==TRUE)
    {
    Position_Cursor(11,5);
    cprintf("Reboot the computer to ensure that any changes take effect.");
    Position_Cursor(0,10);
    }
}

/* Modify Partition Type of Primary Partition */
void Modify_Partition_Type(int drive,int partition_number,int type_number)
{
  long offset;

  Read_Sector(drive,0,0,1);

  offset=((partition_number*16)-16)+(0x1be+4);

  sector_buffer[offset]=type_number;

  Write_Sector(drive,0,0,1);
}

/* Is there more than one hard disk? */
int More_Than_One_Hard_Disk()
{
  int result=1;

  unsigned char temp_sector_buffer[512];

  result=biosdisk(2, 129, 0, 0, 1, 1, temp_sector_buffer);

  if(result==0)
    {
    return(TRUE);
    }
  else
    {
    return(FALSE);
    }
}

/* Pause Routine */
void Pause()
{
  if(menu_in_use==TRUE)
    {
    printf("Press ");
    cprintf("Esc");
    printf(" to continue");
    }
  else
    {
    printf("\nPress any key to continue.\n");
    }

  asm{
    mov ah,7
    int 0x21
    }
}

/* Position cursor on the screen */
void Position_Cursor(int column,int row)
{
  asm{
    /* Get video page number */
    mov ah,0x0f
    int 0x10

    /* Position Cursor */
    mov ah,0x02
    mov dh,BYTE PTR row
    mov dl,BYTE PTR column
    int 0x10
    }
}

/* Read Partition Table */
void Read_Partition_Table(int drive,int view_type)
{
  long index=0x1be;

  int exiting_primary=TRUE;
  int extended=FALSE;
  int extended_pointer=1;
  int partition_designation=PRIMARY;
  int pointer=0;
  int record_extended_info_flag=FALSE;

  int done_flag=FALSE;

  unsigned long extended_cylinder;
  unsigned long extended_head;
  unsigned long extended_sector;

  available_primary_partitions=0;
  whole_extended_partition_size=0;
  extended_partition_space_used=0;
  hard_disk_space_used=0;

  primary_partition_exists=FALSE;
  extended_partition_exists=FALSE;
  logical_drives_exist=FALSE;

  Read_Sector(drive,0,0,1);

  do{
    if(pointer==4) partition_designation=EXTENDED;

    if((pointer>=4) && (extended==TRUE))
      {
      Read_Sector(drive,extended_head,extended_cylinder,extended_sector);
      extended=FALSE;
      index=0x1be;

      if(exiting_primary==FALSE)
	{
	pointer--;
	}
      else
	{
	exiting_primary=FALSE;
	}
      }

    /* Flag boot partition */
    if((sector_buffer[index]==128) && (partition_designation==PRIMARY)) active_flag[pointer]=TRUE;

    /* Determine Partition Type */
    numeric_partition_type[pointer]=sector_buffer[index+4];

    if(sector_buffer[index+4]==0x00)
      {
      if(partition_lookup_table==INTERNAL) strcpy(partition_type[pointer],"Unused       ");

      if(partition_designation==EXTENDED)
	{
	number_of_partitions=pointer;
	done_flag=TRUE;
	}
      }

    if( ( (sector_buffer[index+4]==0x01) || (sector_buffer[index+4]==0x04) || (sector_buffer[index+4]==0x06) || (sector_buffer[index+4]==0x0b) || (sector_buffer[index+4]==0x0c) || (sector_buffer[index+4]==0x0e) ) && (pointer<4) )
      {
      primary_partition_exists=TRUE;
      available_primary_partitions++;
      }

    if( (sector_buffer[index+4]==0x05) || (sector_buffer[index+4]==0x0f) )
      {
      if(partition_lookup_table==INTERNAL) strcpy(partition_type[pointer],"Extended     ");
      extended=TRUE;
      extended_partition_exists=TRUE;
      record_extended_info_flag=TRUE;
      }

    if(partition_lookup_table==EXTERNAL)
      {
      Scan_External_Lookup_Table(sector_buffer[index+4],pointer,view_type);
      }
    else
      {
      if(sector_buffer[index+4]==0x01) strcpy(partition_type[pointer],"DOS-12       ");
      if(sector_buffer[index+4]==0x02) strcpy(partition_type[pointer],"XENIX root   ");
      if(sector_buffer[index+4]==0x03) strcpy(partition_type[pointer],"XENIX usr    ");
      if(sector_buffer[index+4]==0x04) strcpy(partition_type[pointer],"DOS-16 <32M  ");
      if(sector_buffer[index+4]==0x06) strcpy(partition_type[pointer],"DOS-16 >=32M ");
      if(sector_buffer[index+4]==0x07) strcpy(partition_type[pointer],"OS/2 HPFS    ");
      if(sector_buffer[index+4]==0x08) strcpy(partition_type[pointer],"AIX          ");
      if(sector_buffer[index+4]==0x09) strcpy(partition_type[pointer],"AIX bootable ");
      if(sector_buffer[index+4]==0x0a) strcpy(partition_type[pointer],"OS/2 boot mgr");
      if(sector_buffer[index+4]==0x0b) strcpy(partition_type[pointer],"FAT32        ");
      if(sector_buffer[index+4]==0x0c) strcpy(partition_type[pointer],"FAT32 LBA    ");
      if(sector_buffer[index+4]==0x0d) strcpy(partition_type[pointer],"Unknown      ");
      if(sector_buffer[index+4]==0x0e) strcpy(partition_type[pointer],"FAT16 LBA    ");
      if(sector_buffer[index+4]==0x0f) strcpy(partition_type[pointer],"FAT16 EXT LBA");
      if((sector_buffer[index+4]>=0x20) && (sector_buffer[index+4]<0x64)) strcpy(partition_type[pointer],"Unknown      ");
      if(sector_buffer[index+4]==0x64) strcpy(partition_type[pointer],"Novell       ");
      if(sector_buffer[index+4]==0x65) strcpy(partition_type[pointer],"Novell       ");
      if((sector_buffer[index+4]>=0x66) && (sector_buffer[index+4]<0x80)) strcpy(partition_type[pointer],"Unknown      ");
      if(sector_buffer[index+4]==0x80) strcpy(partition_type[pointer],"Old MINIX    ");
      if(sector_buffer[index+4]==0x81) strcpy(partition_type[pointer],"Linux/MINIX  ");
      if(sector_buffer[index+4]==0x82) strcpy(partition_type[pointer],"Linux Swap   ");
      if(sector_buffer[index+4]==0x83) strcpy(partition_type[pointer],"Linux Native ");
      if(sector_buffer[index+4]>0x83) strcpy(partition_type[pointer],"Unknown      ");
      }

    starting_head[pointer] = sector_buffer[index+1];
    ending_head[pointer] = sector_buffer[index+5];

    true_partition_size[pointer]=Decimal_Number(sector_buffer[index+12],sector_buffer[index+13],sector_buffer[index+14],sector_buffer[index+15]);

    partition_size[pointer]=true_partition_size[pointer]/2000;

    if(pointer<4)
      {
      hard_disk_space_used=hard_disk_space_used+true_partition_size[pointer];
      }

    if( ( (sector_buffer[index+0x04]==0x05) || (sector_buffer[index+0x04]==0x0f) )&& (pointer<4) )
      {
      whole_extended_partition_size=true_partition_size[pointer];
      }

    Extract_Cylinder_and_Sector(sector_buffer[index+2],sector_buffer[index+3]);

    starting_cylinder[pointer]=g_cylinder;
    starting_sector[pointer]=g_sector;

    if((extended==TRUE) && (record_extended_info_flag==TRUE))
      {
      extended_cylinder=starting_cylinder[pointer];
      extended_head=starting_head[pointer];
      extended_sector=starting_sector[pointer];

      extended_partition_cylinder[extended_pointer]=extended_cylinder;
      extended_partition_head[extended_pointer]=extended_head;
      extended_partition_sector[extended_pointer]=extended_sector;

      extended_pointer++;

      record_extended_info_flag=FALSE;
      }

    Extract_Cylinder_and_Sector(sector_buffer[index+6],sector_buffer[index+7]);

    ending_cylinder[pointer]=g_cylinder;
    ending_sector[pointer]=g_sector;

    number_of_partitions=pointer;
    pointer++;

    if((extended==FALSE) && (pointer==4))
      {
      number_of_partitions=4;
      done_flag=TRUE;
      }

    index=index + 16;
    } while(done_flag==FALSE);

  if(number_of_partitions>4)
    {
    logical_drives_exist=TRUE;

    index=5;
    do
      {
      extended_partition_space_used=extended_partition_space_used+true_partition_size[index-1];
      index++;
      }while(index<=number_of_partitions);
    }
}

/* Read_Sector */
void Read_Sector(int drive, int head, long cyl, int sector)
{
  int result;

  result=biosdisk(2, drive, head, cyl, sector, 1, sector_buffer);

    if (result!=0)
      {
      printf("\nDisk read error...operation terminated.\n");
      exit(1);
      }
}

/* Remove MBR */
void Remove_MBR(int drive)
{
  long pointer=0;

  Read_Sector(drive,0,0,1);

  do
    {
    sector_buffer[pointer]=0x00;
    pointer++;
    }while(pointer<0x1be);

  Write_Sector(drive,0,0,1);
}

/* Save MBR */
void Save_MBR(int drive)
{
  long index=0;

  FILE *file_pointer;

  Read_Sector(drive,0,0,1);

  file_pointer = fopen("boot.mbr","wb");

  if(!file_pointer)
    {
    printf("\nError opening or creating \"BOOT.BIN\" for writing...Operation Terminated.\n");
    exit(1);
    }

  do
    {
    fputc(sector_buffer[index],file_pointer);
    index++;
    }while(index<0x1be);

  do{
    fputc(0,file_pointer);
    index++;
    }while(index<512);

  fclose(file_pointer);
}

/* Scan External Lookup Table */
void Scan_External_Lookup_Table(int indicated_partition_type,int partition,int view_type)
{
  int counter;
  int index=0;
  int offset=0;

  char possible_partition_type[3];

  do
    {
    /* Copy the possible partition type from the partition_lookup_table_buffer */
    /* into the possible_partition_type */
    offset=0;
    do
      {
      possible_partition_type[offset]=partition_lookup_table_buffer[ ((index*80)+offset) ];
      offset++;
      }while(offset<=3);

    if(indicated_partition_type==atoi(possible_partition_type))
      {
      if(view_type==STANDARD)
	{
	counter=0;
	do
	  {
	  partition_type[partition][counter]=partition_lookup_table_buffer[ (index*80)+(counter+18) ];
	  counter++;
	  }while(counter<50);
	index=256;
	}
      else
	{
	counter=0;
	do
	  {
	  partition_type[partition][counter]=partition_lookup_table_buffer[ (index*80)+(counter+4) ];
	  counter++;
	  }while(counter<13);
	index=256;
	}
      }
    index++;
    }while(index<=255);
}

/* Toggle Active Partition */
void Toggle_Active_Partition(int drive, long partition_number, int toggle)
{
  int value;

  long offset;

  Read_Sector(drive,0,0,1);

  if(1==toggle)
    {
    value=0x80;
    }
  else
    {
    value=0x00;
    }

  offset=((partition_number*16)-16)+0x1be;

  sector_buffer[offset]=value;

  Write_Sector(drive,0,0,1);
}

/* Write Sector */
void Write_Sector(int drive, int head, long cyl, int sector)
{
  int result;

  result=biosdisk(3, drive, head, cyl, sector, 1, sector_buffer);

  if (result!=0)
    {
    printf("\nDisk write error...Operation Terminated.\n");
    exit(1);
    }

  sectors_have_changed=TRUE;
}

/*
/////////////////////////////////////////////////////////////////////////////
//  MAIN ROUTINE
/////////////////////////////////////////////////////////////////////////////
*/
void main(int argc, char *argv[])
{
  int true;
  int true1;
  int true2;

  int drive;

  Allocate_Memory_For_Buffers();

  /* Check for interrupt 0x13 extensions */
  Check_For_INT13_Extensions();

  /* Is there a "part.dat" file? If so, use it. */
  if(External_Lookup_File_Exists()==TRUE)
    {
    partition_lookup_table=EXTERNAL;
    Load_External_Lookup_Table();
    }
  else partition_lookup_table=INTERNAL;

  /* if FDISK is typed without any options */
  if(argc==1)
    {
    Determine_Color_Video_Support();
    Ask_User_About_FAT32_Support();
    Menu_Routine();
    exit(0);
    }

  /* if FDISK is typed with options */
  if(argc>=2)
    {
    /* if "FDISK /MONO" is typed */
    true=1;
    true=stricmp("/MONO",argv[1]);
    if(0==true)
      {
      textcolor(7);
      Menu_Routine();
      exit(0);
      }

    /* if "FDISK /?" is typed */
    true=1;
    true=strcmp("/?",argv[1]);
    if(0==true)
      {
      Display_Help_Screen();
      exit(0);
      }

    /* if "FDISK /L" is typed */
    true=1;
    true=stricmp("/L",argv[1]);
    if(0==true)
      {
      List_Available_Drives();
      exit(0);
      }

    /* if "FDISK /DUMP" is typed */
    /* "/STATUS" is added for MS-FDISK compatibility */
    true1=1;
    true2=1;
    true1=stricmp("/DUMP",argv[1]);
    true2=stricmp("/STATUS",argv[1]);
    if(0==(true1*true2))
      {
      Dump_Partition_Information();
      exit(0);
      }

    /*if there is a 2nd option, convert the option to a drive # */
    if(argc>=3)
      {
      drive=argv[2] [0];

      drive=drive-47;

      /*check to make sure the drive is a legitimate number */
      if( (drive<2) || (drive>6) )
	{
	printf("\nIncorrect drive designation...Operation Terminated.\n");
	exit(1);
	}

      drive=drive+126;
      }

    /*if no drive number is entered, set the drive to the first physical drive */
    if(argc<3)
      {
      drive=128;
      }

    /*if "FDISK /A" is typed */
    true=1;
    true=stricmp("/A",argv[1]);
    if(0==true)
      {
      Automatically_Partition_Hard_Drive(drive);
      exit(0);
      }

    /*if "FDISK /C" is typed */
    true=1;
    true=stricmp("/C",argv[1]);
    if(0==true)
      {
      Clear_Partition_Table(drive);
      exit(0);
      }

    /*if "FDISK /I [/drive# [/tech]]" is typed */
    true=1;
    true=stricmp("/I",argv[1]);

    if(0==true)
      {
      if(argc==4)
	{
	true=1;
	true=stricmp("/TECH",argv[3]);

	if(0==true)
	  {
	  Clear_Screen();
	  Display_Partition_Table(drive,TECHNICAL);
	  exit(0);
	  }
	else
	  {
	  printf("\nSyntax Error...Operation Terminated.\n");
	  exit(1);
	  }
	}
      else
	{
	Clear_Screen();
	Display_Partition_Table(drive,STANDARD);
	exit(0);
	}
      }

    /* if "FDISK /N" is typed*/

    /*   Syntax: FDISK /N drive# partition_type size [/P] [/S type]*/

    true=1;
    true=stricmp("/N",argv[1]);

    if(0==true)
      {
      int partition_type=NULL;
      int percentage_flag=NULL;
      int special_flag=NULL;
      int special_partition_type=NULL;


      unsigned long size;

      if(argc<5)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      /* Determine type */
      true=1;
      true=stricmp("P",argv[3]);
      if(0==true) partition_type = PRIMARY;

      true=1;
      true=stricmp("E",argv[3]);
      if(0==true) partition_type = EXTENDED;

      true=1;
      true=stricmp("L",argv[3]);
      if(0==true) partition_type = LOGICAL;

      if(partition_type==NULL)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      /* Set partition size */

      size=atol(argv[4]);

      /* Set percentage_flag, if necessary. */
      if((argc==6) || (argc==8))
	{
	true=1;
	true=stricmp("/P",argv[5]);
	if(0==true) percentage_flag = PERCENTAGE;

	if(argc==8)
	  {
	  true=1;
	  true=stricmp("/P",argv[7]);
	  if(0==true) percentage_flag = PERCENTAGE;
	  }
	}

      /* Set special partition type, if necessary. */
      if((argc>=6) && (argc<=8))
	{
	true=1;
	true=stricmp("/S",argv[5]);
	if(0==true)
	  {
	  special_flag = SPECIAL;
	  special_partition_type=atoi(argv[6]);
	  }

	if(argc==8)
	  {
	  true=1;
	  true=stricmp("/S",argv[6]);
	  if(0==true)
	    {
	    special_flag = SPECIAL;
	    special_partition_type=atoi(argv[7]);
	    }
	  }
	}

      if( (special_flag==SPECIAL) && ( (special_partition_type<1) || (special_partition_type>255)  )  )
	{
	printf("\nSpecial partition type is out of range...Operation Terminated.\n");
	exit(1);
	}

      if( (special_partition_type==11) || (special_partition_type==12) ) FAT32=TRUE;

      if( (size>2048) && (FAT32==FALSE) )
	{
	printf("\nEntered partition size is too large...Operation Terminated.\n");
	exit(1);
	}

      Create_New_Partition(drive,partition_type,size,percentage_flag,special_flag,special_partition_type);
      exit(0);
      }

    /* if "FDISK /T" is typed */

    /* Syntax:  FDISK [/T drive# partition# active_toggle] */
    true=1;
    true=stricmp("/T",argv[1]);
    if(0==true)
      {
      if(argc<5)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}
      int partition_number=atoi(argv[3]);

      if((partition_number<1) || (partition_number>4))
	{
	printf("\nPartition number is out of range (1-4)...Operation Terminated.\n");
	exit(1);
	}

      int toggle=atoi(argv[4]);

      if((toggle<0) || (toggle>1))
	{
	printf("\nInvalid toggle setting...Operation Terminated.\n");
	exit(1);
	}

      Toggle_Active_Partition(drive,partition_number,toggle);
      exit(0);
      }

    /* if "FDISK /D" is typed */

    /* Syntax:  FDISK [/D drive# {partition# || /L} ] */
    true=1;
    true=stricmp("/D",argv[1]);
    if(0==true)
      {
      int partition_number;
      int by_number=TRUE;
      int type=NULL;
      int result;

      if(argc<4)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      true=1;
      true=stricmp("/L",argv[3]);
      if(0==true)
	{
	Read_Partition_Table(drive,TECHNICAL);

	if(number_of_partitions<5)
	  {
	  printf("\nNo logical partitions exist...Operation Terminated.\n");
	  exit(1);
	  }
	partition_number=LAST;
	by_number=FALSE;
	}

      true=1;
      true=stricmp("/P",argv[3]);
      if(0==true)
	{
	type=PRIMARY;
	by_number=FALSE;
	}

      true=1;
      true=stricmp("/E",argv[3]);
      if(0==true)
	{
	type=EXTENDED;
	by_number=FALSE;
	}

      if(by_number==TRUE)
	{
	partition_number=atoi(argv[3]);

	if((partition_number<1) || (partition_number>4))
	  {
	  printf("\nPartition number is out of range (1-4)...Operation Terminated.\n");
	  exit(1);
	  }
	}

      if(by_number==TRUE)
	{
	Delete_Partition(drive,partition_number);
	}
      else
	{
	result=Delete_P_Or_E_Partition(drive,type);
	if(result==FALSE)
	  {
	  printf("\nError attempting to delete partition...Operation Terminated.\n");
	  exit(1);
	  }
	}

      exit(0);
      }

    /* if "FDISK /MBR" is typed */

    /* Syntax:  FDISK [/MBR drive#] */

    /* "/CMBR" is added for MS-FDISK compatibility */

    true1=1;
    true2=1;
    true1=stricmp("/MBR",argv[1]);
    true2=stricmp("/CMBR",argv[2]);
    if(0==(true1*true2))
      {
      Create_MBR(drive);
      exit(0);
      }

    /* if "FDISK /RMBR" is typed */

    /* Syntax:  FDISK [/RMBR drive#] */
    true=1;
    true=stricmp("/RMBR",argv[1]);
    if(0==true)
      {
      Remove_MBR(drive);
      exit(0);
      }

    /* if "FDISK /AMBR" is typed */

    /* Syntax: FDISK [/AMBR drive#] */
    true=1;
    true=stricmp("/AMBR",argv[1]);
    if(0==true)
      {
      Create_Alternate_MBR(drive);
      exit(0);
      }

    /* if "FDISK /SMBR" is typed */

    /* Syntax: FDISK [/SMBR drive#] */
    true=1;
    true=stricmp("/SMBR",argv[1]);
    if(0==true)
      {
      Save_MBR(drive);
      exit(0);
      }

    /* if "FDISK [/M drive# primarypartition# newtype#]" is typed */
    true=1;
    true=stricmp("/M",argv[1]);
    if(0==true)
      {
      int partition_number;
      int type_number;

      partition_number=atoi(argv[3]);

      if((partition_number<1) || (partition_number>4))
	{
	printf("\nPrimary partition number is out of range...Operation Terminated.\n");
	exit(1);
	}

      type_number=atoi(argv[4]);

      if((type_number<=0) || (type_number>255))
	{
	printf("\nNew partition type is out of range...Operation Terminated.\n");
	exit(1);
	}

      Modify_Partition_Type(drive,partition_number,type_number);
      }
    }
  DeAllocate_Memory_For_Buffers();
}
