/*  XMS.C  Extended memory operations
 *  Copyright (C) 1991-1998  Felix Ritter
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <dos.h>
#include <alloc.h>
#include "defines.h"
#include "xms.h"

#pragma inline

/*-- globale Variablen -----------------------------------------------*/

static void far *XMSPtr;        /* Zeiger auf Extended-Memory-Manager (XMM) */
BYTE XMSErr;                            /* Fehlercode der letzten Operation */
BOOL XMSTreiber;                                /* != 0 wenn XMS-Treiber ok */

void XMSCall( BYTE FktNr, XMSRegs *Xr)
{
   asm mov cx, ds
   asm push cx
   _AH= FktNr;
   _BX= Xr->BX;
   _DX= Xr->DX;
   _SI= Xr->SI;
   _DS= Xr->Segment;
   asm mov es, cx
   asm call es: [ XMSPtr]
   asm mov cx, ds
   Xr->AX=      _AX;
   Xr->BX=      _BX;
   Xr->DX=      _DX;
   Xr->SI=      _SI;
   Xr->Segment= _CX;
   asm pop ds
}

/***********************************************************************
* XMSInit : Initialisiert die Routinen zum Aufruf der XMS-Funktionen   *
**--------------------------------------------------------------------**
* Eingabe : keine                                                      *
* Ausgabe : TRUE, wenn ein XMS-Treiber entdeckt wurde, sonst FALSE     *
* Info    : - Der Aufruf dieser Funktion mu dem Aufruf aller anderen  *
*             Prozeduren und Funktionen aus diesem Programm voraus-    *
*             gehen.                                                   *
***********************************************************************/

BOOL XMSInit( void )
{
 union REGS Regs;            /* Prozessorregister fr Interruptaufruf */
 struct SREGS SRegs;                               /* Segmentregister */
 XMSRegs Xr;                               /* Register fr XMS-Aufruf */

 Regs.x.ax = 0x4300;    /* Verfgbarkeit des XMS-Managers feststellen */
 int86( 0x2F, &Regs, &Regs );              /* DOS-Dispatcher aufrufen */

 if ( Regs.h.al == 0x80 )                 /* XMS-Manager angetroffen? */
  {                                                             /* Ja */
   Regs.x.ax = 0x4310;           /* Einsprungspunkt des XMM ermitteln */
   int86x( 0x2F, &Regs, &Regs, &SRegs );
   XMSPtr = MK_FP( SRegs.es, Regs.x.bx );  /*Adresse in glob. Var. sp.*/
   XMSErr = ERR_NOERR;                /* noch kein Fehler aufgetreten */
   return TRUE;           /* Handler angetroffen, Modul initialisiert */
  }
 else                                 /* kein XMS-Handler installiert */
  return FALSE;
}

/***********************************************************************
* XMSQueryVer: liefert die XMS-Versionsnummer und andere Statusinfor-  *
*              mationen                                                *
**--------------------------------------------------------------------**
* Eingabe : VerNr = nimmt die Versionsnummer nach dem Funktionsaufruf  *
*                   auf (Format: 235 == 2.35)                          *
*           RevNr = nimmt die Revisionsnummer nach dem Funktionsaufruf *
*                   auf                                                *
* Ausgabe : TRUE, wenn eine HMA zur Verfgung steht, sonst FALSE       *
***********************************************************************/

BOOL XMSQueryVer( int * VerNr, int * RevNr)
{
 XMSRegs Xr;                               /* Register fr XMS-Aufruf */

 XMSCall( 0, &Xr );                       /* XMS-Funktion #0 aufrufen */
 *VerNr = HI(Xr.AX)*100 + ( LO(Xr.AX) >> 4 ) * 10 +
	   ( LO(Xr.AX) & 15 );
 *RevNr = HI(Xr.BX)*100 + ( LO(Xr.BX) >> 4 ) * 10 +
	  ( LO(Xr.BX) & 15 );
 return ( Xr.DX == 1 );
}

/***********************************************************************
* XMSQueryFree : Liefert die Gre des freien extended Memory und die  *
*                des grten freien Blocks                             *
**--------------------------------------------------------------------**
* Eingabe : GesFrei: Nimmt die Gesamtgre des freien EM auf.          *
*           MaxBl  : Nimmt die Gre des grten freien Blocks auf.    *
* Info    : - Beide Angabe in KB                                       *
*           - Die Gre der HMA wird nicht mitgezhlt, auch wenn sie   *
*             noch nicht an ein Programm vergeben wurde.               *
***********************************************************************/

void XMSQueryFree( int * GesFrei, int * MaxBl )
{
 XMSRegs Xr;                               /* Register fr XMS-Aufruf */

 XMSCall( 8, &Xr );                       /* XMS-Funktion #8 aufrufen */
 *GesFrei = Xr.AX;                         /* Gesamtgre steht in AX */
 *MaxBl   = Xr.DX;                     /* freier Speicher steht in DX */
}

/***********************************************************************
* XMSGetMem : Allokiert einen extended-Memory-Block (EMB)              *
**--------------------------------------------------------------------**
* Eingabe : LenKB : Gre des angeforderten Blocks in KB               *
* Ausgabe : Handle zum weiteren Zugriff auf dne Block oder 0, wenn     *
*           kein Block allokiert werden konnte. Dann steht auch in der *
*           globalen Variable XMSErr ein entsprechender Fehlercode.    *
***********************************************************************/

int XMSGetMem( int LenKb )
{
 XMSRegs Xr;                               /* Register fr XMS-Aufruf */

 Xr.DX = LenKb;                /* Lnge wird im DX-Register bergeben */
 XMSCall( 9, &Xr );                       /* XMS-Funktion #9 aufrufen */
 return Xr.DX;                                /* Handle zurckliefern */
}

/***********************************************************************
* XMSFreeMem : Gibt einen zuvor allokierten extended-Memory-Block      *
*              (EMB) wieder frei.                                      *
**--------------------------------------------------------------------**
* Eingabe : Handle : Das Handle zum Zugriff auf den Block, das beim    *
*                    Aufruf von XMSGetMem zurckgeliefert wurde.       *
* Info    : - Der Inhalt des EMB geht durch diesen Aufruf unwider-     *
*             bringlich verloren und auch das Handle wird ungltig.    *
*           - Vor der Beendigung eines Programms sollten alle allo-    *
*             kierten Bereich mit Hilfe dieser Prozedur wieder frei-   *
*             gegeben werden, damit sie an nachfolgend aufgerufene     *
*             Programm wieder vergeben werden knnen.                  *
***********************************************************************/

void XMSFreeMem( int Handle )
{
 XMSRegs Xr;                               /* Register fr XMS-Aufruf */

 Xr.DX = Handle;              /* Handle wird im DX-Register bergeben */
 XMSCall( 10, &Xr );                     /* XMS-Funktion #10 aufrufen */
}

/***********************************************************************
* XMSCopy : Kopiert Speicherbereiche zwischen dem extended Memory und  *
*           dem konventionellen Speicher bzw. innerhalb der beiden     *
*           Speichergruppen.                                           *
**--------------------------------------------------------------------**
* Eingabe : VonHandle  : Handle des Speicherbereichs, der verschoben   *
*                        werden soll.                                  *
*           VonOffset  : Offset in diesen Block ab dem verschoben wird.*
*           NachHandle : Handle des Speicherbereichs, der das Ziel der *
*                        Verschiebung darstellt.                       *
*           NachOffset : Offset in den Ziel-Block.                     *
*           LenW       : Anzahl der zu verschiebenden Worte            *
* Info    : - Um normalen Speicher in die Operation einzubeziehen,     *
*             mu als Handle der Wert 0 und fr den Offset die Seg-    *
*             ment und Offsetadresse in der gewohnte Form (Offset vor  *
*             Segment) angegeben werden.                               *
***********************************************************************/

void XMSCopy( int VonHandle, long VonOffset, int NachHandle, long NachOffset, WORD LenW)
{
   XMSRegs Xr;                                   /* Register fr XMS-Aufruf */
   EMMS Mi;                                               /* nimmt EEMS auf */
   void *MiPtr;

   Mi.LenB= LenW<< 1;                      /* zunchst den EMMS aufbereiten */
   Mi.SHandle= VonHandle;
   Mi.SOffset= VonOffset;
   Mi.DHandle= NachHandle;
   Mi.DOffset= NachOffset;
   MiPtr = &Mi;                              /* Far-Zeiger auf die Struktur */
   Xr.SI = FP_OFF( MiPtr );                       /* Offsetadresse des EMMS */
   Xr.Segment = FP_SEG( MiPtr );                 /* Segmentadresse des EMMS */
   XMSCall( 11, &Xr );                         /* XMS-Funktion #11 aufrufen */
}

/***********************************************************************
* XMSLock : Sperrt einen extended-Memory-Block gegen sein Verschiebung *
*           durch den XMM und liefert gleichzeitig seine absolute      *
*           Adresse zurck.                                            *
**--------------------------------------------------------------------**
* Eingabe : Handle : Handle des Speicherbereichs, das bei einem vor-   *
*                    hergehenden Aufruf von XMSGetMem zurckgeliefert  *
*                    wurde.                                            *
* Ausgabe : Die lineare Adresse des zugehrigen Speicherblocks.        *
***********************************************************************/

long XMSLock( int Handle )
{
 XMSRegs Xr;                               /* Register fr XMS-Aufruf */

 Xr.DX = Handle;                                    /* Handle des EMB */
 XMSCall( 12, &Xr );                     /* XMS-Funktion #12 aufrufen */
 return ((long) Xr.DX << 16) + Xr.BX;        /* 32-Bit-Adr. berechnen */
}

/***********************************************************************
* XMSUnlock : Gibt einen gesperrten extended-Memory-Block wieder fr   *
*             die Verschiebung frei.                                   *
**--------------------------------------------------------------------**
* Eingabe : Handle : Handle des Speicherbereichs, das bei einem vor-   *
*                    hergehenden Aufruf von XMSGetMem zurckgeliefert  *
*                    wurde.                                            *
***********************************************************************/

void XMSUnlock( int Handle )
{
 XMSRegs Xr;                               /* Register fr XMS-Aufruf */

 Xr.DX = Handle;                                    /* Handle des EMB */
 XMSCall( 13, &Xr );                     /* XMS-Funktion #13 aufrufen */
}

/***********************************************************************
* XMSQueryInfo : Liefert verschiedene Informationen ber einen bereits *
*                allokierten extended-Memory-Block                     *
**--------------------------------------------------------------------**
* Eingabe : Handle : Handle des Speicherbereichs                       *
*           Lock   : Variable, in die der Lock-Zhler eingetragen wird *
*           LenKB  : Variable, in die die Lnge des Blocks in KB ein-  *
*                    getragen wird                                     *
*           FreeH  : Variable, ind die die Anzahl der noch freien      *
*                    Handles eingetragen wird                          *
* Info    : Die Startadresse eines Blocks knnen Sie mit Hilfe dieser  *
*           Prozedur nicht ermitteln, verwenden Sie dafr die Funktion *
*           XMSLock.                                                   *
***********************************************************************/

void XMSQueryInfo( int Handle, int * Lock, int * LenKB, int * FreeH )

{
 XMSRegs Xr;                               /* Register fr XMS-Aufruf */

 Xr.DX = Handle;                                    /* Handle des EMB */
 XMSCall( 14, &Xr );                     /* XMS-Funktion #14 aufrufen */
 *Lock  = HI( Xr.BX );                          /* Register auswerten */
 *FreeH = LO( Xr.BX );
 *LenKB = Xr.DX;
}

/***********************************************************************
* XMSRealloc : Vergert oder verkleinert einen zuvor ber XMSGetMem   *
*              allokierten extended-Memory-Block                       *
**--------------------------------------------------------------------**
* Eingabe : Handle   : Handle des Speicherbereichs                     *
*           NeuLenKB : Neue Lnge des Speicherbereichs in KB           *
* Ausgabe : TRUE, wenn der Block in seiner Gre verndert werden      *
*           konnte, sonst FALSE                                        *
* Info    : Der angegebene Block darf nicht gesperrt sein!             *
***********************************************************************/

BOOL XMSRealloc( int Handle, int NeuLenKB)
{
 XMSRegs Xr;                               /* Register fr XMS-Aufruf */

 Xr.DX = Handle;                                    /* Handle des EMB */
 Xr.BX = NeuLenKB;               /* die neue Lnge in das BX-Register */
 XMSCall( 15, &Xr );                     /* XMS-Funktion #15 aufrufen */
 return ( XMSErr == ERR_NOERR );
}