/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/*
    MSG2MGT - utility for the message handling library
    Copyright (C) 1998  Steffen Kaiser

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/
/* $RCSfile: MSG2MGT.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	The binary message file management tool.

   ----
	command line usage:

	msg2mgt [{option}] binary_file [{ [{option}] command }]


	binary_file : the file where one or more binary message files are
                  canned together (incl. extension!)

	Options:
		/i=	Where the header is to be expected within the _input_ file
			(see below)
		/o=	Where the header is to be written to within the _output_ file
			(see below)

	There are three types of where to place the header (the structure #0):
		1) 'none' at all,
		2) at the 'begin' ('start', 'head') of the file, and
		3) at the 'end' ('tail') of the file,
		4) for most pleasure 'any' means to search for the header.
	So if the command lines reads:
		MSG2MGT /i=tail /o=tail PROGRAM.EXE /i=none + ENGLISH.MS2
	MSG2MGT shall expect the header at the end of the file PROGRAM.EXE,
	then add the file ENGLISH.MS2 (expecting no header at all), and the
	final result shall overwrite file PROGRAM.EXE placing the header
	at its tail. (To put it into other words, ENGLISH.MS2 is appended
	to PROGRAM.EXE.)

	The "/o=" option must preceed binary_file; any "/i=" option applys
	to all following filenames.
	If "/o=" is omitted, the value of "/i=" is used.
	If "/i=" is omitted or is 'any', the file searched for the header;
	first at its tail, then at its start, if not found, 'none' is
	assumed.

   ---

	Commands:

	They describe what to do with the binary file.

	If omited, all canned message definitions are displayed.

	The following commands are available:
	(The double quotes enclose the command; the angle brackets enclose
	 a placeholder; spaces mark positions where one or more spaces must
	 be inserted.)

	"+ <file>" tread 'file' as another binary message library, which is
	appended to binary_file.

	"+<number> <file>" insert 'file' before the first occurance of
	the language with the country code 'number'.

	"+#<number> <file>" insert 'file' before the 'number'th entry.

	"-<number>" remove any definition of the language code 'number'.

	"-#<number>" remove the 'number'th entry.

	"*<number> <file>" extract any definition of language 'number'
	into the 'file'.

	"*#<number> <file>" extract the 'number'th entry into the 'file'.

	"@#<number>" verbosely output the 'number'th entry.

	"@*" verbosely output any entry.

	"@ <file>" read 'file' line by line and interprete each line as
	exactly one command.

	"!compact" try to compress all of the canned message definitions
	without really using any compression algorithm. Compacting the
	message library is the only command to make sure that a previously
	removed entry is removed entirely (up to its last byte). Though,
	this operation may require a lot of time.

	"!alias:<number1>+<number2>" install the language 'number2' as
	an alias for all entries of the language 'number1'.

	"!alias:<number1>-<number2>" remove the language 'number2' as
	an alias from all entries of the language 'number1'.

	"!append:<file>" same as "+ file".

	"!insert:<number>,<file>" same as "+<number> <file>".

	"!insert:#<number>,<file>" same as "+#<number> <file>".

	"!remove:<number>" same as "-<number>".

	"!remove:#<number>" same as "-#<number>".

	"!view" same as "@*".

	"!view:#<number>" same as "@#<number>".

	"!extract:<number>,<file>" same as "*<number> <file>".

	"!extract:#<number>,<file>" same as "*#<number> <file>".

	"!perform" Perform pending "append"/"add"/"insert"/"remove"
	operations. If there are pending operations upon the termination
	of this program, "!perform" is automatically assumed. The command
	sequence: "+ file1" "*#1 file1" won't work because of the pending
	operations, as 'file1' has not been added before it is extracted.
	"+ file1" "!perform" "*#1 file1" would work instead.

   ----
	Notes

	Currently binary files created by the message compiler 2nd and
	method #2 and in the system's native format are handled by this utility.

	The file is accessed in binary mode.

	If the header is located behind the message it points to (the first
	message defined, though) all bytes between the first message and the
	header are assumed to exclusively belong to the message library.

	Extracted message libraries are always in non-compacted form.

   ----
	Methods

	When MSG2MGT starts, it identifies all languages defined within the binary
	file and internally caches the information:
		+ in what sequence the languages are defined,
		+ which group(s) is defined by which language definition and
		+ which file the definition is located in.

	If definitions are added/inserted or removed or a "!compact" command
	is encountered, only this cached information is altered, no physical
	change is made to any existing binary message library.

	When a "!perform" command is encountered, the whole sequence is
	copied into a temporary file; on success the newly build binary
	message library is stored into the binary_file.

   ----

   	Syntax of the !view commands.

	Example (ignore any whitespace in the beginning o each line):

		Entry #0
		| located in file "english.ms2" at position 2597
		| defining language with code 61 (0x03d)
		| 1 supported character sets:
		||	7bitASCII
		| 2 language aliases defined:
		||	1 (0x001), 44 (0x02c)
		| Group #0 contains 139 strings.
		| Defined message strings:
		|| "E_hlpScreen": Grp: 0, Idx: 1, Exitcode: 127, Length: 790, @2624, Str @3736
		|||FreeDos message compiler 2nd version
		|||
		|||Usage: %s [{option}] pathGlbMsg lng locHeader [msgDcl [msgFeatures]]

	The number of leading '|' characters identify subinformation, e.g.
	"7bitASCII" is the subinformation of "1 supported character sets:".
	However, no more than three '|' characters are syntacically used.

	The message string definitions contain the following information:
	The '||' line contains the control information (from left to right):
		+ "%s" - the symbolic message ID (possibly empty)
		+ Grp: %u - the group of the message
		+ Idx: %u - the index of the message with the group
		+ Exitcode: %u - the associated exitcode (random if none assigned)
		+ Length: %u - the length of the message in _bytes_
		+ @%ld - the offset of the control area with the file
		+ Str @%ld - the offset of the string area
	Any following lines with at least three leading '|' characters specify
	the text of the message. If more than three '|' characters lead a line,
	all beginning with the fourth one are part of the message text.
	The message text is slightly decomposed, some control characters
	are represented in C style (e.g. Tabs -> \t), others are not, e.g.
	newlines (\n), others are displayed in C-style hex, e.g. \x1b.
	If a special enclosure has a count or length field (e.g. %-%A and
	%-%r), those fields are always displayed in hex. The %-%m (static
	message include) is decomposed into the symbolic message ID,
	if it is known, following by  single '%' sign. If the symbolic
	message ID cannot be retrieved, two single-byte hex values are
	displayed in the same order as read from the file.

   ----

   Target compilers:
   		Micro-C v3.14, Borland C++ v3.1+

   ----
	
	Known bugs:

   ----
	
*/

#include <assert.h>
#include <stdio.h>
#ifndef _MICROC_
#include <ctype.h>
#include <io.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#endif /*#	!defined(_MICROC_) */
#include <portable.h>
#include <getopt.h>

#include "savemem.h"
#include "fileIO.h"
#include "copyF.h"
#include "shStr.h"
#include "tmpFile.h"
#include "hashFile.h"
#include "msgIDs.h"

#ifdef __TURBOC__
#pragma hdrstop
#endif /*#	defined(__TURBOC__) */

#define MSG_METHOD1				/* don't want method #2 encoding right now */

#include "../msgconf2.h"		/* method #2 structures */

#include "yerror.h"


#ifndef NUL
#define NUL '\0'
#endif /*#	!defined(NUL) */

/* Where to place the header */
#define MH_BEG 1	/* at the beginning of the file */
#define MH_END 2	/* at the end of the file */
#define MH_NON 3	/* not at all */
#define MH_ERR 255	/* error */
#define MH_ANY 0	/* determine */


struct MsgList {
	struct MsgList *nxt;
	dword pos;		/* position of struct #1 of this definition */
	msgTLngcode lng;	/* language code of this definition */
	char *fnam;		/* the filename of where this language definition
						resides in */
};

struct AliasList {
	struct AliasList *nxt;
	int toadd;		/* if to add; otherwise subtract */
	msgTLngcode alias;		/* alias to add/subtract */
	msgTLngcode lng;		/* language to add/substract the alias to/from */
};
#define DISABLED_ALIAS_MASK (1 << (sizeof(word) * 8 - 1))
#define MATCHED_CHARSET_MASK (1 << (sizeof(word) * 8 - 1))


#ifdef _MICROC_
#define CmdFct int
#else /*#	!(defined(_MICROC_)) */
typedef void (*CmdFct)(void);
#endif /*#	defined(_MICROC_) */
struct CmdOpts {		/* filled by idCmd() */
	int needFnam;		/* caller must fetch the 'fnam' member */
	MSGID msgNoFnam;	/* msg ID of error message if no 'fnam' avail */
	CmdFct fct;			/* function to call to perform the command */
	/* arguments of the command to be performed
		maybe invalid (random values), if unused by the command */
	int entry;			/* found '#<number>' */
	int num1;			/* found a <number> or <number1> */
	int num2;			/* found a <number2> */
	char *fnam;			/* found a <file> */
};

struct MSGLIB_ {
	FILE *ml_fp;
	char *ml_fnam;
	dword ml_lastCntrlArea;
	dword ml_startPos;			/* first language defintion */
	int ml_headerLocation;
};
#ifdef _MICROC_
#define MSGLIB struct MSGLIB_
#else /*#	!(defined(_MICROC_)) */
typedef struct MSGLIB_ MSGLIB;
#endif /*#	defined(_MICROC_) */



extern void sourceIn(void);

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: MSG2MGT.C 2.12 1998/10/04 05:17:36 ska Exp ska $";
#endif /*#	defined(RCS_Version) */


const char extBAK[] = ".BAK";
const char quotes[] = "\"'";	/* which characters are to be recognized
									as quotes */

/* flags */
int libModified = 0;	/* was the library modified? */
int compactMsgLib = 0;	/* compact the library with each libPerform */


/* current state of the /i= and /o= switches */
int oHead;			/* where to place the header at the result */
int iHead;			/* where to expect the file on open */

/* Informations of the binary file */
/* The binary file is not handled as "normal" binary message
	libraries: The binfile is prescanned and startPos and endPos
	are determined according the backup 'bak', if != NULL, or
	the original file, otherwise.
	The message library is located somewhere in the middle of the
	file, but possibly there is no binary file at all, the messages
	are located at front or at the end of the file.
	Also, the messages may be surrounded by control structures.

	If bak == NULL, there is no backup of the original file and, if
	any modification is made to the binary file, a backup is created.
	headLen & endLen are evaluated according the backup.
	The startPos & endPos are updated according the actual bin file.
	If *bak == '\0', there is no input file at all.

	The following position/length information are evaluated:
		startPos: the first language definition
		endPos: the upper limit of the message library
		headLen: the number of bytes at the start of the binary file
		endLen: the number of bytes at the end of the binary file

	The association between message IDs and message names is usually
	the assoc that is the first encountered at all.
*/
struct BinFile {
	int header;		/* where to place the header in the bin file */
	char *fn;		/* name of binary file */
	char *bak;		/* name of backup binary file */
	dword startPos;	/* where the message library is locates within the */
	dword endPos;	/* the binary file */
	dword headLen;
	dword endLen;
	struct MsgIDs *msgIDs;	/* msgID <-> msgName assoc of the output file */
} binf = {MH_ANY};

struct CmdOpts cmdOpts;	/* filled by idCmd() */

/* list of language definitions in the "virtual" binary msg library */
struct MsgList *head, *last = NULL;		/* begin and end of the list of
			included language definitions. If last == NULL, no definition
			is included within the binary message library. */
struct MsgList *actEntry = NULL;
struct MsgList *prvEntry = NULL;
int entries = 0;		/* number of entries within the list */
int oldEntries = 0;		/* entries before calling scanLib() */

struct AliasList *aHead, *aLast = NULL;

/* temporary files */
TempFile *tmpStrings = NULL;
TempFile *tmpIndex = NULL;
HashFile *tmpHash = NULL;


/*********************************************************************/
/**************** Library header functions ***************************/
/*********************************************************************/

/*
 *	Identify if the structure #0 contains valid values
 *	Currently only the member ms0_id is checked.
 */
int identMSGLIB(MsgStru0 *msg0)
{	
	assert(msg0);

	if(!(memcmp(msg0->ms0_id, MSG_IDSTRING, sizeof(MSG_IDSTRING) - 1) == 0
	 && msg0->ms0_magicnumber == MSG_MN_STRU0))
		return 0;
	if(msg0->ms0_version != 0)
		error(E_unknownMsgLibVersion, msg0->ms0_version);
	return 1;
}

/*
 *	Fill the header (identification area) with valid values
 */
void mkStru0(MsgStru0 *msg0)
{	
	assert(msg0);

	clrpobj(msg0);
	memcpy(msg0->ms0_id, MSG_IDSTRING, sizeof(MSG_IDSTRING) - 1);
	msg0->ms0_magicnumber = MSG_MN_STRU0;
}

/*
 *	Check if a magic number might be correct
 */
int isMagicNumber(msgTMagic mnum)
{	register unsigned hi, lo;

	hi = (mnum >> 8) & 0xff;
	lo = mnum & 0xff;

	return hi && hi <= MSG_MN_MAXHI && ((hi + lo) & 0xff) == 0;
}


/*********************************************************************/
/**************** Library writing functions **************************/
/*********************************************************************/

/*
 *	Create a new library file
 *	If oHead == MH_ANY, iHead is used
 */
MSGLIB *mkMsgLib(char *fnam, int oHead)
{	MSGLIB *f;
	MsgStru0 msg0;

	assert(fnam);

	f = cgetmem(sizeof(MSGLIB));
	f->ml_fnam = makeString(fnam);
	f->ml_fp = Efopen(fnam, inM("wrb", "w+b"));
	if((f->ml_headerLocation = oHead == MH_ANY? iHead: oHead) == MH_BEG) {
		/* write the structure #0 */
		mkStru0(aS(msg0));
		Ewrite(aS(msg0), sizeof(msg0), f->ml_fp, f->ml_fnam);
	}
	return f;
}

/*
 *	Create a new, temporary binary message file
 */
MSGLIB *mkTmpMsgLib(void)
{	TempFile *fp;
	char *fn;
	MSGLIB *f;

	fp = tmpFile(TF_RW);
	fn = makeString(fp->fn);
	clTempFile(fp);
	f = mkMsgLib(fn, oHead);
	disposeString(fn);
	return f;
}

/*
 *	Close a library file
 *	If the header is to be placed at the end, append it.
 */
void clMsgLib(MSGLIB *f)
{	MsgStru0 msg0;
	dword pos;

	assert(f);

	if(f->ml_headerLocation == MH_END) {
		mkStru0(aS(msg0));
		/* The "address" field is the negative number of bytes
			written so far (plus the sizeof(msg0)) */
		Egetpos(pos, f->ml_fp, f->ml_fnam);
#ifdef _MICROC_
		longcpy(msg0.ms0_address, f->ml_startPos);
		longsub(msg0.ms0_address, pos);
		longset(pos, sizeof(msg0));
		longsub(msg0.ms0_address, pos);
#else /*#	!(defined(_MICROC_)) */
		msg0.ms0_address = f->ml_startPos - (pos + sizeof(msg0));
#endif /*#	defined(_MICROC_) */
		Ewrite(aS(msg0), sizeof(msg0), f->ml_fp, f->ml_fnam);
	}
	Efclose(f->ml_fp, f->ml_fnam);
	disposeString(f->ml_fnam);
	free(f);
}



/*
 *	The structure of the temporary file that holds the pool of
 *	strings to be dumped later
 */
struct Pool {
	dword nxt;			/* position of the next message of the same length */
	dword pos;			/* position within the temporary string file */
	dword ms3;			/* position of the associated structure #3 */
};

/*
 *	The structure to define where to find a specific group and
 *	how many messages are defined by the group. This information
 *	will later be used to calculate the amount bytes used by the
 *	structure #1 by this language definition.
 */
struct GroupDef {
	char *fnam;			/* file this group definition resides in */
	int entries;		/* number of messages within this group */
	dword pos;			/* position of structure #3 within the file */
};

/*
 *	This structure holds all information about the current language
 *	definition.
 *	The aliases are masked with DISABLED_ALIAS_MASK to detect add/sub
 *	operations on them and to check if later partial language definitions
 *	have the same aliases applied to them.
 */
struct LngDef {
	msgTLngcode lngcode;					/* 0 if invalid/empty */
	struct GroupDef groups[MSGGROUPMAX];
	int numAliases;							/* >= 0 */
	msgTLngcode *aliases;						/* valid if numAliases > 0 */
	int numCharsets;						/* > 0 after reading a language */
	msgTCharset *charsets;						/* valid if numCharsets > 0 */
} lng = {0};			/* make sure it's initialized */


/*
 *	Add all groups of the current entry
 */
void addCntrlGroups(MsgStru1 *msg1, FILE *fp, char *fnam)
{	MsgStru2 msg2;
	int gid;

	assert(msg1);
	assert(fp);
	assert(fnam);

	while(msg1->ms1_groups--) {
		Eread(aS(msg2), sizeof(msg2), fp, fnam);
		if(lng.groups[gid = msg2.ms2_gid].fnam)
			error(E_multGroup, lng.lngcode, gid);
		lng.groups[gid].fnam = fnam;
		lng.groups[gid].entries = msg2.ms2_num;
		/* the field ms2_ms3 is relative to the end of the msg2 struct.
			we need an absolute postion later. */
		Egetpos(lng.groups[gid].pos, fp, fnam);
		longadd(lng.groups[gid].pos, msg2.ms2_ms3);
	}
}


/*
 *	Open the file and read the structure #1 the entry resides in
 */
void openEntry(struct MsgList *entry, MsgStru1 *msg1, FILE **fp
	, char **fnam)
{
	assert(entry);
	assert(msg1);
	assert(fp);
	assert(fnam);
	assert(entry->fnam);

	*fp = Efopen(*fnam = entry->fnam, "rb");
	Esetpos(entry->pos, *fp, entry->fnam);
	Eread(msg1, sizeof(*msg1), *fp, *fnam);
}



/*
 *	Search for an alias within an alias array
 */
msgTLngcode *findAlias(int num, msgTLngcode *aliases, msgTLngcode alias)
{
	assert(aliases);

	while(num--)
		if((*aliases++ & ~DISABLED_ALIAS_MASK) == alias)
			return aliases - 1;
	return NULL;
}


/*
 *	Prepare a new language definition
 */
void prepCntrlArea(struct MsgList *entry)
{	FILE *fp;
	char *fnam;
	MsgStru1 msg1;
	struct AliasList *al;
	msgTLngcode *pAl;

	assert(entry);

	if(lng.lngcode)
		fatal(E_multPrep);
	lng.lngcode = entry->lng;
	clraobj(lng.groups);

	/* open the file the entry is located in */
	openEntry(entry, aS(msg1), &fp, &fnam);

	/* read in the supported character sets */
	lng.numCharsets = msg1.ms1_charsets;		/* must be > 0 */
	if(!lng.numCharsets)
		error(E_noCharsets, lng.lngcode);
	Eread(lng.charsets = getmem(sizeof(msgTCharset) * lng.numCharsets)
	 , (sizeof(msgTCharset) * lng.numCharsets), fp, fnam);

	/* read in the aliases from the entry */
	if((lng.numAliases = msg1.ms1_aliases) != 0) {
		Eread(lng.aliases = getmem(sizeof(msgTLngcode) * lng.numAliases)
		 , (sizeof(msgTLngcode) * lng.numAliases), fp, fnam);
	}
	else lng.aliases = NULL;

	/* Now apply the alias changes */
	if((al = aHead) != NULL) {
		aLast->nxt = NULL;			/* make sure the list is terminated */
		do if(al->lng == lng.lngcode) {
		/* this command applies to the language */
			pAl = findAlias(lng.numAliases, lng.aliases, al->alias);
			if(al->toadd) {			/* add if not currently in the set */
				if(pAl)			/* Make sure it's not disabled */
					*pAl &= ~DISABLED_ALIAS_MASK;
				else {			/* add a new entry */
					lng.aliases = chgmem(lng.aliases
						, sizeof(word) * ++lng.numAliases);
					lng.aliases[lng.numAliases - 1] = al->alias;
				}
			}
			else {					/* subtract alias */
				if(pAl)
					*pAl |= DISABLED_ALIAS_MASK;
			}
		} while((al = al->nxt) != NULL);
	}

	addCntrlGroups(aS(msg1), fp, fnam);
	Efclose(fp, fnam);
}

/*
 *	This is a hack, just assume that the 7-bit character set is
 *	compatible to all code pages, but no code page to another.
 */
int msg_mtchCSet(msgTCharset cs1, msgTCharset cs2)
{	return cs1 == CHARSET_7BITASCII
	 || cs2 == CHARSET_7BITASCII
	 || cs1 == CHARSET_NONE
	 || cs2 == CHARSET_NONE
	 || cs1 == cs2;
}


/*
 *	Add a language definition to a previously initialized one
 *	Return != 0 if this entry was added.
 */
int addCntrlArea(struct MsgList *entry)
{	FILE *fp;
	char *fnam;
	MsgStru1 msg1;
	msgTLngcode al;
	msgTCharset charset;
	int i, found;

	assert(entry);

	/* open the file the entry is located in */
	openEntry(entry, aS(msg1), &fp, &fnam);

	assert(fp);
	assert(fnam);

	if(lng.lngcode != msg1.ms1_lng)
		error(E_joinLng);

	/* check if the supported character sets are identical.
		Note: "identical" does not mean that both set of character sets
		are equal, but both sets must not contain entries that have no
		"identical" character set in the other set.
		Two character sets are identical per definition; that means there
		is a matrix that declares which character sets are to be assumed
		as identical.

		For now: All character sets are un-identical, except if one
		character set is "7bitascii" or both character sets are
		identified by the same number.

		We use the "special" negative values to identify matched identical
		character sets within the lng.charsets.
	*/
	do {
		Eread(&charset, sizeof(charset), fp, fnam);
		for(found = i = 0; i < lng.numCharsets; ++i)
			if(msg_mtchCSet(charset
			 , lng.charsets[i] & ~MATCHED_CHARSET_MASK)) {
				found = 1;
				lng.charsets[i] |= MATCHED_CHARSET_MASK;
			}
	} while(found && --msg1.ms1_charsets);

	/* reset the MATCHED_CHARSET bit */
	for(i = 0; i < lng.numCharsets; ++i)
		if(lng.charsets[i] & MATCHED_CHARSET_MASK)
			lng.charsets[i] &= ~MATCHED_CHARSET_MASK;
		else found = 0;

	if(!found) {		/* at least one character set did not matched */
		warning(W_lngMultCharsets, lng.lngcode);
		Efclose(fp, fnam);
		return 0;
	}

	/* read in the aliases from the entry */
	while(msg1.ms1_aliases--) {
		Eread(&al, sizeof(al), fp, fnam);
		if(!findAlias(lng.numAliases, lng.aliases, al))
			error(E_misLng, lng.lngcode, al);
	}
	addCntrlGroups(aS(msg1), fp, fnam);
	Efclose(fp, fnam);

	return 1;
}

/*
 *	Add a string reference to the pool of strings to be dumped later
 *
 *	We maintain three temporary files:
 *	We save the message strings into a temporary file #1. This is
 *	automatically the string area, if this library is not to be compacted.
 *	In compact mode we have the advantage that we can later re-use the
 *	duplicated message strings without re-open the original source files.
 *
 *	We save the following information into temporary file #2:
 *	+ The absolue position of the structure #3 of this message in the
 *		new file (1)
 *	+ At which position the string resides in the tempfile #1 (2)
 *	+ The absolute position to another message of the same length.
 *
 *	The temporary file #3 acts as an hash table and sorts all entries.
 *
 *	For the compact mode we need to sort all message string by their length.
 *	We accomplissh this task by adding all strings to a table, where
 *	the index into the table is the length of the message string.
 *	Collisions are resolved by chaining all messages of the same length
 *	into a single-linked list.
 *
 *	When the strings are dumped (with "compact" enabled this means after
 *	dump _all_ control areas of _all_ language definitions what makes it
 *	impossible to hold all strings within memory), we can update the
 *	reference in the structure #3 of the message by this method:
 *	1) Determine where the message string is located within the string
 *		area.
 *	2) Seek in the new file to (1) + offsetof(struct3, ms3_string)
 *	3) Dump the (dword) value := absolute_position_of_string
 *		- (1) + sizeof(struc3)
 *	This can be performed in any order of the messages themselves, regardless
 *	if they were compacted or not.
 */
void addStringToPool(MsgStru3 *msg3, FILE *fp, char *fn
	, FILE *msgFp, char *msgFn)
{	struct Pool msg;
	dword pos;				/* position is fp must remain */

	assert(msgFp);
	assert(fp);
	assert(fn);
	assert(msgFn);
	assert(msg3);

	if(!tmpStrings) {
		tmpStrings = tmpFile(TF_RW);
		tmpIndex = tmpFile(TF_RW);
		tmpHash = hashFile();
			/* the position "0" is used to mark the end of the
				hash chain, therefore we cannot use it to store
				an index entry in */
		Ewrite(aS(msg), sizeof(msg), tmpIndex->fp, tmpIndex->fn);
	}

	/* chain the message string into the hash-chain */
	if(msg3->ms3_length < 0 || msg3->ms3_length > 32767)
		error(E_longMsgStr, 32767);

	Egetpos(pos, tmpIndex->fp, tmpIndex->fn);
		/* current head of chain of messages of the same length */
	hfread(msg3->ms3_length, msg.nxt, tmpHash);
	hfrewrite(pos, tmpHash);		/* make this message the new head */

	/* Write the entry in the index file */
	Egetpos(msg.pos, tmpStrings->fp, tmpStrings->fn);
	Egetpos(msg.ms3, msgFp, msgFn);

	Ewrite(aS(msg), sizeof(msg), tmpIndex->fp, tmpIndex->fn);

	/* Duplicate string into the temporary string file */
	/* The position within fp must remain unchanged */
	Egetpos(pos, fp, fn);
	Eseekc(msg3->ms3_string, fp, fn);
	copyfile(msg3->ms3_length, tmpStrings->fp, tmpStrings->fn, fp, fn);
	Esetpos(pos, fp, fn);
}

/*
 *	Compact the message strings
 *
 *	We read the strings in descending order one by one and search them
 *	in this temporary file. If the string could be located within the
 *	file, the string is not appended, but re-uses older message strings.
 *	Because the strings are stored without the trailing '\0' character
 *	a message string can re-use one or more strings that have been
 *	immediately stored following each other.
 */
void compactLib(void)
{	int ch, c;
	struct Pool msg;
	char *buf, *p, *q;
	int size, length;
	TempFile *tmp;	/* will replace tmpStrings after compacting */
#ifdef _MICROC_
	dword tmp;
#endif /*#	defined(_MICROC_) */

	informative(M_compactMsgDB);

	tmp = tmpFile(TF_RW);
	/* Because Micro-C has trouble with read & write files, we
		change to attributes to read-only now */
	setMode(tmp->fp, TF_RD);

	/* The longest entry is placed at the very end of the hash file */
	buf = getmem(length = tmpHash->entries);
	while(length--) {
		hfread(length, msg.nxt, tmpHash);
		while(longtst(msg.nxt)) {
			Esetpos(msg.nxt, tmpIndex->fp, tmpIndex->fn);
			Eread(aS(msg), sizeof(msg), tmpIndex->fp, tmpIndex->fn);
			Esetpos(msg.pos, tmpStrings->fp, tmpStrings->fn);
			Eread(buf, length, tmpStrings->fp, tmpStrings->fn);

			/* Now we must search the new temporary file for the
				buffered message string. To save memory, especially
				for Micro-C, we read the new file character per
				character */
			Eseek(0, SEEK_SET, tmp->fp, tmp->fn);
			c = *buf;
			iM( longset(tmp, 1) );
			do {
				while((ch = getc(tmp->fp)) != EOF && ch != c);
				Egetpos(msg.pos, tmp->fp, tmp->fn);

				if(ch == EOF) {		/* not found -> append */
					setMode(tmp->fp, TF_WR);
					Ewrite(buf, length, tmp->fp, tmp->fn);
					setMode(tmp->fp, TF_RD);
					break;
				}

				/* possible start of message string found */
				/* msg.pos points one charater too far */
				p = buf;
				size = length;
				while(--size && (ch = getc(tmp->fp)) != EOF
				 && ch == *++p);

				if(!size) {		/* we found the complete string */
					inM( longsub(msg.pos, tmp) , --msg.pos );
					break;
				}

				if(ch == EOF) {		/* we found a partial string
									at the end of the area */
					/* append only the characters not in the message */
					setMode(tmp->fp, TF_WR);
					Ewrite(p + 1, size, tmp->fp, tmp->fn);
					setMode(tmp->fp, TF_RD);
					inM( longsub(msg.pos, tmp) , --msg.pos );
					break;
				}

				/* We've found no restart point --> we can newly 
					start scanning */
				Esetpos(msg.pos, tmp->fp, tmp->fn);
			} while(1);

			/* we have found a new offset for "msg.pos"
				==> rewrite the index */
			Eseek(-sizeof(msg), SEEK_CUR, tmpIndex->fp, tmpIndex->fn);
			setMode(tmpIndex->fp, TF_WR);
			Ewrite(aS(msg), sizeof(msg), tmpIndex->fp, tmpIndex->fn);
			setMode(tmpIndex->fp, TF_RD);
		}
	}
	free(buf);

	/* We have re-located all message strings into the new temporary
		file ==> replace the old with the new one */
	clTempFile(tmpStrings);
	tmpStrings = tmp;
}


/*
 *	Dump all strings into the binary message library and adjust the
 *	message references
 *
 *	After appending the string area we update all struct #3's string
 *	reference pointers. The index file contains the pointer within the
 *	string temp file; the offset to the strings themselves is calculated
 *	via 'pos'.
 */
void dumpMsgStrings(MSGLIB *Xfp, int compact)
{	dword pos;
	dword stringsize;
	MsgStru1 msg1;
	MsgStru3 msg3;
	struct Pool msg;
	iM(	dword tmp	);

	assert(Xfp);

	/* Because Micro-C has trouble with read & write files, we
		change to attributes to read-only now */
	setMode(tmpStrings->fp, TF_RD);
	setMode(tmpIndex->fp, TF_RD);
	setMode(tmpHash->fp, TF_RD);

	if(compact)				/* all strings shall be compacted */
		compactLib();

	/* stringsize := size of temporary string file
		This is the amount of bytes that will be appended to
		the binary message library and the offset that is to be
		added to the current displacement in the last structure #1
		of the library */
	Eseek(0, SEEK_END, tmpStrings->fp, tmpStrings->fn);
	Egetpos(stringsize, tmpStrings->fp, tmpStrings->fn);
	Egetpos(pos, Xfp->ml_fp, Xfp->ml_fnam);	/* where the strings are added */
	setMode(Xfp->ml_fp, TF_RD);
	Esetpos(Xfp->ml_lastCntrlArea, Xfp->ml_fp, Xfp->ml_fnam);
	Epeek(aS(msg1), sizeof(msg1), Xfp->ml_fp, Xfp->ml_fnam);
	longadd(msg1.ms1_next, stringsize);
	setMode(Xfp->ml_fp, TF_WR);
	Ewrite(aS(msg1), sizeof(msg1), Xfp->ml_fp, Xfp->ml_fnam);


	/* Rewrite the structures #3 */
	/* The position of the string is calculated as following:
			strpos := start_of_strings - start_of_msg3 - sizeof(msg3)
				+ offset_within_string_area

			start_of_strings == pos
			start_of_msg3 == msg.ms3
			offset_within_string_area == msg.pos
	*/
		/* the first entry is not used */
	setMode(Xfp->ml_fp, TF_RD);
	Eseek(sizeof(msg), SEEK_SET, tmpIndex->fp, tmpIndex->fn);
	while(fread(aS(msg), sizeof(msg), 1, tmpIndex->fp) == 1) {
		Esetpos(msg.ms3, Xfp->ml_fp, Xfp->ml_fnam);
		Epeek(aS(msg3), sizeof(msg3), Xfp->ml_fp, Xfp->ml_fnam);
#ifdef _MICROC_
		longcpy(msg3.ms3_string, pos);
		longsub(msg3.ms3_string, msg.ms3);
		longsub(msg3.ms3_string, tmp);
		longadd(msg3.ms3_string, msg.pos);
#else /*#	!(defined(_MICROC_)) */
		msg3.ms3_string = pos - msg.ms3 - sizeof(msg3) + msg.pos;
#endif /*#	defined(_MICROC_) */
		setMode(Xfp->ml_fp, TF_WR);
		Ewrite(aS(msg3), sizeof(msg3), Xfp->ml_fp, Xfp->ml_fnam);
		setMode(Xfp->ml_fp, TF_RD);
	}
	Esetpos(pos, Xfp->ml_fp, Xfp->ml_fnam);

	/* copy the string area to the message library */
	Eseek(0, SEEK_SET, tmpStrings->fp, tmpStrings->fn);
	setMode(Xfp->ml_fp, TF_WR);
	copyfilel(stringsize, Xfp->ml_fp, Xfp->ml_fnam
			, tmpStrings->fp, tmpStrings->fn);
	clTempFile(tmpStrings);
	clTempFile(tmpIndex);
	clHashFile(tmpHash);
}


/*
 *	Dump control area of current language definition into the
 *	binary message library.
 *	In the global structure "lng" the following information has been
 *	gathered together:
 *	1) All information of structure #1 (language code, number of aliases,
 *		number of groups)
 *	2) All information for the structure #2 of all groups,
 *	3) where to read the information of all the structure #3's from
 *		except the pointers to the strings.
 *	The "nxt" field of the structure #1 is initialized with the
 *	byte immediately following the dump of the complete control area.
 *	This is the correct value for the "compacted mode".
 *	To allow to change this value for the last definition to point
 *	behind the string area, the position of the actual structure #1
 *	is stored within the MSGLIB structure, so the dumpMsgStrings()
 *	function can correct this pointer.
 */
void dumpCntrlArea(MSGLIB *Xfp)
{	MsgStru1 msg1;
	MsgStru2 msg2;
	MsgStru3 msg3;
	int i;
	unsigned entries;
	msgTLngcode *pAl1, *pAl2;
	FILE *msgFp, *fp;
	char *msgFn, *fn;
#ifdef _MICROC_
	dword tmp1, tmp2;
#endif /*#	defined(_MICROC_) */

	assert(Xfp);

	msg1.ms1_magicnumber = MSG_MN_STRU1;
	msg1.ms1_lng = lng.lngcode;
	msg1.ms1_charsets = lng.numCharsets;

	/* Ignore the disabled aliases */
	msg1.ms1_aliases = 0;
	pAl1 = pAl2 = lng.aliases;
	while(lng.numAliases--)
		if(!(*pAl1++ & DISABLED_ALIAS_MASK)) {		/* enabled alias */
			++msg1.ms1_aliases;
			*pAl2++ = pAl1[-1];
		}


	/* Count the defined groups */
	/* If no group containing one message is found, this language definition
		is droped */
	msg1.ms1_groups = 0;
	i = MSGGROUPMAX;
	while(i--)
		if(lng.groups[i].fnam) {
			if(lng.groups[i].entries)
				++msg1.ms1_groups;
			else lng.groups[i].fnam = NULL;		/* NO entry! */
		}
	if(!msg1.ms1_groups) {
		warning(W_noGroup, lng.lngcode);
		goto errRet;
	}

/* Now is the structure #1 completely filled ==> write it (except ms1_next) */
	/* First, we save the current position */
	Egetpos(Xfp->ml_lastCntrlArea, msgFp = Xfp->ml_fp, msgFn = Xfp->ml_fnam);
	Ewrite(aS(msg1), sizeof(msg1), msgFp, msgFn);
	/* write the supported character sets */
	Ewrite(lng.charsets, sizeof(msgTCharset) * msg1.ms1_charsets, msgFp, msgFn);
	/* write the aliases */
	if(msg1.ms1_aliases)
		Ewrite(lng.aliases, sizeof(msgTLngcode) * msg1.ms1_aliases, msgFp, msgFn);

/* Now we write all structure #2's */
	/* The pointer to the associated first structure #3 is calculated
		as following:
			poi := sizeof(struc3) * number_of_entries_of_all_previous_groups
				+ sizeof(struc2) * number_of_all_following_groups
		The number_of_entries is accumulated into 'entries', the number
		of following groups is accumulated into 'msg1.ms1_groups'.
	*/
	entries = 0;
	i = MSGGROUPMAX;
	while(i--) if(lng.groups[i].fnam) {
		msg2.ms2_gid = i;
#ifdef _MICROC_
		longset(msg2.ms2_ms3, sizeof(MsgStru2) * --msg1.ms1_groups);
		longset(tmp1, entries);
		longmul(tmp1, tmp2);		/* tmp2 == sizeof(struc3) */
		longadd(msg2.ms2_ms3, tmp1);
#else /*#	!(defined(_MICROC_)) */
		msg2.ms2_ms3 = sizeof(MsgStru2) * --msg1.ms1_groups
		 + (long)sizeof(MsgStru3) * entries;
#endif /*#	defined(_MICROC_) */
		entries += msg2.ms2_num = lng.groups[i].entries;
		Ewrite(aS(msg2), sizeof(msg2), msgFp, msgFn);
	}

/* Now we copy all structure #3's of all groups and add all strings to
	the pool of strings to be dumped later */
	i = MSGGROUPMAX;
	while(i--) if((fn = lng.groups[i].fnam) != NULL) {
		fp = Efopen(fn, "rb");
		Esetpos(lng.groups[i].pos, fp, fn);		/* seek to 1st msg3 */
		entries = lng.groups[i].entries;
		while(entries--) {
			Eread(aS(msg3), sizeof(msg3), fp, fn);
			addStringToPool(aS(msg3), fp, fn, msgFp, msgFn);
			Ewrite(aS(msg3), sizeof(msg3), msgFp, msgFn);
		}
		Efclose(fp, fn);
	}

	/* Now all data is written -> update next field */
	Egetpos(msg1.ms1_next, msgFp, msgFn);
	longsub(msg1.ms1_next, Xfp->ml_lastCntrlArea);
	Esetpos(Xfp->ml_lastCntrlArea, msgFp, msgFn);
	Ewrite(aS(msg1), offsetof(MsgStru1, ms1_next)
	 + sizeof(dword), msgFp, msgFn);
	Eseek(0, SEEK_END, msgFp, msgFn);

errRet:
	free(lng.aliases);
	free(lng.charsets);
	lng.lngcode = 0;
}


/*********************************************************************/
/**************** Library reading functions **************************/
/*********************************************************************/

/*
 *	See msgIDs.h
 */
struct ScanMsgID {
	unsigned entries;
	FILE *fp;
	char *fn;
};

static int scanNxtMsgID(void *arg, MSGID2 *id, char **name
	, int *length, int *buflen)
{	MsgStruID msgID;
	struct ScanMsgID *a;

	assert(id);
	assert(name);
	assert(length);
	assert(buflen);

	a = cS(struct ScanMsgID *)arg;
	Eread(aS(msgID), sizeof(msgID), a->fp, a->fn);
	if(!msgID.msID_id)		/* end of list reached */
		return 0;
	
	if(*buflen <= msgID.msID_length)
		*name = chgmem(*name, *buflen = msgID.msID_length + 1);

	assert(*name);

	*id = msgID.msID_id;
	*length = msgID.msID_length;
	Eread(*name, *length, a->fp, a->fn);

	return 1;
}

/*
 *	Read all language definitions
 *	The file must be positioned at the first language definition.
 */
void scanLngDefs(FILE *fp, char *fn, dword iM(*) limit)
{	MsgStru1 msg1;
	MsgStruIDS msgIDS;
	dword pos;
	struct ScanMsgID arg;

	assert(fp);
	assert(fn);
	iM(assert(limit));

	/* Read in all the language definitions */
	do {
		Egetpos(pos, fp, fn);
		if(longcmp1(pos, limit) >= 0)
			break;			/* upper limit reached */

		/* here must be another language definition */

		/* read in the structure #1 of this language definition */
		Epeek(aS(msg1), sizeof(msg1), fp, fn);
		if(!isMagicNumber(msg1.ms1_magicnumber))
			error(E_brokenMsgLib, fn, pos);

		switch(msg1.ms1_magicnumber) {
		case MSG_MN_STRU1:
			/* create a new entry */
			++entries;
			actEntry = getmem(sizeof(struct MsgList));
			if(!last) head = last = actEntry;
			else last = last->nxt = actEntry;

			longcpy(last->pos, pos);
			last->fnam = makeString(fn);

			if((last->lng = msg1.ms1_lng) == 0)
				error(E_invldLngCode, fn);
			break;

		case MSG_MN_STRU0:
			goto ret;

		case MSG_MN_IDS:
			Eread(aS(msgIDS), sizeof(msgIDS), fp, fn);
			arg.fp = fp;
			arg.fn = fn;
			binf.msgIDs = scanMsgIDS(binf.msgIDs, aF(scanNxtMsgID), &arg);
			break;
		}

		Esetpos(pos, fp, fn);
		Eseekc(msg1.ms1_next, fp, fn);			/* the start of msg1 */
	} while(longtst(msg1.ms1_next));

ret:
}


/*
 *	Identify if this is a binary message library and where the
 *	header is located and where the endPos is located
 *
 *	Note: I consider the seek & tell operations as very light weight
 *	operation.
 */
int identLib(char *fn, dword inM(*limit, *Xlimit), int *Xheader, FILE **Xfp)
{	int header;
	FILE *fp;
	MsgStru0 msg0;
	dword pos;
	word magicNumber;
#ifndef _MICROC_
	dword limit;
#endif /*#	!defined(_MICROC_) */

	assert(fn);
	assert( inM(limit, Xlimit) );
	assert(Xheader);
	assert(*Xfp);

	header = *Xheader;

	fp = Efopen(fn, "rb");

	/* probe, if the file is large enough */
	Eseek(0, SEEK_END, fp, fn);
	Egetpos(pos, fp, fn);
	Eseek(0, SEEK_SET, fp, fn);

	if(!longtst(pos)) {			/* file is empty */
		warning(W_emptyMsgLib, fn);
		goto errRet;
	}

#ifdef _MICROC_
	if(!(pos->hi || pos->lo >= max(sizeof(msg0), sizeof(msg1))))
#else /*#	!(defined(_MICROC_)) */
	if(pos < max(sizeof(MsgStru0), sizeof(MsgStru1)))
#endif /*#	defined(_MICROC_) */
	{
		warning(W_smallMsgLib, fn);
		goto errRet;
	}

	/* Identify where the header is located */
	switch(header) {
	default:				/* search for it */
		Eread(aS(msg0), sizeof(msg0), fp, fn);
		if(identMSGLIB(aS(msg0))) {
			header = MH_BEG;
			break;
		}
	case MH_END:			/* at the end! */
		Eseek(-sizeof(msg0), SEEK_END, fp, fn);
	case MH_BEG:			/* at the beginning! */
		Eread(aS(msg0), sizeof(msg0), fp, fn);
		if(!identMSGLIB(aS(msg0))) {
			if(header == MH_ANY) {
				Eseek(0, SEEK_SET, fp, fn);
				Eread(&magicNumber, sizeof(magicNumber), fp, fn);
				if(isMagicNumber(magicNumber)) {
					header = MH_NON;
					goto noHeader;
				}
				header = MH_ERR;
				goto okRet;
			}
			goto errRet;
		}
		if(header == MH_ANY)
			header = MH_END;
		break;
	case MH_NON: 		/* no header --> all the file is the libary */
noHeader:
		clrobj(msg0);
		break;
	}

	if(header == MH_END)	{
		Egetpos(limit, fp, fn);
#ifdef _MICROC_
		longset(pos, sizeof(msg0));
		longsub(limit, pos);
#else /*#	!(defined(_MICROC_)) */
		limit -= sizeof(msg0);
#endif /*#	defined(_MICROC_) */
	}
	else {		/* the upper limit is the file size */
		Egetpos(pos, fp, fn);
		Eseek(0, SEEK_END, fp, fn);
		Egetpos(limit, fp, fn);
		Esetpos(pos, fp, fn);
	}

	/* seek to the first language definition */
	if(identMSGLIB(aS(msg0)))
		Eseekc(msg0.ms0_address, fp, fn);
	else Eseek(0, SEEK_SET, fp, fn);

okRet:
	*Xfp = fp;
	*Xheader = header;
#ifndef _MICROC_
	*Xlimit = limit;
#endif /*#	!defined(_MICROC_) */
	return 1;

errRet:
	Efclose(fp, fn);
	return 0;
}



/*
 *	Identify the binary file
 */
void identBinLib(void)
{	char *fn;
	FILE *fp;
#ifdef _MICROC_
	dword tmp;
#endif /*#	defined(_MICROC_) */

	if(!fileExists(fn = binf.fn)) {
		informative(M_creatMsgLib, fn);
		/* To ease the later processing, an empty file is created, if
			the binary file does not exist. */
		if((fp = fopen(fn, "w")) == NULL)
			error(E_creatBinLib, fn);
		fclose(fp);
	}

	binf.header = iHead;
	if(identLib(fn, aS(binf.endPos), &binf.header, &fp)
	 && binf.header != MH_ERR) {
		/* a binary message library was found
			binf.endPos contains the library's upper limit
			the current position of fp is the library's lower limit
			binf.header specifies whether or not the upper/lower limit
				is modified by an header
		*/
		Egetpos(binf.startPos, fp, fn);		/* where the 1. lng def is */
#ifdef _MICROC_			/* calculate the length preceeding the 1. lng def */
		longcpy(binf.headLen, binf.startPos);
		if(binf.header == MH_BEG) {
			longset(tmp, sizeof(MsgStru0));
			longsub(binf.headLen, tmp);
		}
#else /*#	!(defined(_MICROC_)) */
		binf.headLen = binf.header == MH_BEG
			? binf.startPos - sizeof(MsgStru0)
			: binf.startPos;
#endif /*#	defined(_MICROC_) */

		/* calcualte the length of the part of the binary file following
			the binary message library */
		Eseek(0, SEEK_END, fp, fn);
		Egetpos(binf.endLen, fp, fn);
		longsub(binf.endLen, binf.endPos);
		if(binf.header == MH_END) {
#ifdef _MICROC_
			longset(tmp, sizeof(MsgStru0));
			longsub(binf.endLen, tmp);
#else /*#	!(defined(_MICROC_)) */
			binf.endLen -= sizeof(MsgStru0);
#endif /*#	defined(_MICROC_) */
		}
		Esetpos(binf.startPos, fp, fn);
		Efclose(fp, fn);
	}
	else {
		/* In this case, no binary message libary was found
			or contains no messages. */
		longset(binf.startPos, 0);
		longset(binf.endPos, 0);
		longset(binf.endLen, 0);

		if(binf.header == MH_ERR) {
			binf.header = MH_END;		/* by default append stru0 */
			Eseek(0, SEEK_END, fp, fn);
			Egetpos(binf.headLen, fp, fn);
			Efclose(fp, fn);
		}
		else
			longset(binf.headLen, 0);
	}
}


/*
 *	Read in all entries and update oldEntries/entries
 *	Identifies where the header is located
 *	Returns the location of the header.
 */
#define scanLib(fnam) scanLib_(fnam, iHead)
int scanLib_(char *fn, int header)
{	FILE *fp;
	dword limit;

	assert(fn);

	oldEntries = entries;

	if(identLib(fn, aS(limit), &header, &fp)) {
		scanLngDefs(fp, fn, limit);
		Efclose(fp, fn);
	}
	else 
		error(E_noMsgLib, fn, header == MH_BEG? M_headBEG: M_headEND);


	return header;
}


/*
 *	Scan the binary message file
 *	
 */
void scanBinLib(void)
{	FILE *fp;

	fp = Efopen(binf.fn, "rb");
	Esetpos(binf.startPos, fp, binf.fn);
	scanLngDefs(fp, binf.fn, binf.endPos);
	Efclose(fp, binf.fn);
	informative(entries == 1? M_entOne: M_entMultiple, M_entFoundBinLib, entries);
}


/*
 *	Display changes of the list
 */
void dispDiffEnt(void)
{	MSGID op;
	unsigned diff;

	if(oldEntries == entries)
		informative(M_entNoChange);
	else {
		if(oldEntries < entries) {
			op = M_entAdded;
			diff = entries - oldEntries;
		}
		else {
			op = M_entRemoved;
			diff = oldEntries - entries;
		}
		informative(diff == 1? M_entOne: M_entMultiple, op, diff);
		oldEntries = entries;
	}
}

/*
 *	Create the binary file based upon the original binary file and the
 *	temporary binary message library.
 *
 *	If no backup file is already created, one is created.
 */
void makeNewBinFile(char *fnam)
{	dword tmp;
	FILE *fp, *f1, *f2;
	char *p, *q;
	MsgStru0 msg0;

	assert(fnam);

	if(!binf.bak) {		/* create the backup */
		p = getmem(strlen(binf.fn) + 4);
		q = stpcpy(p, binf.fn);
		while(--q >= p && !strchr("\\/:.", *q));
		if(*q == '.')
			strcpy(q, extBAK);
		else	strcat(p, extBAK);

		binf.bak = makeString(p);
		remove(p);
		if(rename(binf.fn, binf.bak))
			error(E_backupBinLib, binf.fn, binf.bak);
	}

	mkStru0(aS(msg0));
	fp = Efopen(binf.fn, "wb");
	f1 = Efopen(binf.bak, "rb");
	f2 = Efopen(fnam, "rb");
	copyfilel(binf.headLen, fp, binf.fn, f1, binf.bak);
	if(binf.header == MH_BEG)
		Ewrite(aS(msg0), sizeof(msg0), fp, binf.fn);
	Egetpos(binf.startPos, fp, binf.fn);
	dumpBinMsgIDS(binf.msgIDs, fp, binf.fn);
	copyfilee(fp, binf.fn, f2, fnam);
	Egetpos(binf.endPos, fp, binf.fn);
	if(binf.header == MH_END) {
#ifdef _MICROC_
		longcpy(msg0.ms0_address, binf.startPos);
		longset(tmp, sizeof(msg0));
		longsub(msg0.ms0_address, tmp);
		longsub(msg0.ms0_address, binf.endPos);
#else /*#	!(defined(_MICROC_)) */
		msg0.ms0_address = binf.startPos - sizeof(msg0) - binf.endPos;
#endif /*#	defined(_MICROC_) */
		Ewrite(aS(msg0), sizeof(msg0), fp, binf.fn);
	}
#ifdef _MICROC_
	longset(tmp, 0);
	longsub(tmp, binf.endLen);
#else /*#	!(defined(_MICROC_)) */
	tmp = -binf.endLen;
#endif /*#	defined(_MICROC_) */
	Eseekx(tmp, SEEK_END, f1, binf.bak);
	copyfilel(tmp, fp, binf.fn, f1, binf.bak);
	Efclose(fp, binf.fn);
	Efclose(f1, binf.bak);
	Efclose(f2, fnam);
}


/*********************************************************************/
/**************** Helper functions ***********************************/
/*********************************************************************/

/*
 *	Writes out one language definition in a uncompacted form
 */
void dumpLngDef(MSGLIB *fp, struct MsgList *entry)
{	assert(fp);
	assert(entry);

	prepCntrlArea(entry);
	dumpCntrlArea(fp);
	dumpMsgStrings(fp, 0);
}


/*
 *	Add a new entry into the alias modification list
 */
void addAliasEntry(int toadd, int lng, int alias)
{	struct AliasList *h;

	h = getmem(sizeof(struct AliasList));
	if(aLast) aLast = aLast->nxt = h;
	else aHead = aLast = h;

	aLast->nxt = NULL;
	aLast->toadd = toadd;
	aLast->lng = lng;
	aLast->alias = alias;
}

/*
 *	Remove one entry from the message definition list
 */
void removeEntry(struct MsgList *entry, struct MsgList *prev)
{
	assert(entry);

	if(!entry) return;

	if(entry == last)
		last = prev;

	if(prev)	prev->nxt = entry->nxt;
	else		head = entry->nxt;

	disposeString(entry->fnam);
	free(entry);
	--entries;
}


/*
 *	Search the message definition list for an entry
 *	"act" is always pointing to the entry probed first.
 *	When a match is found, "nxtEntry()" sets act to the next
 *	entry (or NULL) and returns the found entry.
 *	If act == NULL on entry, a NOP is performed.
 */
struct MsgList *nxtEntry(void)
{	struct MsgList *act;
	int i;

	if((act = actEntry) == NULL) return NULL;

	i = cmdOpts.num1;

	do {
		if(cmdOpts.entry) {
			if(i-- == 0) {
				actEntry = NULL;		/* one direct entry specified */
				return act;
			}
		}
		else if(i == act->lng) {
			actEntry = act == last? NULL: act->nxt;
			return act;
		}
		prvEntry = act;
	} while(act != last && (act = act->nxt) != NULL);

	return actEntry = NULL;
}

struct MsgList *cmdEntry(void)
{
	actEntry = head;
	prvEntry = NULL;
	return nxtEntry();
}

/*
 *	Display an error message for a failed cmdEntry() operation
 */
void errMatch(void)
{
	error(cmdOpts.entry? E_entryTooHigh
				: E_noSuchLng, cmdOpts.num1, entries);
}


/*
 *	Write a single character as hexadecimal value
 */
void writeHex(unsigned ch)
{	printf("\\x%x", ch & 0xff);
}

/*
 *	Write the string of a message onto stdout
 */
void writeMsgString(byte *buf, unsigned length)
{	int ch;
	char *p;
	static char from[] = "\t\v\f\r\n";
	static char to[] = "tvfrn";
	struct MsgIDhash hash;
	char *name;

	assert(buf);

	if(length)
		fputs("|||", stdout);
	while(length--) {
		ch = *buf++;
		if(ch == '%' && length) {		/* handle enclosures */
			if((ch = *buf++) == '-' && *buf == '%' && length >= 4) {
				/* special enclosure */
				fwrite(buf - 2, 4, 1, stdout);	/* the detection part */
				length -= 4;			/* '-%r<count>' == 4 chars */
				switch(*++buf) {		/* which enclosure? */
				case 'm':				/* m<msgid> */
					if(getSymID(binf.msgIDs
					 , *(MSGID2*)(++buf), aS(hash), &name)) {
						name[hash.length] = '\0';
						fputs(name, stdout);
						putchar('%');
						buf += sizeof(MSGID2) - 1;
						length -= sizeof(MSGID2) - 1;
						break;
					}
					writeHex(*++buf);
					--length;
				case 'W':				/* W<size> */
				case 'A':				/* A<size> */
				case 'R':				/* R<count> */
				case 'r':				/* r<count><character> */
					writeHex(*++buf);
					/* the charater(s) is emitted normally */
					break;
				default:				/* assume no hex output */
					++length;
					break;
				}
				++buf;
				continue;
			}
			putchar('%');	/* we emit both characters, that will */
			putchar(ch);	/* handle misinterpretation of '%%' */
			--length;
		}
		else if(ch == '\n')
			fputs("\n|||", stdout);
		else if(!iscntrl(ch)) putchar(ch);
		else {
			if(ch && (p = strchr(from, ch)) != NULL) {
				putchar('\\');
				putchar(to[p - from]);
			}
			else writeHex(ch);
		}
	}
	putchar('\n');
}


/*
 *	View an entry
 */
void viewEntry(struct MsgList *entry, int idx, int verbosely)
{	unsigned j, msgid;
	MsgStru1 msg1;
	MsgStru2 msg2;
	MsgStru3 msg3;
	FILE *fp;
	char *fn;
	msgTLngcode alias;
	msgTCharset charset;
	dword pos, pos1, tmp;
	byte *buf;
	struct MsgIDhash hash;
	char *name;

	assert(entry);

	openEntry(entry, aS(msg1), &fp, &fn);
	if(idx)
		smessage(M_entSepNum);
	smessage(M_entNum, idx, fn = entry->fnam, entry->pos);
	smessage(M_entLng, msg1.ms1_lng, msg1.ms1_lng);

	smessage(M_entCharsets, msg1.ms1_charsets);
	j = 1;
	do {
		if(--j)
			smessage(M_entSepCharset);
		else {
			smessage(M_ent1Charset);
			j = 4;
		}
		Eread(&charset, sizeof(charset), fp, fn);
		switch(charset) {
		case CHARSET_NONE:	smessage(M_entCSNONE); break;
		case CHARSET_7BITASCII:	smessage(M_entCS7bitascii); break;
		case CHARSET_UNICODE16:	smessage(M_entCSUnicode16); break;
		default: smessage(M_entCodepage, charset); break;
		}
	} while(--msg1.ms1_charsets);
	smessage(M_entEndCharset);

	if(msg1.ms1_aliases) {
		smessage(M_entAliases, msg1.ms1_aliases);
		j = 1;
		do {
			if(--j)
				smessage(M_entSepAlias);
			else {
				smessage(M_ent1Alias);
				j = 4;
			}
			Eread(&alias, sizeof(alias), fp, fn);
			smessage(M_entAlias, alias, alias);
		} while(--msg1.ms1_aliases);
		smessage(M_entEndAlias);
	}
	else smessage(M_entNoAliases);

	if((idx = msg1.ms1_groups) != 0) {
		Egetpos(pos, fp, fn);
		do {
			Eread(aS(msg2), sizeof(msg2), fp, fn);
			smessage(M_entGroup, msg2.ms2_gid, msg2.ms2_num);
		} while(--idx);
		if(verbosely) {
			smessage(M_entMsgStrings);
			idx = msg1.ms1_groups;
			do {
				Esetpos(pos, fp, fn);
				Eread(aS(msg2), sizeof(msg2), fp, fn);
				Egetpos(pos, fp, fn);
				Eseekc(msg2.ms2_ms3, fp, fn);
				msgid = 0;
				Egetpos(pos1, fp, fn);		/* Start offset of entry */
				while(msg2.ms2_num--) {
					Eread(aS(msg3), sizeof(msg3), fp, fn);
					Egetpos(tmp, fp, fn);
					longadd(tmp, msg3.ms3_string );
					if(!getSymID(binf.msgIDs, mkMsgID(msg2.ms2_gid, ++msgid)
					 , aS(hash), &name))
					 	name = "";
					else name[hash.length] = '\0';
					smessage(M_entMsg, name, msg2.ms2_gid, msgid
					 , msg3.ms3_exitcode, j = msg3.ms3_length, pos1
					 , tmp );
					Egetpos(pos1, fp, fn);
					if(j) {		/* non-empty message */
						buf = getmem(j);
						Eseekc(msg3.ms3_string, fp, fn);
						Eread(buf, j, fp, fn);
						Esetpos(pos1, fp, fn);
						writeMsgString(buf, j);
						free(buf);
					}
				}
			} while(--idx);
		}
	}
	else smessage(M_entNoGroups);

	Efclose(fp, fn);

}

void viewAllEntries(int verbosely)
{	int i;

	if(!entries || !last || !head) {
		smessage(M_noEntries);
		return;
	}

	i = -1;
	last->nxt = NULL;		/* Make sure the list is terminated */
	actEntry = head;
	do viewEntry(actEntry, (unsigned)(++i), verbosely);
	while((actEntry = actEntry->nxt) != NULL);
}



/*********************************************************************/
/**************** Command implementations ****************************/
/*********************************************************************/

/* The following implementations are available:
	libAppend	to append a binary message file (pc)
	libInsert	to insert a binary message file before an entry/language (pc)
	libRemove	to remove an entry/language (pc)
	libExtract	to extract an entry/language
	libCompact	to compact the binary file (pc)
	libAddAlias	to add an alias language to a language (pc)
	libSubAlias	to subtract an alias language to a language (pc)
	libView		to verbosely output one entry
	libViewAll	to view all entries
	libViewAllVerbose	to verbosely view all entries
	libPerform	to actually perform all pending commands

	Commands marked with (pc) are pending commands that are not performed
	immediately, but by performing a 'libPerform' command all at once.
*/

/*
 *	Make all future 'libPerform's to compact the binary message library.
 *	No field used within cmdOpts.
 */
void libCompact(void)
{	compactMsgLib = 1;
	libModified = 1;
}

/*
 *	Append a binary message library to the current one.
 *	cmdOpts.fnam holds the file to read in.
 */
void libAppend(void)
{	
	scanLib(cmdOpts.fnam);
	if(oldEntries != entries)
		libModified = 1;
	dispDiffEnt();
}


/*
 *	Insert a binary message library nto the current one.
 *	cmdOpts.fnam holds the file to read in.
 *	cmdOpts.num1 specifies the position (either language or entry)
 *	cmdOpts.entry specifes to interprete num1 as entry
 */
void libInsert(void)
{	
	struct MsgList *oldLast, *entry;

	if((entry = cmdEntry()) == NULL) {
		errMatch();
		return;
	}

	/* a matching entry was found */
	/* scanLib() will always append any found definition to 'last'.
		We patch 'last' to 'prev', in order scanLib() will append
		the new entries there, then we patch the last->nxt to
		point to 'entry' and reset 'last' to its original value. */

	oldLast = last;	/* last cannot be NULL as we found an entry! */
	last = prvEntry;

	scanLib(cmdOpts.fnam);
	if(oldEntries != entries) {
		if(last) last->nxt = entry;
		else head = entry;		/* nothing added to top of list */
		libModified = 1;
	}
	last = oldLast;
	dispDiffEnt();
}


/*
 *	Remove a language definition.
 *	cmdOpts.num1 specifies either the language or entry to remove
 *	cmdOpts.entry specifes to interprete num1 as entry
 */
void libRemove(void)
{	
	struct MsgList *entry;

	oldEntries = entries;
	if((entry = cmdEntry()) != NULL) do {
		removeEntry(entry, prvEntry);
	} while((entry = nxtEntry()) != NULL);

	if(oldEntries == entries)
		errMatch();
	else {
		dispDiffEnt();
		libModified = 1;
	}
}


/*
 *	Add a new alias to all definitions of one language
 *	cmdOpts.num1 is the language to alter
 *	cmdOpts.num2 is the alias
 */
void libAddAlias(void)
{	addAliasEntry(1, cmdOpts.num1, cmdOpts.num2);
}

/*
 *	Subtract an alias from all definitions of one language
 *	cmdOpts.num1 is the language to alter
 *	cmdOpts.num2 is the alias
 */
void libSubAlias(void)
{	addAliasEntry(0, cmdOpts.num1, cmdOpts.num2);
}

/*
 *	Extract one or more entries from the binary message library
 *	cmdOpts.fnam holds the file to be written.
 *	cmdOpts.num1 specifies the position (either language or entry)
 *	cmdOpts.entry specifes to interprete num1 as entry
 */
void libExtract(void)
{	MSGLIB *fp;			/* output file */
	struct MsgList *entry;

	if((entry = cmdEntry()) == NULL) {
		errMatch();
		return;
	}

	fp = mkMsgLib(cmdOpts.fnam, oHead);
	dumpBinMsgIDS(binf.msgIDs, fp->ml_fp, fp->ml_fnam);
	do dumpLngDef(fp, entry);
	while((entry = nxtEntry()) != NULL);
	clMsgLib(fp);
}

/*
 *	Perform all pending operations
 */
void libPerform(void)
{	MSGLIB *fp;
	char *fnam;

	if(!libModified) return;

	fp = mkTmpMsgLib();
	fnam = makeString(fp->ml_fnam);
	if(entries && last && head) {
		last->nxt = NULL;		/* make sure the list is terminated */
		while((actEntry = head) != NULL) {
			if(actEntry->lng) { 	/* lng == 0 marks an invalid/empty entry */
				if(compactMsgLib) {
					prepCntrlArea(actEntry);
					/* Now try if we can find another entry of this
						language, we will join both */
					if((last = actEntry->nxt) != NULL) do {
						if(last->lng == actEntry->lng && addCntrlArea(last))
							last->lng = 0;
					} while((last = last->nxt) != NULL);
					dumpCntrlArea(fp);
				}
				else	dumpLngDef(fp, actEntry);
			}
			head = actEntry->nxt;
			removeEntry(actEntry, NULL);
		}

		if(compactMsgLib)
			dumpMsgStrings(fp, compactMsgLib);

	}
	last = NULL;
	entries = 0;
	clMsgLib(fp);
	makeNewBinFile(fnam);
	remove(fnam);
	disposeString(fnam);
	scanBinLib();
}


/*
 *	Display all entries to the screen
 */
void libViewAll(void)
{	viewAllEntries(0);
}
void libViewAllVerbose(void)
{	viewAllEntries(1);
}

/*
 *	Display one entry verbosely
 */
void libView(void)
{	struct MsgList *entry;
	
	if((entry = cmdEntry()) != NULL)
		viewEntry(entry, cmdOpts.num1, 1);
	else
		errMatch();
}

/*********************************************************************/
/**************** Command line parsing tools *************************/
/*********************************************************************/


char *fetchNumber(char *p, int *pnum)
{	int num;

	assert(p);
	assert(pnum);

	if(!isdigit(*p)) return NULL;

	num = 0;
	if((p[1] == 'x' || p[1] == 'X') && isxdigit(p[2])) {
		p += 2;
		do num = (num << 4) | (*p
		 - (isdigit(*p)? '0': isupper(*p)? 'A' - 10: 'a' - 10));
		while(isxdigit(*++p));
	}
	else
		do num = num * 10 + *p - '0';
			while(isdigit(*++p));

	*pnum = num;
	return p;
}


int getNumber(char *p, MSGID errMsg, char *cmd)
{	int num;

	assert(p);
	assert(cmd);

	if((p = fetchNumber(p, &num)) == NULL || *p)
		error(errMsg, cmd);
	return num;
}

/*
 *	Identify a command by the passed argument and fill in the
 *	global structure cmdOpts.
 */
void idCmd(char *Xcmd)
{	char *cmd;
	char *cmds[] = {
		"compact",
		"alias:",
		"append:",
		"insert:",
		"remove:",
		"view",
		"view:",
		"extract:",
		"perform" };
	int cmdIdx, cmdLength;
	MSGID errMsg;

	assert(Xcmd);

	clrobj(cmdOpts);
	cmdOpts.msgNoFnam = E_reqFnam;

	cmd = Xcmd;
	switch(*cmd++) {
	case '+':		/* short hand add/insert */
	/*	"+ <file>" tread 'file' as another binary message library, which is
		appended to binary_file.

		"+<number> <file>" insert 'file' before the first occurance of
		the language with the country code 'number'.

		"+#<number> <file>" insert 'file' before the 'number'th entry.
	*/
		if(*cmd == NUL) {		/* first version */
			cmdOpts.fct = aF(libAppend);
			cmdOpts.needFnam = 1;
			break;
		}
		cmdOpts.fct = aF(libInsert);
		errMsg = E_cmdInsert;
	fetchNumFnam:
		cmdOpts.needFnam = 1;
	fetchNum:
		if(*cmd == '#') {
			++cmd;
			cmdOpts.entry = 1;
		}
		cmdOpts.num1 = getNumber(cmd, errMsg, Xcmd);
		break;

	case '-':			/* short hand removeable */
	/*	"-<number>" remove any definition of the language code 'number'.

		"-#<number>" remove the 'number'th entry.
	*/
	cmdRemove:
		cmdOpts.fct = aF(libRemove);
		errMsg = E_cmdRemove;
		goto fetchNum;

	case '*':			/* short hand extraction */
	/*	"*<number> <file>" extract any definition of language 'number'
		into the 'file'.

		"*#<number> <file>" extract the 'number'th entry into the 'file'.
	*/
		cmdOpts.fct = aF(libExtract);
		errMsg = E_cmdExtract;
		goto fetchNumFnam;

	case '@':			/* output or command file */
	/*	"@#<number>" verbosely output the 'number'th entry.

		"@*" verbosely output any entry.

		"@ <file>" read 'file' line by line and interprete each line as
		exactly one command.
	*/
	cmdView:
		switch(*cmd) {
		case NUL:		/* source command file */
			cmdOpts.fct = aF(sourceIn);
			cmdOpts.needFnam = 1;
			break;
		case '*': 		/* view verbosely */
			cmdOpts.fct = aF(libViewAllVerbose);
			if(!cmd[1]) break;
		default:
			error(E_cmdView, Xcmd);
		case '#':
			cmdOpts.fct = aF(libView);
			errMsg = E_cmdView;
			goto fetchNum;
		}
		break;

	case '!':			/* long format command */
	/*	"!compact" try to compress all of the canned message definition
		without really using any compression algorithm. Compacting the
		message library is the only command to make sure that a previously
		removed entry is removed entirely (up to its last byte). Though,
		this operation may require a lot of time.

		"!alias:<number1>+<number2>" install the language 'number2' as
		an alias for all entries of the language 'number1'.

		"!alias:<number1>-<number2>" remove the language 'number2' as
		an alias for all entries of the language 'number1'.

		"!append:<file>" same as "+ file".

		"!insert:<number>,<file>" same as "+<number> <file>".

		"!insert:#<number>,<file>" same as "+#<number> <file>".

		"!remove:<number>" same as "-<number>".

		"!remove:#<number>" same as "-#<number>".

		"!view" same as "@*".

		"!view:#<number>" same as "@#<number>".

		"!extract:<number>,<file>" same as "*<number> <file>".

		"!extract:#<number>,<file>" same as "*#<number> <file>".

		"!perform" Perform pending "append"/"add"/"insert"/"remove"
		operations. If there are pending operations upon the termination
		of this program, "!perform" is automatically assumed. The command
		sequence: "+ file1" "*#1 file1" won't work because of the pending
		operations, as 'file1' has not been added before it is extracted.
		"+ file1" "!perform" "*#1 file1" would work instead.
	*/

		cmdOpts.needFnam = 0;
		for(cmdIdx = 0; cmdIdx < sizeof(cmds) / sizeof(char *); ++cmdIdx) {
			cmdLength = strlen(cmds[cmdIdx]);
			if(memcmp(cmd, cmds[cmdIdx], cmdLength) == 0
			 && (cmds[cmdIdx][cmdLength - 1] == ':'
			     || cmd[cmdLength] == NUL)) {
			    cmd += cmdLength;
				switch(cmdIdx) {
				case 0:			/* !compact */
					cmdOpts.fct = aF(libCompact);
					break;
				case 1:			/* !alias: */
					cmd = fetchNumber(cmd, &cmdOpts.num1);
					switch(cmd? *cmd: '?') {
					case '+':	cmdOpts.fct = aF(libAddAlias); break;
					case '-':	cmdOpts.fct = aF(libSubAlias); break;
					default:	error(E_cmdAlias, Xcmd);
					}
					cmdOpts.num2 = getNumber(cmd + 1, E_cmdAlias, Xcmd);
					break;
				case 2:			/* !append: */
					cmdOpts.fct = aF(libAppend);
					if(!*cmd) error(E_cmdAppend, Xcmd);
					cmdOpts.fnam = cmd;
					break;
				case 3:			/* !insert: */
					cmdOpts.fct = aF(libInsert);
					if(*cmd == '#') {
						++cmd;
						cmdOpts.entry = 1;
					}
					if((cmd = fetchNumber(cmd, &cmdOpts.num1)) == NULL
					 || *cmd != ',' || !*++cmd)
						error(E_cmdInsert, Xcmd);
					cmdOpts.fnam = cmd;
					break;
				case 4:			/* !remove: */
					goto cmdRemove;
				case 5:			/* !view */
					cmdOpts.fct = aF(libViewAllVerbose);
					break;
				case 6:			/* !view: */
					if(*cmd != '#')	error(E_cmdView, Xcmd);
					goto cmdView;
				case 7:			/* !extract: */
					cmdOpts.fct = aF(libExtract);
				case 8:			/* !perform */
					cmdOpts.fct = aF(libPerform);
					break;
				}
				return;
			}
		}
	default:
		error(E_nocmd, Xcmd);
	}
}


/*
 *	Evaluate /i= and /o= arguments
 */
int getIOArg(int opt, char *arg)
{	static char *args[] = {
		"none", "beginning", "start", "head", "end", "tail", "any" };
	int i, length;

	assert(arg);

	strlwr(arg);
	length = strlen(arg);
	for(i = 0; i < sizeof(args) / sizeof(args[0])
	 && memcmp(args[i], arg, length) != 0; ++i);
	switch(i) {
	case 0: return MH_NON;
	case 1: case 2: case 3: return MH_BEG;
	case 4: case 5: return MH_END;
	case 6: return MH_ANY;
	}
	error(E_IOarg, opt, arg);
}


/*
 *	Scan options during fetching the commands
 */
void scanOpts(int argc, char **argv)
{	int opt;

	assert(argv);

	while((opt = getoptG(argc, argv)) != EOF) switch(opt) {
	case 'I':	iHead = getIOArg('I', optarg); break;
	case 'O':	error(E_optOatstart);
	default:	hlpScreen(); break;
	}
}

/*
 *	Read a word out of the file and duplicate it into the dynamic memory
 *	The single und double quotes are recognized. The quotes are removed
 *	from the word.
 */
char *readWord(FILE *fp, char *fn)
{	char *buf, *bufend, *p;
	int ch, quote, length;

	assert(fp);
	assert(fn);

	bufend = 10 + ( p = buf = getmem(length = 10) );
	quote = 0;
	while((ch = getc(fp)) != EOF && (quote || !isspace(ch)))
		if(quote == ch)		/* quote ends here */
			quote = 0;
		else if(!quote && strchr(quotes, ch))	/* quote starts here */
			quote = ch;
		else {					/* new character for the word */
			*p++ = ch;
			if(p == bufend) {	/* extend the buffer */
				buf = chgmem(buf, length += 10);
				p = (bufend = buf + length) - 10;
			}
		}
	*p = '\0';
	return buf;
}


/*
 *	Source in a command file
 */
void sourceIn(void)
{	char *fn;
	FILE *fp;
	char *buf, *buf1;

	fn = makeString(cmdOpts.fnam);
	fp = Efopen(fn, "rt");
	while((buf = readWord(fp, fn)) != NULL) {
		idCmd(buf);
		if(cmdOpts.needFnam) {
			if((cmdOpts.fnam = buf1 = readWord(fp, fn)) != NULL)
				error(cmdOpts.msgNoFnam);
		}
		else buf1 = NULL;
		(cmdOpts.fct)();
		free(buf1);
		free(buf);
	}
	Efclose(fp, fn);
	disposeString(fn);
}

void errRet(void)
{
	clMsgIDs(binf.msgIDs);
	clTempFile(tmpStrings);
	clTempFile(tmpIndex);
	clHashFile(tmpHash);
}

char *opt1st = "";
char *opt2nd = "IO";
char *optSWCHAR = "/";
main(int argc, char **argv)
{
	int opt;

	msgInit();
	msgErrFct(errRet);

	oHead = iHead = MH_ANY;

	while((opt = getoptG(argc, argv)) != EOF) switch(opt) {
	case 'I':	iHead = getIOArg('I', optarg); break;
	case 'O':	oHead = getIOArg('O', optarg); break;
	default:	hlpScreen(); break;
	}

	if((binf.fn = argv[optind++]) == NULL)	hlpScreen();

	informative(M_binMsgLib, binf.fn);
	identBinLib();
	scanBinLib();
	libModified = binf.header != MH_ANY && binf.header != oHead && oHead != MH_ANY;
	if(oHead != MH_ANY)
		binf.header = oHead;
	if(binf.header == MH_ANY)
		binf.header = MH_NON;

	iHead = oHead = MH_ANY;

	/* reading and performing commands at the command line */
	if(!argv[optind])			/* default: view all */
		libViewAll();
	else do {
		/* scan any option preceeding the command */
		scanOpts(argc, argv);
		if(!argv[optind]) break;

		/* read in and identify the command */
		idCmd(argv[optind++]);
		if(cmdOpts.needFnam) {
			scanOpts(argc, argv);
			if(!argv[optind])
				error(cmdOpts.msgNoFnam);
			cmdOpts.fnam = argv[optind++];
		}
		(cmdOpts.fct)();
	} while(argv[optind]);

	if(libModified)
		libPerform();

	errRet();

	nM(return 0);
}
/*********************************************************************/
/*********************************************************************/
/*********************************************************************/
void hlpScreen(void)
{	message(stdout, E_2mgtHlpScreen, appName());
	exit(msgErrorNumber(E_2mgtHlpScreen));
}
