/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/*
    MSGDIFF - utility for the message handling library
    Copyright (C) 97  Steffen Kaiser

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/
/* $RCSfile: MSGDIFF.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	The comparator for two message description files and reorder the
	file to ensure that each message gains the same message number.

   ----
	command line usage:

	msgDiff [/l=] [/o[=#]] lng [lng2]

	lng		: language to be compared
	lng2	: language to be considered "OK"
	/l=		: defines a file into which the log is dumped
	/o[=#]	: set output file. If omitted, the renaming process is done.
			: If '=#' is omitted, the output is ignored ==> test only.
			: Otherwise the new messages are dumped into '#'.

	The file extension of both files defaults to ".MSG".
	<lng2> defaults to "english.msg".

	From <lng> all messages are scanned. Then while scanning <lng2>:
	- If a message of <lng2> is also cantained within <lng>, the
		message from <lng> is copied into the new file.
	- If a message of <lng2> is not contained in <lng>, the message
		from <lng2> is copied.
	- If a message of <lng> is not contained in <lng2>, it is removed.

	If this process finishes successfully, the old <lng> file is renamed
	to <filename>.MG overwriting an already existing file without notice
	and the new file is renamed to <lng>.

	If the error number of a message differs between both files, the one
	of <lng2> is copied.

	If the logfile is given, it is overwritten without notice and contains
	lines that look like this:
		+<name> [<errnr>]
		-<name> [<errnr>]
		#<name> [<errnr>]
	<name> is the name of the message. <errnr> the error number, if one
	is associated with that message.
	'+' means: new message added from <lng2>
	'-' means: message deleted from <lng>
	'#' means: error number changed from <lng2> but message body remained

	If the logfile is enabled, but no changes need to be made, the logfile
	is deleted.

	Errorlevel:
		0: logfile deleted: No difference between <lng> and <lng2>
		0: logfile not deleted: Differences found
		else: an error occured

   ----

   Target compilers:
   		Micro-C v3.14, Borland C++ v3.1+

   ----
	
*/

#include <stdio.h>
#ifndef _MICROC_
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <io.h>
#endif /*#	!defined(_MICROC_) */
#include <portable.h>
#include <dfn.h>
#include <getopt.h>

#ifdef __TURBOC__
#pragma hdrstop
#endif /*#	defined(__TURBOC__) */

#include "yerror.h"

#ifndef NUL
#define NUL '\0'
#endif /*#	!defined(NUL) */
#define BUF 2048 	/* buffer size while copying a file */
#define NO_NR 0xFFFF

struct MSGHEAD {
	struct MSGHEAD *nxt;
	int flags;
	int errnr;
	dword fpos;
	char name[1];
};


/*
 *	default extension for the message description files
 */
char dfltLng2[] = "ENGLISH.MSG";
const char extMsg[] = "MSG";

char *msgFile; /* extension-less local message file; with space for ext */
char *msgBody = NULL;			/* current message body */
char msgHeader[40];				/* current message header */

unsigned errNr = 0;			/* error level of current message header */
unsigned lineNr = 0;		/* keep it NOT (long) because of Micro-C */
FLAG binary = 0;


/* log stuff */
FILE *logFp, *outFp;
char *logFile = NULL;
FLAG emptyLog = !NUL;
struct MSGHEAD lhead = {NULL};


void hlpScreen(void)
{	message(stdout, E_diffHlpScreen, appName());
	exit(msgErrorNumber(E_diffHlpScreen));
}

/*
 *	Allocate a chunk of memory
 *	On failure: Terminate with an error message
 */
byte *getmem(int len)
{	byte *h;

	if((h = malloc(len)) == NULL)
		error(E_noMem);

	return h;
}


/*
 *	Duplicate a string into the local heap.
 *	On failure: Terminate with an error message
 */
char *dupstr(char *str)
{	char *h;

	if((h = strdup(str)) == NULL)
		error(E_noMem);

	return h;
}


/*
 *	StrCat() within dynamic memory
 */
#define addStr(a,b) addStr_(&(a),(b))
char *addStr_(char **str, char *s)
{
	if(s == NULL) return *str;
	if(*str == NULL)
		return *str = dupstr(s);

	if((*str = realloc(*str, strlen(*str) + strlen(s) + 1)) == NULL)
		fatal(E_noMem);
#ifdef _MICROC_
	stpcat(*str, s);
	return *str;
#else /*#	!(defined(_MICROC_)) */
	return strcat(*str, s);
#endif /*#	defined(_MICROC_) */
}



/*
 *	Open a file
 *	On failure terminate with an error message
 */
FILE *Efopen(char **fname, const char *mode)
{	FILE *fp;
	char *p;

	/* open file & emit error on failure */
	if((fp = fopen(*fname, mode)) == NULL) {
		if(!dfnsplit(*fname, NULL, NULL, NULL, &p))
			fatal(E_noMem);
		if(!p) {
			if((p = dfnmerge(NULL, NULL, NULL, *fname, extMsg)) == NULL)
				fatal(E_noMem);
			fp = fopen(p, mode);
		}
		if(!fp) fatal(E_openFile, *fname);
		*fname = p;
	}

	return fp;
}

/*
 *	Close a file
 *	If an error occured while writing/reading the file, terminate
 *	with an error message.
 */
void Efclose(FILE *fp, char *fnam)
{	/* ensure, fp->buffer is empty, so ferror() is up to date */
	fflush(fp);

	if(ferror(fp))
		fatal(E_writeFile, fnam);

	fclose(fp);
}


/*
 *	Return if the current message has an error number attached.
 */
int haveErrNr(void)
{	return *msgHeader == 'E' || (*msgHeader == 'I' && errNr != NO_NR);
}



char *mkFname(char *fname, char *ex)
{	char *dr, *pa, *fi, *fn;

	if(!dfnsplit(fname, &dr, &pa, &fi, NULL)
	 || (fn = dfnmerge(NULL, dr, pa, fi, ex)) == NULL)
	 	fatal(E_noMem);

	free(dr);
	free(pa);
	free(fi);
	return fn;
}

char *readMsgBody(FILE *fp)
/*	read next lines, until EOF or an message header encountered */
{	char line[80];
	char *p;
	int cnt;
	int c;

	free(msgBody);
	msgBody = NULL;
	do {
		p = line - 1;
		cnt = 1;
		++lineNr;
		while((c = fgetc(fp)) != EOF && (*++p = c) != '\n')
			if(p > line + 78) { 
				p[1] = NUL;
				addStr(msgBody, line);
				p = line - 1;
			} 
			else if(c == '>') {
				if(cnt && ++cnt == 7) {	/* message header found */
					p[-5] = NUL;	/* cut out '>>>>>>' */
					c = EOF;	/* stop reading */
					break;
				}
			}
			else cnt = 0;	/* no message header */

		p[1] = NUL;
		addStr(msgBody, line);
	} while(c != EOF);

	/* strip last '\n' */
	if(*(p = strchr(msgBody, NUL) - 1) == '\n')
		*p = NUL;
	return msgBody;
}

char *readMsgHeader(FILE *fp)
/* treat the next line as a message header, the '>>>>>>" are already
   read */
{	
	char *p;
	int c;

	*(p = msgHeader) = NUL;
	while((isalnum(c = fgetc(fp)) || c == '_') && p < &msgHeader[39])
		*p++ = c;
	*p = NUL;

	while(c != '\n' && isspace(c))
		c = fgetc(fp);

	if(*msgHeader == 'E' || *msgHeader == 'I') { /* scan error level */
		if(isdigit(c)) {
			errNr = 0;
			do errNr = errNr * 10 + c - '0';
			while(isdigit(c = fgetc(fp)));

			while(c != '\n' && isspace(c))
				c = fgetc(fp);
		}
		else if(*msgHeader == 'I')
			errNr = NO_NR;
		else
			error(E_msgFileHeadErrLvl, lineNr);
	}

	if(*msgHeader == NUL && c == EOF)
		return NULL;

	if(*msgHeader < 'A' || *msgHeader > 'Z' || msgHeader[1] != '_' 
	  || c != '\n' && c != EOF)
		error(E_msgFileHeadStruc, lineNr);

	readMsgBody(fp);	/* scan in the body and supply it in the global msgBody */
	return msgHeader;
}


void writeBody(void)
{	fputs(msgBody, outFp);
	fputc('\n', outFp);
}
void writeMsg(void)
{	fprintf(outFp, haveErrNr()? ">>>>>>%s %u\n" : ">>>>>>%s\n"
	 , msgHeader, errNr);
	writeBody();
}


void setFpos(FILE *fp, dword iM(*)fpos)
{
#ifdef _MICROC_
	fseek(fp, fpos->hi, fpos->lo, 0);
#else /*#	!(defined(_MICROC_)) */
	fseek(fp, fpos, 0);
#endif /*#	defined(_MICROC_) */
}


void getFpos(FILE *fp, dword *fpos)
{
#ifdef _MICROC_
	ftell(fp, &fpos->hi, &fpos->lo);
#else /*#	!(defined(_MICROC_)) */
	*fpos = ftell(fp);
#endif /*#	defined(_MICROC_) */
}


void log(char typ)
{	emptyLog = 0;
	if(logFile)
		fprintf(logFp, haveErrNr()? "%c%s %u\n": "%c%s\n"
		 , typ, msgHeader, errNr);
}



char *opt1st = "O";
char *opt2nd = "LO";

main(int argc, char **argv)
{	char *lng, *lng2, *ofnam;
	FILE *fp, *fp2;
	int c, oOpt;
	dword fpos;
	struct MSGHEAD *p;


    msgInit();

	logFile = NULL;

	oOpt = 0;
	while((c = getoptG(argc, argv)) != EOF) switch(c) {
	case 'L':				/* logfile */
		logFile = optarg;
		break;
	case 'O':				/* output file */
		oOpt = 1;
		ofnam = optarg;
		break;
	default:
		hlpScreen();
	}

	lng2 = dfltLng2;

	switch(argc - optind) {
	default:	hlpScreen();
	case 2:		lng2 = argv[optind + 1];
	case 1:		lng = argv[optind + 0];
				break;
	}

	if(!oOpt) {
		for(c = 0; c < 0xFFF; ++c) {
			sprintf(msgHeader, "%03x", c);
			ofnam = mkFname(lng, msgHeader);
			if((outFp = fopen(ofnam, "rb")) == NULL)
				goto found;
			fclose(outFp);
			free(ofnam);
		}
		error(E_tmpFile);
		ofnam = dupstr(msgHeader);
	found:
	}
	else if(ofnam == NULL)
		ofnam = "nul";

	/* read message file & create the message definition files */
	fp = Efopen(&lng, "rt");
	fp2 = Efopen(&lng2, "rt");
	outFp = Efopen(&ofnam, "wt");

	if(logFile)
		logFp = Efopen(&logFile, "wt");

	/* first: Scan lng for all defined messages */
	informative(M_scanLng, lng);
	lineNr = 0;
	readMsgBody(fp);	/* skip header */
	getFpos(fp, aS(fpos));

	p = aS(lhead);
	while(readMsgHeader(fp)) {
		p = p->nxt = cS(struct MSGHEAD*)getmem(sizeof(struct MSGHEAD)
		 + strlen(msgHeader));
		strcpy(p->name, msgHeader);
		p->errnr = errNr;
		p->flags = 0;
		longcpy(p->fpos, fpos);
		getFpos(fp, aS(fpos));
	}
	p->nxt = NULL;

	informative(M_copyLng, lng2);
	lineNr = 0;

	rewind(fp);
	readMsgBody(fp);	/* skip header */
	if(*msgBody) {
		writeBody();
		readMsgBody(fp);
	}
	else {
		rewind(fp2);
		readMsgBody(fp2);
		writeBody();
	}

	while(readMsgHeader(fp2)) {
		p = aS(lhead);
		while((p = p->nxt) != NULL && strcmp(p->name, msgHeader) != 0);
		if(p) {
			if(p->errnr != errNr
			 && (*msgHeader == 'E' || *msgHeader == 'I')) {
				errNr = p->errnr;
				log('#');
			}
			setFpos(fp, p->fpos);
			readMsgHeader(fp);
			p->flags = 1;
		}
		else
			log('+');
		writeMsg();
	}

	p = aS(lhead);
	while((p = p->nxt) != NULL) if(!p->flags) {
		errNr = p->errnr;
		strcpy(msgHeader, p->name);
		log('-');
	}

	Efclose(fp, lng);
	Efclose(fp2, lng2);
	Efclose(outFp, ofnam);
	if(logFile)
		Efclose(logFp, logFile);

	if(emptyLog) {
		if(logFile)
			unlink(logFile);
		if(!oOpt)
			unlink(ofnam);
	}
	else if(!oOpt) {
		lng2 = mkFname(lng, "mg");
#ifndef _MICROC_
		chmod(lng2, 0777);
#else /*#	!(!defined(_MICROC_)) */
		set_attr(lng2, 0);
#endif /*#	!defined(_MICROC_) */
		unlink(lng2);
		if(rename(lng, lng2))
			error(E_renameFile, lng, lng2);
		if(rename(ofnam, lng))
			error(E_renameFile, ofnam, lng);
	}

#ifndef _MICROC_
	return 0;	/* standard for Micro-C */
#endif /*#	!defined(_MICROC_) */
}
