/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/*
    MSGIDS - Management of msgID association mappings
    Copyright (C) 1998  Steffen Kaiser

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/
/* $RCSfile: MSGIDS.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	Management to handle the association between numerical and
	symbolical message ID.

   ----

   Target compilers:
   		Micro-C v3.14, Borland C++ v3.1+

   ----
	
	Known bugs:

   ----
	
*/

#include <assert.h>
#include <stdio.h>
#ifndef _MICROC_
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#endif /*#	!defined(_MICROC_) */
#include <portable.h>

#include "savemem.h"
#include "fileIO.h"
#include "tmpFile.h"
#include "hashFile.h"
#include "msgIDs.h"

#ifdef __TURBOC__
#pragma hdrstop
#endif /*#	defined(__TURBOC__) */

#define MSG_METHOD1				/* don't want method #2 encoding right now */

#include "../msgconf2.h"		/* method #2 structures */

#include "yerror.h"


#ifndef NUL
#define NUL '\0'
#endif /*#	!defined(NUL) */


#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: MSGIDS.C 2.4 1998/10/04 05:17:36 ska Exp ska $";
#endif /*#	defined(RCS_Version) */


static char *getBuf = NULL;		/* symbolic message ID buffer for get*() */


/*
 *	Unlink one MsgIDs entry
 */
void clMsgIDs(struct MsgIDs *f)
{	if(f) {
		clTempFile(f->dat);
		clHashFile(f->idx1);
		clHashFile(f->idx2);
		free(f);
	}
}


/* see below */
unsigned hashMsgId(MSGID2 msgNum)
{	
/* Currently the msgID is constructed as follows:
		0xYXXX
	Y is the message group &
	XXX is the message sub-id

	because it's not sure that XXX is greater than 0x7FF, but Y is
	definitely equal to 8 sometimes, we just reconstruct the number into:
		0xXXXY

	Because not all message groups are in use today, that leaves some
	unused entries within the index file, but if we'd use msgNum itself,
	we would require 0x8001 hash entries at minimum.

	Note: hasMsgId _must_ return unique hash values!
*/
	return (msgIndex(msgNum) << 4) | msgGroup(msgNum);
}

/* see below */
unsigned hashMsgName(char *msgName, int length)
{	unsigned val;

	assert(msgName);
	assert(strlen(msgName) >= length);

/* to make things easier we sum up all bytes and use the remainder
	modulo the maximum number of hash entries we want to support
*/
	for(val = 0; length--;)
		val += *(byte*)msgName++;

	return val % 512;
}


/*
 *	Search the msgIDs for a specific numerical message ID
 *	Note: the hashMsgId() is an one-to-one relationship.
 */
int getSymID(struct MsgIDs *msgIDs, MSGID2 id, struct MsgIDhash *hash
	, char**name)
{	dword pos;
	TempFile *fd;

	assert(hash);
	assert(name);

	if(!msgIDs)
		return 0;

	if(msgIDs->grpIDs[msgGroup(id)] < msgIndex(id)) /* id too high */
		return 0;

	hfread(hashMsgId(id), pos, msgIDs->idx1);
	if(!longtst(pos))		/* non-defined message */
		return 0;

	/* read the hash entry */
	fd = msgIDs->dat;
	Esetpos(pos, fd->fp, fd->fn);
	Eread(hash, sizeof(struct MsgIDhash), fd->fp, fd->fn);
	Eread(*name = getBuf = chgmem(getBuf, hash->length + 1)
	 , hash->length, fd->fp, fd->fn);
	return 1;
}


/*
 *	Search the msgIDs for the symbolic message ID
 *	Symbolic IDs are chained within a single-linked list.
 */
int getSymNam(struct MsgIDs *msgIDs, char *name, int length
	, struct MsgIDhash *hash)
{
	TempFile *fd;

	assert(name);
	assert(hash);

	if(!msgIDs)
		return 0;

	getBuf = chgmem(getBuf, length);

	fd = msgIDs->dat;
	hfread(hashMsgName(name, length), hash->next, msgIDs->idx2);
	while(longtst(hash->next)) {
		Esetpos(hash->next, fd->fp, fd->fn);
		Eread(hash, sizeof(struct MsgIDhash), fd->fp, fd->fn);
		if(hash->length != length)	/* cannot match! */
			continue;
		Eread(getBuf, length, fd->fp, fd->fn);
		if(memcmp(getBuf, name, length) == 0)		/* matched! */
			return 1;
	}

	return 0;
}


/*
 *	Read the association between message IDs and message names.
 *
 *	Generate the binary message library's map information and/or
 *	verify the local one. We will also check if the information as
 *	mismatches within itself.
 *
 *	Two file format modes are supported:
 *		0: binary message library
 *		1: text file
 *
 *	We generate a double-indexed data file in this format:
 *		DATfile: a sequence of message name entries: (struct msgIDhash)
 *			1x dword		pointer to next hash entry (0 if end of chain)
 *			1x MSGID2		the msgID of this entry
 *			1x (int)		N length of message name
 *			Nx character	message name
 *		IDXfile #1: maps message IDs into message names
 *			1x dword		pointer into DATfile (dword field)
 *			The function unsigned hashMsgId(MSGID2 msgNum) maps
 *			a message ID into the number of the record within this IDXfile.
 *		IDXfile #2: chains message names into an hashed chained single
 *			linked list of messages
 *			1x dword		pointer into DATfile (dword field)
 *			The function unsigned hashMsgName(char *msgName)
 *			maps a message name into the number of the record, where the
 *			chain of hashed messages starts in.
 */
struct MsgIDs *scanMsgIDS(struct MsgIDs *currMsgIDs, ReadNxtMsgID nxt, void *arg)
{	
	char *buf, *name;
	int length, buflen;
	struct MsgIDhash hash;
	dword pos;
	TempFile *fd;
	HashFile *fi1, *fi2;
	FLAG verify;
	MSGID2 id;

	buf = NULL;
	buflen = 0;

	if(currMsgIDs == NULL) {		/* create new mapping */
		verify = 0;
		currMsgIDs = cgetmem(sizeof(struct MsgIDs));
		fd = currMsgIDs->dat = tmpFile(TF_RW);
			/* we cannot use address "0" */
		Eseek(1, SEEK_SET, fd->fp, fd->fn);
		setMode(fd, TF_RD);	/* we set write perm when necessary */
		fi1 = currMsgIDs->idx1 = hashFile();
		fi2 = currMsgIDs->idx2 = hashFile();
	}
	else {			/* verify local mapping information with global one */
		verify = 1;
		fd = currMsgIDs->dat;
		fi1 = currMsgIDs->idx1;
		fi2 = currMsgIDs->idx2;
	}

	/* There is one fixed mapping:
		E_hlpScreen mkMsgID(MSG_GLOBAL, 1)
	*/
	id = mkMsgID(LOCAL_GROUP, 1);
	buflen = length = strlen(buf = dupstr("E_hlpScreen"));
	++buflen;
	do {
		if(verify) {			/* verify the msgID */
			if(getSymID(currMsgIDs, id, aS(hash), &name)) {
				/* check if the local numMsgID matches the global one */
				if(hash.length != length
				 || memcmp(buf, name, length) != 0) {
					buf[length] = '\0';
					name[hash.length] = '\0';
				 	error(E_mapMismatch, id, buf, name);
				}
				continue;		/* matched */
			}
			/* add new entry! */
		}
		if(getSymNam(currMsgIDs, buf, length, aS(hash))) {
			/* already found. Check the numMsgID */
			if(id != hash.msgid) {
				buf[length] = '\0';
				error(E_doubleMsg, buf, id, hash.msgid);
			}
			continue;		/* equal */
		}
			/* keep track of the amount of messages per group */
		if(currMsgIDs->grpIDs[msgGroup(id)] < msgIndex(id))
			currMsgIDs->grpIDs[msgGroup(id)] = msgIndex(id);
		
			/* where the symMsgID is located within the DAT file */
		Eseek(0, SEEK_END, fd->fp, fd->fn);
		Egetpos(pos, fd->fp, fd->fn);
		
			/* add this symMsgID into file indexing the numerical currMsgIDs */
		hfwrite(hashMsgId(id), pos, fi1);
		
			/* Chain the symbolic msgID into the indexed chain of singled
				linked list of IDs. */
		hfread(hashMsgName(buf, length), hash.next, fi2);
		hfrewrite(pos, fi2);
		
			/* actual add the mapping */
		hash.length = length;
		hash.msgid = id;
		Ewrite(aS(hash), sizeof(hash), fd->fp, fd->fn);
		Ewrite(buf, length, fd->fp, fd->fn);

	} while(nxt && (nxt)(arg, &id, &buf, &length, &buflen));

	free(buf);
	return currMsgIDs;
}

struct AddMsgArgs {
	char *name;
	int length;
	MSGID2 msgid;
};

static int addSingleMsgID(void *arg, MSGID2 *id, char **buf, int *length
	, int *buflen)
{	struct AddMsgArgs *a;

	assert(id);
	assert(buf);
	assert(length);
	assert(buflen);

	a = cS(struct AddMsgArgs*)arg;
	if(a->msgid) {
		*id = a->msgid;
		*buflen = 1 + (*length = a->length);
		memcpy(*buf = chgmem(*buf, *buflen), a->name, *length);
		a->msgid = 0;
		return 1;
	}
	return 0;
}

/*
 *	Return the msg index the symbolic msgID.
 *	If the message does not already exist, a new one is created and
 *	added into the msgIDs list.
 */
 int addMsgID(struct MsgIDs *msgIDs, char *name, int length, int group)
 {	struct MsgIDhash hash;
 	struct AddMsgArgs arg;
 	MSGID2 id;

 	assert(msgIDs);
 	assert(name);

 	if(getSymNam(msgIDs, name, length, aS(hash))) {
 		if(msgGroup(hash.msgid) != group)
 			error(E_collGroup, name, group, hash.msgid);
 		return msgIndex(hash.msgid);
 	}

 	/* add a new message */
	arg.name = name;
	arg.length = length;
	arg.msgid = id = mkMsgID(group, ++msgIDs->grpIDs[group]);
	scanMsgIDS(msgIDs, aF(addSingleMsgID), aS(arg));
	return msgIndex(id);
}

/*
 *	Dump all messages in ascending order to the 'nxt' function.
 */
void dumpMsgIDS(struct MsgIDs *currMsgIDs, DumpNxtMsgID nxt, void *arg)
{	unsigned group;
	unsigned index;
	MSGID2 id;
	struct MsgIDhash hash;
	char *name;

	assert(currMsgIDs);
	assert(nxt);

	for(group = 0; group < MSGGROUPMAX; ++group)
		for(index = 0; ++index <= currMsgIDs->grpIDs[group];)
			if(getSymID(currMsgIDs, id = mkMsgID(group, index)
			 , aS(hash), &name)) {
			 	if(!(nxt)(arg, id, name, hash.length))
			 		return;
			 }

}

struct binMsgIDSDump {
	FILE *fp;
	char *fn;
};

static int binNxtMsgID(inM(struct binMsgIDSDump, void) *arg
	, MSGID2 id, char *name, int length)
{	MsgStruID msgID;

	assert(name);
	assert(arg);

	msgID.msID_length = length;
	msgID.msID_id = id;
	Ewrite(aS(msgID), sizeof(msgID)
	 , (cS(struct binMsgIDSDump*)arg)->fp
	 , (cS(struct binMsgIDSDump*)arg)->fn);
	if(length) {
		Ewrite(name, length
		 , (cS(struct binMsgIDSDump*)arg)->fp
		 , (cS(struct binMsgIDSDump*)arg)->fn);
	}
	return 1;
}


void dumpBinMsgIDS(struct MsgIDs *msgIDs, FILE *fp, char *fn)
{
	MsgStruIDS msgIDS;
	MsgStruID msgID;
	struct binMsgIDSDump arg;
	dword posA, posE;

	if(!msgIDs)
		return;

	/* the 'next' member will be updated later */
	Egetpos(posA, fp, fn);
	msgIDS.msIDS_magicnumber = MSG_MN_IDS;
	Ewrite(aS(msgIDS), sizeof(msgIDS), arg.fp = fp, arg.fn = fn);

	dumpMsgIDS(msgIDs, aF(binNxtMsgID), aS(arg));

	memset(aS(msgID), 0, sizeof(msgID));
	Ewrite(aS(msgID), sizeof(msgID), fp, fn);

	/* update the 'next' member */
	Egetpos(posE, fp, fn);
	Esetpos(posA, fp, fn);
	longcpy(msgIDS.msIDS_next, posE);
	longsub(msgIDS.msIDS_next, posA);
	Ewrite(aS(msgIDS), sizeof(msgIDS), fp, fn);
	Esetpos(posE, fp, fn);
}
