/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/*
    SAVEMEM - Wrappers of malloc() and assoc. functions
    Copyright (C) 1998  Steffen Kaiser

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/
/* $RCSfile: SAVEMEM.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	Wrapper of malloc() and associated functions.
	All functions issue an error message on failure and terminate the
	program.

   ----

   Available functions:

   ----

   Target compilers:
   		Micro-C v3.14, Borland C++ v3.1+

   ----
	
	Known bugs:

   ----
	
*/

#include <assert.h>
#ifndef _MICROC_
#include <stdlib.h>
#include <string.h>
#endif /*#	!defined(_MICROC_) */
#include <portable.h>

#ifdef __TURBOC__
#pragma hdrstop
#endif /*#	defined(__TURBOC__) */

#include "savemem.h"
#include "yerror.h"


#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: SAVEMEM.C 2.3 1998/10/04 05:17:36 ska Exp ska $";
#endif /*#	defined(RCS_Version) */

/*
 *	Allocate a chunk of memory
 *	On failure: Terminate with an error message
 */
void *getmem(int len)
{	byte *h;

	if((h = malloc(len)) == NULL)
		error(E_noMem);

#ifdef TEST
printf("malloc(%u) = %p\n", len, h);
#endif /*#	defined(TEST) */
	return h;
}


/*
 *	Allocate a chunk of memory and erase the contents to zero
 *	On failure: Termiante with an error message
 */
void *cgetmem(int len)
{	byte *h;

	memset(h = getmem(len), 0, len);
	return h;
}


/*
 *	Reallocate a chunk of memory and terminate the application on failure
 */
void *chgmem(void *p1, int size)
{	void *p;

	if((p = realloc(p1, size)) == NULL)
		error(E_noMem);
#ifdef TEST
printf("realloc(%p, %u) = %p\n", p1, size, p);
#endif /*#	defined(TEST) */
	return p;
}

/*
 *	Duplicate a string into the local heap.
 *	On failure: Terminate with an error message
 */
char *dupstr(char *str)
{	char *h;

	assert(str);

	if((h = strdup(str)) == NULL)
		error(E_noMem);


#ifdef TEST
printf("malloc(%u) = %p\n", strlen(str) + 1, h);
#endif /*#	defined(TEST) */

	return h;
}

/*
 *	Replace the contents of a string variable with another string
 */
char *newstr_(char **var, char *str)
{	assert(var);

	free(*var);
	return *var = str? dupstr(str): NULL;
}


/*
 *	Duplicate a portion of a string
 */
char *strdupe(char *start, char *end)
{	char c;

	assert(start);
	assert(end);

	c = *end;
	*end = NUL;
	start = dupstr(start);
	*end = c;

	return start;
}


/*
 *	strcat() within dynamic memory
 */
char *addStr_(char **str, char *s)
{
	assert(str);

	if(s == NULL) return *str;
	if(*str == NULL)
		return *str = dupstr(s);

	*str = chgmem(*str, strlen(*str) + strlen(s) + 1);

#ifdef _MICROC_
	strcat(*str, s);
	return *str;
#else /*#	!(defined(_MICROC_)) */
	return strcat(*str, s);
#endif /*#	defined(_MICROC_) */
}

#ifdef TEST
void free__(void *p)
{
printf("free(%p)\n", p);
#undef free
	free(p);
}
#endif /*#	defined(TEST) */
