/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/*
    SHSTR - Shared string implementation
    Copyright (C) 1998  Steffen Kaiser

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/
/* $RCSfile: SHSTR.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	Strings are duplicated into the dynamic memory and, if more than
	one string is equal to another string, both strings will share the
	same memory area.
	The internal structure (control area) of the shared strings is
	hidden, the shared strings are addressed by the pointer to the
	(char*) string a shared strings holds.

   ----

	String *findString(int length, int hashnum, char *s)
		Search a string within the shared string heap.
		Return NULL if not found.
		hashnum _must_ be generated by "genStrHashID(s, strlen(s))"
		to consistently use the shared strings.

   ----

   Target compilers:
   		Micro-C v3.14, Borland C++ v3.1+

   ----
	
	Known bugs:

   ----
	
*/

#include <assert.h>
#ifndef _MICROC_
#include <stdlib.h>
#include <string.h>
#endif /*#	!defined(_MICROC_) */
#include <portable.h>

#include "savemem.h"
#include "shStr.h"


#ifdef __TURBOC__
#pragma hdrstop
#endif /*#	defined(__TURBOC__) */

#include "yerror.h"


struct String_ {
	struct String_ *nxt;
	int linkcount;
	int hashnum;
	int length;
	char string[1];
};
#ifdef _MICROC_
#define String struct String_
#else /*#	!(defined(_MICROC_)) */
typedef struct String_ String;
#endif /*#	defined(_MICROC_) */


#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: SHSTR.C 2.2 1998/10/04 05:17:36 ska Exp ska $";
#endif /*#	defined(RCS_Version) */


static String *strings = NULL;	/* first shared string */

/*
 *	Find a string within the string heap
 *	Each string has two values assigned:
 *	Its length and an CRC/hash number. When a string is searched for,
 *	first both numbers are compared and, if both match, then the
 *	string itself.
 */
static String *findString(int length, int hashnum, char *s)
{	String *p;

	assert(s);

	if((p = strings) != NULL) do {
		if(p->length == length && p->hashnum == hashnum
		 && memcmp(p->string, s, length) == 0)
		 	return p;
	} while((p = p->nxt) != NULL);
	return NULL;
}

/*
 *	Create a hash value for the string
 */
int genStrHashID(byte *s, unsigned length)
{	int num;

	assert(s);

	/* to make things a bit easier, we just sum up all the string bytes */
	num = 0;
	while(length--)
		num += *s++;

	return num;
}


/*
 *	Put a string into the string heap
 */
char *makeString(char *s)
{	int hashnum;				/* string hash/CRC/checksum number */
	int length;					/* length of string */
	String *p;
	
	assert(s);

	hashnum = genStrHashID((byte*)s, length = strlen(s));
	if((p = findString(length, hashnum, s)) != NULL) {
		/* string found, if linkcount exceeds its range -> create new */
		if(++p->linkcount)
			return &p->string[0];
		--p->linkcount;		/* overflow */
	}

	/* not found --> create a new String node */
	p = getmem(sizeof(String) + length);
	p->nxt = strings;
	strings = p;
	p->length = length;
	p->hashnum = hashnum;
	p->linkcount = 1;
	return strcpy(&p->string[0], s);
}

/*
 *	Dispose a string from the string heap
 */
void disposeString(char *s)
{	String *p, *prev;

	assert(s);

	prev = NULL;
	if((p = strings) != NULL) do {
		if(&p->string[0] == s) {		/* string to be disposed */
			if(!--p->linkcount) {		/* not in use anymore */
				/* string node can be free'ed */
				if(prev) prev->nxt = p->nxt;
				else strings = p->nxt;
				free(p);
			}
			return;
		}
		prev = p;
	} while((p = p->nxt) != NULL);
}
