/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $RCSfile: MSGEMIT.INC $
   $Locker: ska $	$Name:  $	$State: Rel $

   Message emitting functions. Wrappers of the skprintf() function.

   Due to the useage of the MR depend msgLock() function this function
   must be sourced-in once per access method.

*/

#include <stdio.h>
#ifndef _MICROC_
#include <string.h>
#include <process.h>
#include <dos.h>
#include <io.h>
#endif /*#	!defined(_MICROC_) */
#include <portable.h>
#include "yerror.h"

#define NUL '\0'

#ifdef __TURBOC__
	/* Don't let the declaration of __msgOutput() come into the
		precompiled header information. Borland tends to act
		strangely otherwise. */
#pragma hdrstop
#endif /*#	defined(__TURBOC__) */

#if 0
#if MSG_METHOD == 1
#define __msgOutput  __msgOutput1
#else /*#	!(MSG_METHOD == 1) */
#define __msgOutput  __msgOutput2
#endif /*#	MSG_METHOD == 1 */
#endif /*#	0 */

static void __msgOutput();			/* low-level emit function */


#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: MSGEMIT.INC 1.3 1998/10/07 04:42:53 ska Rel ska $";
#endif /*#	defined(RCS_Version) */

/*ska no format string error reporting
const unsigned char E_hlpFmtStr[] = "Error in format string near '%s'";
*/


static unsigned char noiseDisallow[ENoise_Nr] = {NOISE_TEST};
/* The rest of the array is filled with zeros meaning: allowed */



/*
 * defines a function, which is executed right before exit(),
 * in the case an error() or fatal() message has been emitted
 */
#ifdef _MICROC_
static int msgErrRet = NULL;
void _msgErrFct(int fct)

#else /*#	!(defined(_MICROC_)) */
static void (*msgErrRet)(void) = NULL;
void msgErrFct(void (*fct)(void))

#endif /*#	defined(_MICROC_) */

{	msgErrRet = fct; }



#ifdef _MICROC_			/* Micro-C appereance of the functions */


#include <file.h>	/* for fileno() */


#if 0		/*ska Unfortunately, this macro won't work in Micro-C because
				the ENoise_*** macro is not expanded. */

/* Micro-C's K&R-style / ** / is the equivalent for ANSI-C's ## */
	/* need std stack frame */

#define msgFct(a) register void a(MSGID msg)								\
	asm {																	\
		mov cx, ENoise_/**/a												\
  		jmp short __warning_emit_msg										\
	}
#endif /*#	0 */

#define msgFct(a,b) register void a(MSGID msg)	/* need std stack frame */	\
	asm {																	\
		mov cx, b															\
  		jmp short __warning_emit_msg										\
	}


/*
 * interactive means, that the function should be checked, wether or not
 * stdin is redirected. If stdin is redirected this message is ignored.
 */
register void interactive(MSGID msg)
{	asm "push ax";			/* preserve the number of passed arguments */
	if(noiseDisallow[ENoise_interactive] == NOISE_TEST) {
		if(!isatty(fileno(stdin))) {
			asm "pop ax";
			return;
		}
	}
	/* else the application disallowed the test */

	asm {
		pop ax		; get the number of arguments
		mov cx, ENoise_interactive
__warning_emit_msg:	; entry point for the wrappers of the lower-level function
		shl ax, 1	; amount of bytes used by all arguments
		lea bx, 2[bp]	; address below last arg
		add ax, bx	; address of format string
		push ax		; __msgOutput()''s first arg (addr of format string)
		push cx		; __msgOutput()''s second arg (noise level)
		push ax		; dummy argument (fout, unused)
		call ___msgOutput
		add sp, 6
	}
}


/*
 * message that is emitted into a stream
 */
register void message(FILE *fp, MSGID msg)
{
	asm {
		shl ax, 1	; amount of bytes used by all arguments
		lea bx, 2[bp]	; address below last arg
		add ax, bx	; address of file pointer
		mov bx, ax
		dec ax
		dec ax		; address of format string
		push ax		; __msgOutput()''s first arg	(fmt str)
		mov ax, ENoise_none	; no immediate form tells ASM here
		push ax		; __msgOutput()''s second arg	(noise level)
		push Word Ptr [bx]	; file pointer to write to
		call ___msgOutput
		add sp, 6
	}
}

#else /*#	!(defined(_MICROC_)) */

/* I don't care about about the waste of mem here; it's portable and
   not very overweighted.
*/
#if 0		/* Micro-C cannot handle a macro of this sort */
#define msgFct(a) void a(MSGID msg, ...)							\
	{	__msgOutput(&msg, ENoise_##a); }
#endif /*#	0 */

#define msgFct(a,b) void a(MSGID msg, ...)							\
	{	__msgOutput(&msg, b); 	}


/*
 * interactive means, that the function should be checked, wether or not
 * stdin is redirected. If stdin is redirected this message is ignored.
 */
void interactive(MSGID msg, ...)
{	if(noiseDisallow[ENoise_interactive] == NOISE_TEST) {
		if(!isatty(fileno(stdin)))
			return;
	}
	/* else the application disallowed the test */

	__msgOutput(&msg, ENoise_interactive);
}



/*
 * message that is emitted into a stream
 */
void message(FILE *fp, MSGID msg, ...)
{	__msgOutput(&msg, ENoise_none, fp);	}

#endif /*#	defined(_MICROC_) */



/* declare *normal* message wrapper functions */
msgFct(error, ENoise_error)
msgFct(fatal, ENoise_fatal)
msgFct(warning, ENoise_warning)
msgFct(informative, ENoise_informative)
msgFct(smessage, ENoise_screen)



/*ska
   Micro-C pushes parameters left->right; therefore right parameters
   cannot be omitted by default to access the leftmost parameters.

   To place the function definition behind its usage allows ANSI-C
   compilers to stuff as many argument into the function as supplied 
   on function call (less than three!).
   Well, that's because the prototype at the start of the source is
   in K&R style :)
 */

/*
 *	mid-level function to emit a message.
 *
 *	1) Check if it is allowed to emit this message;
 *	2) Make assumption where to emit the message to;
 *	3) Prepend standard text;
 *	4) Perform the emitting by invoking skprintf();
 *	5) Finish up.
 *
 *	Parameters:
 *		fout: file stream to emit into (Micro-C only)
 *		*arg: message ID
 *		arg[1..]: arguments of the message
 *		noiseLevel: noise level according to ENoise_????
 *		fp: output file stream; (for ENoise_none & non-Micro-C only!)
 */
static void __msgOutput(const MSGID *arg, int noiseLevel, FILE *inM(fout,fp))
{	MSGID noiseMsg;
#ifndef _MICROC_	/* Under Micro-C this function is always called with three
					   parameters. */
	FILE *fout;			/* because fp exists only for ENoise_none */
#endif /*#	!defined(_MICROC_) */

	/* Check if message is allowed to be emitted */
	if(!noiseDisallow[noiseLevel]) {

/* nxtArg() returns the address of the successor */
#ifndef _MICROC_	/* 2 arg has greater address */
#define nxtArg(arg) (unsigned char*)(arg + 1)
#else /*#	!(!defined(_MICROC_)) */ /* 2 arg has lesser address */
#define nxtArg(arg) (arg - 1 - 1 /* Micro-C won't sub 2 here */)
#endif /*#	!defined(_MICROC_) */

		/* Prepend standard message:
			none:
			interactive:
			screen:
			informative:	<appName>:
			warning:		<appName>: <I_warning>:
			error:			<appName>: <I_error>:
			fatal:			<appName>: <I_fatal>:

		  Also figure out where to emit to:
			none:			fp
			interactive:	stdout
			screen:			stdout
			else:			stderr
		*/
		switch(noiseLevel) {
			case ENoise_informative: noiseMsg = NULL; goto emitHead;
			case ENoise_warning: noiseMsg = I_warning; goto emitHead;
			case ENoise_error: noiseMsg = I_error; goto emitHead;
			case ENoise_fatal: noiseMsg = I_fatal; 
			emitHead:
				/* emit application's name */
				message(fout = stderr, I_appName, appName());
				if(noiseMsg) {
					/* emit type of message */
					message(stderr, I_noise, msgLock(noiseMsg));
					msgUnlock(noiseMsg);
				}
				break;

			case ENoise_screen:
			case ENoise_interactive:
				fout = stdout;
				break;

	#ifndef _MICROC_		/* under Micro-C no fp needed */
			case ENoise_none: fout = fp; break;	/* file pointer supplied
				on function call */
	#endif /*#	!defined(_MICROC_) */
		}

		/* message emitting */
		skprintf((unsigned char*)fout, 0, msgLock(*arg), nxtArg(arg));	/* stuff into fstream */
		msgUnlock(*arg);

		/* Finish up message emitting */
		switch(noiseLevel) {
			case ENoise_informative:
			case ENoise_warning:
			case ENoise_error:
			case ENoise_fatal:
				putc('\n', fout);
				break;
			/*ska
			case ENoise_interactive:
			case ENoise_screen:
			case ENoise_none:
				both are programmer defined, thus, leave them alone.
			*/
		}
	}

	/* check if this is an aborting message type */
	switch(noiseLevel) {
	case ENoise_error:
	case ENoise_fatal:
		if(msgErrRet)
			msgErrRet();
		exit(msgErrorNumber(*arg));
	}
}


/*
 *	set the allow/deny/test flag for a noise level
 *	Parameters:
 *		noiseLevel: one of macros ENoise_?????
 *			If too high, the call is ignored.
 *		flag: one of the macros NOISE_????
 */
void setNoiseLevel(unsigned noiseLevel, int flag)
{	if(noiseLevel < ENoise_Nr)
		noiseDisallow[noiseLevel] = flag; 
}


/*
 *	return the allow/deny/test flag for a noise level
 *	Parameters:
 *		noiseLevel: one of macros ENoise_?????
 *
 *	Return:
 *		NOISE_ERR: If noiseLevel is too high
 *		one of the macros NOISE_????: otherwise
 */
int getNoiseLevel(unsigned noiseLevel)
{	return noiseLevel < ENoise_Nr? noiseDisallow[noiseLevel]: NOISE_ERR; 
}


/*
 *	Initialize the message emitting system.
 *
 *	It performs the following steps:
 *	1) Perform a redirection test for the interactive noise level;
 *	2) Check if ANSI is installed.
 */
void msgInitMV(void)
{
/*	Check for redirected stdin */
	if(noiseDisallow[ENoise_interactive] == NOISE_TEST) {
		noiseDisallow[ENoise_interactive] = isatty(fileno(stdin))?
			NOISE_ALLOW: NOISE_DENY;
	}

	appNameSave();		/* Initialize appName() */
}
