/*
    SUPPL - a supplemental library for common useage
    Copyright (C) 1995,1996  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $RCSfile: MCB_MSTR.C $
   $Locker:  $	$Name:  $	$State: Exp $

	mcb_primaryShell(void)

	Return the MCB segment of the master shell (1st launched shell).

	Target compilers: Micro-C v3.13, v3.14; Borland C v2, v3.1, v4.52

*/

#include "initsupl.inc"

#ifndef _MICROC_
#include <dos.h>
#endif
#ifndef NDEBUG
#include <stdio.h>
#endif
#include <portable.h>
#include "environ.h"
#include "mcb.h"

static int firstShell(void *arg, unsigned mcb)
/* mcb_walk function for "1st shell" */
{	if(isShell(mcb)			/* shell found */
	 && peekw(mcb, SEG_OFFSET + 0x2c)) {	/* environment found */
	 	*(unsigned *)arg = mcb;
	 	return 1;		/* don't cycle any further */
	}
	return 0;			/* cycle further */
}


static lastShell(void *arg, unsigned mcb)
/* mcb_walk function for "last shell" */
{	if(isShell(mcb)				/* a shell found */
	 && peekw(mcb, SEG_OFFSET + 0x2c) > mcb)	/* its env block is behind shell */
	 	*(unsigned *)arg = mcb;
	return 0;					/* because last shell is searched -> cycle */
}

unsigned mcb_primaryShell(void)
/* There is no known way, which will never fail.

	There are three ways of locating the primary shell and with it the
	the master environment:
		1) Check the program INT-2E points to;
		2) Check the first shell in the MCB chain;
		3) Check the last shell that has its environment behind its PSP.

	With all the memory saving programs moving their data freely through
	the memory, all three may or may not return the correct environment.

	Therefore, all three checks are made and the result will be assumed
	as true, which comes twice.
*/
{	unsigned shell2E, shellFirst, shellLast;	/* found shells via 1-3 */

#ifdef _MICROC_
	get_vector(0x2E, &shell2E, &shellFirst);
#else
	isr vec2E;

	get_isr(0x2E, vec2E);
	shell2E = FP_SEG(vec2E);
	shellFirst = FP_OFF(vec2E);
#endif
	if(!shell2E || shell2E == 0xffff		/* no valid segment */
	 || (unsigned)peekb(shell2E, shellFirst) == 0xCF) /* iret */
		shell2E = 0;		/* there is no INT-2E handler! */
	else {					/* there is a 2E handler installed */
		/* move to the handler's shell */
		shell2E = mcb_shell(SEG2MCB(shell2E)); /* OK, shell found */
	}

	shellFirst = shellLast = 0;
	mcb_walk(0, aF(firstShell), &shellFirst);
	mcb_walk(0, aF(lastShell), &shellLast);

	if(shell2E == shellFirst && shell2E == shellLast)
		return shell2E;

	if(shellLast == shellFirst)
		return shellLast;

	return 0;				/* nothing found */
}
