;
;  Hard disk I/O functions
;

public	disk_get_info
public	disk_read
public	disk_write
public	disk_verify

public  open
public  close
public  lseek
public  read
public  write


disk_info		struc
 hd_flags	dd	?
 hd_num		dd	?
 num_cyls	dd	?
 num_heads	dd	?
 sect_per_track	dd	?
 sect_per_cyl	dd	?
 total_sects	dd	?
ends

disk_lba_info	struc
 lba_inf_buf_sz	DW	?
 lba_inf_flags	DW	?
 lba_inf_cyls	DD	?
 lba_inf_heads	DD	?
 lba_inf_sects	DD	?
 lba_inf_total	DQ	?
 lba_inf_ssize	DW	?
ends

disk_lba_addr	struc
 lba_buf_size	DB	?
 lba_reserved	DB	?
 lba_num_sect	DW	?
 lba_buf_addr	DD	?
 lba_rel_sect	DQ	?
ends

_DATA16		segment
disk_param_buf	DB	20h Dup(?)
_DATA16		ends


;----------------------------------------------------------------------
_TEXT 		segment			; 32-bit protected mode code
;----------------------------------------------------------------------

disk_get_info	proc	C	near

		arg	@@hd: dword, @@disk_info: dword
		local	@@num_disks: dword

		push	ebx
		push	esi
		push	edi
		
		mov	esi, @@disk_info
		mov	ah, 08h			; get hard disk info
		mov	dl, byte ptr @@hd

		movzx	edx, dl
		mov	[esi].hd_num, edx

		rm_int	13h, 0

		jc	@@error		

		xor	eax, eax
		mov	al, dl
		mov	@@num_disks, eax

		movzx	eax, dh
		inc	eax
		mov	[esi].num_heads, eax

		mov	eax, ecx
		and	eax, 3Fh
		mov	[esi].sect_per_track, eax

		mov	eax, ecx
		xchg	ah, al
		shr	ah, 6
		movzx	eax, ax
		inc	eax
		mov	[esi].num_cyls, eax
		;
		;  Now we try to read last sector on last+1 cylinder of disk
		;  If we can do that - BIOS hides an extra cylinder
		;
		mov	ax, 0201	; read 1 sector
		mov	bx, 0		; beginning of the buffer
		mov	dl, byte ptr @@hd
		add	ch, 1
		jnc	@@skip
		add	cl, 40h
	@@skip:
		jc	@@count_total	; there is already 1024 cylinders

		rm_int	13h, 0

		jc	@@count_total	; error - no extra cylinder

		inc	[esi].num_cyls
@@count_total:
		mov	eax, [esi].sect_per_track
		mul	[esi].num_heads
		mov	[esi].sect_per_cyl, eax
		mul	[esi].num_cyls
		mov	[esi].total_sects, eax
		mov	[esi].hd_flags, 0
		;
		; Now lets check LBA interface
		;
		mov	edi, esi

		mov	ah, 41h		; Installation check
		mov	bx, 055AAh
		mov	dl, byte ptr @@hd

		rm_int	13h, 0
		jc	@@no_lba
		
		cmp	bx, 0AA55h
		jne	@@no_lba

		test	cx, 1
		jz	@@no_lba

		mov	ah, 48h		; Get drive parameters
		mov	bx, 055AAh
		mov	dl, byte ptr @@hd

		mov	esi, offset disk_param_buf
		mov	gs:[esi].lba_inf_buf_sz, 1Ah
		
		rm_int	13h, 0
		jc	@@no_lba

 		or	[edi].hd_flags, 1

		mov	eax, dword ptr gs:[esi].lba_inf_total
		cmp	eax, [edi].total_sects
		jb	@@dont_save_total
		mov	[edi].total_sects, eax
@@dont_save_total:	

		mov	ax, gs:[esi].lba_inf_flags
		test	ax, 2
		jz	@@calc_chs

		; Need to comment it because some BIOSes report 255 heads
		; for CHS interface and 14-16 for LBA - we will use CHS
		
		;mov	eax, gs:[esi].lba_inf_heads
		;mov	[edi].num_heads, eax
		;mov	ebx, gs:[esi].lba_inf_sects
		;mov	[edi].sect_per_track, ebx
		;mul	bx
		;mov	[edi].sect_per_cyl, eax
		;mov	eax, gs:[esi].lba_inf_cyls
		;jmp	@@check_cyls
@@calc_chs:
		mov	eax, [edi].total_sects
		mov	edx, eax
		shr	edx, 16
		mov	ebx, [edi].sect_per_cyl
		div	bx
		movzx	eax, ax
@@check_cyls:
		cmp	eax, [edi].num_cyls
		jb	@@dont_save_cyls
		mov	[edi].num_cyls, eax
@@dont_save_cyls:
@@no_lba:
		mov	eax, @@num_disks
		jmp	@@end
@@error:
		mov	eax, -1
@@end:
		pop	edi
		pop	esi
		pop	ebx

		ret
disk_get_info	endp

;----------------------------------------------------------------------

disk_operation	proc	near
		;
		;  Input:    ah  - operation (read/write/verify)
		;            ecx - number of sectors
		;            edi - relative sector
		;            esi - address of hd_info
		;
		;  Output:   CF  - set on error
		;            ecx - number of sectors successfully
		;			   (read/written/verified)
		;
		;  Destroys: eax, ebx, edx, esi
		;

		test	[esi].hd_flags, 1
		jz	@@chs_mode
@@lba_mode:
		push	eax
		mov	 dl, byte ptr [esi].hd_num
		mov	esi, offset disk_param_buf
		mov	gs:[esi].lba_buf_size, 10h
		mov	gs:[esi].lba_reserved, 00h
		mov	ah, 0
		mov	gs:[esi].lba_num_sect, cx
		mov	ax, buffer_seg
		mov	word ptr gs:[esi].lba_buf_addr, 0
		mov	word ptr gs:[esi].lba_buf_addr+2, ax
		mov	dword ptr gs:[esi].lba_rel_sect, edi
		mov	dword ptr gs:[esi].lba_rel_sect+4, 0
		pop	eax
		add	ah, 40h	; LBA operations
		mov	al, 00h ; zero flags if writing (no verify)
		rm_int  13h, 0
		movzx	ecx, gs:[esi].lba_num_sect
		jmp	@@end
@@chs_mode:
		mov	al, cl	  ; num_sect
		push	eax
		mov	eax, edi  ; relative
		xor	edx, edx
		mov	ebx, [esi].sect_per_track
		div	ebx
		mov	cl, dl	; sector
		inc	cl
		xor	edx, edx
		mov	ebx, [esi].num_heads
		div	ebx
		mov	dh, dl	; head
		mov	ch, al	; cyl-lo
		shr	eax, 2
		and	al, 0C0h
		or	cl, al	; cyl-hi
		mov	dl, byte ptr [esi].hd_num
		pop	eax

		xor	ebx, ebx	; beginning of the buffer
		rm_int  13h, 0
		movzx	ecx, al
@@end:
		ret

disk_operation	endp

;----------------------------------------------------------------------

copy_buffer_to_data_seg macro
		;
		; Input: ecx - number of bytes to copy
		;        edi - address of buffer in data segment
		;
		xor	esi, esi
		push	ds
		mov	ds, buffer_sel
		push	ecx
		shr	ecx, 2
		cld
		rep
		  movsd
		pop	ecx
		test	ecx, 2
		jz	@@skip2
		movsw
@@skip2:
		test	ecx, 1
		jz	@@skip1
		movsb
@@skip1:
		pop	ds
		
endm

;----------------------------------------------------------------------

copy_data_seg_to_buffer	macro
		;
		; Input: ecx - number of bytes to copy
		;        esi - address of buffer in data segment
		;
		xor	edi, edi
		push	es
		mov	es, buffer_sel
		push	ecx
		shr	ecx, 2
		cld
		rep
		  movsd
		pop	ecx
		test	ecx, 2
		jz	@@skip2
		movsw
@@skip2:
		test	ecx, 1
		jz	@@skip1
		movsb
@@skip1:
		pop	es

endm

;----------------------------------------------------------------------

disk_read	proc	C	near

		arg	@@hd_info: dword, @@rel: dword, @@buf: dword, \
							@@num: dword
		push	ebx
		push	esi
		push	edi
		
@@read_more:
		mov	ecx, @@num
		mov	edi, @@rel
		mov	esi, @@hd_info
		
		cmp	ecx, 128  ; max num of sectors that 64k buffer can hold
		jbe	@@num_sect_ok
		mov	ecx, 128
@@num_sect_ok:

		push	ecx
		mov	ah, 2   	; read
		call	disk_operation
		pop	ecx		; number of sectors read
		jc	@@error

		push	ecx
		shl	ecx, 9		;   512 byted per sector
		mov	edi, @@buf
		copy_buffer_to_data_seg
		mov	@@buf, edi	; remember next position
		pop	ecx

		add	@@rel, ecx
		sub	@@num, ecx
		jnz	@@read_more

		xor	eax, eax

		jmp	@@end
@@error:
		mov	eax, -1
@@end:
		pop	edi
		pop	esi
		pop	ebx

		ret
disk_read	endp

;----------------------------------------------------------------------

disk_write	proc	C	near

		arg	@@hd_info: dword, @@rel: dword, @@buf: dword, \
							@@num: dword
		push	ebx
		push	esi
		push	edi
		
@@write_more:
		mov	ecx, @@num
		cmp	ecx, 128  ; max num of sectors that 64k buffer can hold
		jbe	@@num_sect_ok
		mov	ecx, 128
@@num_sect_ok:
		push	ecx		; number of sectors to write
		shl	ecx, 9		; 512 byted per sector
		mov	esi, @@buf
		copy_data_seg_to_buffer
		mov	@@buf, esi	; remember next position
		pop	ecx
		
		mov	edi, @@rel
		mov	esi, @@hd_info

		push	ecx
		mov	ah, 3  ; write
		call	disk_operation
		pop	ecx
		jc	@@error

		add	@@rel, ecx
		sub	@@num, ecx
		jnz	@@write_more

		xor	eax, eax

		jmp	@@end
@@error:
		mov	eax, -1
@@end:
		pop	edi
		pop	esi
		pop	ebx

		ret
disk_write	endp

;----------------------------------------------------------------------

disk_verify	proc	C	near

		arg	@@hd_info: dword, @@rel: dword, @@num: dword
		local	@@num_done

		push	ebx
		push	esi
		push	edi
		
		mov	@@num_done, 0

@@verify_more:
		mov	ecx, @@num
		mov	edi, @@rel
		mov	esi, @@hd_info
		test	[esi].hd_flags, 1 
		mov	ebx, 255   ; max num of sectors to verify for CHS mode
		jz	@@chs_mode
		mov	ebx, 65535 ; max num of sectors to verify for LBA mode
@@chs_mode:
		cmp	ecx, ebx
		jbe	@@num_sect_ok
		mov	ecx, ebx
@@num_sect_ok:
		mov	edi, @@rel
		mov	esi, @@hd_info
		push	ecx
		mov	ah, 4 ; verify
		call	disk_operation
		pop	ecx
		jc	@@error

		add	@@num_done, ecx
		add	@@rel, ecx
		sub	@@num, ecx

		jnz	@@verify_more
		jmp	@@end
@@error:
		cmp	ecx, 1
		je	@@end
		inc	ecx
		shr	ecx, 1
;		cmp	ecx, 1
;		je	@@num_sect_ok
;		inc	ecx
;		shr	ecx, 1
		jmp	@@num_sect_ok
@@end:
		mov	eax, @@num_done

		pop	edi
		pop	esi
		pop	ebx

		ret
disk_verify	endp

;----------------------------------------------------------------------

open		proc	C	near

		arg	@@path: dword, @@mode: dword

		push	ebx
		push	esi
		push	edi
		
		mov	esi, @@path	; copy path to buffer
		xor	edi, edi
		push	es
		mov	es, buffer_sel
		cld
@@next_char:
		lodsb
		stosb
		or	al, al
		jnz	@@next_char
		pop	es

		test	@@mode, 0200h ; O_TRUNC
		jnz	@@create

;		mov	ah, 41h  ; delete file
;		xor	edx, edx ; address of the path
;		rm_int	21h, 1   ; pass buffer_seg in ds
;		jnc	@@create

		mov	eax, @@mode
		mov	ah, 3Dh  ; open file
		xor	edx, edx ; address of the path
		rm_int	21h, 1   ; pass buffer_seg in ds
		movzx	eax, ax
		jnc	@@end
		
		; Error
		
		cmp	eax, 2   ; File not found
		jne	@@error
		
		test	@@mode, 0100h ; O_CREAT
		jz	@@error
@@create:
		mov	ah, 3Ch  ; create a new file / truncate existing file 
		xor	ecx, ecx ; normal attributes
		xor	edx, edx ; address of the path
		rm_int	21h, 1   ; pass buffer_seg in ds
		movzx	eax, ax
		jnc	@@end
@@error:
		mov	eax, -1
@@end:
		pop	edi
		pop	esi
		pop	ebx

		ret
open		endp

;----------------------------------------------------------------------

close		proc	C	near

		arg	@@fd: dword

		push	ebx
		push	esi
		push	edi
		
		xor	eax, eax
		mov	 ah, 3Eh	; close file
		mov	ebx, @@fd
		
		rm_int	21h, 0
		
		mov	eax, 0

		jnc	@@end
		
		mov	eax, -1
@@end:
		pop	edi
		pop	esi
		pop	ebx

		ret
close		endp

;----------------------------------------------------------------------

lseek		proc	C	near

		arg	@@fd: dword, @@pos: dword, @@mode: dword

		push	ebx
		push	esi
		push	edi
		
		mov	eax, @@mode
		mov	 ah, 42h	; lseek
		mov	ebx, @@fd
		mov	ecx, @@pos	; pos in CX:DX
		mov	edx, ecx
		shr	ecx, 16
		
		rm_int	21h, 0
		
		jc	@@error
		
		shl	edx, 16		; returns pos in DX:AX
		mov	 dx, ax
		mov	eax, edx
		
		jmp	@@end
@@error:
		mov	eax, -1
@@end:
		pop	edi
		pop	esi
		pop	ebx

		ret
lseek		endp

;----------------------------------------------------------------------

read		proc	C	near

		arg	@@fd: dword, @@buf: dword, @@len: dword
		local	@@read: dword

		push	ebx
		push	esi
		push	edi
		
		mov	@@read, 0
@@read_more:
		mov	ah, 3Fh  ; read
		mov	ebx, @@fd
		mov	ecx, @@len
		cmp	ecx, 65024 ; 65536-512
		jbe	@@num_bytes_ok
		mov	ecx, 65024
@@num_bytes_ok:
		xor	edx, edx ; address of the buffer
		rm_int	21h, 1   ; pass buffer_seg in ds
		jc	@@error

		mov	ecx, eax
		mov	edi, @@buf
		copy_buffer_to_data_seg
		mov	@@buf, edi
	
		add	@@read, eax
		sub	@@len, eax
		cmp	eax, 0
		je	@@eof
		cmp	@@len, 0
		jne	@@read_more
@@done:
@@eof:
		mov	eax, @@read
		jmp	@@end
@@error:
		mov	eax, -1
@@end:
		pop	edi
		pop	esi
		pop	ebx

		ret
read		endp

;----------------------------------------------------------------------

write		proc	C	near

		arg	@@fd: dword, @@buf: dword, @@len: dword
		local	@@written: dword

		push	ebx
		push	esi
		push	edi
		
		mov	@@written, 0
@@write_more:
		mov	ah, 40h  ; write
		mov	ebx, @@fd
		mov	ecx, @@len
		cmp	ecx, 65024 ; 65536-512
		jbe	@@num_bytes_ok
		mov	ecx, 65024
@@num_bytes_ok:
		push	ecx
		mov	esi, @@buf
		copy_data_seg_to_buffer
		mov	@@buf, esi
		pop	ecx
		
		xor	edx, edx ; address of the buffer
		rm_int	21h, 1   ; pass buffer_seg in ds
		jc	@@error

		add	@@written, eax
		sub	@@len, eax
		cmp	eax, ecx
		jne	@@done   ; not all bytes were written - disk full
		cmp	@@len, 0
		jne	@@write_more
@@done:
		mov	eax, @@written
		jmp	@@end
@@error:
		mov	eax, -1
@@end:
		pop	edi
		pop	esi
		pop	ebx

		ret
write		endp

;----------------------------------------------------------------------
_TEXT 		ends
;----------------------------------------------------------------------
