#ifndef PART_H
#define PART_H

#include "ext32.h"
#include "part_msg.h"

#define SECT_SIZE  512		/* Must not be changed */


typedef struct 			/* Partition record in MBR */
    {
     unsigned char active;
     unsigned char start_head;
     unsigned      start_sect:6;
     unsigned      start_cylH:2;
     unsigned char start_cylL;
     unsigned char fs_id;
     unsigned char end_head;
     unsigned      end_sect:6;
     unsigned      end_cylH:2;
     unsigned char end_cylL;
     unsigned long rel_sect;	/* Number os sectors prior to partition */
     unsigned long num_sect;	/* Number of sectors in the partition */

    }MbrPartRec;

#define FD_CHS	    (0x00010000)
#define BM_MAGIC2   (0x4D42)
#define MAGIC_NUM   (0xAA55)
#define MAGIC_NUM32 (0xAA550000)

typedef struct
    {
     unsigned char code[0x1AE];
     unsigned long bm_chs;
     unsigned long fd_chs;
     unsigned long bm_rel_sect;
     unsigned char ipl_timeout;
     unsigned char ipl_default;
     unsigned short  bm_magic2; /* 'BM' = 0x4D42 */

     MbrPartRec      part_rec[4];
     unsigned short  magic_num;  /* Magic number (must be 0xAA55) */

    }Mbr;

#define N_ROWS 48

#define P_FS_FLAGS	0x0000FFFF /* Flags that depends on file system */

#define OP_ANY_TYPE	0x000003FF /* Methods to work with file systems */ 

#define OP_VALIDATE	0x00000001
#define OP_PRINT	0x00000002
#define OP_PREVIEW	0x00000004
#define OP_SETUP	0x00000008
#define OP_CHKDSK	0x00000010
#define OP_FORMAT	0x00000020
#define OP_VERIFY	0x00000040
#define OP_DEFRAG	0x00000080
#define OP_RESIZE	0x00000100
#define OP_MOVE_FIX	0x00000200

#define P_HIGHLIGHT	0x00001000 /* Predefined file system attributes */
#define P_EXTENDED	0x00002000
#define P_HIDABLE	0x00004000
#define P_UNHIDABLE	0x00008000


#define P_DYN_FLAGS	0x00FF0000 /* Dynamic attributes of partitions */

#define P_NOT_EMPTY	0x00010000
#define P_LOGICAL	0x00020000
#define P_BTS_LOADED	0x00040000


#define P_ANY_ERROR     0xFF000000

#define P_LOC_ERROR     0x1F000000 /* Problems with partition location */
#define P_ERR_ZERO      0x01000000
#define P_ERR_RANGE     0x02000000
#define P_ERR_OVERLAP   0x04000000
#define P_ERR_LOG_ORPH  0x08000000
#define P_ERR_LOG_STRU  0x10000000

#define P_BTS_ERROR     0xE0000000 /* Problems with partition contents */
#define P_ERR_CANT_READ 0x20000000
#define P_ERR_BOOT_SECT 0x40000000


#define EMPTY(p)       (!((p)->flags & P_NOT_EMPTY))
#define VALID(p)       (!((p)->flags & P_ANY_ERROR))
#define VALID_LOC(p)   (!((p)->flags & P_LOC_ERROR))
#define VALID_BTS(p)   (!((p)->flags & P_BTS_ERROR))
#define UNUSED(p)        ((p)->fs_id == FS_UNUSED)
#define LOGICAL(p)       ((p)->flags & P_LOGICAL)
#define EXTENDED(p)      ((p)->flags & P_EXTENDED)
#define BTS_LOADED(p)    ((p)->flags & P_BTS_LOADED)


typedef struct
    {
     unsigned long  flags;
     unsigned short fs_num;	/* fs number in the file system type table */
     unsigned short fs_id;
     unsigned short mbr_row;
     unsigned short tag;

     Sector start_cyl;
     Sector start_head;
     Sector start_sect;

     Sector end_cyl;
     Sector end_head;
     Sector end_sect;

     Sector rel_sect;
     Sector num_sect;
     Sector last_sect;
     Sector ext_base;

     Sector rel_sect_loaded;
     void   *part_boot_sect;
     void   *part_misc_info;

    }Part;

#define FS_UNUSED    0x0000
#define FS_BM        0x0001
#define FS_DOSEXT    0x0500
#define FS_HIDDEN    0xFF80
#define FS_MBR       0xFF81
#define FS_UNKNOWN   0xFFFF

struct fs_desc
    {
     unsigned short fs_id;
     unsigned short flags;
     char           *name;
     int (*handler)(int op, DiskInfo *, Part *, char **argv);
    };

#define OP_HANDLER(op,di,p,v) (fs_desc[(p)->fs_num].handler((op),(di),(p),(v)))

struct mode_desc
    {
     char tab_x;
     char tab_y;
     char n_rows;
     char n_cols;
     char pos[8];
     char len[8];
     char cur_pos[8];
    };


#define MAX_PART_ROWS  32
#define MAX_MENU_ROWS  20

typedef struct			/* 12 bytes */
    {
     unsigned short fs_id;
     unsigned char  flags;
     unsigned char  mbr_row;
     unsigned long rel_sect;
     unsigned long num_sect;
    }BmPartRec;

typedef struct		/* 80 bytes */
   {
    unsigned char boot_dev;	/* if 0 - empty */
    unsigned char boot_part;
    unsigned char name[30];
    unsigned char options;
    unsigned char num_keys;
    unsigned short keys[15];
    unsigned long visible[4];
   }BmMenuRec;


#define BM_MAGIC (0xEA4A4D42)   /* BMJ      */
#define BM_JADDR (0x08400000)   /* 0840:0000 */
#define BM_VERSION   (23800L)   /* v 2.38.00 */

#define BM_OPT_VIRUS_CHECK   0x0001
#define BM_OPT_INSTALL_PM    0x0002
#define BM_OPT_USE_PART      0x0004
#define BM_OPT_USE_MENU      0x0008
#define BM_OPT_USE_DEFAULT   0x0010
#define BM_OPT_USE_PASSWORD  0x0020

typedef struct 			/* 64+384+1600 = 2048 bytes   */
    {
     unsigned long bm_magic;   /* BMJ           16+16+32=64 */
     unsigned long bm_jaddr;
     unsigned long version;
     unsigned long options;

     unsigned char menu_row;
     unsigned char menu_col;
     unsigned char bm_timeout;
     unsigned char bm_default;

     unsigned char cur_ipl;
     unsigned char new_ipl;
     unsigned char ipl_timeout;
     unsigned char ipl_default;

     unsigned char disks[4];
     unsigned long password;

     unsigned char title[32];

     BmPartRec part[MAX_PART_ROWS];	/* 12*32= 384 */
     BmMenuRec menu[MAX_MENU_ROWS];	/* 80*20=1600 */

    }BmData;


#define IPL_TYPE_UNKN  0
#define IPL_TYPE_BM    1
#define IPL_TYPE_STD   2

#define BM_IPL_SIZE	0x1AE
#define STD_IPL_SIZE	0x0DB
#define EMP_IPL_SIZE	0x020
#define FAT16_BOOT_SIZE 0x1C0

extern char BM_IPL[];			/* external assembly code */
extern char STD_IPL[];
extern char EMP_IPL[];
extern char FAT16_BOOT[];

#define max(x,y)   ((x)>(y)?(x):(y))
#define min(x,y)   ((x)<(y)?(x):(y))


#define MODE_CHS     0    /* modes for displaying partition table */
#define MODE_LBA     1
#define MODE_EXT     2
#define MODE_MINI    3
#define MODE_MISC    4
#define MODE_EXACT   5
#define MODE_TITLE   6


typedef struct
    {
     Sector rel_sect_loaded;
     int  sys;
     int  boot_fix;
     char sysid[12];
     char fs_id[12];
     char label[12];
     int  fat_size;
     int  clust_size;
     
     Sector rel_sect;
     Sector exp_rel_sect;
     Sector num_sect;
     Sector exp_num_sect;
     Sector max_num_sect;
     Sector drive_num;
     Sector exp_drive_num;

     Sector fat_rel_sect;
     Sector fat_sect_read;
     Sector num_data_sect;
     Sector last_data_sect;
     Sector first_data_sect;
    }FatInfo;


/* part.c */

void start_gui(int hd);
int edit_part_col( Part *p, int col, int mode );

int h_deflt    ( int op, DiskInfo *, Part *, char **argv );
int ipl_handler( int op, DiskInfo *, Part *, char **argv );
int ext_handler( int op, DiskInfo *, Part *, char **argv );
int fat_handler( int op, DiskInfo *, Part *, char **argv );


/* part_aux.c */

int  prepare_mbr_records( Part *src, Part *dst );

void   pack_part_rec( Part *, MbrPartRec * );
void unpack_part_rec( MbrPartRec *, Part * );

void   pack_bm_part_rec( Part *, BmPartRec * );
void unpack_bm_part_rec( BmPartRec *, Part * );

void recalculate_partition( DiskInfo *, Part *, int mode);
void part_import( DiskInfo *, Part *, Sector ext_base, Sector log_base);
void part_export( DiskInfo *, Part *, Sector ext_base, Sector log_base);
int sort_part_table( DiskInfo *, Part *, int *row);

void sprintf_partition(char *tmp, int num, Part *p, int mode);

int  find_ipl_type( Mbr * );
void find_fs_num( Part *p );
void free_part_info( Part *p );
int count_zeros( void *buf, int len );
unsigned long calc_hash( void *p, int n );

/* part_cmd.c */

void start_cmd( int hd, int argc, char **argv );
void print_part_table( DiskInfo *, Part *, int details );
void print_fat_details( FatInfo * );

/* part_dsk.c */

int read_part_table( DiskInfo *, Part * );
int save_part_table( DiskInfo *, Part * );
void read_boot_sect( DiskInfo *, Part *, int num_sect );
int write_boot_sect( DiskInfo *, Part *, void *buf, int num_sect );

int verify_part( DiskInfo *, Part *, Sector *bbt, int bbt_size );
int read_part_rel( DiskInfo *, Part *, Sector rel, void *buf, int num_sect );
int write_part_rel( DiskInfo *, Part *, Sector rel, void *buf, int num_sect );

/* part_gui.c */


#define EV_CANCEL -2
#define EV_FAILED -1
#define EV_OK      0

void gui_init(void);
void gui_done(void);

#define HINT    0
#define MESSAGE 1
#define WARNING 2

void redraw_main_screen( DiskInfo *, int mode );
void redraw_main_prompt( int save );
void redraw_main_hint( int h_type, char *hint );
void redraw_part_table( int first_row, int row, int col, Part *, int n_rows, int mode );

void clear_error( int p );
void popup_error( char *err );

void show_help( int topic );
void show_error( char *msg );
int dialog_box( char *msg, char *ch1, ... );

int   edit_int( unsigned long lim, unsigned long *n );
void write_int( int attr, int x, int y, int w, int n );

int progress( char *msg );
void select_fs_type( Part * );

void clear_preview_box( char *);

void   edit_ipl_options( BmData * );
void redraw_ipl_options( BmData *, int act );

void redraw_ext_details( Part * );

void redraw_fat_details( FatInfo *, int act );
void redraw_fat_prompt( int changed );

void update_fat_info( DiskInfo *, Part *, FatInfo * );
void read_fat_table( DiskInfo *, Part *, FatInfo * );
void bgview_fat_details( DiskInfo *, Part *, FatInfo *, int act );
void edit_fat_details( DiskInfo *, Part *, FatInfo * );
void save_fat_boot_sect( DiskInfo *, Part *, FatInfo * );
void fat_format( DiskInfo *, Part *, FatInfo * );

void show_bad_blocks( DiskInfo *, Part *, Sector *bbt, int bbt_size );


/* part_wiz.c */

void part_ins_wiz( DiskInfo *, Part *, int *row, int *n );
void part_new_wiz( DiskInfo *, Part *, int *row, int *n );


#define TMP_SZ	1024

#ifdef PART_C
  char tmp[TMP_SZ];
#else
  extern char tmp[TMP_SZ];
#endif

#ifndef PART_TAB_C
  extern struct fs_desc   fs_desc[];
  extern struct mode_desc mode_desc[];
#endif


#endif
