/*
 Ranish Partition Manager   Copyright (C) 1998 by Mikhail Ranish
 This code is distributed under GNU General Public License (GPL)
 The full text of the license can be found at http://www.gnu.org
*/

#define PART_DSK_C
#include "part.h"


int read_part_table( DiskInfo *di, Part *part )
{
 Mbr *mbr;
 BmData *bm;
 int i, curr, n, top, num;
 Part *p, part2[4];
 Sector ext_base, log_base;

 if( (bm=malloc(sizeof(BmData)))==0 )
   {
    show_error(ERROR_MALLOC);
    return -1;
   }

 memset(part,0,N_ROWS*sizeof( Part));

 p=&part[0];
 ext_base=0;
 log_base=0;

 p->fs_id=FS_MBR;
 find_fs_num(p);
 p->rel_sect=0;
 p->num_sect=1;
 p->part_misc_info=bm;
 recalculate_partition(di,p,MODE_LBA);

 /* We are reading MBR and then EMBRs on each cycle of this loop */
 /* curr is an index in part[] of (E)MBR that is now processed   */
 /* n - number of records that have been read so far             */

 n=1;
 curr=0;

 while(1)
    {
     p = &part[curr];
     read_boot_sect(di,p,1);
     if( !(BTS_LOADED(p)) ) break;

     mbr = (Mbr *) p->part_boot_sect;

     curr=0;

     for( i=0 ; i<4 ; i++ )
      if( mbr->part_rec[i].fs_id!=0 )   /* not unused - so we import it */
        {
         unpack_part_rec( &mbr->part_rec[i], &part[n] );
         part[n].mbr_row = i+1;
         if( EXTENDED(p) )
          part[n].flags |= P_LOGICAL;
         part_import(di, &part[n], ext_base, log_base );

         if( EXTENDED(&part[n]) && curr==0 ) curr = n;

         n++;
         if( n==N_ROWS ) break;
        }

     if( n==N_ROWS ) break;
     if( curr==0 ) break;
     log_base=part[curr].rel_sect;
     if( ext_base==0 ) ext_base=log_base;
    }

 if( !BTS_LOADED(&part[0]) ) /* didn't even read MBR */
   {
    return -1;
   }

 // guess smart on the real partition types

 mbr = part[0].part_boot_sect;

 memset(bm,0,sizeof(BmData));
 
 bm->bm_magic = BM_MAGIC;
 bm->bm_jaddr = BM_JADDR;
 bm->version = BM_VERSION;
 
 bm->ipl_timeout = 6;                /* Default settings */
 bm->options = BM_OPT_VIRUS_CHECK;

 if( mbr->bm_magic2==BM_MAGIC2 && mbr->bm_rel_sect!=0 )
   {
    if( disk_read(di,mbr->bm_rel_sect,bm,4)==-1 )
      {
       show_error("Error reading bm_data"); //ERROR_READ_MBR);
      }
    else
      {
       // import new partitions here (if any)
      }
   }

 if( mbr->bm_magic2==BM_MAGIC2 )
   {
    bm->options    |= mbr->ipl_timeout&1;
    bm->ipl_timeout = mbr->ipl_timeout/2;
    bm->ipl_default = mbr->ipl_default;
   }

 bm->cur_ipl = find_ipl_type(mbr);
 bm->new_ipl = bm->cur_ipl;

 return 0;
}/* read_part_table */


int save_part_table( DiskInfo *di, Part *part )
{
 int i, j, curr, next;
 Sector ext_base, log_base;
 Part part2[4];
 Mbr *mbr = (Mbr*)tmp;
 BmData *bm = part[0].part_misc_info;

 if( bm->new_ipl==IPL_TYPE_UNKN )
   {
    memset(mbr,0,0x1BE);
    memmove(mbr,part[0].part_boot_sect,0x1BE);
   }
 else if( bm->new_ipl==IPL_TYPE_STD )
   {
    memset(mbr,0,0x1BE);
    memmove(mbr,STD_IPL,STD_IPL_SIZE);
    bm->cur_ipl=IPL_TYPE_STD;
   }
 else if( bm->new_ipl==IPL_TYPE_BM )
   {
    memset(mbr,0,0x1BE);
    memmove(mbr,BM_IPL,BM_IPL_SIZE);
    mbr->bm_chs=0;
    mbr->fd_chs=FD_CHS;
    mbr->bm_rel_sect=0;
    mbr->ipl_default=bm->ipl_default;
    mbr->ipl_timeout=(bm->ipl_timeout<<1)+(bm->options&1);
    mbr->bm_magic2=BM_MAGIC2;
    bm->cur_ipl=IPL_TYPE_BM;
   }

 curr=0;       /* current MBR or EMBR that is being saved */
 next=0;       /* EMBR that will be processed next        */
 log_base=0;
 ext_base=0;

 prepare_mbr_records(part,part2);

 for( i=0 ; i<N_ROWS ; i++ )
  if( EXTENDED(&part[i]) )
    {
     next=i;
     break;
    }

 while(1)
    {
     for( i=0 ; i<4 ; i++ )
        {
         part_export(di,&part2[i],ext_base,log_base);
         pack_part_rec(&part2[i],&mbr->part_rec[i]);
        }

     mbr->magic_num=MAGIC_NUM;

     if( write_boot_sect(di,&part[curr],mbr,1)==-1 )
       {
        return -1;
       }

     if( next==0 ) break;
                                             /* preparing logical partitions */
     curr=next;
    
     log_base = part[curr].rel_sect;

     memset(mbr,0,512);
     memset(part2,0,4*sizeof(Part)); 

     if( !LOGICAL(&part[curr]) )
       {
        ext_base = log_base;
        memmove( mbr, EMP_IPL, EMP_IPL_SIZE );
        strcpy( ((char*)mbr) + EMP_IPL_SIZE, EXT_NOT_BOOTABLE );
       }
     
     j=0;
     for( next=curr+1 ; next<N_ROWS ; next++ )
      if( !UNUSED(&part[next]) && (LOGICAL(&part[next]) ||
                                  EXTENDED(&part[next]) ) )
        {
         if( j<3 || j==3 && EXTENDED(&part[next]) )
           {
            part2[j]=part[next];
            j++;
           }
         if( EXTENDED(&part[next]) ) break;
        }
 
     if( next==N_ROWS ) /* no more extended partitions */
       next=0;
    }/* while */

 return 0;
}/* save_part_table */



void read_boot_sect( DiskInfo *di, Part *p, int n )
{
 if(  p->rel_sect == p->rel_sect_loaded  &&
    ( BTS_LOADED(p) || (p->flags & P_ERR_CANT_READ) ) ) return;
 
 p->flags &= ~(P_ERR_CANT_READ + P_BTS_LOADED); /* clearing flags */
 
 if( p->part_boot_sect==0 )
   {
    if( (p->part_boot_sect=malloc(512*n))==0 )
      {
       p->flags |= P_ERR_CANT_READ;
       return;
      }
   }

 p->rel_sect_loaded = p->rel_sect;

 if( disk_read(di,p->rel_sect,p->part_boot_sect,n)==-1 )
   {
    p->flags |= P_ERR_CANT_READ;
    return;
   }

 p->flags |= P_BTS_LOADED;
}/* read_boot_sect */



int write_boot_sect( DiskInfo *di, Part *p, void *buf, int n)
{
 if( buf==0 ) buf = p->part_boot_sect;

 if( buf==0 ) return -1;

 if( disk_write(di,p->rel_sect,buf,n)==-1 )
   {
    return -1;
   }

 if( p->part_boot_sect!=0 && p->part_boot_sect!=buf )
   {
    memmove(p->part_boot_sect,buf,n*512);
    p->rel_sect_loaded = p->rel_sect;
    p->flags |= P_BTS_LOADED;
    p->flags &= ~P_ERR_CANT_READ;
   }

 return 0;
}/* write_boot_sect */


int read_part_rel( DiskInfo *di, Part *p, Sector rel,
                                                   void *buf, int num_sect)
{
 return disk_read(di, p->rel_sect + rel, buf, num_sect);
}


int write_part_rel( DiskInfo *di, Part *p, Sector rel, void *buf, int num_sect)
{
 return disk_write(di, p->rel_sect + rel, buf, num_sect);
}


int verify_part( DiskInfo *di, Part *p, Sector *bbt, int  bbt_size)
{
 Sector base_sect;
 int x, num_sect, num_bad, num_done;

 num_bad = 0;
 base_sect = 0;

 // Win95 prevents to write/verify its drives to protect long file names
 // disk_lock(hd); it seems like we dont have to lock drives from 32-bit code
 
 progress(MESG_VERIFYING);
 progress("%   0%");
 
 if( p->num_sect==0 ) return EV_OK;
 
 while( base_sect < p->num_sect )
    {
     num_sect = min( p->num_sect-base_sect, di->sect_per_cyl );

     num_done = disk_verify(di, p->rel_sect+base_sect, num_sect);

     base_sect += num_done;
     x = base_sect*100/p->num_sect;

     if( num_done!=num_sect )
       {
        if( num_bad==bbt_size ) /* BBT is full */
          {
           break;
          }
        bbt[num_bad] = base_sect;
        num_bad++;
        base_sect++; /* skipping bad sector */
        sprintf(tmp,"%% %3d%% Bad sector:%3d", x, num_bad );
       }
     else
       {
        sprintf(tmp,"%% %3d%% Cylinder:%5d", x,
                           (p->rel_sect+base_sect)/di->sect_per_cyl );
       }

     if( progress(tmp)==EV_CANCEL )
       {
        //disk_unlock(hd);
        return EV_CANCEL;
       }
    }
    
 //disk_unlock(hd);
 return num_bad;
}/* verify_part */


