/*
Copyright (C) 1997-1999 Erwin Waterlander

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "std_macro.h"
#include "structures.h"
#include "nameset.h"
#include "display.h"
#include "wcd.h"


/************************************************************************
 *
 *  swap(), ssort() and sort_list()
 *
 *  Sort a nameset list alphabeticly
 *
 ************************************************************************/

void swap(nameset list, int i, int j)
{  text temp;

   temp = list->array[i];
   list->array[i] = list->array[j];
   list->array[j] = temp;
}

void ssort (nameset list, int left, int right)
{
 int i, last;

  if (left >= right) return; /* fewer than 2 elements */

  swap(list, left, (left + right)/2);
  last = left;

  for (i = left+1; i <=right; i++)
	if  (strcmp(list->array[i],list->array[left])<0)
	   swap(list, ++last, i);

  swap(list, left, last);
  ssort(list, left, last-1);
  ssort(list, last+1, right);
}

void sort_list(nameset list)
{
 ssort(list,0,(list->size)-1);
}

/************************************************************************
 *
 *  maxLength()
 *  Get the longest string in a nameset list.
 *
 ************************************************************************/

#ifdef WCD_USECURSES
int maxLength(nameset list)
{
	int i,len,maxlen = 0;

   for (i=0;i<list->size;i++)
   {
    if( (len=strlen(list->array[i])) > maxlen)
       maxlen=len;
   }
	if (maxlen > 32)
      return(maxlen);
	else
	   return 32 ;     /* minimal width for help screen */
}

/************************************************************************
 *
 *  wcd_prefresh()
 *  wcd wrapper around the ncurses prefresh() function to prevent
 *  reads and writes over the WINDOW array bounds.
 *
 ************************************************************************/

int wcd_prefresh(WINDOW *pad, int pminrow, int pmincol,
             int sminrow, int smincol, int smaxrow, int smaxcol, int pmaxcol, int shift)
{
	int p_mincol,s_maxcol;

	/* We are only going to shift sideways */

	if (shift > pmaxcol)    /* shift too far */
		shift = pmaxcol;

	p_mincol = pmincol + shift;

	s_maxcol = pmaxcol - shift;

	if (s_maxcol > smaxcol)  /* screen smaller than displayed part of pad */
		s_maxcol = smaxcol;

   return prefresh(pad,pminrow,p_mincol,sminrow,smincol,smaxrow,s_maxcol);
}
#endif

/************************************************************************
 *
 *  display_list(nameset list)
 *
 *  Display a match list on screen.
 *
 *  There are three versions of the function:
 *
 *  1) CONIO version
 *     Default used for DOS, optional for Windows NT.
 *     Scroll back is programmed in.
 *
 *  2) NCURSES version
 *     Optional for Unix. Very similar to CONIO version.
 *     Scroll back is programmed in.
 *
 *  3) stdout version
 *     This version will work on any platform. Default for Unix.
 *     Under unix we make use of the standard scroll-back
 *     capability of the terminal.
 *
 ************************************************************************/

#ifdef WCD_USECONIO

/* upper left corner is (1,1) */

int display_list(nameset list,int perfect)
{
int i, top, bottom, c=0, extended, gtxt=0;
int page ;
char *buffer;
struct text_info ti;

	gettextinfo(&ti);

		buffer = (char *) malloc(ti.screenwidth * ti.screenheight * 2);

		if (buffer!=NULL)   /* get total screen */
		  gtxt = gettext(1,1,ti.screenwidth,ti.screenheight,buffer);


		sort_list(list);

		i= -1;

		if (list->size > SCROLL_WIN_HEIGHT )
		{
			clrscr();

			bottom = list->size -1;
			top = list->size - SCROLL_WIN_HEIGHT;

			window(1,1,ti.screenwidth,SCROLL_WIN_HEIGHT+1);

			clrscr();

			for (i=top;i<=bottom;i++)
			  cprintf("%c  %s\n\r",(char)(((i-top)%SCROLL_WIN_HEIGHT) + 'a'),list->array[i]);

			page = bottom / SCROLL_WIN_HEIGHT + 1 ;

			window (1,SCROLL_WIN_HEIGHT+1,ti.screenwidth,25);

			clrscr();

			if(perfect)
				cprintf("\n\rPerfect ");
			else
				cprintf("\n\rWild ");
			cprintf("match for %d directories.\n\r",list->size);
			cprintf("Please choose one (<Enter> to abort): ");
			fflush(stdout);
			gotoxy (PAGEOFFSET, 2);
			cprintf("w=up x=down  Page %d/%d      ",page,(list->size -1)/SCROLL_WIN_HEIGHT +1);
			gotoxy (OFFSET, 3);

			while ((c != 13 )&&(( c < 'a' ) || ( c > 'v' )))
			{

			  c = getch();

			  switch(c)
			  {
			  case 'x':
			  case 'w':
			  case 0:   /* extended key */

				if(c==0)
				  extended = getch();
				else
				{
				 if (c=='w')
					extended = 73;  /* Page Up */
				 else
					extended = 81;  /* Page Down */
				}


				if ((extended == 73) || /* Page Up */
					(extended == 72))   /* Arrow Up */
				{
				  window(1,1,ti.screenwidth,SCROLL_WIN_HEIGHT+1);
				  clrscr();

				  if(bottom > (SCROLL_WIN_HEIGHT -1))
				  {
					bottom = bottom - SCROLL_WIN_HEIGHT ;
					top = top - SCROLL_WIN_HEIGHT ;
				  }

				  if (top<0) top = 0;

				  if (bottom < (SCROLL_WIN_HEIGHT -1) )
				  gotoxy(1,SCROLL_WIN_HEIGHT-bottom);

				  for (i=top;i<=bottom;i++)
					  cprintf("%c  %s\n\r",(char)(((i-top)%SCROLL_WIN_HEIGHT) + 'a'),list->array[i]);

					page = bottom / SCROLL_WIN_HEIGHT + 1 ;

					window (1,SCROLL_WIN_HEIGHT+1,ti.screenwidth,25);
					gotoxy (PAGEOFFSET, 2);
					cprintf("w=up x=down  Page %d/%d      ",page,(list->size -1)/SCROLL_WIN_HEIGHT +1);
					gotoxy (OFFSET, 3);


				} /* Page Up */

				if ((extended == 81) || /* Page down */
					(extended == 80))   /* Arrow down */
				{
				  window(1,1,ti.screenwidth,SCROLL_WIN_HEIGHT+1);
				  clrscr();

				  if(bottom < (list->size -1 ))
				  {
					bottom = bottom + SCROLL_WIN_HEIGHT ;
					top = bottom - SCROLL_WIN_HEIGHT + 1;
				  }

				  for (i=top;i<=bottom;i++)
					cprintf("%c  %s\n\r",(char)(((i-top)%SCROLL_WIN_HEIGHT) + 'a'),list->array[i]);

					page = bottom / SCROLL_WIN_HEIGHT + 1 ;

					window (1,SCROLL_WIN_HEIGHT+1,ti.screenwidth,25);
					gotoxy (PAGEOFFSET, 2);
					cprintf("w=up x=down  Page %d/%d      ",page,(list->size -1)/SCROLL_WIN_HEIGHT +1);
					gotoxy (OFFSET, 3);
				}/* Page down */

				window (1,SCROLL_WIN_HEIGHT+1,ti.screenwidth,25);
				gotoxy (OFFSET, 3);
			  break;
			  case 3:  /* Control-C */
			  case 13: /* Enter */
			  case 27: /* Escape */
				c = 13;
				i = -1;
				break;
			  default:
				i=c+top-'a'+1;
			  break;
			  }
			}
		window (1,1,ti.screenwidth,ti.screenheight);
		gotoxy (ti.curx, ti.cury);
		}
		else
		{
			for (i=0;i<list->size;i++)
				printf("%c  %s\n",(char)( (i%SCROLL_WIN_HEIGHT) + 'a'),list->array[i]);

			if(perfect)
				printf("\nPerfect ");
			else
				printf("\nWild ");
			printf("match for %d directories.\n",list->size);
			printf("Please choose one (<Enter> to abort): ");
			fflush(stdout);


			while ((c != 13 )&&(( c < 'a' ) || ( c > ('a'+list->size -1) )))
			{
			  c = getch();

			  switch(c)
			  {
			  case 0:
				  extended = getch();
			  break;
			  case 3:  /* Control-C */
			  case 13: /* Enter */
			  case 27: /* Escape */
				c = 13;
				i = -1;
				break;
			  default:
				i=c -'a'+1;
			  break;
			 }
			}
		}

		printf("\n");

		if (gtxt ==1)
		  puttext(1,1,ti.screenwidth,ti.screenheight,buffer);

		if (buffer!=NULL)
		  free(buffer);
/*
	cprintf("window left      %2d\r\n",ti.winleft);
	cprintf("window top       %2d\r\n",ti.wintop);
	cprintf("window right     %2d\r\n",ti.winright);
	cprintf("window bottom    %2d\r\n",ti.winbottom);
	cprintf("attribute        %2d\r\n",ti.attribute);
	cprintf("normal attribute %2d\r\n",ti.normattr);
	cprintf("current mode     %2d\r\n",ti.currmode);
	cprintf("screen height    %2d\r\n",ti.screenheight);
	cprintf("screen width     %2d\r\n",ti.screenwidth);

	cprintf("current x        %2d\r\n",ti.curx);
	cprintf("current y        %2d\r\n",ti.cury); */

		return i;
}

#elif defined WCD_USECURSES

/* upper left corner is (0,0) */

int display_list(nameset list,int perfect)
{
  int i, line, top, bottom, c=0;
  int scrollWinHeight, page, len, shift=0, err ;
  int scrollWinLen, inputWinLen ;
  WINDOW *scrollWin, *inputWin ;

/* Notice that list->size > 1 when this function is called. */

	sort_list(list);

   initscr();
   keypad(stdscr, TRUE);
   intrflush(stdscr, FALSE);
   cbreak();
   noecho();
   nonl();
	scrollok(stdscr, TRUE);	/* enable scrolling */

   if (LINES < 4)
   {
      endwin();
#ifdef XCURSES
      XCursesExit();
#endif
      fprintf(stderr,"Wcd: error, screen height must be > 3.\n");
      return -1;
   }

   scrollWinHeight = LINES - INPUT_WIN_HEIGHT;
   if (scrollWinHeight > SCROLL_WIN_HEIGHT)
      scrollWinHeight = SCROLL_WIN_HEIGHT;

	len = maxLength(list);
	scrollWinLen = len + BLANKING;
   refresh();

   scrollWin = newpad(scrollWinHeight,scrollWinLen);
   if (scrollWin == NULL)
   {
      endwin();
#ifdef XCURSES
      XCursesExit();
#endif
      fprintf(stderr,"Wcd: error creating scroll window.\n");
      return -1;
   }

   scrollok(scrollWin, TRUE);

	bottom = list->size -1;
	top = list->size - scrollWinHeight;
	if (top < 0)
		top = 0;

	if (bottom < (scrollWinHeight -1) )
      line = scrollWinHeight - bottom - 1;
   else
      line = 0;

	for (i=top;i<=bottom;i++)
	{
	  mvwprintw(scrollWin,line,0,"%c  %s",(char)(((i-top)%scrollWinHeight) + 'a'),list->array[i]);
	  line++;
	}
   err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);

	inputWinLen = INPUT_WIN_LEN + BLANKING;

   inputWin = newpad(INPUT_WIN_HEIGHT,inputWinLen);
   if (inputWin == NULL)
   {
      endwin();
#ifdef XCURSES
      XCursesExit();
#endif
      fprintf(stderr,"Wcd: error creating input window.\n");
      return -1;
   }

   scrollok(inputWin, TRUE);

	if(perfect)
	  mvwprintw(inputWin,1,0,"Perfect ");
	else
	  mvwprintw(inputWin,1,0,"Wild ");
	wprintw(inputWin,"match for %d directories.",list->size);
	mvwprintw(inputWin,2,0,"Please choose one (<Enter> to abort): ");

	page = bottom / scrollWinHeight + 1 ;

	wmove (inputWin, 1, PAGEOFFSET);
	wprintw(inputWin,"w=up x=down ?=help  Page %d/%d      ",page,(list->size -1)/scrollWinHeight +1);
	wmove (inputWin, 2, OFFSET);
   err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);

	while ((c != 13 )&&(( c < 'a' ) || ( c > ('a'+scrollWinHeight-1) )))
	{

	  c = getch();

	  switch(c)
	  {
	  case 'w':
	  case KEY_UP:  /* Arrow Up */
     case KEY_PPAGE: /* Page Up */

		  if(bottom > (scrollWinHeight -1))
		  {
			bottom = bottom - scrollWinHeight ;
			top = top - scrollWinHeight ;
		  }

		  if (top<0) top = 0;

        wclear(scrollWin);

	     if (bottom < (scrollWinHeight -1) )
           line = scrollWinHeight - bottom - 1;
        else
           line = 0;

	     for (i=top;i<=bottom;i++)
		  {
	        mvwprintw(scrollWin,line,0,"%c  %s",(char)(((i-top)%scrollWinHeight) + 'a'),list->array[i]);
		     line++;
		  }
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);

		   page = bottom / scrollWinHeight + 1 ;

			wmove (inputWin, 1, PAGEOFFSET);
			wprintw(inputWin,"w=up x=down ?=help  Page %d/%d      ",page,(list->size -1)/scrollWinHeight +1);
			wmove (inputWin, 2, OFFSET);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);

			break;

		case 'x':
		case KEY_DOWN: /* Arrow down */
      case KEY_NPAGE: /* Page down */

		  if(bottom < (list->size -1 ))
		  {
			bottom = bottom + scrollWinHeight ;
			top = bottom - scrollWinHeight + 1;
		  }

        wclear(scrollWin);
		  line=0;
	     for (i=top;i<=bottom;i++)
		  {
	        mvwprintw(scrollWin,line,0,"%c  %s",(char)(((i-top)%scrollWinHeight) + 'a'),list->array[i]);
			  line++;
		  }
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);

			page = bottom / scrollWinHeight + 1 ;

			wmove (inputWin, 1, PAGEOFFSET);
			wprintw(inputWin,"w=up x=down ?=help  Page %d/%d      ",page,(list->size -1)/scrollWinHeight +1);
			wmove (inputWin, 2, OFFSET);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);
	  break;

      case ',':
      case KEY_LEFT:
          if (shift > 0)
             shift--;
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);
         break;
      case '.':
      case KEY_RIGHT:
         if (shift < len)
            shift++;
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);
         break;
       case '<':
       case '[':
          shift -=10;
          if (shift < 0)
             shift=0;
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);
         break;
      case ']':
      case '>':
         shift +=10;
         if (shift > len)
            shift=len;
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);
         break;
      case '?':

         wclear(scrollWin);
         if (scrollWinHeight < 17)
				mvwprintw(scrollWin,0,0,"Screenheight must be > 20 for help.");
         else
         {
				mvwprintw(scrollWin, 0,0,"<Up>            Page Up.");
				mvwprintw(scrollWin, 1,0,"w               Page Up.");
				mvwprintw(scrollWin, 2,0,"<Down>          Page Down.");
				mvwprintw(scrollWin, 3,0,"x               Page Down.");
				mvwprintw(scrollWin, 4,0,"<Left>          Scroll 1 left.");
				mvwprintw(scrollWin, 5,0,",               Scroll 1 left.");
				mvwprintw(scrollWin, 6,0,"<Right>         Scroll 1 right.");
				mvwprintw(scrollWin, 7,0,".               Scroll 1 right.");
				mvwprintw(scrollWin, 8,0,"<               Scroll 10 left.");
				mvwprintw(scrollWin, 9,0,"[               Scroll 10 left.");
				mvwprintw(scrollWin,10,0,">               Scroll 10 right.");
				mvwprintw(scrollWin,11,0,"]               Scroll 10 right.");
				mvwprintw(scrollWin,12,0,"^C              Abort.");
				mvwprintw(scrollWin,13,0,"<Esc><Esc>      Abort.");
				mvwprintw(scrollWin,14,0,"<Enter>         Abort.");
				mvwprintw(scrollWin,16,0,"Type w or x to quit help.");
         }
			err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);
	      break;
	  case 3:  /* Control-C */
	  case 13: /* Enter */
	  case KEY_ENTER:
	  case 27: /* Escape */
		c = 13;
		i = -1;
		break;
	  default:
		i=c+top-'a'+1;
	  break;
	  }
	}

   endwin();
#ifdef XCURSES
   XCursesExit();
#endif

	printf("\n");
   return i;
}

#else

 /* stdout version */

int display_list(nameset list,int perfect)
{
  int i;

	sort_list(list);

	for (i=0;i<list->size;i++)
		printf("%d  %s\n",i+1,list->array[i]);

	if(perfect)
	  printf("\nPerfect ");
	else
	  printf("\nWild ");
	printf("match for %d directories.\n",list->size);
	printf("Please choose one (<Enter> to abort): ");

	return wcd_get_int();
}
#endif
