/*
Copyright (C) 1997-2000 Erwin Waterlander

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "std_macro.h"
#include "structures.h"
#include "nameset.h"
#include "display.h"
#include "dosdir.h"
#include "WcdStack.h"
#include "text.h"
#include "wcd.h"
#include "stack.h"

#ifdef WCD_USECURSES
int maxLengthStack(WcdStack s)
{
	int i,len,maxlen = 0;

   for (i=0;i<s->size;i++)
   {
    if( (len=strlen(s->dir[i])) > maxlen)
       maxlen=len;
   }
	if (maxlen > 32)
      return(maxlen);
	else
	   return 32 ;     /* minimal width for help screen */
}
#endif

/********************************************************************
 *
 *                 stack_add
 *
 ********************************************************************/

int stack_add(WcdStack ws, char *dir)
{
	ws->lastadded++;

	if(ws->lastadded == ws->maxsize)
		ws->lastadded = 0;

	ws->current = ws->lastadded;

	/* printWcdStack("add ", ws, stdout); */

	/* free old dir string if present */
	if (ws->dir != NULL)
	if((ws->dir[ws->lastadded] != NULL) && (ws->size == ws->maxsize))
	free (ws->dir[ws->lastadded]);

	putTextelementAtWcdStack(textNew(dir), ws->lastadded, ws);

	return(0);
}
/********************************************************************
 *
 *                 stack_read
 *
 ********************************************************************/

int stack_read(WcdStack ws,char *stackfilename)
{

	FILE *infile;
	char tmp[DD_MAXDIR];


	/* open stack-file */
	if  ((ws->maxsize <= 0)||((infile = fopen(stackfilename,"r")) == NULL))
	{
		/* fprintf(stderr,"Wcd: Error opening stack-file\n"); */
		ws->lastadded = -1;
		ws->current = -1;
	}
	else
	{
		if(fscanf(infile,"%d %d",&ws->lastadded,&ws->current)==2)
		{

			while( !feof(infile)&&(ws->size < ws->maxsize) )
			{
			int len ;
			/* read a line */
			len = getline(tmp,DD_MAXDIR,infile);

			if (len > 0 )
				addTexttoWcdStack(textNew(tmp), ws);
			}
		}
		else
		  {
			fprintf(stderr,"Wcd: Error parsing stack\n");
			ws->lastadded = -1;
			ws->current = -1;
		  }

		fclose(infile);

		if (ws->lastadded >= ws->size)
		ws->lastadded = 0;
		if (ws->current >= ws->size)
		ws->current = 0;
	}

 /*	printWcdStack("READ ", ws, stdout); */
	return(0);
}
/********************************************************************
 *
 *                 stack_print
 *
 * Returns: On succes: the index number of the chosen stack entry
 *          On fail: -1
 *
 * This function has a modified version of display_list().
 * See file display.c for more comments.
 *
 ********************************************************************/

int stack_print(WcdStack ws)
{

	int  i, start, j;
#ifdef WCD_USECONIO
	char *buffer;
	struct text_info ti;
	int gtxt=0,c=0,extended;
#endif
#ifdef WCD_USECURSES
  int top, bottom, line, c=0;
  int scrollWinHeight, page, len, shift=0, err ;
  int scrollWinLen, inputWinLen ;
  WINDOW *scrollWin, *inputWin ;
#else
  int k;
#endif

/*	printWcdStack("XXX ", ws, stdout); */


	if((ws == NULL) || (ws->maxsize <= 0))
	return (-1);
	else
		if( ((ws->size) <= 0) || ((ws->size) > ws->maxsize) )
		return (-1);
		else
		{

#ifdef WCD_USECONIO
			gettextinfo(&ti);
			buffer = (char *) malloc(ti.screenwidth * ti.screenheight * 2);

			if (buffer!=NULL)
			{
			  gtxt = gettext(1,1,ti.screenwidth,ti.screenheight,buffer);
			}
#endif
			if (ws->size < ws->maxsize)
				 start = 0;
			 else
				  start = ws->lastadded + 1;

			 if (ws->lastadded >= ws->maxsize)
				  start = 0;

#ifdef WCD_USECONIO
			k = 0;
			for(i=0; i < (ws->size) ; i++)
			{
			j = (i + start)%(ws->size);
			printf("%c ",(char)( (k%26) + 'a'));
			k++;

			printf("%s",ws->dir[j]);
			if (j == ws->current)
				printf(" *");
			printf("\n");
			}

			printf("\nPlease choose one (<Enter> to abort): ");
			fflush(stdout);

			while ((c != 13 )&&(( c < 'a' ) || ( c > ('a'+ws->size -1) )))
			{
			  c = getch();

			  switch(c)
			  {
			  case 0:
				  extended = getch();
			  break;
			  case 3:  /* Control-C */
			  case 13: /* Enter */
			  case 27: /* Escape */
				c = 13;
				i = -1;
				break;
			  default:
				i=c -'a'+1;
			  break;
			  }
			}

#elif defined(WCD_USECURSES)

   initscr();
   keypad(stdscr, TRUE);
   intrflush(stdscr, FALSE);
   cbreak();
   noecho();
   nonl();
	scrollok(stdscr, TRUE);	/* enable scrolling */

   if (LINES < 4)
   {
      endwin();
#ifdef XCURSES
      XCursesExit();
#endif
      fprintf(stderr,"Wcd: error, screen height must be > 3.\n");
      return -1;
   }

   scrollWinHeight = LINES - INPUT_WIN_HEIGHT;
   if (scrollWinHeight > SCROLL_WIN_HEIGHT)
      scrollWinHeight = SCROLL_WIN_HEIGHT;

   len = maxLengthStack(ws);
	scrollWinLen = len + BLANKING;
   refresh();

   scrollWin = newpad(scrollWinHeight,scrollWinLen);
   if (scrollWin == NULL)
   {
      endwin();
#ifdef XCURSES
      XCursesExit();
#endif
      fprintf(stderr,"Wcd: error creating scroll window.\n");
      return -1;
   }

   scrollok(scrollWin, TRUE);

	bottom = ws->size -1;
	top = ws->size - scrollWinHeight;
	if (top < 0)
		top = 0;

	if (bottom < (scrollWinHeight -1) )
      line = scrollWinHeight - bottom - 1;
   else
      line = 0;

	for (i=top;i<=bottom;i++)
	{
	  j = (i + start)%(ws->size);
	  mvwprintw(scrollWin,line,0,"%c  %s",(char)(((i-top)%scrollWinHeight) + 'a'),ws->dir[j]);
	  if (j == ws->current)
	     wprintw(scrollWin," *");
	  line++;
   }
   err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);

	inputWinLen = INPUT_WIN_LEN + BLANKING;

   inputWin = newpad(INPUT_WIN_HEIGHT,inputWinLen);
   if (inputWin == NULL)
   {
      endwin();
#ifdef XCURSES
      XCursesExit();
#endif
      fprintf(stderr,"Wcd: error creating input window.\n");
      return -1;
   }

   scrollok(inputWin, TRUE);

	mvwprintw(inputWin,2,0,"Please choose one (<Enter> to abort): ");

	page = bottom / scrollWinHeight + 1 ;

	wmove (inputWin, 1, PAGEOFFSET);
	wprintw(inputWin,"w=up x=down ?=help  Page %d/%d      ",page,(ws->size -1)/scrollWinHeight +1);
	wmove (inputWin, 2, OFFSET);
   err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);

	while ((c != 13 )&&(( c < 'a' ) || ( c > ('a'+scrollWinHeight-1) )))
	{

	  c = getch();

	  switch(c)
	  {
	  case 'w':
	  case KEY_UP:  /* Arrow Up */
     case KEY_PPAGE: /* Page Up */

		  if(bottom > (scrollWinHeight -1))
		  {
			bottom = bottom - scrollWinHeight ;
			top = top - scrollWinHeight ;
		  }

		  if (top<0) top = 0;

        wclear(scrollWin);

	     if (bottom < (scrollWinHeight -1) )
           line = scrollWinHeight - bottom - 1;
        else
           line = 0;

	     for (i=top;i<=bottom;i++)
	     {
	        j = (i + start)%(ws->size);
	        mvwprintw(scrollWin,line,0,"%c  %s",(char)(((i-top)%scrollWinHeight) + 'a'),ws->dir[j]);
	        if (j == ws->current)
	           wprintw(scrollWin," *");
	        line++;
			}
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);

		   page = bottom / scrollWinHeight + 1 ;

			wmove (inputWin, 1, PAGEOFFSET);
			wprintw(inputWin,"w=up x=down ?=help  Page %d/%d      ",page,(ws->size -1)/scrollWinHeight +1);
			wmove (inputWin, 2, OFFSET);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);

			break;

		case 'x':
		case KEY_DOWN: /* Arrow down */
      case KEY_NPAGE: /* Page down */

		  if(bottom < (ws->size -1 ))
		  {
			bottom = bottom + scrollWinHeight ;
			top = bottom - scrollWinHeight + 1;
		  }

        wclear(scrollWin);
        line=0;
	     for (i=top;i<=bottom;i++)
	     {
	        j = (i + start)%(ws->size);
	        mvwprintw(scrollWin,line,0,"%c  %s",(char)(((i-top)%scrollWinHeight) + 'a'),ws->dir[j]);
	        if (j == ws->current)
	           wprintw(scrollWin," *");
	        line++;
		   }
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);

			page = bottom / scrollWinHeight + 1 ;

			wmove (inputWin, 1, PAGEOFFSET);
			wprintw(inputWin,"w=up x=down ?=help  Page %d/%d      ",page,(ws->size -1)/scrollWinHeight +1);
			wmove (inputWin, 2, OFFSET);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);
	  break;

      case ',':
      case KEY_LEFT:
          if (shift > 0)
             shift--;
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);
         break;
      case '.':
      case KEY_RIGHT:
         if (shift < len)
            shift++;
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);
         break;
      case '<':
      case '[':
         shift -=10;
         if (shift < 0)
            shift=0;
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);
        break;
      case '>':
      case ']':
         shift +=10;
         if (shift > len)
            shift=len;
         err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);
         err = wcd_prefresh(inputWin,0,0,scrollWinHeight,0,scrollWinHeight+INPUT_WIN_HEIGHT-1,COLS-1,inputWinLen-1,shift);
         break;
      case '?':

         wclear(scrollWin);
         if (scrollWinHeight < 17)
				mvwprintw(scrollWin,0,0,"Screenheight must be > 20 for help.");
         else
         {
				mvwprintw(scrollWin, 0,0,"<Up>            Page Up.");
				mvwprintw(scrollWin, 1,0,"w               Page Up.");
				mvwprintw(scrollWin, 2,0,"<Down>          Page Down.");
				mvwprintw(scrollWin, 3,0,"x               Page Down.");
				mvwprintw(scrollWin, 4,0,"<Left>          Scroll 1 left.");
				mvwprintw(scrollWin, 5,0,",               Scroll 1 left.");
				mvwprintw(scrollWin, 6,0,"<Right>         Scroll 1 right.");
				mvwprintw(scrollWin, 7,0,".               Scroll 1 right.");
				mvwprintw(scrollWin, 8,0,"<               Scroll 10 left.");
				mvwprintw(scrollWin, 9,0,"[               Scroll 10 left.");
				mvwprintw(scrollWin,10,0,">               Scroll 10 right.");
				mvwprintw(scrollWin,11,0,"]               Scroll 10 right.");
				mvwprintw(scrollWin,12,0,"^C              Abort.");
				mvwprintw(scrollWin,13,0,"<Esc><Esc>      Abort.");
				mvwprintw(scrollWin,14,0,"<Enter>         Abort.");
				mvwprintw(scrollWin,16,0,"Type w or x to quit help.");
				err = wcd_prefresh(scrollWin,0,0,0,0,scrollWinHeight-1,COLS-1,scrollWinLen-1,shift);
         }
	     break;
	  case 3:  /* Control-C */
	  case 13: /* Enter */
	  case KEY_ENTER:
	  case 27: /* Escape */
		c = 13;
		i = -1;
		break;
	  default:
		i=c+top-'a'+1;
	  break;
	  }
	}

   endwin();
#ifdef XCURSES
   XCursesExit();
#endif

	printf("\n");
#else
			k=1;
			for(i=0; i < (ws->size) ; i++)
			{
			j = (i + start)%(ws->size);
			printf("%2d ",k);
			k++;

			printf("%s",ws->dir[j]);
			if (j == ws->current)
				printf(" *");
			printf("\n");
			}

			printf("\nPlease choose one (<Enter> to abort): ");
			i = wcd_get_int();
#endif

#ifdef WCD_USECONIO
			printf("\n");

			if (gtxt ==1)
			  puttext(1,1,ti.screenwidth,ti.screenheight,buffer);

			if (buffer!=NULL)
				free(buffer);
#endif

			if (( i <=0)||(i > ws->size)) /* fail */
			{
			  return(-1);
			 }
			else    /* succes */
			{
			  i = ( i - 1 + start)%(ws->size);
			  ws->current = i;
			  return(i);
			}
		}
}
/********************************************************************
 *
 *                 stack_push
 *
 ********************************************************************/

char* stack_push(WcdStack ws, int push_ntimes)
{

	int  new_stack_nr;

	if(ws == NULL)
	return (NULL);
	else
		if( ((ws->size) <= 0) || ((ws->size) > ws->maxsize) )
		return (NULL);
		else
		{

			  push_ntimes = push_ntimes % ws->size;

			  new_stack_nr = ws->current - push_ntimes;

			  if(new_stack_nr < 0)
			  new_stack_nr = ws->size + new_stack_nr;

			  ws->current = new_stack_nr;

			  return(ws->dir[ws->current]);
		}
}
/********************************************************************
 *
 *                 stack_pop
 *
 *
 *
 ********************************************************************/

char* stack_pop(WcdStack ws, int pop_ntimes)
{

	int  new_stack_nr;


	if(ws == NULL)
	return (NULL);
	else
		if( ((ws->size) <= 0) || ((ws->size) > ws->maxsize) )
		return (NULL);
		else
		{
	         pop_ntimes = pop_ntimes % ws->size;

	         new_stack_nr = ws->current + pop_ntimes;

	         if(new_stack_nr > (ws->size -1))
	         new_stack_nr =  new_stack_nr - ws->size;

	         ws->current = new_stack_nr;
	         return(ws->dir[ws->current]);
		}
}
/********************************************************************
 *
 *                 stack_write
 *
 ********************************************************************/

int stack_write(WcdStack ws,char *stackfilename)
{
	FILE *outfile;
	int  i;

	if (ws->maxsize <= 0)
	   return(0);
	else
	if ( (outfile = fopen(stackfilename,"w")) == NULL)
	{
		fprintf(stderr,"Wcd: Error opening stack-file for write.\n");
		return(0);
	}
	else
	{
		fprintf(outfile,"%d %d\n",ws->lastadded,ws->current);
		for(i=0;((i<ws->size)&&(i<ws->maxsize));i++)
		{
	  /*		printf("writing line %d\n",i);  */
			fprintf(outfile,"%s\n",ws->dir[i]);
		}
		fclose(outfile);
	}
	return(0);
}

