/*
// Program:  Format
// Written By:  Brian E. Reifsnyder
// Version:  0.7
// Copyright:  1999 - 2001 under the terms of the GNU GPL, Version 2
*/

/*
/////////////////////////////////////////////////////////////////////////////
//  SPECIAL
/////////////////////////////////////////////////////////////////////////////
*/

#define NAME "Format"
#define VERSION "0.7"

int debug_prog=0;            /* set to 1 for debug mode */

#define MAIN
#define PD_OTHER             /* set in order to use pdiskio library */

/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <bios.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>

#include "bootcode.h"
#include "bpb.h"
#include "clstsize.h"
#include "format.h"

#include "pdiskio.h"

/*
/////////////////////////////////////////////////////////////////////////////
//  PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/

void ASCII_CD_Number(unsigned long number);
void Ask_User_To_Insert_Disk();
void Clear_Sector_Buffer();
void Clear_Huge_Sector_Buffer();
void Compute_Interleave_Factor();
void Compute_Sector_Skew();
void Confirm_Hard_Drive_Formatting();
void Critical_Error_Handler(int error_code);
void Display_Drive_Statistics();
void Display_Percentage_Formatted(unsigned long percentage);
void Format_Floppy_Cylinder(int cylinder,int head);
void Initialization();
void Read_Sectors(unsigned long sector_number,int number_of_sectors);
void Set_Floppy_Media_Type();
void Set_Hard_Drive_Media_Parameters();
void Setup_DDPT();

void Unconditional_Format();
void Unconditional_Floppy_Format();
void Unconditional_Hard_Disk_Format();
void Write_Sectors(unsigned long sector_number,int number_of_sectors);
void Write_System_Files();

extern void Convert_Logical_To_Physical(unsigned long sector
 ,unsigned long total_heads,unsigned long total_sectors);

unsigned int Sectors_Per_FAT(int fat_type, unsigned long number_of_sectors
 ,unsigned char sectors_per_cluster, unsigned int bytes_per_sector);

unsigned long Total_Logical_Sectors(unsigned char drive_letter[2]);

/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

void ASCII_CD_Number(unsigned long number)
{
  int comma_counter;
  int index;
  int right_index;
  int shift_counter;
  int shift_register;
  int shift_temp_register;
  int start_shift;


  index=0;

  do
    {
    ascii_cd_number[index]=0;

    index++;
    }while(index<14);

  ultoa(number,ascii_cd_number,10);

  /* Add Commas */
  index=13;
  right_index=13;
  start_shift=FALSE;

  do
    {
    if(ascii_cd_number[index]>0) start_shift=TRUE;

    if(start_shift==TRUE)
      {
      ascii_cd_number[right_index]=ascii_cd_number[index];
      ascii_cd_number[index]=0;
      right_index--;
      }

    index--;
    }while(index>=0);

  comma_counter=0;
  index=13;
  do
    {
    comma_counter++;

    if(ascii_cd_number[index]==0)
      {
      comma_counter=5;
      ascii_cd_number[index]=' ';
      }

    if(comma_counter==4)
      {
      shift_counter=index-1;
      shift_register=ascii_cd_number[index];
      ascii_cd_number[index]=',';
      do
	{
	shift_temp_register=ascii_cd_number[shift_counter];
	ascii_cd_number[shift_counter]=shift_register;
	shift_register=shift_temp_register;

	shift_counter--;
	}while(shift_counter>=0);

      comma_counter=0;
      }

    index--;
    }while(index>=0);

  ascii_cd_number[14]=0;
}

void Ask_User_To_Insert_Disk()
{
  printf(" Insert new diskette for drive %c:\n",param.drive_letter[0]);
  printf(" and press ENTER when ready...");

  /* Wait for a key */
  asm{
    mov ah,0x08
    int 0x21
    }

  printf("\n\n");
}

/* Clear Huge Sector Buffer */
void Clear_Huge_Sector_Buffer()
{
  long loop=0;

  do
    {
    huge_sector_buffer[loop]=0;

    loop++;
    }while(loop<(512*32));
}

void Compute_Interleave_Factor()
{
  int index;
  int starting_sector;


  low_level.interleave_factor=drive_specs[param.media_type].interleave_factor;

  index=1;
  starting_sector=0;
  low_level.interleave_index=0;
  do
    {
    low_level.interleave_map[low_level.interleave_index]=index;
    low_level.interleave_index
     =low_level.interleave_index+low_level.interleave_factor;
    if(low_level.interleave_index>=drive_specs[param.media_type].sectors_per_cylinder)
      {
      starting_sector=starting_sector+1;
      low_level.interleave_index=starting_sector;
      }

    index++;
    }while(index<=drive_specs[param.media_type].sectors_per_cylinder);

  if(debug_prog==TRUE)
    {
    printf("\n[DEBUG]  Initial Interleave Map:  \n          ");

    index=0;
    do
      {
      printf("%d ",low_level.interleave_map[index]);
      index++;
      }while(index<=(drive_specs[param.media_type].sectors_per_cylinder-1));
    printf("\n");
    }
}

void Compute_Sector_Skew()
{
  int carry;
  int index;
  int skew_counter;

  if(drive_specs[param.media_type].sector_skew==0) return;

  skew_counter=0;

  do
    {
    carry
     =low_level.interleave_map[(drive_specs[param.media_type].sectors_per_cylinder-1)];

    index=(drive_specs[param.media_type].sectors_per_cylinder-1);

    do
      {
      low_level.interleave_map[index]=low_level.interleave_map[(index-1)];

      index--;
      }while(index>0);

    low_level.interleave_map[0]=carry;

    skew_counter++;
    }while(skew_counter<drive_specs[param.media_type].sector_skew);

  if(debug_prog==TRUE)
    {
    printf("\n[DEBUG]  Interleave Map After Sector Skew:  \n          ");

    index=0;
    do
      {
      printf("%d ",low_level.interleave_map[index]);
      index++;
      }while(index<=(drive_specs[param.media_type].sectors_per_cylinder-1));
    printf("\n");
    }
}

void Confirm_Hard_Drive_Formatting()
{
  int character=0;

  printf("\n WARNING: ALL DATA ON NON-REMOVABLE DISK\n");
  printf(" DRIVE %c: WILL BE LOST!\n",param.drive_letter[0]);
  printf(" Proceed with Format (Y/N)?");

  /* Get keypress */
  asm{
    mov ah,0x07
    int 0x21
    mov BYTE PTR character,al
    }

  printf("\n");
  if( (character!='Y') && (character!='y') ) exit(10);
  printf("\n");
}

/* Convert Huge number into 4 LMB integer values */
void Convert_Huge_To_Integers(unsigned long number)
{
  unsigned long temp=number;
  unsigned long mask=0x000000ff;

  integer1=temp & mask;

  temp=number;
  mask=0x0000ff00;
  integer2=(temp & mask) >> 8;

  temp=number;
  mask=0x00ff0000;
  integer3=(temp & mask) >> 16;

  temp=number;
  mask=0xff000000;
  integer4=(temp & mask) >> 24;
}

/* Create File System */
void Create_File_System()
{
  char far *bpb;

  int index;
  int root_directory_first_sector;
  int root_directory_num_sectors;
  int space_fill;

  unsigned bpb_seg;
  unsigned bpb_off;

  unsigned char oem_id[8];

  unsigned long sector;

  void *bpb_p;

  /* Flush DOS disk buffers. */
  asm{
    mov ah,0x0d
    int 0x21
    }

  oem_id[0]='F';
  oem_id[1]='D';
  oem_id[2]='O';
  oem_id[3]='S';
  oem_id[4]='_';
  oem_id[5]='1';
  oem_id[6]='.';
  oem_id[7]='0';

  if(debug_prog==TRUE) printf("\n[DEBUG]  Creating File System\n");

  Clear_Sector_Buffer();

  /* Add Jump Instruction */
  sector_buffer[0]=0xeb;
  sector_buffer[1]=0x3c;
  sector_buffer[2]=0x90;

  /* Add OEM ID */
  index=3;
  do
    {
    sector_buffer[index]=oem_id[(index-3)];
    index++;
    }while(index<11);

  /* *** Add BPB */
  (void *)bpb_p=&drive_specs[param.media_type];

  bpb_seg=FP_SEG(bpb_p);
  bpb_off=FP_OFF(bpb_p);

  (char far *)bpb=(char far *)MK_FP(bpb_seg,bpb_off);

  index=11;
  do
    {
    sector_buffer[index]=bpb[(index-10)];

    index++;
    }while(index<=35);

  /* *** Add Extended BPB */
  if(param.media_type==HD) sector_buffer[36]=0x80;

  /* Add Signature Byte */
  sector_buffer[38]=0x29;

  /* Add Serial Number */
  randomize();
  sector_buffer[39]=rand();
  sector_buffer[40]=rand();
  sector_buffer[41]=rand();
  sector_buffer[42]=rand();

  drive_statistics.serial_number_low
   =(sector_buffer[39]) + (sector_buffer[40]<<8);

  drive_statistics.serial_number_high
   =(sector_buffer[41]) + (sector_buffer[42]<<8);

  /* Add Volume Label */
  if(param.v==TRUE)
    {
    index=43;
    space_fill=FALSE;
    do
      {
      if(param.volume_label[(index-43)]==0x00) space_fill=TRUE;

      if(space_fill==FALSE) sector_buffer[index]=param.volume_label[(index-43)];
      else sector_buffer[index]=' ';

      index++;
      }while(index<=53);
    }
  else
    {
    sector_buffer[43]='N';
    sector_buffer[44]='O';
    sector_buffer[45]=' ';
    sector_buffer[46]='N';
    sector_buffer[47]='A';
    sector_buffer[48]='M';
    sector_buffer[49]='E';
    sector_buffer[50]=' ';
    sector_buffer[51]=' ';
    sector_buffer[52]=' ';
    sector_buffer[53]=' ';
    }

  /* Add System ID */
  sector_buffer[54]='F';
  sector_buffer[55]='A';
  sector_buffer[56]='T';
  sector_buffer[57]='1';

  if(param.fat_type==FAT12) sector_buffer[58]='2';
  else sector_buffer[58]='6';

  sector_buffer[59]=' ';
  sector_buffer[60]=' ';
  sector_buffer[61]=' ';

  /* Add Boot Code */
  index=62;
  do
    {
    sector_buffer[index]=boot_code[(index-62)];

    index++;
    }while(index<=(130+62));

  /* Add Flag */
  sector_buffer[510]=0x55;
  sector_buffer[511]=0xaa;

  /* Write boot sector to the first sector of the disk */
  if(debug_prog==TRUE) printf("\n[DEBUG]  Writing Boot Sector->    0\n");
  Write_Sectors(0,1);

  /* Configure FAT Tables */
  Clear_Sector_Buffer();
  sector=1;
  do
    {
    if(debug_prog==TRUE) printf("[DEBUG]  Clearing FAT Sector->  %3d\n",sector);
    Write_Sectors(sector,1);

    sector++;
    }while(sector<(1+(2*drive_specs[param.media_type].sectors_per_fat)));

  sector_buffer[0]=drive_specs[param.media_type].media_descriptor;
  sector_buffer[1]=0xff;
  sector_buffer[2]=0xff;
  if(param.fat_type==FAT16) sector_buffer[3]=0xff;

  if(debug_prog==TRUE) printf("[DEBUG]  Write FAT Headers To Sector->    1\n");
  Write_Sectors(1,1);

  if(debug_prog==TRUE) printf("[DEBUG]  Write FAT Headers To Sector->    %3d\n",(1+drive_specs[param.media_type].sectors_per_fat));
  Write_Sectors( (1+drive_specs[param.media_type].sectors_per_fat),1 );

  /* Clear Root Directory Area */
  Clear_Sector_Buffer();
  sector=(drive_specs[param.media_type].sectors_per_fat*2)+1;
  root_directory_first_sector=sector;
  root_directory_num_sectors=drive_specs[param.media_type].root_directory_entries/16;
  do
    {
    if(debug_prog==TRUE) printf("[DEBUG]  Clearing Root Directory Sector->  %3d\n",sector);
    Write_Sectors(sector,1);
    sector++;
    }while(sector<(root_directory_first_sector+root_directory_num_sectors));

  /* Add Volume Label to Root Directory */
  if(param.v==TRUE)
    {
    Clear_Sector_Buffer();
    sector=(drive_specs[param.media_type].sectors_per_fat*2)+1;
    index=0;
    space_fill=FALSE;
    do
      {
      if(param.volume_label[(index)]==0x00) space_fill=TRUE;

      if(space_fill==FALSE) sector_buffer[index]=param.volume_label[(index)];
      else sector_buffer[index]=' ';

      index++;
      }while(index<=10);

    sector_buffer[11]=0x08;

    if(debug_prog==TRUE) printf("[DEBUG]  Writing Volume Label To Root Directory Sector->  %3d\n",sector);
    Write_Sectors(sector,1);
    }

  /* Flush DOS disk buffers so there is not any phantom information in */
  /* the directory.                                                    */
  asm{
    mov ah,0x0d
    int 0x21
    }
}


void Critical_Error_Handler(int error_code)
{
  if(error_code==2)                /* bad sector id mark              */
    {
    printf("\nBad sector ID\n");
    exit(1);
    }

  if(error_code==3)                /* write protect error             */
    {
    printf("\nWrite protect error\n");
    exit(1);
    }

  if(error_code==8)                /* DMA overrun                     */
    {
    printf("\nDMA overrun\n");
    exit(1);
    }

  if(error_code==9)                /* DMA error--64k boundary crossed */
    {
    printf("\nDMA error--64k boundary crossed\n");
    exit(1);
    }

  if(error_code==16)               /* CRC error                       */
    {
    printf("\nCRC error\n");
    exit(1);
    }

  if(error_code==32)               /* controller malfunction          */
    {
    printf("\nController malfunction\n");
    exit(1);
    }

  if(error_code==64)               /* seek failure                    */
    {
    printf("\nSeek failure\n");
    exit(1);
    }

  if(error_code==128)              /* time out                        */
    {
    printf("\nDevice timeout\n");
    exit(1);
    }

  if(error_code==255)
    {
    printf("\nInvalid drive specification\n");
    exit(1);
    }
}


void Display_Drive_Statistics()
{
  drive_statistics.allocation_units_available_on_disk
   =drive_statistics.bytes_available_on_disk
   /drive_statistics.bytes_in_each_allocation_unit;


  /* The 2 lines below will eventually need to be re-located. */
  drive_statistics.bytes_total_disk_space
   =drive_statistics.allocation_units_available_on_disk
   *drive_statistics.bytes_in_each_allocation_unit;
  drive_statistics.bytes_available_on_disk
   =drive_statistics.bytes_total_disk_space;

  ASCII_CD_Number(drive_statistics.bytes_total_disk_space);
  printf("\n%13s bytes total disk space\n",ascii_cd_number);

  ASCII_CD_Number(drive_statistics.bytes_available_on_disk);
  printf("%13s bytes available on disk\n",ascii_cd_number);

  printf("\n");

  ASCII_CD_Number(drive_statistics.bytes_in_each_allocation_unit);
  printf("%13s bytes in each allocation unit.\n",ascii_cd_number);

  ASCII_CD_Number(drive_statistics.allocation_units_available_on_disk);
  printf("%13s allocation units available on disk.\n",ascii_cd_number);

  printf("\n");
  printf(" Volume Serial Number is %04X-%04X\n"
   ,drive_statistics.serial_number_high,drive_statistics.serial_number_low);
}

void Display_Invalid_Combination()
{
  printf("\n Invalid combination of options...please consult documentation.\n");
  printf(" Operation Terminated.\n");
  exit(4);
}

/* Help Routine */
void Display_Help_Screen()
{
  printf("\n%6s Version %s\n",NAME,VERSION);
  printf("Written By:  Brian E. Reifsnyder\n");
  printf("Copyright 1999 - 2001 under the terms of the GNU GPL, Version 2.\n\n");
  printf("Syntax:\n\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/F:size] [/B | /S]\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/T:tracks /N:sectors] [/B | /S]\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/1] [/4] [/B | /S]\n");
  printf("FORMAT drive: [/Q] [/U] [/1] [/4] [/8] [/B | /S]\n\n");
  printf(" /V:label   Specifies a volume label for the disk.\n");
  printf(" /Q         Quick formats the disk.  The disk can be UNFORMATed.\n");
  printf(" /U         Unconditionally formats the disk.  The disk cannot be UNFORMATted\n");
  printf(" /F:size    Specifies the size of the floppy disk to format.  Valid sizes are:\n");
  printf("              160, 180, 320, 360, 720, 1200, 1440, or 2880.\n");
  printf(" /B         Kept for compatibility, formally reserved space for the boot files.\n");
  printf(" /S         Copies the operating system files to make the disk bootable.\n");
  printf(" /T:tracks  Specifies the number of tracks on a floppy disk.\n");
  printf(" /N:sectors Specifies the number of sectors on a floppy disk.\n");
  printf(" /1         Formats a single side of a floppy disk.\n");
  printf(" /4         Formats a 360K floppy disk in a 1.2 MB floppy drive.\n");
  printf(" /8         Formats a 5.25\" disk with 8 sectors per track.\n");
}

void Display_Percentage_Formatted(unsigned long percentage)
{
  printf("%3d percent completed.\n",percentage);

  if(debug_prog!=TRUE)
    {
    /* Re-position cursor back to the beginning of the line */
    asm{
      /* Get current video display mode */
      mov ah,0x0f
      int 0x10

      /* Get cursor position */
      mov ah,0x03
      int 0x10

      /* Set cursor position to beginning of line */
      mov ah,0x02
      sub dh,1
      int 0x10
      }
    }
}

void Format_Floppy_Cylinder(int cylinder,int head)
{
  int drive_number;

  unsigned int result;

  /* Set Up Track Address Fields */
  TAF track_address_fields[36];
  void *track_address_fields_p=track_address_fields;

  unsigned track_address_fields_seg=FP_SEG(track_address_fields_p);
  unsigned track_address_fields_off=FP_OFF(track_address_fields_p);

  int index=0;
  do
    {
    track_address_fields[index].cylinder=cylinder;
    track_address_fields[index].head=head;
    track_address_fields[index].sector=low_level.interleave_map[index];
    track_address_fields[index].size_code=0x02;

    index++;
    }while(index<drive_specs[param.media_type].sectors_per_cylinder);

  drive_number=param.drive_number;

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Formatting:  Cylinder->  %2d    Head->  %2d\n",cylinder,head);
    }

  /* Format the Track */
  asm{
    mov ah,0x05
    mov ch,BYTE PTR cylinder
    mov dh,BYTE PTR head
    mov dl,BYTE PTR drive_number
    mov es,WORD PTR track_address_fields_seg
    mov bx,WORD PTR track_address_fields_off
    int 0x13
    mov BYTE PTR result,ah
    }

  if(result!=0) Critical_Error_Handler(result);

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Result From Interrupt 0x13, Service 0x05->  %X\n",result);
    }
}

void Initialization()
{
  flags.version=SIX;
  debug.write=TRUE;

  param.drive_letter[0]=NULL;
  param.volume_label[0]=NULL;

  param.drive_type=NULL;
  param.drive_number=NULL;
  param.fat_type=NULL;
  param.media_type=UNKNOWN;

  param.force_yes=FALSE;

  param.v=FALSE;
  param.q=FALSE;
  param.u=FALSE;
  param.f=FALSE;
  param.b=FALSE;
  param.s=FALSE;
  param.t=FALSE;
  param.n=FALSE;
  param.one=FALSE;
  param.four=FALSE;
  param.eight=FALSE;

  param.size=UNKNOWN;
  param.cylinders=0;
  param.sectors=0;

  Setup_DDPT();
}

/* Read Sector(s) from drive */
void Read_Sectors(unsigned long sector_number,int number_of_sectors)
{
  unsigned int result;

  if(param.media_type!=HD)
    {
    Convert_Logical_To_Physical(sector_number
     ,drive_specs[param.media_type].number_of_heads
     ,drive_specs[param.media_type].sectors_per_cylinder);

    /* Read the sector(s). */
    result=biosdisk(2, param.drive_number, translated_head
     ,translated_cylinder, translated_sector, number_of_sectors, sector_buffer);

    if(debug_prog==TRUE)
      {
      printf("[DEBUG]  Result From Write_Physical_Sectors()->  %X\n",result);
      }
    }
  else
    {
    int error_code=Read_Logical_Sectors(param.drive_letter,sector_number,
     number_of_sectors);

    if(error_code==255)
      {
      printf("\n Invalid drive specification\n");
      exit(1);
      }
    }
}

/* Save the old file system for possible recovery with unformat */
void Save_File_System()
{
  int bad_boot_sector=TRUE;
  int end_of_root_directory_flag;

  long loop=512;

  unsigned char mirror_map[5120];

  unsigned long pointer=0;

  unsigned long offset_from_end=5;

  unsigned long destination_sector;
  unsigned long source_sector;

  unsigned long mirror_beginning;
  unsigned long mirror_map_beginning;
  unsigned long mirror_map_size;
  unsigned long mirror_size;
  unsigned long number_of_bytes_in_mirror_map;
  unsigned long number_of_logical_sectors_on_drive;
  unsigned long number_of_root_directory_entries;

  unsigned long beginning_of_fat;
  unsigned long beginning_of_root_directory;

  unsigned long sectors_per_fat;

  unsigned long end_of_root_directory;

  unsigned long temp_buffer_1=0;
  unsigned long temp_buffer_2=0;
  unsigned long temp_buffer_3=0;
  unsigned long temp_buffer_4=0;

  unsigned int sector_flag[] = {
  'A','M','S','E','S','L','I','F','V','A','S',
  'R','O','R','I','M','E','S','A','E','P'};

  unsigned int mirror_map_header[] = {
  ':','\\','M','I','R','R','O','R','.','F','I','L'};

  if(param.drive_type==HARD) offset_from_end=20;

  /* Get the boot sector, compute the FAT size, compute the root dir size,*/
  /* and get the end of the logical drive. */
  Read_Sectors(0,1);

  if(  (sector_buffer[510]==0x55) && (sector_buffer[511]==0xaa) )
    {
    temp_buffer_1=sector_buffer[17];
    temp_buffer_2=sector_buffer[18];
    number_of_root_directory_entries=temp_buffer_1|(temp_buffer_2<<8);

    temp_buffer_1=sector_buffer[19];
    temp_buffer_2=sector_buffer[20];
    number_of_logical_sectors_on_drive=temp_buffer_1|(temp_buffer_2<<8);

    if(number_of_logical_sectors_on_drive==0)
      {
      temp_buffer_1=sector_buffer[32];
      temp_buffer_2=sector_buffer[33];
      temp_buffer_3=sector_buffer[34];
      temp_buffer_4=sector_buffer[35];

      number_of_logical_sectors_on_drive=temp_buffer_1|(temp_buffer_2<<8)
       |(temp_buffer_3<<16)|(temp_buffer_4<<24);
      }

    temp_buffer_1=sector_buffer[22];
    temp_buffer_2=sector_buffer[23];
    sectors_per_fat=temp_buffer_1|(temp_buffer_2<<8);

    bad_boot_sector=FALSE;
    }

  /* If the boot sector is not any good, don't save the file system. */
  if(bad_boot_sector==FALSE)
    {
    printf(" Saving UNFORMAT information\n");

    /* Compute the beginning sector of the mirror map and the size of */
    /* the mirror image.     */
    mirror_size=1+sectors_per_fat+(number_of_root_directory_entries/16);

    mirror_map_size=(mirror_size/64)+1;

    mirror_beginning=(number_of_logical_sectors_on_drive-mirror_size)-offset_from_end;
    mirror_map_beginning=mirror_beginning-mirror_map_size;

    /* Compute the locations of the first FAT and the root directory */
    beginning_of_fat=sectors_per_fat+1;

    beginning_of_root_directory=(sectors_per_fat*2)+1;
    end_of_root_directory=beginning_of_root_directory+(number_of_root_directory_entries/16)-1;

    /* Write the mirror map pointer to the last sectors of the logical drive. */
    Clear_Sector_Buffer();

    Convert_Huge_To_Integers(mirror_map_beginning);

    sector_buffer[0]=integer1;
    sector_buffer[1]=integer2;
    sector_buffer[2]=integer3;
    sector_buffer[3]=integer4;

    pointer=4;

    do                                           /* Add pointer sector flag */
      {
      sector_buffer[pointer]=sector_flag[pointer-4];
      pointer++;
      }while(pointer<=24);

    if(debug_prog==TRUE)
      {
      printf("[DEBUG]  Writing mirror map pointer to sector->  %ld\n",
       (number_of_logical_sectors_on_drive-offset_from_end));
      printf("[DEBUG]  Writing mirror map beginning at sector->  %ld\n",mirror_map_beginning);
      }

    Write_Sectors( (number_of_logical_sectors_on_drive-offset_from_end),1);

    /* Create the mirror map and copy the file system to the mirror.  */
    Clear_Sector_Buffer();

    pointer=0;

    do                                           /* Clear mirror_map buffer */
      {
      mirror_map[pointer]=0;
      pointer++;
      }while(pointer<=5119);

    mirror_map[0]=param.drive_letter[0];

    pointer=1;

    do                                           /* Add mirror map header */
      {
      mirror_map[pointer]=mirror_map_header[pointer-1];
      pointer++;
      }while(pointer<=12);

					       /* Main mirror map creation */
					       /* and copying loop.        */
    pointer=84;
    source_sector=0;
    destination_sector=mirror_beginning;

    end_of_root_directory_flag=FALSE;
    number_of_bytes_in_mirror_map=0;

    do
      {
      if( (source_sector>0) && (source_sector<beginning_of_fat) )
       source_sector=beginning_of_fat;

      /* Copy mirror image one sector at a time */
      Read_Sectors(source_sector,1);

      Write_Sectors(destination_sector,1);

      /* Enter mapping information into mirror map buffer */

      Convert_Huge_To_Integers(source_sector);

      mirror_map[pointer+0]=integer1;
      mirror_map[pointer+1]=integer2;
      mirror_map[pointer+2]=integer3;
      mirror_map[pointer+3]=integer4;

      Convert_Huge_To_Integers(destination_sector);

      mirror_map[pointer+4]=integer1;
      mirror_map[pointer+5]=integer2;
      mirror_map[pointer+6]=integer3;
      mirror_map[pointer+7]=integer4;

      source_sector++;
      destination_sector++;
      pointer=pointer+8;
      number_of_bytes_in_mirror_map=pointer;

      if(source_sector>=end_of_root_directory) end_of_root_directory_flag=TRUE;

      }while(end_of_root_directory_flag==FALSE);

    /* Write trailer in mirror map */

    mirror_map[pointer+0]=0;
    mirror_map[pointer+1]=0;
    mirror_map[pointer+2]=0;
    mirror_map[pointer+3]=0;

    /* Write the mirror map onto the disk.   */

    pointer=0;
    destination_sector=mirror_map_beginning;

    do
      {
      loop=0;

      do                                         /* Load the sector buffer */
	{
	sector_buffer[loop]=mirror_map[pointer+loop];

	loop++;
	}while(loop<512);

      Write_Sectors(destination_sector,1);       /* Write the mirror map   */
						 /* sector.                */
      destination_sector++;
      pointer=pointer+512;
      }while(pointer < number_of_bytes_in_mirror_map);

    }
  else
    {
    printf(" Drive appears unformatted, UNFORMAT information not saved.\n");
    }
}

unsigned int Sectors_Per_FAT(int fat_type, unsigned long number_of_sectors
 ,unsigned char sectors_per_cluster, unsigned int bytes_per_sector)
{
  int loop;

  unsigned long byte_counter;
  unsigned long cluster_counter;
  unsigned long fat_entry_counter;
  unsigned long fat_sector_counter;
  unsigned long last_cluster_counter;
  unsigned long number_data_sectors;
  unsigned long number_fat_bytes;
  unsigned long number_fat_entries;
  unsigned long number_fat_sectors;
  unsigned long number_of_clusters;
  unsigned long sector_counter;

  /* Determine the number total of clusters on the logical drive. */
  number_of_clusters=(number_of_sectors/sectors_per_cluster);

  /* Calculate the starting number of FAT sectors. */
  if(fat_type==FAT12)
    {
    number_fat_bytes=3+((number_of_clusters*2)/3);
    }

  if(fat_type==FAT16)
    {
    number_fat_bytes=2+(number_of_clusters*2);
    }

  number_fat_sectors=(number_fat_bytes/bytes_per_sector) + 2;

  /* Find the best fit for the file system.                                */

  do
    {
    number_fat_sectors--;

    number_data_sectors = number_of_sectors
     - (number_fat_sectors * 2) - 1;

    number_of_clusters = number_data_sectors/sectors_per_cluster;

    /* Calculate the number of entries given the fat table size.          */

    if(fat_type==FAT12)
      {
      number_fat_entries=(((number_fat_sectors*bytes_per_sector)*2)/3)-3;
      }

    if(fat_type==FAT16)
      {
      number_fat_entries=((number_fat_sectors*bytes_per_sector)/2)-2;
      }

    }while( number_fat_entries > number_of_clusters);

  number_fat_sectors++;

  return( (unsigned int) number_fat_sectors);
}

void Set_Floppy_Media_Type()
{
  int drive_number=param.drive_number;
  int number_of_cylinders;
  int sectors_per_cylinder;

  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Current Disk Drive Parameter Table Values:\n");
    printf("[DEBUG]       Step Rate:                 %d\n",ddpt->step_rate);
    printf("[DEBUG]       Head Unload Time:          %d\n",ddpt->head_unload_time);
    printf("[DEBUG]       DMA Flag:                  %d\n",ddpt->dma_flag);
    printf("[DEBUG]       Post R. T. of Disk Motor:  %d\n",ddpt->post_rt_of_disk_motor);
    printf("[DEBUG]       Sector Size:               %d\n",ddpt->sector_size);
    printf("[DEBUG]       Sectors Per Cylinder:      %d\n",ddpt->sectors_per_cylinder);
    printf("[DEBUG]       Intersector Gap Length:    %d\n",ddpt->gap3_length_rw);
    printf("[DEBUG]       Data Length:               %d\n",ddpt->dtl);
    printf("[DEBUG]       Intersect. Gap Len. Xmat:  %d\n",ddpt->gap3_length_xmat);
    printf("[DEBUG]       Fill Character:            %d\n",ddpt->fill_char_xmat);
    printf("[DEBUG]       Head Settle Time:          %d\n",ddpt->head_settle_time);
    printf("[DEBUG]       Motor start=up time:       %d\n",ddpt->run_up_time);
    }

  param.fat_type=FAT12;

  if(param.f==TRUE)
    {
    /* Standard Format Types */
    if(param.size==160) param.media_type=FD160;
    if(param.size==180) param.media_type=FD180;
    if(param.size==320) param.media_type=FD320;
    if(param.size==360) param.media_type=FD360;
    if(param.size==720) param.media_type=FD720;
    if(param.size==1200) param.media_type=FD1200;
    if(param.size==1440) param.media_type=FD1440;
    if(param.size==2880) param.media_type=FD2880;
    /* Non-Standard Format Types */
    if(param.size==400) param.media_type=FD400;
    if(param.size==800) param.media_type=FD800;
    if(param.size==1680) param.media_type=FD1680;
    if(param.size==3360) param.media_type=FD3360;
    if(param.size==1494) param.media_type=FD1494;
    if(param.size==1743) param.media_type=FD1743;
    if(param.size==3486) param.media_type=FD3486;
    if(param.size==1700) param.media_type=FD1700;
    }

  if(param.t==TRUE)
    {
    int index=0;
    do
      {
      if( (param.cylinders==drive_specs[index].cylinders)
       && (param.sectors==drive_specs[index].sectors_per_cylinder) )
	{
	param.media_type=index;
	index=20;
	}

      index++;
      }while(index<8);
    }

  drive_number=param.drive_number;

  if(param.media_type==UNKNOWN)
    {
    /* Attempt to automatically detect the media type */
    int drive_number=param.drive_number;
    int drive_type=0;

    asm{
      mov ah,0x08
      mov dl,BYTE PTR drive_number
      int 0x13

      mov BYTE PTR drive_type,bl
      }

    if(drive_type==0x01)
      {
      param.size=360;
      param.media_type=FD360;
      }
    if(drive_type==0x02)
      {
      param.size=1200;
      param.media_type=FD1200;
      }
    if(drive_type==0x03)
      {
      param.size=720;
      param.media_type=FD720;
      }
    if(drive_type==0x04)
      {
      param.size=1440;
      param.media_type=FD1440;
      }
    if(drive_type==0x05)     /* Originally for floppy tape drives */
      {
      param.size=2880;
      param.media_type=FD2880;
      }
    if(drive_type==0x06)
      {
      param.size=2880;
      param.media_type=FD2880;
      }
    }

  number_of_cylinders=drive_specs[param.media_type].cylinders;
  sectors_per_cylinder=drive_specs[param.media_type].sectors_per_cylinder;

  asm{
    mov ah,0x18
    mov ch,BYTE PTR number_of_cylinders
    mov cl,BYTE PTR sectors_per_cylinder
    mov dl,BYTE PTR drive_number
    int 0x13
    jnc int_supported
    }

  ddpt->sectors_per_cylinder=sectors_per_cylinder;

  int_supported:

  if(param.media_type>HD)
    {
    /* Adjust for non-standard formats. */

    }

  drive_statistics.bytes_total_disk_space
   =((unsigned long)drive_specs[param.media_type].bytes_per_sector
   *(unsigned long)drive_specs[param.media_type].total_sectors)
   -((1+(2*(unsigned long)drive_specs[param.media_type].sectors_per_fat)
   +((unsigned long)drive_specs[param.media_type].root_directory_entries/16))
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector);

  drive_statistics.bytes_available_on_disk
   =drive_statistics.bytes_total_disk_space;

  drive_statistics.bytes_in_each_allocation_unit
   =(unsigned long)drive_specs[param.media_type].sectors_per_cluster
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector;

  Compute_Interleave_Factor();

  if(debug_prog==TRUE)
    {
    printf("\n[DEBUG]  Configured Disk Drive Parameter Table Values:\n");
    printf("[DEBUG]       Step Rate:                 %d\n",ddpt->step_rate);
    printf("[DEBUG]       Head Unload Time:          %d\n",ddpt->head_unload_time);
    printf("[DEBUG]       DMA Flag:                  %d\n",ddpt->dma_flag);
    printf("[DEBUG]       Post R. T. of Disk Motor:  %d\n",ddpt->post_rt_of_disk_motor);
    printf("[DEBUG]       Sector Size:               %d\n",ddpt->sector_size);
    printf("[DEBUG]       Sectors Per Cylinder:      %d\n",ddpt->sectors_per_cylinder);
    printf("[DEBUG]       Intersector Gap Length:    %d\n",ddpt->gap3_length_rw);
    printf("[DEBUG]       Data Length:               %d\n",ddpt->dtl);
    printf("[DEBUG]       Intersect. Gap Len. Xmat:  %d\n",ddpt->gap3_length_xmat);
    printf("[DEBUG]       Fill Character:            %d\n",ddpt->fill_char_xmat);
    printf("[DEBUG]       Head Settle Time:          %d\n",ddpt->head_settle_time);
    printf("[DEBUG]       Motor start=up time:       %d\n",ddpt->run_up_time);
    }
}

void Set_Hard_Drive_Media_Parameters()
{
  int index=0;
  int sub_index=0;

  int drive_number=UNUSED;
  int partition_number=UNUSED;

  unsigned long number_of_sectors;

  param.media_type=HD;
  param.physical_drive_number=UNUSED;

  Read_Partition_Tables();

  /* Search drive_lettering_buffer[8] [27] for the drive letter */
  do
    {
    sub_index=0;
    do
      {
      if(param.drive_letter[0]==drive_lettering_buffer[index][sub_index])
	{
	drive_number=index;
	param.physical_drive_number=index+0x80;
	partition_number=sub_index;
	}

      sub_index++;
      }while(sub_index<27);

    index++;
    }while(index<8);

  if(drive_number==UNUSED)
    {
    printf("\n Invalid drive specification\n");
    exit(1);
    }

  if(partition_number<4)
    {
    /* Drive is in the Primary Partition Table */

    number_of_sectors
     =part_table[drive_number].pri_part_num_sect[partition_number];

    index=0;
    do
      {
      if( (number_of_sectors>=cluster_size_table[index].min_num_sectors)
       && (number_of_sectors<cluster_size_table[index].max_num_sectors) )
	drive_specs[HD].sectors_per_cluster=cluster_size_table[index].sectors_per_cluster;

      index++;
      }while(index<7);

    if(part_table[drive_number].pri_part_num_type[partition_number]==1)
      {
      param.fat_type=FAT12;

      drive_specs[HD].sectors_per_fat
       =Sectors_Per_FAT(FAT12, number_of_sectors
       ,drive_specs[HD].sectors_per_cluster, drive_specs[HD].bytes_per_sector);
      }

    if(part_table[drive_number].pri_part_num_type[partition_number]==4)
      {
      param.fat_type=FAT16;

      drive_specs[HD].sectors_per_fat
       =Sectors_Per_FAT(FAT16, number_of_sectors
       ,drive_specs[HD].sectors_per_cluster, drive_specs[HD].bytes_per_sector);
      }
    if(part_table[drive_number].pri_part_num_type[partition_number]==6)
      {
      param.fat_type=FAT16;

      drive_specs[HD].sectors_per_fat
       =Sectors_Per_FAT(FAT16, number_of_sectors
       ,drive_specs[HD].sectors_per_cluster, drive_specs[HD].bytes_per_sector);
      }

    drive_specs[HD].cylinders=part_table[drive_number].total_cyl;
    drive_specs[HD].sectors_per_cylinder=part_table[drive_number].total_sect;
    drive_specs[HD].number_of_heads=part_table[drive_number].total_head+1;

    drive_specs[HD].hidden_sectors_low=part_table[drive_number].total_sect;
    drive_specs[HD].hidden_sectors_high=0;

    if(number_of_sectors<=0xffff)
      {
      drive_specs[HD].total_sectors=number_of_sectors;
      }
    else
      {
      drive_specs[HD].large_sector_count_low=number_of_sectors&0x0000ffff;
      drive_specs[HD].large_sector_count_high=(number_of_sectors&0xffff0000)>>16;
      }

    param.size=part_table[drive_number].pri_part_size_in_MB[partition_number];

    param.partition_number=partition_number;
    }
  else
    {
    /* Drive is in the Extended Partition Table */
    partition_number=partition_number-4;

    number_of_sectors
     =part_table[drive_number].log_drive_num_sect[partition_number];

    index=0;
    do
      {
      if( (number_of_sectors>=cluster_size_table[index].min_num_sectors)
       && (number_of_sectors<cluster_size_table[index].max_num_sectors) )
	drive_specs[HD].sectors_per_cluster=cluster_size_table[index].sectors_per_cluster;

      index++;
      }while(index<9);

    if(part_table[drive_number].log_drive_num_type[partition_number]==1)
      {
      param.fat_type=FAT12;

      drive_specs[HD].sectors_per_fat
       =Sectors_Per_FAT(FAT12, number_of_sectors
       ,drive_specs[HD].sectors_per_cluster, drive_specs[HD].bytes_per_sector);
      }
    if(part_table[drive_number].log_drive_num_type[partition_number]==4)
      {
      param.fat_type=FAT16;

      drive_specs[HD].sectors_per_fat
       =Sectors_Per_FAT(FAT16, number_of_sectors
       ,drive_specs[HD].sectors_per_cluster, drive_specs[HD].bytes_per_sector);
      }
    if(part_table[drive_number].log_drive_num_type[partition_number]==6)
      {
      param.fat_type=FAT16;

      drive_specs[HD].sectors_per_fat
       =Sectors_Per_FAT(FAT16, number_of_sectors
       ,drive_specs[HD].sectors_per_cluster, drive_specs[HD].bytes_per_sector);
      }

    drive_specs[HD].cylinders=part_table[drive_number].total_cyl;
    drive_specs[HD].sectors_per_cylinder=part_table[drive_number].total_sect;
    drive_specs[HD].number_of_heads=part_table[drive_number].total_head+1;

    drive_specs[HD].hidden_sectors_low=part_table[drive_number].total_sect;
    drive_specs[HD].hidden_sectors_high=0;

    if(number_of_sectors<=0xffff)
      {
      drive_specs[HD].total_sectors=number_of_sectors;
      }
    else
      {
      drive_specs[HD].large_sector_count_low=number_of_sectors&0x0000ffff;
      drive_specs[HD].large_sector_count_high=(number_of_sectors&0xffff0000)>>16;
      }

    param.size=part_table[drive_number].log_drive_size_in_MB[partition_number];

    param.partition_number=partition_number+4;
    }

  drive_statistics.bytes_total_disk_space
   =((unsigned long)drive_specs[param.media_type].bytes_per_sector*number_of_sectors)
   -((1+(2*(unsigned long)drive_specs[param.media_type].sectors_per_fat)
   +((unsigned long)drive_specs[param.media_type].root_directory_entries/16))
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector);
  drive_statistics.bytes_available_on_disk
   =drive_statistics.bytes_total_disk_space;

  drive_statistics.bytes_in_each_allocation_unit
   =(unsigned long)drive_specs[param.media_type].sectors_per_cluster
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector;
}

void Setup_DDPT()
{
  unsigned ddpt_seg;
  unsigned ddpt_off;

  /* Get the location of the DDPT */
    asm{
    mov ah,0x35
    mov al,0x1e
    int 0x21

    mov WORD PTR ddpt_seg,es
    mov WORD PTR ddpt_off,bx
    }

  (void far *)ddpt = MK_FP(ddpt_seg,ddpt_off);
}

/* Return Total Logical Sectors Function */
unsigned long Total_Logical_Sectors(unsigned char drive_letter[2])
{
  /* Returns the total number of logical sectors, if successful and */
  /* 0 if the drive is not found.                                   */

  int index=0;
  int part_num=255;
  int physical_drive=255;
  int sub_index=0;

  unsigned long total_logical_sectors;

  /* Ensure that drive_letter[0] is upper case. */
  if(drive_letter[0]>='a') drive_letter[0]=drive_letter[0]-32;

  /* Determine where the logical drive is located */
  Determine_Drive_Letters();

  do
    {
    sub_index=0;
    do
      {
      if(drive_letter[0]==drive_lettering_buffer[index][sub_index])
	{
	physical_drive=index;
	part_num=sub_index;
	}

      sub_index++;
      }while(sub_index<27);

    index++;
    }while(index<8);

  if(physical_drive==255) return(0);

  if(part_num<4) total_logical_sectors
   =part_table[physical_drive].pri_part_num_sect[part_num];
  else total_logical_sectors
   =part_table[physical_drive].log_drive_num_sect[(part_num-4)];

  return(total_logical_sectors);
}

/* Unconditionally Format the Drive */
void Unconditional_Format()
{
  if(param.drive_type==FLOPPY) Unconditional_Floppy_Format();
  else Unconditional_Hard_Disk_Format();
}

void Unconditional_Floppy_Format()
{
  int drive_number=param.drive_number;
  int index=0;

  unsigned long percentage;

  /* Reset the floppy disk controller */
  asm{
    mov ah,0x00
    mov dl,BYTE PTR drive_number
    int 0x13
    }

  do
    {
    if(drive_specs[param.media_type].number_of_heads==2)
     Format_Floppy_Cylinder(index,0);
    Format_Floppy_Cylinder(index,1);

    percentage
     =((100*index)/(drive_specs[param.media_type].cylinders));
    Display_Percentage_Formatted(percentage);

    Compute_Sector_Skew();

    index++;
    }while(index<=drive_specs[param.media_type].cylinders);
}

void Unconditional_Hard_Disk_Format()
{
  int error_code;
  int number_of_sectors=32;

  int percentage_refresh_rate=10;
  int percentage_refresh_counter=10;

  long index=0;

  unsigned long sector_number=0;
  unsigned long max_logical_sector=Total_Logical_Sectors(param.drive_letter);
  unsigned long percentage;

  /* Clear huge_sector_buffer */
  do
    {
    huge_sector_buffer[index]=0;

    index++;
    }while(index<(512*32));

  /* Clear sectors (maximum of 32 at a time) */
  do
    {
    error_code=Write_Logical_Sectors(param.drive_letter,sector_number,
     number_of_sectors);

    if(error_code>0)
      {
      if(error_code!=4) Critical_Error_Handler(error_code);
      else
	{
	/* Scan and Record Bad Sectors */

	}
      }

    if(percentage_refresh_counter==0)
      {
      percentage=(100*sector_number)/max_logical_sector;
      Display_Percentage_Formatted(percentage);
      }
    if(percentage_refresh_counter==percentage_refresh_rate)
     percentage_refresh_counter=-1;
    percentage_refresh_counter++;

    sector_number=sector_number+32;
    }while(sector_number<max_logical_sector);
}

/* Write System Files */
void Write_System_Files()
{
  char sys[9] = {'s','y','s',' ','x',':',13,0,0};

  sys[4]=param.drive_letter[0];

  system(sys);
}

void Write_Sectors(unsigned long sector_number,int number_of_sectors)
{
  unsigned int result;

  if(param.media_type!=HD)
    {
    Convert_Logical_To_Physical(sector_number
     ,drive_specs[param.media_type].number_of_heads
     ,drive_specs[param.media_type].sectors_per_cylinder);

    /* Write the sector(s). */
    result=biosdisk(3, param.drive_number, translated_head
     ,translated_cylinder, translated_sector, number_of_sectors, sector_buffer);

    if(debug_prog==TRUE)
      {
      printf("[DEBUG]  Result From Write_Physical_Sectors()->  %X\n",result);
      }
    }
  else
    {
    int error_code=Write_Logical_Sectors(param.drive_letter,sector_number,
     number_of_sectors);

    if(error_code==255)
      {
      printf("\n Invalid drive specification\n");
      exit(1);
      }
    }
}

/*
/////////////////////////////////////////////////////////////////////////////
//  MAIN ROUTINE
/////////////////////////////////////////////////////////////////////////////
*/
void main(int argc, char *argv[])
{
  int loop=0;

  Initialization();

  /* if FORMAT is typed without any options or with /? */
  if( (argc==1) || (argv[1][1]=='?') )
    {
    Display_Help_Screen();
    exit(1);
    }

  /*if FORMAT is typed with a drive letter */
  if(argc>=2)
    {
    int index=3;

    char compare_character[2];

    param.drive_letter[0]=argv[1] [0];

    /* change to upper case, if necessary. */
    if( (param.drive_letter[0]>='a') && (param.drive_letter[0]<='z') )
     param.drive_letter[0]=param.drive_letter[0]-32;

    /* set the drive number */
    if( (param.drive_letter[0]>='A') && (param.drive_letter[0]<='Z') )
     param.drive_number = param.drive_letter[0] - 'A';
    else param.drive_number=255;

    if( (param.drive_number<0) || (param.drive_number> 25) )
      {
      printf("\n Invalid drive specification\n");
      exit(4);
      }

    if(param.drive_letter[0]>='a') param.drive_letter[0]=param.drive_letter[0]-32;
    param.drive_letter[1] = ':';

    if(debug_prog==TRUE) printf("\n[DEBUG]  Drive To Format->  %s \n\n",param.drive_letter);

    /* Set the type of disk */
    if(param.drive_number>1) param.drive_type=HARD;
    else param.drive_type=FLOPPY;

    if(argc>=3)
      {
      index=3;
      do
	{
	compare_character[0]=argv[index-1][1];

	/* Determine which switches were entered */
	if( ('v'==compare_character[0]) || ('V'==compare_character[0]) )
	  {
	  param.v=TRUE;

	  loop=0;
	  do
	    {
	    param.volume_label[loop]=argv[index-1][loop+3];
	    if(param.volume_label[loop]>96) param.volume_label[loop]=param.volume_label[loop]-32;

	    loop++;
	    }while(loop<11);
	  }

	if( ('q'==compare_character[0]) || ('Q'==compare_character[0]) )
	  {
	  param.q=TRUE;
	  }

	if( ('u'==compare_character[0]) || ('U'==compare_character[0]) )
	  {
	  param.u=TRUE;
	  }

	if( ('f'==compare_character[0]) || ('F'==compare_character[0]) )
	  {
	  param.f=TRUE;
	  if( ('1'==argv[index-1][3]) && ('6'==argv[index-1][4]) ) param.size=160;
	  if( ('1'==argv[index-1][3]) && ('8'==argv[index-1][4]) ) param.size=180;
	  if( ('3'==argv[index-1][3]) && ('2'==argv[index-1][4]) ) param.size=320;
	  if( ('3'==argv[index-1][3]) && ('6'==argv[index-1][4]) ) param.size=360;
	  if( ('7'==argv[index-1][3]) ) param.size=720;
	  if( ('1'==argv[index-1][3]) && ('2'==argv[index-1][4]) ) param.size=1200;
	  if( ('1'==argv[index-1][3]) && ('2'==argv[index-1][5]) ) param.size=1200;
	  if( ('1'==argv[index-1][3]) && ('4'==argv[index-1][4]) ) param.size=1440;
	  if( ('1'==argv[index-1][3]) && ('4'==argv[index-1][5]) ) param.size=1440;
	  if( ('2'==argv[index-1][3]) ) param.size=2880;
	  }

	if( ('b'==compare_character[0]) || ('B'==compare_character[0]) )
	  {
	  param.b=TRUE;
	  }

	if( ('s'==compare_character[0]) || ('S'==compare_character[0]) )
	  {
	  param.s=TRUE;
	  }

	if( ('t'==compare_character[0]) || ('T'==compare_character[0]) )
	  {
	  param.t=TRUE;
	  if('4'==argv[index-1][3]) param.cylinders=40;
	  if('8'==argv[index-1][3]) param.cylinders=80;
	  }

	if( ('n'==compare_character[0]) || ('N'==compare_character[0]) )
	  {
	  param.n=TRUE;
	  if('8'==argv[index-1][3]) param.sectors=8;
	  if('9'==argv[index-1][3]) param.sectors=9;
	  if( ('1'==argv[index-1][3]) && ('5'==argv[index-1][4]) ) param.sectors=15;
	  if( ('1'==argv[index-1][3]) && ('8'==argv[index-1][4]) ) param.sectors=18;
	  if('3'==argv[index-1][3]) param.sectors=36;
	  }

	if( ('y'==compare_character[0]) || ('Y'==compare_character[0]) )
	  {
	  param.force_yes=TRUE;
	  }

	if('1'==compare_character[0])
	  {
	  param.one=TRUE;
	  }

	if('4'==compare_character[0])
	  {
	  param.four=TRUE;
	  }

	if('8'==compare_character[0])
	  {
	  param.eight=TRUE;
	  }

	index++;
	}while(index<=argc);
      }

    /* Ensure that valid switch combinations were entered */
    if( (param.b==TRUE) && (param.s==TRUE) ) Display_Invalid_Combination();
    if( (param.v==TRUE) && (param.eight==TRUE) ) Display_Invalid_Combination();
    if( ( (param.one==TRUE) || (param.four==TRUE) ) && ( (param.f==TRUE) || (param.t==TRUE) || (param.n==TRUE) ) ) Display_Invalid_Combination();
    if( ( (param.t==TRUE) && (param.n!=TRUE) ) || ( (param.n==TRUE) && (param.t!=TRUE) ) ) Display_Invalid_Combination();
    if( (param.f==TRUE) && ( (param.t==TRUE) || (param.n==TRUE) ) )Display_Invalid_Combination();
    if( ( (param.one==TRUE) || (param.four==TRUE) ) && (param.eight==TRUE) )Display_Invalid_Combination();
    if( ( (param.four==TRUE) || (param.eight==TRUE) ) && (param.one==TRUE) )Display_Invalid_Combination();
    if( ( (param.eight==TRUE) || (param.one==TRUE) ) && (param.four==TRUE) )Display_Invalid_Combination();

    /* Make sure that a floppy disk is unconditionally formatted if /f /t */
    /* or /n are selected.                                                */
    if( (param.f==TRUE) || (param.t==TRUE) || (param.n==TRUE) )
      {
      param.u=TRUE;
      param.q=FALSE;
      }

    /* Set the media parameters */
    if(param.drive_type==FLOPPY)
      {
      if(param.force_yes==FALSE) Ask_User_To_Insert_Disk();
      Set_Floppy_Media_Type();
      }
    else
      {
      if(param.force_yes==FALSE) Confirm_Hard_Drive_Formatting();
      Set_Hard_Drive_Media_Parameters();
      }

    /* Format Drive */
    if( (param.u==TRUE) && (param.q==FALSE) )
      {
      /* Unconditional Format */
      Unconditional_Format();
      Create_File_System();
      }

    if( (param.u==FALSE) && (param.q==TRUE) )
      {
      /* Quick Format */
      if(param.drive_type==HARD) Save_File_System();
      Create_File_System();
      }

    if( (param.u==TRUE) && (param.q==TRUE) )
      {
      /* Quick Unconditional format */
      printf(" QuickFormatting \n");
      Create_File_System();
      }

    if( (param.u==FALSE) && (param.q==FALSE) )
      {
      /* Safe Format */
      if(param.drive_type==HARD) Save_File_System();
      Create_File_System();
      }

    printf(" Format complete.          \n");

    if(param.s==TRUE) Write_System_Files();

    Display_Drive_Statistics();

    exit(0);
    }
}
