/*  XCopy

    Copyright (c) Joe Cosentino 1997-2001.
    All Rights Reserved.

    Some code borrowed from Replace by Rene Ableidinger.

*/

// I N C L U D E S //////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <io.h>
#include <dir.h>
#include <dos.h>
#include <sys/stat.h>

// D E F I N E S ////////////////////////////////////////////////////////////

#define DIR_SEPARATOR "\\"

#define F_OK 0                          // Exists.
#define R_OK 4                          // Readable.
#define W_OK 2                          // Writeable.
#define X_OK 1                          // Executable.

// G L O B A L S ////////////////////////////////////////////////////////////

char opt_archive=0,opt_cont=0,opt_full=0,opt_m=0,opt_preview=0,opt_prompt=0,opt_quiet=0,opt_verify=0,opt_wait=0,opt_help=0,opt_subdir=0,opt_update=0,opt_hidden=0,dest_drive;
int file_counter=0;

// P R O T O T Y P E S //////////////////////////////////////////////////////

void classify_args(char, char *[], char *, char *[], char *, char *[]);
void error(char *, char *, const char *);
void exit_fn(void);
void preview_file(char []);
void replace_files(char [], char [], char []);
void replace_file(char [], char []);
void copy_file(char [], char []);

// F U N C T I O N S ////////////////////////////////////////////////////////

void classify_args(char argc, char *argv[], char *fileargc, char *fileargv[], char *optargc, char *optargv[])
{
    char i, *argptr;

    *fileargc=0;
    *optargc=0;
    for (i=1;i<argc;i++)
        {
        argptr=argv[i];
        if (argptr[0] == '/')
            {
            optargv[*optargc]=argptr + 1;
            *optargc=*optargc + 1;
            } // end if.
        else
            {
            fileargv[*fileargc]=argptr;
            *fileargc=*fileargc + 1;
            } // end else.

        } // end for.

} // end classify_args.

/////////////////////////////////////////////////////////////////////////////

void error(char *source, char *dest, const char *error)
{
    printf("Error copying %s to %s\n%s.\n", strupr(source), strupr(dest), error);

} // end write_protect.

/////////////////////////////////////////////////////////////////////////////

void exit_fn(void)
{
    char preview_txt[255]="";

    if (opt_preview)
        {
        strcpy(preview_txt, "would be ");
        } // end if.

    printf("\t%d File(s) %scopied\n", file_counter, preview_txt);

} // end exit_fn.

/////////////////////////////////////////////////////////////////////////////

void preview_file(char dest_filename[])
{
    if (opt_update)
        printf("Would update %s\n", dest_filename);
    else
        printf("Would add %s\n", dest_filename);

} // end preview_file.

/////////////////////////////////////////////////////////////////////////////

void replace_files(char src_pathname[], char src_filemask[], char dest_pathname[])
{
    char filemask[MAXPATH],new_dest_pathname[MAXPATH],src_filename[MAXPATH],dest_filename[MAXPATH];
    struct ffblk fileblock;
    int fileattrib,done;

    fileattrib=FA_RDONLY + FA_ARCH;
    if (opt_hidden)
        {
        fileattrib=fileattrib + FA_HIDDEN + FA_SYSTEM;
        } // end if.

    strcpy(filemask, src_pathname);
    strcat(filemask, src_filemask);
    done=findfirst(filemask, &fileblock, fileattrib);
    while (!done)
        {
        // Rebuild source filename.
        strcpy(src_filename, src_pathname);
        strcat(src_filename, fileblock.ff_name);

        // Build destination filename.
        strcpy(dest_filename, dest_pathname);
        strcat(dest_filename, fileblock.ff_name);

        replace_file(src_filename, dest_filename);
        done=findnext(&fileblock);
        } // end while.

    if (opt_subdir)
        {
        strcpy(filemask, dest_pathname);
        strcat(filemask, "*.*");
        done=findfirst(filemask, &fileblock, FA_DIREC);
        while (!done)
            {
            if (fileblock.ff_attrib == FA_DIREC && strcmp(fileblock.ff_name, ".") != 0 && strcmp(fileblock.ff_name, "..") != 0)
                {
                strcpy(new_dest_pathname, dest_pathname);
                strcat(new_dest_pathname, fileblock.ff_name);
                strcat(new_dest_pathname, DIR_SEPARATOR);
                replace_files(src_pathname, src_filemask, new_dest_pathname);
                } // end if.

            done=findnext(&fileblock);
            } // end while.

        } // end if.

} // end replace_files.

/////////////////////////////////////////////////////////////////////////////

void replace_file(char src_filename[], char dest_filename[])
{
    char ch,msg_prompt[255];
    int dest_file_exists,fileattrib;
    struct stat src_statbuf,dest_statbuf;
    struct dfree disktable;
    unsigned long free_diskspace;

    // Check source file for read permission.
    if (access(src_filename, R_OK) != 0)
        {
        error(src_filename, dest_filename, "Access denied");
        if (!opt_cont)
            exit(5);

        } // end if.

    // Get info of source and destination file.
    stat(src_filename, &src_statbuf);
    dest_file_exists=!stat(dest_filename, &dest_statbuf);

    // Get amount of free disk space in destination drive.
    getdfree(dest_drive, &disktable);
    free_diskspace=(unsigned long) disktable.df_avail * disktable.df_sclus * disktable.df_bsec;

    if (dest_file_exists)
        {
        return;
        } // end if.

    if (opt_update)
        {
        if (src_statbuf.st_mtime <= dest_statbuf.st_mtime)
            {
            return;
            } // end if.

        } // end if.

    // Check free space on destination disk.
    if (src_statbuf.st_size > free_diskspace - dest_statbuf.st_size)
        {
        error(src_filename, dest_filename, "Insufficient disk space");
        if (!opt_cont)
            exit(39);

        } // end if.

    if (opt_prompt)
        {
        strcpy(msg_prompt, "%s (Y/N)? ");
        do
            {
            printf(msg_prompt, dest_filename);
            scanf("%c", &ch);
            fflush(stdin);
            ch=toupper(ch);
            } // end do.
        while (ch != 'Y' && ch != 'N');

        if (ch != 'Y')
            {
            return;
            } // end if.

        } // end if.

    if (opt_preview)
        preview_file(dest_filename);
    else
        copy_file(src_filename, dest_filename);

    file_counter=file_counter + 1;

} // end replace_file.

/////////////////////////////////////////////////////////////////////////////

void copy_file(char src_filename[], char dest_filename[])
{
    FILE *src_file,*dest_file;
    char buffer[BUFSIZ];
    int buffersize,fileattrib;
    struct ftime filetime;

    // Open source file.
    src_file=fopen(src_filename, "rb");
    if (src_file == NULL)
        {
        error(src_filename, dest_filename, "Cannot open source file");
        if (!opt_cont)
            exit(30);

        } // end if.

    // Open destination file.
    dest_file=fopen(dest_filename, "wb");
    if (dest_file == NULL)
        {
        error(src_filename, dest_filename, "Cannot create destination file");
        fclose(src_file);
        if (!opt_cont)
            exit(29);

        } // end if.

    if (!opt_quiet && opt_full)
        printf("%s -> %s\n", src_filename, dest_filename);
    else if (!opt_quiet)
        printf("%s\n", dest_file);

    // Copy file data.
    buffersize=fread(buffer, sizeof(char), BUFSIZ, src_file);
    while (buffersize > 0)
        {
        if (fwrite(buffer, sizeof(char), buffersize, dest_file) != buffersize)
            {
            error(src_filename, dest_filename, "Error writing destination file");
            fclose(src_file);
            fclose(dest_file);
            if (!opt_cont)
                exit(29);

            } // end if.

        buffersize=fread(buffer, sizeof(char), BUFSIZ, src_file);
        } // end while.

    // Copy file timestamp.
    getftime(fileno(src_file), &filetime);
    setftime(fileno(dest_file), &filetime);

    // Close files.
    fclose(src_file);
    fclose(dest_file);

    // Copy file attributes.
    if (opt_m)
        {
        fileattrib=FA_ARCH;
        _chmod(dest_filename, -1, fileattrib);
        } // end if.
    else
        {
        fileattrib=_chmod(src_filename, 0);
        _chmod(dest_filename, 1, fileattrib);
        } // end else.

} // end copy_file.

/////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
    char fileargc,*fileargv[255],optargc,*optargv[255],cur_pathname[MAXPATH]="",src_pathname[MAXPATH]="",src_filemask[MAXPATH]="",dest_pathname[MAXPATH]="",*ptr,option[255]="",i;

    classify_args(argc, argv, &fileargc, fileargv, &optargc, optargv);
    for (i=0;i<optargc;i++)             // Get options.
        {
        strcpy(option, optargv[i]);
        if (stricmp(option, "a") == 0)
            opt_archive=-1;
        else if (stricmp(option, "c") == 0)
            opt_cont=-1;
        else if (stricmp(option, "f") == 0)
            opt_full=-1;
        else if (stricmp(option, "m") == 0)
            opt_m=-1;
        else if (stricmp(option, "l") == 0)
            opt_preview=-1;
        else if (stricmp(option, "h") == 0)
            opt_hidden=-1;
        else if (stricmp(option, "p") == 0)
            opt_prompt=-1;
        else if (stricmp(option, "q") == 0)
            opt_quiet=-1;
        else if (stricmp(option, "v") == 0)
            opt_verify=-1;
        else if (stricmp(option, "s") == 0)
            opt_subdir=-1;
        else if (stricmp(option, "u") == 0)
            opt_update=-1;
        else if (stricmp(option, "w") == 0)
            opt_wait=-1;
        else if (stricmp(option, "?") == 0)
            opt_help=-1;
        else
            {
            printf("Invalid parameter - %s\n", optargv[i]);
            exit(11);
            } // end else.

        } // end for.

    if (opt_help)
        {
        printf("Copies files and directories.\n\n"
               "Syntax: XCOPY source destination [/Y | /-Y] [/A | /M] [/P] [/V] [/W] [/C] [/Q]\n"
               "                                 [/F] [/L] [/H] [/U] [/V] [/?]\n"
               "  source      Specifies the location and names of the files you want to\n"
               "              copy\n"
               "  destination Specifies the destination of the files you want to copy\n"
               "  /A          Copies only source files that have their archive attributes\n"
               "              set\n"
               "  /M          Copies source files that have their archive attributes set and\n"
               "              turns off the archive attribute\n"
               "  /P          Prompts for confirmation about whether you want to create\n"
               "              each destination file\n"
               "  /C          Continues copying even if errors occur\n"
               "  /S          Copies directories and subdirectories\n"
               "  /F          Displays full source and destination file names while copying\n"
               "  /L          Displays files that would be copied\n"
               "  /H          Copies hidden and system files also\n"
               "  /Q          Does not display file names while copying\n"
               "  /U          Updates files that already exist in destination\n"
               "  /V          Verifies each file as it is written to the destination file\n"
               "              to make sure that the destination files are identical to the\n"
               "              source files\n"
               "  /W          Prompts you to press a key before copying\n"
               "  /?          Displays this help message\n");
        exit(11);
        } // end if.

    if (argc < 2 || fileargc < 1)
        {
        printf("Required parameter missing\n");
        return 1;
        } // end if.

    atexit(exit_fn);
    getcwd(cur_pathname, MAXPATH);
    _fullpath(src_pathname, fileargv[0], MAXPATH);
    ptr=strrchr(src_pathname, (char)* DIR_SEPARATOR);
    strcpy(src_filemask, ptr + 1);
    *ptr='\0';
    if (strlen(src_pathname) <= 2)
        {
        strcat(src_pathname, DIR_SEPARATOR);
        } // end if.

    if (chdir(src_pathname) != 0)       // Check source path.
        {
        error(src_pathname, dest_pathname, "Path not found");
        if (!opt_cont)
            exit(3);

        } // end if.

    chdir(cur_pathname);                // Restore path.
    if (strlen(src_pathname) > 3)
        {
        // Source path is not a root directory -> add backslash.
        strcat(src_pathname, DIR_SEPARATOR);
        } // end if.

    if (fileargc < 2)                   // Get destination pathname.
        {
        // No destination path specified -> use current.
        strcpy(dest_pathname, cur_pathname);
        } // end if.
    else
        {
        _fullpath(dest_pathname, fileargv[1], MAXPATH);
        if (chdir(dest_pathname) != 0)
            {
            error(src_pathname, dest_pathname, "Path not found");
            if (!opt_cont)
                exit(3);

            } // end if.

        chdir(cur_pathname);            // Restore path.
        } // end else.

    strcat(dest_pathname, DIR_SEPARATOR);
    if (strcmp(src_pathname, dest_pathname) == 0)
        {
        error(src_pathname, dest_pathname, "Source and destination path must not be the same");
        if (!opt_cont)
            exit(11);

        } // end if.

    dest_drive=toupper(dest_pathname[0]) - 64;
    if (opt_archive && opt_m)
        {
        printf("Invalid combination of parameters\n");
        if (!opt_cont)
            exit(11);

        } // end if.

    if (opt_verify)
        setverify(1);

    if (opt_wait)
        {
        printf("Press any key to begin copying file(s)...\n");
        getch();
        fflush(stdin);
        } // end if.

    replace_files(src_pathname, src_filemask, dest_pathname);
    return 0;

} // end main.
