/*    
   Copyrigt.c - Abstractions for a common interface between absolute disk 
                access and image files.
   
   Copyright (C) 2000 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <io.h>
#include <dos.h>
#include <ctype.h>
#include <fcntl.h>
#include <alloc.h>
#include <string.h>

#include "..\header\rdwrsect.h"
#include "..\header\bool.h"
#include "..\header\drive.h"
#include "..\header\FTEerr.h"

#define BYTESPERSECTOR 512

/* Image file modifiers. */
static int ReadFromImageFile(int handle, int nsects, SECTOR lsect,
                             void* buffer)
{
  if (lseek(handle, lsect*BYTESPERSECTOR, SEEK_SET) == -1L)
  {
     SetFTEerror(FTE_READ_ERR);
     return -1;
  }

  if (read(handle, buffer, nsects*BYTESPERSECTOR) == -1)
  {
     SetFTEerror(FTE_READ_ERR);
     return -1;
  }

  return 0;
}

static int WriteToImageFile(int handle, int nsects, SECTOR lsect,
                            void* buffer)
{
  if (lseek(handle, lsect*BYTESPERSECTOR, SEEK_SET) == -1L)
  {
     SetFTEerror(FTE_WRITE_ERR);
     return -1;
  }

  if (write(handle, buffer, nsects*BYTESPERSECTOR) == -1)
  {
     SetFTEerror(FTE_WRITE_ERR);
     return -1;
  }

  return 0;
}

/* Real disk sector modifiers. */
static int ReadFromRealDisk(int handle, int nsects, SECTOR lsect,
                            void* buffer)
{
  if (absread(handle, nsects, (int) lsect, buffer) == -1)
  {
     SetFTEerror(FTE_READ_ERR);
     return -1;
  }

  return 0;
}

static int WriteToRealDisk(int handle, int nsects, SECTOR lsect, void* buffer)
{
  if (abswrite(handle, nsects, (int) lsect, buffer) == -1)
  {
     SetFTEerror(FTE_WRITE_ERR);
     return -1;
  }

  return 0;
}

/*
** Public read sector function.
*/
int ReadSectors(RDWRHandle handle, int nsects, SECTOR lsect, void* buffer)
{
    return handle->ReadFunc(handle->handle, nsects, lsect, buffer);
}

/*
** Public write sector function.
*/
int WriteSectors(RDWRHandle handle, int nsects, SECTOR lsect, void* buffer)
{
    return handle->WriteFunc(handle->handle, nsects, lsect, buffer);
}

/*
** Private function to initialise the reading and writing of sectors.
*/
static int PrivateInitReadWriteSectors(char* driveorfile, int modus,
                                       RDWRHandle* handle)
{
    *handle = (RDWRHandle) malloc(sizeof(struct RDWRHandleStruct));
    if (*handle == NULL) return FALSE;
    memset(*handle, 0, sizeof(struct RDWRHandleStruct));

    if (HasAllFloppyForm(driveorfile))
    {
       (*handle)->ReadFunc    = ReadFromRealDisk;
       (*handle)->WriteFunc   = WriteToRealDisk;
       (*handle)->handle      = toupper(driveorfile[0]) - 'A';
    }
    else
    {
       (*handle)->ReadFunc    = ReadFromImageFile;
       (*handle)->WriteFunc   = WriteToImageFile;
       (*handle)->handle      = open (driveorfile, modus|O_BINARY);
       (*handle)->IsImageFile = TRUE;

       if ((*handle)->handle == -1)
       {
          free(*handle);
          *handle = NULL;
          return FALSE;
       }
    }
    return TRUE;
}

/*
**  Init for reading and writing of sectors.
*/
int InitReadWriteSectors(char* driveorfile, RDWRHandle* handle)
{
    return PrivateInitReadWriteSectors(driveorfile, O_RDWR, handle);
}

/*
**  Init for reading only.
*/
int InitReadSectors(char* driveorfile, RDWRHandle* handle)
{
    return PrivateInitReadWriteSectors(driveorfile, O_RDONLY, handle);
}

/*
**  Init for writing only (reasonably stupid).
*
int InitWriteSectors(char* driveorfile, RDWRHandle* handle)
{
    return PrivateInitReadWriteSectors(driveorfile, O_WRONLY, handle);
}
*/

/*
**  Close the sector read/write system.
*/
void CloseReadWriteSectors(RDWRHandle* handle)
{
    if (*handle != NULL)
    {
       if ((*handle)->IsImageFile) close((*handle)->handle);
       free(*handle);
       *handle = NULL;
    }
}
