/*
   Fat.c - FAT manipulation code.
   Copyright (C) 2000 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/


#include <string.h>

#include "..\header\bool.h"
#include "..\header\rdwrsect.h"
#include "..\header\fat.h"
#include "..\header\boot.h"
#include "..\header\direct.h"
#include "..\header\fatconst.h"

int GetFatStart(RDWRHandle handle)
{
    return GetReservedSectors(handle);
}

int GetFatLabelSize(RDWRHandle handle)
{
    unsigned long disksize;

    disksize = (long) GetSectorsPerCluster(handle) *
	       (long) GetNumberOfSectors(handle)   *
	       (long) BYTESPERSECTOR;

    if (!disksize) return FALSE;

    return (disksize > HD32MB) ? FAT16 : FAT12;
}

CLUSTER DataSectorToCluster(RDWRHandle handle, SECTOR sector)
{
    SECTOR dirstart, dirlen, sectorspercluster;

    dirstart = GetDirectoryStart(handle);
    if (dirstart == FALSE) return FALSE;

    sectorspercluster = (SECTOR) GetSectorsPerCluster(handle);
    if (!sectorspercluster) return FALSE;

    dirlen = (GetNumberOfRootEntries(handle) / ENTRIESPERSECTOR);
    if (!dirlen) return FALSE;

    return (CLUSTER) ((sector + (2 * sectorspercluster) - dirstart - dirlen) /
                     sectorspercluster);
}

SECTOR ConvertToDataSector(RDWRHandle handle,
			   CLUSTER fatcluster)
{
    SECTOR sector, dirstart, dirlen, sectorspercluster;

    dirstart = GetDirectoryStart(handle);
    if (dirstart == FALSE) return FALSE;

    sectorspercluster = (SECTOR) GetSectorsPerCluster(handle);
    if (!sectorspercluster) return FALSE;

    sector = ((SECTOR) (fatcluster - 2)) * sectorspercluster;

    dirlen = (GetNumberOfRootEntries(handle) / ENTRIESPERSECTOR);

    return sector + dirstart + dirlen;
}

static int GetFatLabel(RDWRHandle handle,
		       char* readbuf, CLUSTER labelnr,
		       CLUSTER* label)
{
    int     labelsize;
    CLUSTER cluster;

    labelsize = GetFatLabelSize(handle);
    if (!labelsize) return FALSE;

    labelnr %= (FATREADBUFSIZE * 8 / labelsize);

    if (labelsize == FAT12)
    {
       cluster = labelnr * 1.5;
       memcpy(label, &readbuf[(unsigned) cluster], sizeof(CLUSTER));

       if (labelnr % 2 == 0)
	  *label &= 0xfff;
       else
	  *label = (*label >> 4) & 0xffff;
    }
    else if (labelsize == FAT16)
    {
       memcpy(label, &readbuf[(unsigned) labelnr << 1], sizeof(CLUSTER));
    }
    else
       return FALSE;

    return TRUE;
}

static int LastFatLabel(RDWRHandle handle, CLUSTER label)
{
    int labelsize;

    labelsize = GetFatLabelSize(handle);

    if (labelsize == FAT12)
       return (FAT12_LAST(label));
    else if (labelsize == FAT16)
       return (FAT16_LAST(label));

    return FALSE;
}

static CLUSTER GeneralizeLabel(CLUSTER label, RDWRHandle handle)
{
   int labelsize = GetFatLabelSize(handle);

   if (labelsize == FAT12)
   {
      if (FAT12_FREE(label))     return FAT_FREE_LABEL;
      if (FAT12_BAD(label))      return FAT_BAD_LABEL;
      if (FAT12_RESERVED(label)) return FAT_RESERVED_LABEL;
      if (FAT12_LAST(label))     return FAT_LAST_LABEL;
      return label;
   }
   else
      return label;
}

int LinearTraverseFat(RDWRHandle handle,
		      int (*func) (RDWRHandle handle,
				   CLUSTER label,
				   SECTOR datasector,
				   void** structure),
		      void** structure)
{
    int      i, j = 2, iterations, rest, fatstart;
    unsigned toreadsectors, toreadlabels, labelsinbuf, dataclusters;
    char     buffer[FATREADBUFSIZE];

    CLUSTER  label;
    SECTOR   datasector;

    unsigned short sectorsperfat;
    unsigned char  sectorspercluster;

    fatstart = GetFatStart(handle);
    if (!fatstart) return FALSE;

    sectorsperfat     = GetSectorsPerFat(handle);
    if (!sectorsperfat) return FALSE;

    sectorspercluster = GetSectorsPerCluster(handle);
    if (!sectorspercluster) return FALSE;

    dataclusters = GetClustersInDataArea(handle);
    if (!dataclusters) return FALSE;

    iterations = sectorsperfat / SECTORSPERREADBUF;
    rest       = sectorsperfat % SECTORSPERREADBUF;

    toreadsectors = SECTORSPERREADBUF;
    toreadlabels  = labelsinbuf = FATREADBUFSIZE * 8 / GetFatLabelSize(handle);

    for (i = 0; i < iterations + (rest > 0); i++)
    {
	toreadsectors = (i == iterations) ? rest : SECTORSPERREADBUF;

	if (ReadSectors(handle, toreadsectors,
			fatstart + (i*SECTORSPERREADBUF), buffer) == -1)
	   return FALSE;

	for (; j < toreadlabels; j++)
	{
	    if (j >= dataclusters) break;

	    if (!GetFatLabel(handle, buffer, j, &label)) return FALSE;
	    datasector = ConvertToDataSector(handle, j);
	    if (!func (handle, GeneralizeLabel(label, handle), datasector,
		       structure))
	       return TRUE;
	}
	toreadlabels += labelsinbuf;
    }

    return TRUE;
}

int FileTraverseFat(RDWRHandle handle, CLUSTER startcluster,
		    int (*func) (RDWRHandle handle,
				 CLUSTER label,
				 SECTOR  datasector,
				 void** structure),
		    void** structure)
{
    char buffer[FATREADBUFSIZE];
    CLUSTER cluster, seeking, prevpart = -1;
    SECTOR  sector;

    int  fatstart;

    cluster = startcluster;
    fatstart = GetFatStart(handle);

    while (!LastFatLabel(handle, cluster))
    {
	seeking = cluster / (FATREADBUFSIZE * 8 / GetFatLabelSize(handle));

	if (prevpart != seeking)
	{
	   if (ReadSectors(handle, SECTORSPERREADBUF,
			   fatstart + (unsigned) seeking * SECTORSPERREADBUF,
			   buffer) == -1)
	      return FALSE;

	   prevpart = seeking;
	}

	sector = ConvertToDataSector(handle, cluster);
	if (!GetFatLabel(handle, buffer, cluster, &cluster)) return FALSE;
	if (!func(handle, GeneralizeLabel(cluster, handle), sector,
		  structure))
	   return TRUE;
    }

    return TRUE;
}

int ReadFatLabel(RDWRHandle handle, CLUSTER labelnr,
		 CLUSTER* label)
{
    int     labelsize;
    char    buffer[FATREADBUFSIZE];
    int     sectorstart;
    int     labelsinbuf;

    labelsize = GetFatLabelSize(handle);
    if (!labelsize) return FALSE;

    sectorstart = GetFatStart(handle);
    if (!sectorstart) return FALSE;

    labelsinbuf  = FATREADBUFSIZE * 8 / labelsize;
    sectorstart += (unsigned) labelnr / labelsinbuf;

    if (ReadSectors(handle, SECTORSPERREADBUF, sectorstart, buffer) == -1)
	return FALSE;

    return GetFatLabel(handle, buffer, labelnr, label);
}

int WriteFatLabel(RDWRHandle handle, CLUSTER labelnr,
		  CLUSTER label)
{
    int      labelsize;
    char     buffer[FATREADBUFSIZE];
    int      sectorstart;
    int      labelsinbuf, temp;
    
    labelsize = GetFatLabelSize(handle);
    if (!labelsize) return FALSE;

    sectorstart = GetFatStart(handle);
    if (!sectorstart) return FALSE;

    labelsinbuf  = FATREADBUFSIZE * 8 / labelsize;
    sectorstart += (unsigned) labelnr / labelsinbuf;

    if (ReadSectors(handle, SECTORSPERREADBUF, sectorstart, buffer) == -1)
	return FALSE;

    if (labelsize == FAT12)
    {
       labelnr *= 12;
       labelnr %= 8 * FATREADBUFSIZE;

       if (labelnr % 8 == 0)
       {
	  memcpy(&temp, &buffer[(unsigned) labelnr / 8], 2);
	  temp &= 0xf000;
	  temp |= ((unsigned) label & 0xfff);
	  memcpy(&buffer[(unsigned) labelnr / 8], &temp, 2);
       }
       else
       {
	  memcpy(&temp, &buffer[(unsigned) labelnr / 8], 2);
	  temp &= 0x000f;
	  temp |= ((unsigned) label << 4);
	  memcpy(&buffer[(unsigned) labelnr / 8], &temp, 2);
       }
    }
    else if (labelsize == FAT16)
    {
       memcpy(&buffer[(unsigned) labelnr << 1], &label, 2);
    }
    else
       return FALSE;

    return WriteSectors(handle, SECTORSPERREADBUF,
			sectorstart, buffer) != -1;
}

unsigned long GetBytesInFat(RDWRHandle handle)
{
    return ((long) GetFatLabelSize(handle) *
            (long) GetClustersInDataArea(handle)) / 8L;
}
