/* This is a simple little program that users can use to 
   easily get the version of the distribution installed,
   and if wanted the version of the specific program
   that CAME WITH the distribution this program is 
   distributed with.  This program is updated with each
   release, and simply displays hard coded information.
   Only covers boot and base.
*/

#define DIST_VERSION "FreeDOS Ripcord BETA 7 H2 (\"SPEARS\") Distribution"

#define NA 1                /* not currently available                          */
#define BETA7 2             /* indicates available released in Beta7            */
#define BETA6 4             /* indicates available in Beta6 or earlier          */
#define RIPCORD 8           /* updated in a ripcord, but not the current one    */
#define UPDATED 16          /* updated in this ripcord release                  */
#define INTERNAL 32768      /* command is internal to FreeCom                   */
#define ALL (NA | BETA7 | BETA6 | RIPCORD | UPDATED)
#define BETA (BETA7 | BETA6)

#define NUM_PROGRAMS 68
typedef struct PROGINFO
{
  char * name;              /* The name of the program                    */
  char * version;           /* String representing the version            */
  unsigned int  updated;    /* indicates if updated this release or not   */
} PROGINFO;

PROGINFO programs[NUM_PROGRAMS] = 
{
  { "Kernel",   "Build 2.0.25.c                 ", UPDATED },
  { "Install",  "3.76 (kjd - not official)      ", UPDATED },
  { "TE",       "1.0 Revision 1.4               ", RIPCORD },
  { "Assign",   "1.4                            ", BETA6 },
  { "Attrib",   "1.20                           ", BETA7 },
  { "Backup",   "Not Available                  ", NA },
  { "break",    "internal                       ", INTERNAL },
  { "cd",       "internal (aka chdir)           ", INTERNAL },
  { "Chcp",     "GNU Chcp 1.0                   ", RIPCORD },
  { "Chkdsk",   "Not Available                  ", NA },
  { "cls",      "internal                       ", INTERNAL },
  { "Choice",   "3.3                            ", BETA7 },
  { "FreeCom",  "0.83 beta 32                   ", UPDATED },
  { "Comp",     "0.1pv2                         ", BETA6 },
  { "copy",     "internal                       ", INTERNAL },
  { "Debug",    "0.95c                          ", UPDATED },
  { "Defrag",   "alpha 0.2                      ", BETA6 },
  { "del",      "internal (aka erase)           ", INTERNAL },
  { "Deltree",  "1.02c                          ", BETA6 },
  { "dir",      "internal                       ", INTERNAL },
  { "Diskcomp", "0.72                           ", BETA7 },
  { "Diskcopy", "0.6                            ", BETA7 },
  { "echo",     "internal                       ", INTERNAL },
  { "Edit",     "0.5                            ", BETA7 },
  { "Exe2bin",  "1.0                            ", BETA6 },
  { "Fc",       "2.0                            ", BETA6 },
  { "Fdisk",    "1.0                            ", UPDATED },
  { "FDXMS",    "0.8.Grains                     ", BETA7 },
  { "Find",     "2.3                            ", UPDATED },
  { "Fremm386", "Not Available                  ", NA },
  { "Format",   "0.85                           ", UPDATED },
  { "Graphics", "Not Available                  ", NA },
  { "Help",     "3.2                            ", BETA7 },
  { "Htmlhelp", "4.1                            ", BETA6 },
  { "Label",    "1.3                            ", BETA6 },
  { "loadhigh", "internal                       ", INTERNAL },
  { "Mem",      "1.3                            ", BETA7 },
  { "md",       "internal (aka mkdir)           ", INTERNAL },
  { "Mirror",   "0.2                            ", BETA6 },
  { "Mode",     "1.04                           ", BETA6 },
  { "More",     "3.3                            ", UPDATED },
  { "Move",     "1.1                            ", BETA6 },
  { "Nansi",    "3.4                            ", BETA6 },
  { "Nlsfunc",  "Not Available                  ", NA },
  { "pause",    "internal                       ", INTERNAL },
  { "Print",    "1.0                            ", BETA6 },
  { "PrintQ",   "05-Jul-1997                    ", BETA6 },
  { "Replace",  "1.2                            ", BETA6 },
  { "ren",      "internal (aka rename)          ", INTERNAL },
  { "rd",       "internal (aka rmdir)           ", INTERNAL },
  { "Restore",  "Not Available                  ", NA },
  { "Scandisk", "0.5                            ", BETA6 },
  { "Share",    "FreeDOS share, from Kernel     ", BETA7 },
  { "Shsucdex", "1.4b (aka fdcdex)              ", BETA6 },
  { "Sort",     "1.0                            ", BETA6 },
  { "Swsubst",  "3.1                            ", BETA6 },
  { "Sys",      "from kernel 2025c              ", UPDATED },
  { "Tdsk",     "2.3                            ", BETA7 },
  { "time",     "internal                       ", INTERNAL },
  { "Tree",     "3.7.1                          ", UPDATED },
  { "type",     "internal                       ", INTERNAL },
  { "Undelete", "Not Available                  ", NA },
  { "Unformat", "0.8                            ", BETA6 },
  { "ver",      "internal                       ", INTERNAL },
  { "verify",   "internal                       ", INTERNAL },
  { "vol",      "internal                       ", INTERNAL },
  { "Xcopy",    "1.0 (Rene's)                   ", UPDATED },
  { "Xkeyb",    "1.7                            ", BETA7 }
};

#include <stdio.h>
#include <string.h>

void print_usage(void)
{
  printf("ripcord [PROGRAM | ALL | BETA[6|7] | RIPCORD | UPDATED | NA | INTERNAL]\n");
  printf("  PROGRAM is the program to display version information for.\n");
  printf("  Do NOT include extension, eg use FORMAT not FORMAT.EXE\n");
  printf("\n");
  printf("  The remaining choices are keywords that will display\n");
  printf("  version information for the corresponding programs included\n");
  printf("  on the boot disk and in the base package:\n");
  printf("    ALL     : all programs on boot disk & in base (except INTERNAL)\n");
  printf("    BETA#   : only those not updated since Beta [6/7]\n");
  printf("    RIPCORD : all programs updated since Beta 7\n");
  printf("    UPDATED : all programs updated in this Ripcord release\n");
  printf("    NA      : \"standard\" programs not yet available\n");
  printf("    INTERNAL: programs internal to FreeCOM\n");
  printf("\n");
  printf("This program displays hard coded information and only\n");
  printf("reflects information for the distribution it is included with.\n");
  printf("\n");
  printf("Maintained by Kenneth J Davis <jeremyd@computer.org>\n");
  printf("It is public domain.\n");
}


/* Procedures */

int getIndexFor(char *progName)
{
  register int i;
  for (i = 0; i < NUM_PROGRAMS; i++)
    if (strncmpi(progName, programs[i].name, 8) == 0)
      return i;

  return -1;
}

void display_info(int index)
{
  printf("%8s : %s [%s]\n", programs[index].name, programs[index].version,
         ((programs[index].updated & BETA6)?"Beta6" :
         ((programs[index].updated & BETA7)?"Beta7" :
         ((programs[index].updated & RIPCORD)?"Updated since Beta7" : 
         ((programs[index].updated & UPDATED)?"Updated this release" :
         ((programs[index].updated & NA)?"NA" :
         ((programs[index].updated & INTERNAL)?"Internal to FreeCom" : "Unknown status" )))))));
}

void display(int flags)
{
  register int i;
  for (i = 0; i < NUM_PROGRAMS; i++)
    if (programs[i].updated & flags)
      display_info(i);
}

int main(int argc, char *argv[])
{
  register int i;
  register int index;

  printf(DIST_VERSION "\n\n");

  for (i = 1; i < argc; i++)
  {
    if ((argv[i][1] == '?') || (strcmpi(argv[i]+1, "HELP") == 0))
    {
      print_usage();
      return 0;
    }
    else if (strcmpi(argv[i], "ALL") == 0)
    {
      display(ALL);
    }
    else if (strcmpi(argv[i], "NA") == 0)
    {
      display(NA);
    }
    else if (strcmpi(argv[i], "INTERNAL") == 0)
    {
      display(INTERNAL);
    }
    else if (strcmpi(argv[i], "BETA") == 0)
    {
      display(BETA);
    }
    else if (strcmpi(argv[i], "BETA6") == 0)
    {
      display(BETA6);
    }
    else if (strcmpi(argv[i], "BETA7") == 0)
    {
      display(BETA7);
    }
    else if (strcmpi(argv[i], "RIPCORD") == 0)
    {
      display(RIPCORD | UPDATED);
    }
    else if (strcmpi(argv[i], "UPDATED") == 0)
    {
      display(UPDATED);
    }
    else /* probably a specific program */
    {
      index = getIndexFor(argv[i]);
      if ((index >= 0) && (index < NUM_PROGRAMS))
        display_info(index);
      else
        printf("Unknown program [%s]\n", argv[i]);
    }
  }

  return 0;
}
