/*
// Module:  PDISKIO.C
// Module Description:  Disk Input/Output Code Module
//                      All functions that access the hard disk via
//                      interrupt 0x13 are here, including LBA support.
// Written By:  Brian E. Reifsnyder
// Module Version:  2.0
// Copyright:  2001 under the terms of the GNU GPL, Version 2
*/

/*
/////////////////////////////////////////////////////////////////////////////
//  SPECIAL
/////////////////////////////////////////////////////////////////////////////
*/

#define PDISKIO
#include "main.h"

/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <conio.h>
#include <bios.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "pdiskio.h"

/*
/////////////////////////////////////////////////////////////////////////////
//  PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/

/* Module Prototype Declarations */
/* ***************************** */
int Get_Hard_Drive_Parameters(int physical_drive);
int Read_Physical_Sectors_CHS(int drive,long cylinder, long head, long sector, int number_of_sectors);
int Read_Physical_Sectors_LBA(int drive,long cylinder, long head, long sector, int number_of_sectors);
int Write_Physical_Sectors_CHS(int drive, long cylinder, long head, long sector, int number_of_sectors);
int Write_Physical_Sectors_LBA(int drive, long cylinder, long head, long sector, int number_of_sectors);

long double Translate_CHS_To_LBA(unsigned long cylinder,unsigned long head
 ,unsigned long sector,unsigned long total_heads,unsigned long total_sectors);

void Clear_Partition_Table_Area_Of_Sector_Buffer();
void Check_For_INT13_Extensions();
void Convert_Logical_To_Physical(unsigned long sector
 ,unsigned long total_heads,unsigned long total_sectors);
void Get_Partition_Information();

int Determine_Drive_Letters();
int Read_Logical_Sectors(unsigned char drive_letter[2]
 , unsigned long logical_sector_number, int number_of_sectors);
int Read_Partition_Tables();
int Read_Physical_Sectors(int drive, long cylinder, long head, long sector, int number_of_sectors);
int Write_Logical_Sectors(unsigned char drive_letter[2]
 , unsigned long logical_sector_number, int number_of_sectors);
int Write_Partition_Tables();
int Write_Physical_Sectors(int drive, long cylinder, long head, long sector, int number_of_sectors);

unsigned long Decimal_Number(unsigned long hex1, unsigned long hex2, unsigned long hex3, unsigned long hex4);

void Check_For_INT13_Extensions();
void Clear_Sector_Buffer();
void Initialize_LBA_Structures();
void Load_Brief_Partition_Table();

  /* External Prototype Declarations */
  /* ******************************* */
  extern void Clear_Extended_Partition_Table(int drive);
  extern void Clear_Screen(int type);
  extern void Convert_Long_To_Integer(long number);
  extern void Pause();
  extern void Print_Centered(int y,char *text,int style);
  extern void Position_Cursor(int row,int column);

/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

/* Check for interrupt 0x13 extensions */
void Check_For_INT13_Extensions()
{
  int carry;
  int drive_number=0x80;

  unsigned int ah_register;
  unsigned int bx_register;
  unsigned int cx_register;

#ifdef DEBUG
  if(debug.lba==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(0,"void Check_For_INT13_Extensions() debugging screen",BOLD);
    printf("\n\n    drive     int 0x13 ext?     access w/packet\n\n");
    }
#endif

  do
    {
    carry=99;

    asm{
      mov ah,0x41
      mov bx,0x55aa
      mov dl,BYTE PTR drive_number
      int 0x13

      mov BYTE PTR ah_register,ah
      mov WORD PTR bx_register,bx
      mov WORD PTR cx_register,cx

      jnc carry_flag_not_set    /* Jump if the carry flag is clear  */
      }                         /* If the carry flag is clear, then */
				/* the extensions exist.            */
    carry=1;
    part_table[(drive_number-128)].ext_int_13=FALSE;

    carry_flag_not_set:
    if( (carry==99) && (bx_register==0xaa55) )
      {
      flags.use_extended_int_13=TRUE;
      part_table[(drive_number-128)].ext_int_13=TRUE;

      if((cx_register&0x0001)==1) part_table[(drive_number-128)].device_access_using_packet_structure=TRUE;
      else part_table[(drive_number-128)].device_access_using_packet_structure=FALSE;

      part_table[(drive_number-128)].ext_int_13_version=ah_register;

#ifdef DEBUG
      if(debug.lba==TRUE)
	{
	printf("     0x%2x          yes",drive_number);

	if((cx_register&0x0001)==1) printf("                 yes");
	else printf("                  no");

	printf("\n");
	}
#endif

      }
#ifdef DEBUG
    else if(debug.lba==TRUE) printf("     0x%2x           no\n",drive_number);
#endif

    drive_number++;
    }while(drive_number<0x88);

#ifdef DEBUG
  if(debug.lba==TRUE)
    {
    printf("\n\n\n");
    Pause();
    }
#endif

}

/* Clear the Boot Sector of a partition */
void Clear_Boot_Sector(int drive,long cylinder,long head,long sector)
{
  unsigned char stored_sector_buffer[512];
  long index;

  /* Save sector_buffer[512] into stored_sector_buffer[512] */
  index=0;
  do
    {
    stored_sector_buffer[index]=sector_buffer[index];
    index++;
    }while(index<512);

  /* Write all 0xf6 values to sector_buffer[index] */
  index=0;
  do
    {
    sector_buffer[index]=0xf6;
    index++;
    }while(index<512);

  Write_Physical_Sectors(drive,cylinder,head,sector,1);

  /* Restore sector_buffer[512] to its original contents */
  index=0;
  do
    {
    sector_buffer[index]=stored_sector_buffer[index];
    index++;
    }while(index<512);
}

/* Clear The Partition Table Area Of sector_buffer only. */
void Clear_Partition_Table_Area_Of_Sector_Buffer()
{
  long index=0x1be;

  do
    {
    sector_buffer[index]=0;
    index++;
    }while(index<=0x1fd);
}

/* Clear Sector Buffer */
void Clear_Sector_Buffer()
{
  int index=0;

  do
    {
    sector_buffer[index]=0;
    index ++;
    } while(index<512);
}

/* Combine Cylinder and Sector Values */
unsigned long Combine_Cylinder_and_Sector(unsigned long cylinder, unsigned long sector)
{
  long value = 0;

  asm{
    mov ax,WORD PTR cylinder
    mov bx,WORD PTR sector

    mov dl,ah
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1

    mov dh,al

    add dx,bx

    mov WORD PTR value,dx
    }

  return(value);
}

/* Convert a logical sector to a physical drive location */
void Convert_Logical_To_Physical(unsigned long sector
 ,unsigned long total_heads,unsigned long total_sectors)
{
  unsigned long remaining;

  translated_cylinder=sector/(total_sectors*(total_heads+1));
  remaining=sector % (total_sectors*(total_heads+1));
  translated_head=remaining/(total_sectors);
  translated_sector=remaining % (total_sectors);
  translated_sector++;
}

/* Convert Hexidecimal Number to a Decimal Number */
unsigned long Decimal_Number(unsigned long hex1, unsigned long hex2, unsigned long hex3, unsigned long hex4)
{
  return((hex1) + (hex2*256) + (hex3*65536) + (hex4*16777216));
}

/* Determine drive letters */
int Determine_Drive_Letters()
/* Normally returns a 0, returns a 1 if current_letter>='Z' */
{
  int active_found=FALSE;
  int current_letter='C';
  int drive_found=FALSE;
  int index=0;
  int non_dos_partition;
  int non_dos_partition_counter;
  int sub_index=0;

  Load_Brief_Partition_Table();

  /* Clear drive_lettering_buffer[8] [27] */
  do
    {
    sub_index=0;
    do
      {
      drive_lettering_buffer[index] [sub_index]=0;

      sub_index++;
      }while(sub_index<27);

    index++;
    }while(index<8);

  /* Begin placement of drive letters */

  /* First, look for an active primary partition, on the first hard drive... */

  sub_index=0;
  do
    {
    if(part_table[0].active_status[sub_index]==0x80)
      {
      if( (brief_partition_table[0] [sub_index]==1)
       || (brief_partition_table[0] [sub_index]==4)
       || (brief_partition_table[0] [sub_index]==6) )
	{
	drive_lettering_buffer[0] [sub_index]=current_letter;
	current_letter++;
	active_found=TRUE;
	}

      if( (brief_partition_table[0] [sub_index]==0x0e)
       && ( (flags.version==W95) || (flags.version==W95B)
       || (flags.version==W98) ) )
	{
	drive_lettering_buffer[0] [sub_index]=current_letter;
	current_letter++;
	active_found=TRUE;
	}

      if( ( (brief_partition_table[0] [sub_index]==0x0b)
       || (brief_partition_table[0] [sub_index]==0x0c) )
       && ( (flags.version==W95B) || (flags.version==W98) ) )
	{
	drive_lettering_buffer[0] [sub_index]=current_letter;
	current_letter++;
	active_found=TRUE;
	}
      }

    sub_index++;
    }while(sub_index<4);

  /* Next, check all of the primary partitions... */
  index=0;
  if(active_found==TRUE) index=1;
  do
    {
    sub_index=0;
    drive_found=FALSE;

    if(part_table[index].active_status[sub_index]==0x80)
      {
      drive_found=TRUE;
      goto Skip_First_Scan;
      }

    do
      {
      if( (brief_partition_table[index] [sub_index]==1)
       || (brief_partition_table[index] [sub_index]==4)
       || (brief_partition_table[index] [sub_index]==6) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	drive_found=TRUE;
	}

      if( (brief_partition_table[index] [sub_index]==0x0e)
       && ( (flags.version==W95) || (flags.version==W95B)
       || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	drive_found=TRUE;
	}

      if( ( (brief_partition_table[index] [sub_index]==0x0b)
       || (brief_partition_table[index] [sub_index]==0x0c) )
       && ( (flags.version==W95B) || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	drive_found=TRUE;
	}

      sub_index++;
      }while( (sub_index<4) && (drive_found==FALSE) );

    Skip_First_Scan:

    index++;
    }while(index<8);

  /* Next examine the extended partitions... */
  index=0;
  do
    {
    sub_index=4;
    drive_found=FALSE;
    do
      {
      if( (brief_partition_table[index] [sub_index]==1)
       || (brief_partition_table[index] [sub_index]==4)
       || (brief_partition_table[index] [sub_index]==6) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      if( (brief_partition_table[index] [sub_index]==0x0e)
       && ( (flags.version==W95) || (flags.version==W95B)
       || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      if( ( (brief_partition_table[index] [sub_index]==0x0b)
       || (brief_partition_table[index] [sub_index]==0x0c) )
       && ( (flags.version==W95B) || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      sub_index++;
      }while(sub_index<27);

    index++;
    }while(index<8);

  /* Return to the primary partitions... */
  index=0;
  do
    {
    sub_index=0;

    do
      {
      if( (drive_lettering_buffer[index] [sub_index]==0)
       && ( (brief_partition_table[index] [sub_index]==1)
       || (brief_partition_table[index] [sub_index]==4)
       || (brief_partition_table[index] [sub_index]==6) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      if( (drive_lettering_buffer[index] [sub_index]==0)
       && (brief_partition_table[index] [sub_index]==0x0e)
       && ( (flags.version==W95) || (flags.version==W95B)
       || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      if( (drive_lettering_buffer[index] [sub_index]==0)
       && ( (brief_partition_table[index] [sub_index]==0x0b)
       || (brief_partition_table[index] [sub_index]==0x0c) )
       && ( (flags.version==W95B) || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      sub_index++;
      }while(sub_index<4);

    index++;
    }while(index<8);

  /* Find the Non-DOS Logical Drives in the Extended Partition Table */
  non_dos_partition_counter='1';
  index=0;
  do
    {
    part_table[index].num_of_non_dos_log_drives=0;
    sub_index=4;
    drive_found=FALSE;
    do
      {
      if(brief_partition_table[index] [sub_index]>0)
	{
	non_dos_partition=TRUE;

	if( (brief_partition_table[index] [sub_index]==1)
	 || (brief_partition_table[index] [sub_index]==4)
	 || (brief_partition_table[index] [sub_index]==6) )
	  {
	  non_dos_partition=FALSE;
	  }

	if( (brief_partition_table[index] [sub_index]==0x0e)
	 && ( (flags.version==W95) || (flags.version==W95B)
	 || (flags.version==W98) ) )
	  {
	  non_dos_partition=FALSE;
	  }

	if( ( (brief_partition_table[index] [sub_index]==0x0b)
	 || (brief_partition_table[index] [sub_index]==0x0c) )
	 && ( (flags.version==W95B) || (flags.version==W98) ) )
	  {
	  non_dos_partition=FALSE;
	  }

	if( (non_dos_partition==TRUE) && (non_dos_partition_counter<='9') )
	  {
	  drive_lettering_buffer[index] [sub_index]=non_dos_partition_counter;
	  part_table[index].num_of_non_dos_log_drives++;
	  non_dos_partition_counter++;
	  }
	}
      sub_index++;
      }while(sub_index<27);

    non_dos_partition_counter='1';
    index++;
    }while(index<8);

  if(current_letter>'Z') return(1);
  else return(0);
}

/* Extract Cylinder */
unsigned long Extract_Cylinder(unsigned long hex1, unsigned long hex2)
{
  unsigned long cylinder_and_sector = ( (256*hex2) + hex1 );
  unsigned long extracted_cylinder = ( ( (cylinder_and_sector*4) & 768) + (cylinder_and_sector /256) );

  return(extracted_cylinder);
}

/* Extract the Cylinder from an LBA Value */
unsigned long Extract_Cylinder_From_LBA_Value(long double lba_value
 ,long double head,long double sector,long double total_heads
 ,long double total_sectors)
{
  return( ( ( ( (lba_value-(sector-1))/total_sectors)-head)/(total_heads+1) ) );
}

/* Extract Sector */
unsigned long Extract_Sector(unsigned long hex1, unsigned long hex2)
{
  unsigned long cylinder_and_sector = ( (256*hex2) + hex1 );
  unsigned long extracted_sector = cylinder_and_sector % 64;

  return(extracted_sector);
}

/* Get the parameters of the hard disk */
int Get_Hard_Drive_Parameters(int physical_drive)
{
  int error_code=0;

  unsigned int total_number_hard_disks;

  unsigned long total_cylinders=0;
  unsigned long total_heads=0;
  unsigned long total_sectors=0;

  if( (physical_drive-0x80)>=flags.total_number_hard_disks) return(255);

  if(user_defined_chs_settings[(physical_drive-128)].defined==TRUE)
    {
    part_table[(physical_drive-0x80)].total_cyl
     =user_defined_chs_settings[(physical_drive-0x80)].total_cylinders;
    part_table[(physical_drive-0x80)].total_head
     =user_defined_chs_settings[(physical_drive-0x80)].total_heads;
    part_table[(physical_drive-0x80)].total_sect
     =user_defined_chs_settings[(physical_drive-0x80)].total_sectors;
    part_table[(physical_drive-0x80)].num_of_log_drives=0;

    return(0);
    }

  /* Get the hard drive parameters with normal int 0x13 calls. */
  asm{
    mov ah, 0x08
    mov dl, BYTE PTR physical_drive
    int 0x13

    mov bl,cl
    and bl,00111111B

    mov BYTE PTR error_code, ah
    mov BYTE PTR total_sectors, bl
    mov BYTE PTR total_number_hard_disks,dl

    mov bl,cl
    mov cl,ch
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1
    shr bl,1

    mov ch,bl

    mov WORD PTR total_cylinders, cx
    mov BYTE PTR total_heads, dh
    }

  if(flags.total_number_hard_disks==255)
   flags.total_number_hard_disks = total_number_hard_disks;
  if(total_number_hard_disks==0) return(255);


  if(error_code>0) return(error_code);

  part_table[(physical_drive-0x80)].total_head=total_heads;
  part_table[(physical_drive-0x80)].total_sect=total_sectors;
  part_table[(physical_drive-0x80)].num_of_log_drives=0;

  if(part_table[(physical_drive-128)].ext_int_13==TRUE)
    {
    /* Get the hard drive parameters with extended int 0x13 calls. */

    /* Note:  Supported interrupt 0x13 extensions have already been        */
    /* checked for in the function Check_For_INT13_Extensions().           */

    unsigned int result_buffer_segment=FP_SEG(result_buffer);
    unsigned int result_buffer_offset=FP_OFF(result_buffer);

    unsigned long legacy_total_cylinders=total_cylinders;
    unsigned long number_of_physical_sectors;

    asm {
      mov ah,0x48
      mov dl,BYTE PTR physical_drive
      mov ds,result_buffer_segment
      mov si,result_buffer_offset
      int 0x13

      mov BYTE PTR error_code,ah
      }

    if(error_code>0) return(error_code);

    /* Compute the total number of logical cylinders based upon the number */
    /* of physical sectors returned from service 0x48.                     */

    number_of_physical_sectors
     =Decimal_Number(result_buffer[16],result_buffer[17],result_buffer[18]
     ,result_buffer[19]);

    total_cylinders=((number_of_physical_sectors/total_sectors)/(total_heads+1))-1;

    /* If the number of cylinders calculated using service 0x48 is in error,*/
    /* use the total cylinders reported by service 0x08.                    */
    if(legacy_total_cylinders>total_cylinders)
      total_cylinders=legacy_total_cylinders;
    }

  /* Check for an extra cylinder */
  if(0==Read_Physical_Sectors(physical_drive,(total_cylinders+1)
   ,total_heads,total_sectors,1))
   total_cylinders++;

  part_table[(physical_drive-0x80)].total_cyl=total_cylinders;

  return(0);
}

/* Get the volume labels and file system types from the boot sectors */
void Get_Partition_Information()
{
  int index;
  int sub_index;

  long sector_buffer_index;

  /* First get the information from the primary partitions. */
  index=0;
  do
    {
    sub_index=0;
    do
      {
      /* Check for and get the volume label on a FAT12/FAT16 partition. */
      if( ( (part_table[index].pri_part_num_type[sub_index]==1)
       || (part_table[index].pri_part_num_type[sub_index]==4)
       || (part_table[index].pri_part_num_type[sub_index]==6) )
       || ( ( (flags.version==W95) || (flags.version==W95B)
       || (flags.version==W98) )
       && (part_table[index].pri_part_num_type[sub_index]==14) ) )
	{
	Read_Physical_Sectors((index+128)
	 ,part_table[index].pri_part_start_cyl[sub_index]
	 ,part_table[index].pri_part_start_head[sub_index]
	 ,part_table[index].pri_part_start_sect[sub_index],1);

	if( (sector_buffer[53]>=32) && (sector_buffer[53]<=122) )
	  {
	  /* Get Volume Label */
	  sector_buffer_index=43;
	  do
	    {
	    part_table[index].pri_part_vol_label[sub_index]
	     [(sector_buffer_index-43)]=sector_buffer[sector_buffer_index];
	    sector_buffer_index++;
	    }while(sector_buffer_index<=53);
	  }
	}
      else
	{
	strcpy(part_table[index].pri_part_vol_label[sub_index],"           ");
	}

      /* Check for and get the volume label on a FAT32 partition. */
      if( ( (flags.version==W95B) || (flags.version==W98) )
       && ( (part_table[index].pri_part_num_type[sub_index]==11)
       || (part_table[index].pri_part_num_type[sub_index]==12) ) )
	{
	Read_Physical_Sectors((index+128)
	 ,part_table[index].pri_part_start_cyl[sub_index]
	 ,part_table[index].pri_part_start_head[sub_index]
	 ,part_table[index].pri_part_start_sect[sub_index],1);

	if( (sector_buffer[81]>=32) && (sector_buffer[81]<=122) )
	  {
	  /* Get Volume Label */
	  sector_buffer_index=71;
	  do
	    {
	    part_table[index].pri_part_vol_label[sub_index]
	     [(sector_buffer_index-71)]=sector_buffer[sector_buffer_index];
	    sector_buffer_index++;
	    }while(sector_buffer_index<=81);
	  }
	}

      sub_index++;
      }while(sub_index<4);

    index++;
    }while(index<8);

  /* Get the information from the extended partitions. */
  index=0;
  do
    {
    sub_index=0;
    do
      {
      if( ( (part_table[index].log_drive_num_type[sub_index]==1)
       || (part_table[index].log_drive_num_type[sub_index]==4)
       || (part_table[index].log_drive_num_type[sub_index]==6) )
       || ( ( (flags.version==W95) || (flags.version==W95B)
       || (flags.version==W98) )
       && (part_table[index].log_drive_num_type[sub_index]==14) ) )
	{
	Read_Physical_Sectors((index+128)
	 ,part_table[index].log_drive_start_cyl[sub_index]
	 ,part_table[index].log_drive_start_head[sub_index]
	 ,part_table[index].log_drive_start_sect[sub_index],1);

	if( (sector_buffer[53]>=32) && (sector_buffer[53]<=122) )
	  {
	  /* Get Volume Label */
	  sector_buffer_index=43;
	  do
	    {
	    part_table[index].log_drive_vol_label[sub_index]
	     [(sector_buffer_index-43)]=sector_buffer[sector_buffer_index];
	    sector_buffer_index++;
	    }while(sector_buffer_index<=53);

	  }
	}
      else
	{
	strcpy(part_table[index].log_drive_vol_label[sub_index],"           ");
	}

      /* Check for and get the volume label on a FAT32 partition. */
      if( ( (flags.version==W95B) || (flags.version==W98) )
       && ( (part_table[index].log_drive_num_type[sub_index]==11)
       || (part_table[index].log_drive_num_type[sub_index]==12) ) )
	{
	Read_Physical_Sectors((index+128)
	 ,part_table[index].log_drive_start_cyl[sub_index]
	 ,part_table[index].log_drive_start_head[sub_index]
	 ,part_table[index].log_drive_start_sect[sub_index],1);

	if( (sector_buffer[81]>=32) && (sector_buffer[81]<=122) )
	  {
	  /* Get Volume Label */
	  sector_buffer_index=71;
	  do
	    {
	    part_table[index].log_drive_vol_label[sub_index]
	     [(sector_buffer_index-71)]=sector_buffer[sector_buffer_index];
	    sector_buffer_index++;
	    }while(sector_buffer_index<=81);
	  }
	}

      sub_index++;
      }while(sub_index<23);

    index++;
    }while(index<8);
}

/* Initialize the LBA Structures */
void Initialize_LBA_Structures()
{
int index;

  /* Initialize the Disk Address Packet */
  /* ---------------------------------- */

  /* Packet size = 16 */
  disk_address_packet[0]=16;

  /* Reserved, must be 0 */
  disk_address_packet[1]=0;

  /* Number of blocks to transfer = 1 */
  disk_address_packet[2]=1;

  /* Reserved, must be 0 */
  disk_address_packet[3]=0;

  /* Zero out the offsets 8-11 to clear the LBA address field. */
  index=8;
  do
    {
    disk_address_packet[index]=0;
    index++;
    }while(index<12);

  /* Zero out offsets 12-15 because they are currently unused. They are  */
  /* part of the 64 bit LBA address...at the present time only the lower */
  /* 28 bits are used.                                                   */
  index=12;
  do
    {
    disk_address_packet[index]=0;
    index++;
    }while(index<16);

  /* Initialize the Result Buffer */
  /* ---------------------------- */

  /* Buffer Size = 26 */
  result_buffer[0]=26;
}

/* Load the brief_partition_table[8] [27] */
void Load_Brief_Partition_Table()
{
  int index=0;
  int sub_index=0;

  /* Clear brief_partition_table[8] [27] */
  index=0;
  do
    {
    sub_index=0;
    do
      {
      brief_partition_table[index] [sub_index]=0;

      sub_index++;
      }while(sub_index<27);
    index++;
    }while(index<8);

  index=0;
  do
    {
    /* Load the primary partitions into brief_partition_table[8] [27] */
    sub_index=0;
    do
      {
      brief_partition_table[index] [sub_index]=part_table[index].pri_part_num_type[sub_index];

      sub_index++;
      }while(sub_index<4);

    /* Load the extended partitions into brief_partition_table[8] [27] */
    sub_index=0;
    do
      {
      brief_partition_table[index] [(sub_index+4)]=part_table[index].log_drive_num_type[sub_index];

      sub_index++;
      }while(sub_index<23);

    index++;
    }while(index<8);
}

/* Read Logical Sector Function */
int Read_Logical_Sectors(unsigned char drive_letter[2]
 , unsigned long logical_sector_number, int number_of_sectors)
{
  /* Returns 0 if successful and 255 if the drive is not found. */
  /* Otherwise, returns the error code from the BIOS.           */

  int index=0;
  int part_num=255;
  int physical_drive=255;
  int sub_index=0;

  /* Ensure that drive_letter[0] is upper case. */
  if(drive_letter[0]>='a') drive_letter[0]=drive_letter[0]-32;

  /* Determine where the logical drive is located */
  Determine_Drive_Letters();

  do
    {
    sub_index=0;
    do
      {
      if(drive_letter[0]==drive_lettering_buffer[index][sub_index])
	{
	physical_drive=index;
	part_num=sub_index;
	}

      sub_index++;
      }while(sub_index<27);

    index++;
    }while(index<8);

  if(physical_drive==255) return(255);

  /* Compute the physical starting cylinders, heads, and sectors. */
  if(part_num<4)
    {
    Convert_Logical_To_Physical( ((Translate_CHS_To_LBA(
     part_table[physical_drive].pri_part_start_cyl[part_num]
     ,part_table[physical_drive].pri_part_start_head[part_num]
     ,part_table[physical_drive].pri_part_start_sect[part_num]
     ,part_table[physical_drive].total_head
     ,part_table[physical_drive].total_sect))+logical_sector_number)
     ,part_table[physical_drive].total_head
     ,part_table[physical_drive].total_sect);
    }
  else
    {
    part_num=part_num-4;

    Convert_Logical_To_Physical( ((Translate_CHS_To_LBA(
     part_table[physical_drive].log_drive_start_cyl[part_num]
     ,part_table[physical_drive].log_drive_start_head[part_num]
     ,part_table[physical_drive].log_drive_start_sect[part_num]
     ,part_table[physical_drive].total_head
     ,part_table[physical_drive].total_sect))+logical_sector_number)
     ,part_table[physical_drive].total_head
     ,part_table[physical_drive].total_sect);
    }

  /* Write the sector(s). */
  return(Read_Physical_Sectors((physical_drive+0x80),translated_cylinder,translated_head,translated_sector,number_of_sectors));
}

/* Load the Partition Tables and get information on all drives */
int Read_Partition_Tables()
{
  int drive=0;
  int error_code=0;
  int index;
  int sub_index;

  long entry_offset;

  int physical_drive=0x80;

  flags.more_than_one_drive=FALSE;

  do
    {
    /* Get the hard drive parameters and ensure that the drive exists. */
    error_code=Get_Hard_Drive_Parameters(physical_drive);

    /* If there was an error accessing the drive, skip that drive. */
    /* If this drive is emulated, then load the emulation values instead. */
    if( (error_code==0)
#ifdef DEBUG
     && (debug.emulate_disk!=(drive+1) )
#endif
     )
      {
      /* Pre-compute the total size of the hard drive */
      /* */
      part_table[drive].total_hard_disk_size_in_log_sect
       =(part_table[drive].total_cyl+1)
       *(part_table[drive].total_head+1)
       *part_table[drive].total_sect;
      part_table[drive].total_hard_disk_size_in_MB
       =part_table[drive].total_hard_disk_size_in_log_sect/2048;
      }
    else
      {
#ifdef DEBUG
      if(debug.emulate_disk==(drive+1) )
	{
	/* If this is an emulated drive, set it up. */
	if( (flags.version==FOUR)
	 || (flags.version==FIVE)
	 || (flags.version==SIX) )
	  {
#pragma warn -ccc
#pragma warn -rch
	  if(EMULATED_CYLINDERS>1023)
	   part_table[drive].total_cyl=1023;
#pragma warn +ccc
#pragma warn +rch
	  }
	else
	  {
	  part_table[drive].total_cyl=EMULATED_CYLINDERS;
	  }

	part_table[drive].total_head=EMULATED_HEADS;
	part_table[drive].total_sect=EMULATED_SECTORS;

	part_table[drive].total_hard_disk_size_in_log_sectors
	 =(part_table[drive].total_cyl+1)
	 *(part_table[drive].total_head+1)
	 *part_table[drive].total_sect;

	part_table[drive].total_hard_disk_size_in_MB
	 =part_table[drive].total_hard_disk_size_in_log_sectors/2048;

	flags.maximum_drive_number=drive+128;

	if(drive>0) flags.more_than_one_drive=TRUE;
	}
      else
	{
#endif
	if(drive==0)
	  {
	  cprintf("\n    No fixed disks present.\n");
	  exit(6);
	  }
	part_table[drive].total_cyl=0;
	part_table[drive].total_head=0;
	part_table[drive].total_sect=0;
#ifdef DEBUG
	}
#endif
      }

    /* Clear the partition_table_structure structure. */
    part_table[drive].pri_part_largest_free_space=0;

    part_table[drive].pp_largest_free_space_start_cyl=0;
    part_table[drive].pp_largest_free_space_start_head=0;
    part_table[drive].pp_largest_free_space_start_sect=0;

    part_table[drive].pp_largest_free_space_end_cyl=0;

    index=0;
    do
      {
      part_table[drive].active_status[index]=0;
      part_table[drive].pri_part_num_type[index]=0;

      part_table[drive].pri_part_start_cyl[index]=0;
      part_table[drive].pri_part_start_head[index]=0;
      part_table[drive].pri_part_start_sect[index]=0;

      part_table[drive].pri_part_end_cyl[index]=0;
      part_table[drive].pri_part_end_head[index]=0;
      part_table[drive].pri_part_end_sect[index]=0;

      part_table[drive].pri_part_rel_sect[index]=0;
      /* */
      part_table[drive].pri_part_num_sect[index]=0;

      part_table[drive].pri_part_size_in_MB[index]=0;

      part_table[drive].pri_part_physical_order[index]=index;
      part_table[drive].pri_part_created[index]=FALSE;

      index++;
      }while(index<4);

    Clear_Extended_Partition_Table(drive);

    /* Read the Primary Partition Table. */
    if(error_code==0)
      {
      error_code=Read_Physical_Sectors(physical_drive,0,0,1,1);

      if(error_code!=0) return(error_code);

      flags.maximum_drive_number=drive+128;

      if(drive>0) flags.more_than_one_drive=TRUE;

      index=0;
      do
	{
	entry_offset=0x1be+(index*16);

	part_table[drive].active_status[index]=sector_buffer[(entry_offset+0x00)];

	part_table[drive].pri_part_num_type[index]=sector_buffer[(entry_offset+0x04)];

	if(part_table[drive].ext_int_13==FALSE)
	  {
	  /* If int 0x13 extensions are not used get the CHS values. */
	  part_table[drive].pri_part_start_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);
	  part_table[drive].pri_part_start_head[index]=sector_buffer[(entry_offset+0x01)];
	  part_table[drive].pri_part_start_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);

	  part_table[drive].pri_part_end_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
	  part_table[drive].pri_part_end_head[index]=sector_buffer[(entry_offset+0x05)];
	  part_table[drive].pri_part_end_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
	  }

	part_table[drive].pri_part_rel_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x08)],sector_buffer[(entry_offset+0x09)],sector_buffer[(entry_offset+0x0a)],sector_buffer[(entry_offset+0x0b)]);
	part_table[drive].pri_part_num_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x0c)],sector_buffer[(entry_offset+0x0d)],sector_buffer[(entry_offset+0x0e)],sector_buffer[(entry_offset+0x0f)]);

	if( (part_table[drive].ext_int_13==TRUE)
	 && (part_table[drive].pri_part_num_type[index]!=0) )
	  {
	  /* If int 0x13 extensions are used compute the virtual CHS values. */

	  /* The head number is 0 unless the cyl is 0...then the head is 1. */
	  if(part_table[drive].pri_part_rel_sect[index]==part_table[drive].total_sect)
	    {
	    part_table[drive].pri_part_start_head[index]=1;
	    }
	  else part_table[drive].pri_part_start_head[index]=0;
	  part_table[drive].pri_part_start_sect[index]=1;
	  part_table[drive].pri_part_start_cyl[index]
	   =Extract_Cylinder_From_LBA_Value(
	   part_table[drive].pri_part_rel_sect[index]
	   ,part_table[drive].pri_part_start_head[index]
	   ,part_table[drive].pri_part_start_sect[index]
	   ,part_table[drive].total_head
	   ,part_table[drive].total_sect);

	  part_table[drive].pri_part_end_head[index]=part_table[drive].total_head;
	  part_table[drive].pri_part_end_sect[index]=part_table[drive].total_sect;
	  part_table[drive].pri_part_end_cyl[index]
	   =Extract_Cylinder_From_LBA_Value(
	   /* */
	   (part_table[drive].pri_part_rel_sect[index]
	   +part_table[drive].pri_part_num_sect[index])
	   ,part_table[drive].pri_part_end_head[index]
	   ,part_table[drive].pri_part_end_sect[index]
	   ,part_table[drive].total_head
	   ,part_table[drive].total_sect);
	  }

	part_table[drive].pri_part_size_in_MB[index]=(part_table[drive].pri_part_num_sect[index]/2048);

	/* Record the necessary information to easilly and quickly find the */
	/* extended partition when it is time to read it.                   */
	if( (part_table[drive].pri_part_num_type[index]==0x05)
	 || ( (part_table[drive].pri_part_num_type[index]==0x0f)
	 && ( (flags.version==W95) || (flags.version==W95B)
	 || (flags.version==W98) ) ) )
	  {
	  part_table[drive].ext_part_exists=TRUE;
	  part_table[drive].num_of_ext_part=index;
	  part_table[drive].ext_part_num_sect=part_table[drive].pri_part_num_sect[index];
	  part_table[drive].ext_part_size_in_MB=part_table[drive].pri_part_size_in_MB[index];
	  }

	index++;
	}while(index<4);

      /* Read the Extended Partition Table, if applicable. */
      if(part_table[drive].ext_part_exists==TRUE)
	{
	error_code=Read_Physical_Sectors(
	 physical_drive,part_table[drive]
	 .pri_part_start_cyl[part_table[drive].num_of_ext_part]
	 ,part_table[drive].pri_part_start_head[part_table[drive]
	 .num_of_ext_part],part_table[drive]
	 .pri_part_start_sect[part_table[drive].num_of_ext_part],1);

	if(error_code!=0) return(error_code);

	/* Ensure that the sector has a valid partition table before        */
	/* any information is loaded into part_table[drive].           */
	if( (sector_buffer[0x1fe]==0x55) && (sector_buffer[0x1ff]==0xaa) )
	  {
	  index=0;
	  do
	    {
	    entry_offset=0x1be;

	    if(sector_buffer[(entry_offset+0x04)]>0)
	     part_table[drive].num_of_log_drives++;

	    part_table[drive].log_drive_num_type[index]
	     =sector_buffer[(entry_offset+0x04)];

	    if(part_table[drive].ext_int_13==FALSE)
	      {
	      /* If int 0x13 extensions are not used get the CHS values. */

	      part_table[drive].log_drive_start_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);
	      part_table[drive].log_drive_start_head[index]=sector_buffer[(entry_offset+0x01)];
	      part_table[drive].log_drive_start_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);

	      part_table[drive].log_drive_end_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
	      part_table[drive].log_drive_end_head[index]=sector_buffer[(entry_offset+0x05)];
	      part_table[drive].log_drive_end_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
	      }

	    part_table[drive].log_drive_rel_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x08)],sector_buffer[(entry_offset+0x09)],sector_buffer[(entry_offset+0x0a)],sector_buffer[(entry_offset+0x0b)]);
	    part_table[drive].log_drive_num_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x0c)],sector_buffer[(entry_offset+0x0d)],sector_buffer[(entry_offset+0x0e)],sector_buffer[(entry_offset+0x0f)]);

	    if( (part_table[drive].ext_int_13==TRUE)
	     && (part_table[drive].log_drive_num_type[index]!=0) )
	      {
	      /* If int 0x13 extensions are used compute the virtual CHS values. */

	      /* The head number is 0 unless the cyl is 0...then the head is 1. */
	      if(part_table[drive].log_drive_rel_sect[index]==part_table[drive].total_sect)
		{
		part_table[drive].log_drive_start_head[index]=1;
		}
	      else part_table[drive].log_drive_start_head[index]=0;
	      part_table[drive].log_drive_start_sect[index]=1;

	      if(index==0)
		part_table[drive].log_drive_start_cyl[index]
		 =Extract_Cylinder_From_LBA_Value(
		 (part_table[drive].log_drive_rel_sect[index]
		 +part_table[drive].pri_part_rel_sect[part_table
		 [drive].num_of_ext_part])
		 ,part_table[drive].log_drive_start_head[index]
		 ,part_table[drive].log_drive_start_sect[index]
		 ,part_table[drive].total_head
		 ,part_table[drive].total_sect);
	      else
		part_table[drive].log_drive_start_cyl[index]
		 =Extract_Cylinder_From_LBA_Value(
		 (part_table[drive].log_drive_rel_sect[index]
		 +part_table[drive].pri_part_rel_sect[part_table[drive]
		 .num_of_ext_part]
		 +part_table[drive].next_ext_rel_sect[index-1])
		 ,part_table[drive].log_drive_start_head[index]
		 ,part_table[drive].log_drive_start_sect[index]
		 ,part_table[drive].total_head
		 ,part_table[drive].total_sect);


	      part_table[drive].log_drive_end_head[index]=part_table[drive].total_head;
	      part_table[drive].log_drive_end_sect[index]=part_table[drive].total_sect;

	      if(index==0)
		part_table[drive].log_drive_end_cyl[index]
		 =Extract_Cylinder_From_LBA_Value(
		 (part_table[drive].log_drive_rel_sect[index]
		 +part_table[drive].pri_part_rel_sect
		 [part_table[drive].num_of_ext_part]
		 +part_table[drive].log_drive_num_sect[index])
		 ,part_table[drive].log_drive_end_head[index]
		 ,part_table[drive].log_drive_end_sect[index]
		 ,part_table[drive].total_head
		 ,part_table[drive].total_sect);
	      else
		part_table[drive].log_drive_end_cyl[index]
		 =Extract_Cylinder_From_LBA_Value(
		 (part_table[drive].log_drive_rel_sect[index]
		 +part_table[drive].pri_part_rel_sect
		 [part_table[drive].num_of_ext_part]
		 +part_table[drive].log_drive_num_sect[index]
		 +part_table[drive].next_ext_rel_sect[index-1])
		 ,part_table[drive].log_drive_end_head[index]
		 ,part_table[drive].log_drive_end_sect[index]
		 ,part_table[drive].total_head
		 ,part_table[drive].total_sect);
	      }

	    part_table[drive].log_drive_size_in_MB[index]
	     =part_table[drive].log_drive_num_sect[index]/2048;

	    entry_offset=entry_offset+16;
	    if(sector_buffer[(entry_offset+0x04)]==0x05)
	      {
	      part_table[drive].next_ext_exists[index]=TRUE;

	      part_table[drive].next_ext_num_type[index]=sector_buffer[(entry_offset+0x04)];

	      if(part_table[drive].ext_int_13==FALSE)
		{
		/* If int 0x13 extensions are not used get the CHS values. */

		part_table[drive].next_ext_start_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);
		part_table[drive].next_ext_start_head[index]=sector_buffer[(entry_offset+0x01)];
		part_table[drive].next_ext_start_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);

		part_table[drive].next_ext_end_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
		part_table[drive].next_ext_end_head[index]=sector_buffer[(entry_offset+0x05)];
		part_table[drive].next_ext_end_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
		}

	      part_table[drive].next_ext_rel_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x08)],sector_buffer[(entry_offset+0x09)],sector_buffer[(entry_offset+0x0a)],sector_buffer[(entry_offset+0x0b)]);
	      part_table[drive].next_ext_num_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x0c)],sector_buffer[(entry_offset+0x0d)],sector_buffer[(entry_offset+0x0e)],sector_buffer[(entry_offset+0x0f)]);

	    if( (part_table[drive].ext_int_13==TRUE)
	     && (part_table[drive].next_ext_num_type[index]!=0) )
	      {
	      /* If int 0x13 extensions are used compute the virtual CHS values. */

	      part_table[drive].next_ext_start_head[index]=0;
	      part_table[drive].next_ext_start_sect[index]=1;
	      /* */
	      part_table[drive].next_ext_start_cyl[index]
	       =Extract_Cylinder_From_LBA_Value(
	       (part_table[drive].next_ext_rel_sect[index]
	       +part_table[drive].pri_part_rel_sect
	       [part_table[drive].num_of_ext_part])
	       ,part_table[drive].next_ext_start_head[index]
	       ,part_table[drive].next_ext_start_sect[index]
	       ,part_table[drive].total_head
	       ,part_table[drive].total_sect);

	      part_table[drive].next_ext_end_head[index]=part_table[drive].total_head;
	      part_table[drive].next_ext_end_sect[index]=part_table[drive].total_sect;
	      part_table[drive].next_ext_end_cyl[index]
	       =Extract_Cylinder_From_LBA_Value(
	       (part_table[drive].next_ext_rel_sect[index]
	       +part_table[drive].pri_part_rel_sect
	       [part_table[drive].num_of_ext_part]
	       +part_table[drive].next_ext_num_sect[index])
	       ,part_table[drive].next_ext_end_head[index]
	       ,part_table[drive].next_ext_end_sect[index]
	       ,part_table[drive].total_head
	       ,part_table[drive].total_sect);
	      }

	      error_code=Read_Physical_Sectors(physical_drive
	       ,part_table[drive].next_ext_start_cyl[index]
	       ,part_table[drive].next_ext_start_head[index]
	       ,part_table[drive].next_ext_start_sect[index],1);

	      if(error_code!=0) return(error_code);
	      }
	    else index=24;

	    index++;
	    }while(index<24);
	  }
	}
     }
  drive++;
  physical_drive=drive+0x80;
  }while(drive<8);

  Determine_Drive_Letters();

  Get_Partition_Information();
  return(0);
}

/* Read_Physical_Sector */
int Read_Physical_Sectors(int drive, long cylinder, long head
 , long sector, int number_of_sectors)
{
  int current_line;
  int error_code;
  int index;
  int offset;

  number_of_sectors=1;

  if(flags.use_extended_int_13==FALSE)
    {
    error_code=Read_Physical_Sectors_CHS(drive,cylinder,head,sector,number_of_sectors);
    }
  else
    {
    error_code=Read_Physical_Sectors_LBA(drive,cylinder,head,sector,number_of_sectors);
    }

#ifdef DEBUG
  if(debug.read_sector==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(4,"Read_Physical_Sector() function debugging screen",BOLD);

    Position_Cursor(4,10);
    printf("Information passed to this function:");

    Position_Cursor(50,11);
    printf("Drive:     0x%X",drive);
    Position_Cursor(50,12);
    printf("Cylinder:  %d",cylinder);
    Position_Cursor(50,13);
    printf("Head:      %d",head);
    Position_Cursor(50,14);
    printf("Sector:    %d",sector);
    Position_Cursor(4,16);
    printf("Contents of partition table area in sector_buffer[]:");

    current_line=0;
    offset=0x1be;
    do
      {
      index=0;

      Position_Cursor(4,(current_line+18));
      printf("%d:  ",(current_line+1));
      do
	{
	printf("%02X ",sector_buffer[(index+offset)]);
	index++;
	}while(index<16);

      current_line++;
      offset=offset+16;
      }while(offset<(0x1be+64));

    Position_Cursor(4,23);
    printf("Press any key to continue.");

    asm{
      mov ah,7
      int 0x21
      }
    }
#endif

  return(error_code);
}

/* Read physical sector using CHS values */
int Read_Physical_Sectors_CHS(int drive, long cylinder, long head
 , long sector, int number_of_sectors)
{
  int error_code;

  if(number_of_sectors==1)
    {
    error_code=biosdisk(2, drive, head, cylinder, sector, number_of_sectors, sector_buffer);
    }
  else
    {
    error_code=biosdisk(2, drive, head, cylinder, sector, number_of_sectors, huge_sector_buffer);
    }

  return(error_code);
}

/* Read a physical sector using LBA values */
int Read_Physical_Sectors_LBA(int drive, long cylinder, long head
 , long sector, int number_of_sectors)
{
  unsigned int error_code=0;

  /* Get the segment and offset of disk_address_packet. */
  unsigned int disk_address_packet_address_segment=FP_SEG(disk_address_packet);
  unsigned int disk_address_packet_address_offset=FP_OFF(disk_address_packet);

  unsigned long LBA_address_high;
  unsigned long LBA_address_low;

  /* Translate CHS values to LBA values. */
  unsigned long LBA_address=Translate_CHS_To_LBA(cylinder,head,sector
  ,part_table[(drive-128)].total_head
  ,part_table[(drive-128)].total_sect);

  /* Add number_of_sectors to disk_address_packet */
  disk_address_packet[2]=number_of_sectors;

  if(number_of_sectors==1)
    {
    Convert_Long_To_Integer(FP_OFF(sector_buffer));
    disk_address_packet[4]=integer1;
    disk_address_packet[5]=integer2;

    Convert_Long_To_Integer(FP_SEG(sector_buffer));
    disk_address_packet[6]=integer1;
    disk_address_packet[7]=integer2;
    }
  else
    {
    Convert_Long_To_Integer(FP_OFF(huge_sector_buffer));
    disk_address_packet[4]=integer1;
    disk_address_packet[5]=integer2;

    Convert_Long_To_Integer(FP_SEG(huge_sector_buffer));
    disk_address_packet[6]=integer1;
    disk_address_packet[7]=integer2;
    }

  /* Transfer LBA_address to disk_address_packet */
  LBA_address_high=(LBA_address&0xffff0000)>>16;
  LBA_address_low=LBA_address&0x0000ffff;

  Convert_Long_To_Integer(LBA_address_low);
  disk_address_packet[8]=integer1;
  disk_address_packet[9]=integer2;

  Convert_Long_To_Integer(LBA_address_high);
  disk_address_packet[10]=integer1;
  disk_address_packet[11]=integer2;

  /* Load the registers and call the interrupt. */
  asm {
    mov ah,0x42
    mov dl,BYTE PTR drive
    mov ds,disk_address_packet_address_segment
    mov si,disk_address_packet_address_offset
    int 0x13

    mov BYTE PTR error_code,ah
    }

  return(error_code);
}

/* Translate a CHS value to an LBA value. */
long double Translate_CHS_To_LBA(unsigned long cylinder,unsigned long head
 ,unsigned long sector,unsigned long total_heads,unsigned long total_sectors)
{
  return( ( (cylinder*(total_heads+1)+head)*total_sectors+sector-1) );
}

/* Write Logical Sector Function */
int Write_Logical_Sectors(unsigned char drive_letter[2]
 , unsigned long logical_sector_number, int number_of_sectors)
{
  /* Returns 0 if successful and 255 if the drive is not found. */
  /* Otherwise, returns the error code from the BIOS.           */

  int index=0;
  int part_num=255;
  int physical_drive=255;
  int sub_index=0;

  /* Ensure that drive_letter[0] is upper case. */
  if(drive_letter[0]>='a') drive_letter[0]=drive_letter[0]-32;

  /* Determine where the logical drive is located */
  Determine_Drive_Letters();

  do
    {
    sub_index=0;
    do
      {
      if(drive_letter[0]==drive_lettering_buffer[index][sub_index])
	{
	physical_drive=index;
	part_num=sub_index;
	}

      sub_index++;
      }while(sub_index<27);

    index++;
    }while(index<8);

  if(physical_drive==255) return(255);

  /* Compute the physical starting cylinders, heads, and sectors. */
  if(part_num<4)
    {
    Convert_Logical_To_Physical( ((Translate_CHS_To_LBA(
     part_table[physical_drive].pri_part_start_cyl[part_num]
     ,part_table[physical_drive].pri_part_start_head[part_num]
     ,part_table[physical_drive].pri_part_start_sect[part_num]
     ,part_table[physical_drive].total_head
     ,part_table[physical_drive].total_sect))+logical_sector_number)
     ,part_table[physical_drive].total_head
     ,part_table[physical_drive].total_sect);
    }
  else
    {
    part_num=part_num-4;

    Convert_Logical_To_Physical( ((Translate_CHS_To_LBA(
     part_table[physical_drive].log_drive_start_cyl[part_num]
     ,part_table[physical_drive].log_drive_start_head[part_num]
     ,part_table[physical_drive].log_drive_start_sect[part_num]
     ,part_table[physical_drive].total_head
     ,part_table[physical_drive].total_sect))+logical_sector_number)
     ,part_table[physical_drive].total_head
     ,part_table[physical_drive].total_sect);
    }

  /* Write the sector(s). */
  return(Write_Physical_Sectors((physical_drive+0x80),translated_cylinder,translated_head,translated_sector,number_of_sectors));
}

/* Write partition tables */
int Write_Partition_Tables()
{
  int error_code;
  int index;

  int drive_index=0;

  long entry_offset;

  unsigned long high;
  unsigned long low;
  unsigned long temp;

  long extended_cylinder;
  long extended_head;
  long extended_sector;

  long next_extended_cylinder;
  long next_extended_head;
  long next_extended_sector;

  do
    {
    if(part_table[drive_index].part_values_changed==TRUE)
      {
      index=0;

      Clear_Sector_Buffer();

#ifdef DEBUG
      if(debug.write==TRUE)
#endif
       error_code=Read_Physical_Sectors((drive_index+128),0,0,1,1);
#ifdef DEBUG
      else error_code=0;
#endif
      if(error_code!=0) return(error_code);

      Clear_Partition_Table_Area_Of_Sector_Buffer();

      do
	{
	/* If this partition was just created, clear its boot sector. */
	if(part_table[drive_index].pri_part_created[index]==TRUE)
	  {
	  Clear_Boot_Sector((drive_index+128),part_table[drive_index].pri_part_start_cyl[index],part_table[drive_index].pri_part_start_head[index],part_table[drive_index].pri_part_start_sect[index]);
	  }

	if( (part_table[drive_index]
	 .pri_part_num_type[index]==0x05)
	 || (part_table[drive_index]
	 .pri_part_num_type[index]==0x0f) )

	  {
	  extended_cylinder=part_table[drive_index].pri_part_start_cyl[index];
	  extended_head=part_table[drive_index].pri_part_start_head[index];
	  extended_sector=part_table[drive_index].pri_part_start_sect[index];
	  }

	entry_offset=0x1be+(index*16);

	sector_buffer[(entry_offset+0x00)]=part_table[drive_index].active_status[index];

	sector_buffer[(entry_offset+0x01)]=part_table[drive_index].pri_part_start_head[index];

	Convert_Long_To_Integer(Combine_Cylinder_and_Sector
	 (part_table[drive_index]
	 .pri_part_start_cyl[index]
	 ,part_table[drive_index]
	 .pri_part_start_sect[index]));

	sector_buffer[(entry_offset+0x02)]=integer1;
	sector_buffer[(entry_offset+0x03)]=integer2;

	sector_buffer[(entry_offset+0x04)]=part_table[drive_index].pri_part_num_type[index];

	sector_buffer[(entry_offset+0x05)]
	 =part_table[drive_index].pri_part_end_head[index];

	Convert_Long_To_Integer(Combine_Cylinder_and_Sector
	 (part_table[drive_index]
	 .pri_part_end_cyl[index]
	 ,part_table[drive_index]
	 .pri_part_end_sect[index]));

	sector_buffer[(entry_offset+0x06)]=integer1;
	sector_buffer[(entry_offset+0x07)]=integer2;

	high = part_table[drive_index].pri_part_rel_sect[index] >> 16;
	temp=high << 16;

	if(part_table[drive_index].pri_part_rel_sect[index] > 0xffff)
	  {
	  low = part_table[drive_index].pri_part_rel_sect[index] - temp;
	  }
	else
	  {
	  low = part_table[drive_index].pri_part_rel_sect[index];
	  }

	Convert_Long_To_Integer(low);

	sector_buffer[entry_offset+0x08]=integer1;
	sector_buffer[entry_offset+0x09]=integer2;

	Convert_Long_To_Integer(high);

	sector_buffer[entry_offset+0x0a]=integer1;
	sector_buffer[entry_offset+0x0b]=integer2;

	high = part_table[drive_index].pri_part_num_sect[index] >> 16;
	temp=high << 16;

	if(part_table[drive_index].pri_part_num_sect[index] > 0xffff)
	  {
	  low = part_table[drive_index].pri_part_num_sect[index] - temp;
	  }
	else
	  {
	  low = part_table[drive_index].pri_part_num_sect[index];
	  }

	Convert_Long_To_Integer(low);

	sector_buffer[entry_offset+0x0c]=integer1;
	sector_buffer[entry_offset+0x0d]=integer2;

	Convert_Long_To_Integer(high);

	sector_buffer[entry_offset+0x0e]=integer1;
	sector_buffer[entry_offset+0x0f]=integer2;

	index++;
	}while(index<4);

      /* Add the partition table marker values */
      sector_buffer[0x1fe]=0x55;
      sector_buffer[0x1ff]=0xaa;

      error_code=Write_Physical_Sectors((drive_index+128),0,0,1,1);
      if(error_code>0) return(error_code);

      /* Write the Extended Partition Table, if applicable. */

      if(part_table[drive_index].ext_part_exists==TRUE)
	{
	index=0;
	do
	  {
	  /* If this logical drive was just created, clear its boot sector. */
	  if(part_table[drive_index].log_drive_created[index]==TRUE)
	    {
	    Clear_Boot_Sector((drive_index+128),part_table[drive_index].log_drive_start_cyl[index],part_table[drive_index].log_drive_start_head[index],part_table[drive_index].log_drive_start_sect[index]);
	    }

	  Clear_Sector_Buffer();

	  entry_offset=0x1be;

	  /* Add the partition table marker values */
	  sector_buffer[0x1fe]=0x55;
	  sector_buffer[0x1ff]=0xaa;

	  sector_buffer[(entry_offset+0x01)]=part_table[drive_index].log_drive_start_head[index];

	  Convert_Long_To_Integer(Combine_Cylinder_and_Sector(part_table[drive_index].log_drive_start_cyl[index],part_table[drive_index].log_drive_start_sect[index]));
	  sector_buffer[(entry_offset+0x02)]=integer1;
	  sector_buffer[(entry_offset+0x03)]=integer2;

	  sector_buffer[(entry_offset+0x04)]=part_table[drive_index].log_drive_num_type[index];

	  sector_buffer[(entry_offset+0x05)]=part_table[drive_index].log_drive_end_head[index];

	  Convert_Long_To_Integer(Combine_Cylinder_and_Sector(part_table[drive_index].log_drive_end_cyl[index],part_table[drive_index].log_drive_end_sect[index]));
	  sector_buffer[(entry_offset+0x06)]=integer1;
	  sector_buffer[(entry_offset+0x07)]=integer2;

	  high = part_table[drive_index].log_drive_rel_sect[index] >> 16;
	  temp=high << 16;

	  if(part_table[drive_index].log_drive_rel_sect[index] > 0xffff)
	    {
	    low = part_table[drive_index].log_drive_rel_sect[index] - temp;
	    }
	  else
	    {
	    low = part_table[drive_index].log_drive_rel_sect[index];
	    }

	  Convert_Long_To_Integer(low);

	  sector_buffer[entry_offset+0x08]=integer1;
	  sector_buffer[entry_offset+0x09]=integer2;

	  Convert_Long_To_Integer(high);

	  sector_buffer[entry_offset+0x0a]=integer1;
	  sector_buffer[entry_offset+0x0b]=integer2;

	  high = part_table[drive_index].log_drive_num_sect[index] >> 16;
	  temp=high << 16;

	  if(part_table[drive_index].log_drive_num_sect[index] > 0xffff)
	    {
	    low = part_table[drive_index].log_drive_num_sect[index] - temp;
	    }
	  else
	    {
	    low = part_table[drive_index].log_drive_num_sect[index];
	    }

	  Convert_Long_To_Integer(low);

	  sector_buffer[entry_offset+0x0c]=integer1;
	  sector_buffer[entry_offset+0x0d]=integer2;

	  Convert_Long_To_Integer(high);

	  sector_buffer[entry_offset+0x0e]=integer1;
	  sector_buffer[entry_offset+0x0f]=integer2;

	  if(part_table[drive_index].next_ext_exists[index]==TRUE)
	    {
	    next_extended_cylinder=part_table[drive_index]
	     .next_ext_start_cyl[index];

	    next_extended_head=part_table[drive_index]
	     .next_ext_start_head[index];

	    next_extended_sector=part_table[drive_index]
	     .next_ext_start_sect[index];

	    entry_offset=entry_offset+16;

	    sector_buffer[(entry_offset+0x01)]
	     =part_table[drive_index].next_ext_start_head[index];

	    Convert_Long_To_Integer(Combine_Cylinder_and_Sector(
	     part_table[drive_index].next_ext_start_cyl
	     [index],part_table[drive_index]
	     .next_ext_start_sect[index]));

	    sector_buffer[(entry_offset+0x02)]=integer1;
	    sector_buffer[(entry_offset+0x03)]=integer2;

	    sector_buffer[(entry_offset+0x04)]=part_table[drive_index]
	     .next_ext_num_type[index];

	    sector_buffer[(entry_offset+0x05)]=part_table[drive_index]
	     .next_ext_end_head[index];

	    Convert_Long_To_Integer(Combine_Cylinder_and_Sector(
	     part_table[drive_index].next_ext_end_cyl
	     [index],part_table[drive_index].next_ext_end_sect
	     [index]));

	    sector_buffer[(entry_offset+0x06)]=integer1;
	    sector_buffer[(entry_offset+0x07)]=integer2;

	    high = part_table[drive_index].next_ext_rel_sect[index] >> 16;
	    temp=high << 16;

	    if(part_table[drive_index].next_ext_rel_sect[index] > 0xffff)
	      {
	      low = part_table[drive_index].next_ext_rel_sect[index] - temp;
	      }
	    else
	      {
	      low = part_table[drive_index].next_ext_rel_sect[index];
	      }

	    Convert_Long_To_Integer(low);

	    sector_buffer[entry_offset+0x08]=integer1;
	    sector_buffer[entry_offset+0x09]=integer2;

	    Convert_Long_To_Integer(high);

	    sector_buffer[entry_offset+0x0a]=integer1;
	    sector_buffer[entry_offset+0x0b]=integer2;

	    high = part_table[drive_index].next_ext_num_sect[index] >> 16;
	    temp=high << 16;

	    if(part_table[drive_index].next_ext_num_sect[index] > 0xffff)
	      {
	      low = part_table[drive_index].next_ext_num_sect[index] - temp;
	      }
	    else
	      {
	      low = part_table[drive_index].next_ext_num_sect[index];
	      }

	    Convert_Long_To_Integer(low);

	    sector_buffer[entry_offset+0x0c]=integer1;
	    sector_buffer[entry_offset+0x0d]=integer2;

	    Convert_Long_To_Integer(high);

	    sector_buffer[entry_offset+0x0e]=integer1;
	    sector_buffer[entry_offset+0x0f]=integer2;

	    error_code
	     =Write_Physical_Sectors((drive_index+128)
	     ,extended_cylinder,extended_head,extended_sector,1);
	    if(error_code!=0) return(error_code);

	    extended_cylinder=next_extended_cylinder;
	    extended_head=next_extended_head;
	    extended_sector=next_extended_sector;
	    }
	  else
	    {
	    error_code=Write_Physical_Sectors((drive_index+128)
	     ,extended_cylinder,extended_head,extended_sector,1);

	    if(error_code!=0) return(error_code);

	    index=24;
	    }

	  index++;
	  }while(index<24);
	}
      }
    drive_index++;
    }while(drive_index<7);

  return(0);
}

/* Write physical sectors */
int Write_Physical_Sectors(int drive, long cylinder, long head, long sector, int number_of_sectors)
{
  int error_code;

  number_of_sectors=1;

  if(part_table[(drive-128)].ext_int_13==FALSE)
    {
    error_code=Write_Physical_Sectors_CHS(drive,cylinder,head,sector,number_of_sectors);
    }
  else
    {
    error_code=Write_Physical_Sectors_LBA(drive,cylinder,head,sector,number_of_sectors);
    }
  return(error_code);
}

/* Write physical sectors using CHS format. */
int Write_Physical_Sectors_CHS(int drive, long cylinder, long head, long sector, int number_of_sectors)
{
  int error_code;

#ifdef DEBUG
  if(debug.write==TRUE)
    {
#endif

    if(number_of_sectors==1)
      {
      error_code=biosdisk(3, drive, head, cylinder, sector, number_of_sectors, sector_buffer);
      }
    else
      {
      error_code=biosdisk(3, drive, head, cylinder, sector, number_of_sectors, huge_sector_buffer);
      }

#ifdef DEBUG
    }
#endif

#ifdef DEBUG
  else
    {
    int current_line=0;

    long index=0;
    long offset=0x1be;

    Clear_Screen(NULL);
    Print_Centered(4,"Write_Physical_Sector_CHS() function debugging screen",BOLD);
    Position_Cursor(4,6);
    printf("Note:  WRITE=OFF is set or an emulated disk is in existence...no");
    Position_Cursor(4,7);
    printf("       changes will be made.  Please check the \"fdisk.ini\" file");
    Position_Cursor(4,8);
    printf("       for details.");

    Position_Cursor(4,10);
    printf("Information passed to this function:");

    Position_Cursor(50,11);
    printf("Drive:     0x%X",drive);
    Position_Cursor(50,12);
    printf("Cylinder:  %d",cylinder);
    Position_Cursor(50,13);
    printf("Head:      %d",head);
    Position_Cursor(50,14);
    printf("Sector:    %d",sector);
    Position_Cursor(4,16);
    printf("Contents of partition table area in sector_buffer[]:");

    do
      {
      index=0;

      Position_Cursor(4,(current_line+18));
      printf("%d:  ",(current_line+1));
      do
	{
	printf("%02X ",sector_buffer[(index+offset)]);
	index++;
	}while(index<16);

      current_line++;
      offset=offset+16;
      }while(offset<(0x1be+64));

    Position_Cursor(4,23);
    printf("Press any key to continue.");

    asm{
      mov ah,7
      int 0x21
      }

    error_code=0;
    }
#endif

  return(error_code);
}

/* Write physical sectors using LBA format. */
int Write_Physical_Sectors_LBA(int drive, long cylinder, long head, long sector, int number_of_sectors)
{
  unsigned int error_code=0;

  /* Get the segment and offset of disk_address_packet. */
  unsigned int disk_address_packet_address_seg=FP_SEG(disk_address_packet);
  unsigned int disk_address_packet_address_off=FP_OFF(disk_address_packet);

  unsigned long LBA_address_high;
  unsigned long LBA_address_low;

  /* Translate CHS values to LBA values. */
  unsigned long LBA_address=Translate_CHS_To_LBA(cylinder,head,sector
   ,part_table[(drive-128)].total_head
   ,part_table[(drive-128)].total_sect);

  /* Determine the location of sector_buffer[512]             */
  /* and place the address of sector_buffer[512] into the DAP */

  /* Add number_of_sectors to disk_address_packet */
  disk_address_packet[2]=number_of_sectors;

  if(number_of_sectors==1)
    {
    Convert_Long_To_Integer(FP_OFF(sector_buffer));
    disk_address_packet[4]=integer1;
    disk_address_packet[5]=integer2;

    Convert_Long_To_Integer(FP_SEG(sector_buffer));
    disk_address_packet[6]=integer1;
    disk_address_packet[7]=integer2;
    }
  else
    {
    Convert_Long_To_Integer(FP_OFF(huge_sector_buffer));
    disk_address_packet[4]=integer1;
    disk_address_packet[5]=integer2;

    Convert_Long_To_Integer(FP_SEG(huge_sector_buffer));
    disk_address_packet[6]=integer1;
    disk_address_packet[7]=integer2;
    }

  /* Transfer LBA_address to disk_address_packet */
  LBA_address_high=(LBA_address&0xffff0000)>>16;
  LBA_address_low=LBA_address&0x0000ffff;

  Convert_Long_To_Integer(LBA_address_low);
  disk_address_packet[8]=integer1;
  disk_address_packet[9]=integer2;

  Convert_Long_To_Integer(LBA_address_high);
  disk_address_packet[10]=integer1;
  disk_address_packet[11]=integer2;

#ifdef DEBUG
  if(debug.write==TRUE)
    {
#endif

    /* Load the registers and call the interrupt. */
    asm {
      mov ah,0x43
      mov al,0x00
      mov dl,BYTE PTR drive
      mov ds,disk_address_packet_address_seg
      mov si,disk_address_packet_address_off
      int 0x13

      mov BYTE PTR error_code,ah
      }
#ifdef DEBUG
    }
#endif

#ifdef DEBUG
  else
    {
    int current_line=0;
    int index=0;

    long offset=0x1be;

    Clear_Screen(NULL);
    Print_Centered(4,"Write_Physical_Sector_LBA() function debugging screen",BOLD);
    Position_Cursor(4,6);
    printf("Note:  WRITE=OFF is set or an emulated disk is in existence...no");
    Position_Cursor(4,7);
    printf("       changes will be made.  Please check the \"fdisk.ini\" file");
    Position_Cursor(4,8);
    printf("       for details.");

    Position_Cursor(4,10);
    printf("Information passed to this function:");

    Position_Cursor(50,11);
    printf("Drive:     0x%X",drive);
    Position_Cursor(50,12);
    printf("Cylinder:  %d",cylinder);
    Position_Cursor(50,13);
    printf("Head:      %d",head);
    Position_Cursor(50,14);
    printf("Sector:    %d",sector);
    Position_Cursor(4,16);
    printf("Contents of partition table area in sector_buffer[]:");

    do
      {
      index=0;

      Position_Cursor(4,(current_line+18));
      printf("%d:  ",(current_line+1));
      do
	{
	printf("%02X ",sector_buffer[(index+offset)]);
	index++;
	}while(index<16);

      current_line++;
      offset=offset+16;
      }while(offset<(0x1be+64));

    Position_Cursor(4,23);
    printf("Press any key to continue.");

    asm{
      mov ah,7
      int 0x21
      }

    error_code=0;
    }
#endif

  return(error_code);
}
