;------------------------------------------------------------------------------
;		F R E E - D O S		X M S - D R I V E R
;------------------------------------------------------------------------------
;
; Improvments by Martin Strmberg.
; Copyright 2001, Martin Strmberg.
; 
; Martin can be reached at <ams@ludd.luth.se>. Please start the subject line 
; with "FDXMS". If you have a bug report, use bugtrack on the FreeDOS site,
; <http://www.freedos.org>. Make sure to include the FDXMS.SYS version number.
;
;
; Written by Till Gerken for the Free-DOS project.
;
; If you would like to use parts of this driver in one of your projects, please
; check up with me first.
;
; Till can be reached at:	till@tantalo.net (Internet)
;
; For questions concerning Free-DOS, mail the coordinator
; Morgan "Hannibal" Toal <hannibal@iastate.edu>
;
; Comments and bug reports are always appreciated.
;
; Copyright (c) 1995, Till Gerken
;------------------------------------------------------------------------------
; -- IMPLEMENTATION NOTES --
;
; - The Protected Mode handling is very simple, it fits only for the least
;   needs
; - I didn't care about reentrancy. If this thing should be used in
;   Multitasking Environments, well, somebody got to take a look at the code.
; - INT15h, Func. 87h (Move Block) has been re-implemented to preserve
;   the state of A20. (very slow!)
; - INT15h, Func. 88h (Extended Memory Size) has been re-implemented to
;   return 0.
; - Function 0Bh (Move Block) uses it's own Protected Mode handling.
;   It doesn't provide Interrupt windows.
; - The code is not very optimised, I just wrote it down for now. Later, when
;   everything is tested, I'm going to see what can be done
; - Some ideas were taken from the original XMS driver written by
;   Mark E. Huss (meh@bis.adp.com), but everything has been completely
;   rewritten, so if there are bugs, they are mine, not his. ;)
;------------------------------------------------------------------------------

ideal					; switch on ideal mode syntax
jumps
P8086				; We start in 8086 mode
	
include "fdxms.inc"			; this file contains structures,
					; constants and the like

;------------------------------------------------------------------------------
; 16-bit resident code and data
;------------------------------------------------------------------------------

segment code16
assume cs:code16, ds:code16, es:nothing

;------------------------------------------------------------------------------
; device driver header

		dd	-1			; last driver in list
		dw	8000h			; driver flags
		dw	offset strategy		; pointer to strategy routine
		dw	offset interrupt	; pointer to interrupt handler
		db	'XMSXXXX0'		; device driver name

;------------------------------------------------------------------------------
; global data

request_ptr	dd	?			; pointer to request header

gdt32		dw	gdt_size,dummy,0
dummy		dq	0
code16dsc	db	0ffh,0ffh,0,0,0,9ah,0,0
core32dsc	db	0ffh,0ffh,0,0,0,92h,0cfh,0
gdt_size=$-(offset dummy)

code16idx	=	08h
core32idx	=	10h

bios_gdt:
		db	0,0,0,0,0,0,0,0		; Dummy entries for BIOS.
		db	0,0,0,0,0,0,0,0		; Dummy entries for BIOS.
		dw	0ffffh			; Bits 0-15 of source segment length.
bios_src_low	dw	0			; Bits 0-15 of source address.
bios_src_middle	db	0			; Bits 16-23 of source address.
		db	93h			; Source access rights.
		db	0fh			; More type bits and bits 16-19 of source segment length.
bios_src_high	db	0			; Bits 24-31 of source address.
		dw	0ffffh			; Bits 0-15 of dest. segment length.
bios_dst_low	dw	0			; Bits 0-15 of dest. address.
bios_dst_middle	db	0			; Bits 16-23 of dest. address.
		db	93h			; Dest. access rights.
		db	0fh			; More type bits and bits 16-19 of dest segment length.
bios_dst_high	db	0			; Bits 24-31 of dest. address.
		db	0,0,0,0,0,0,0,0		; Dummy entries for BIOS.
		db	0,0,0,0,0,0,0,0		; Dummy entries for BIOS.
	

old_int15	dd	?			; old INT15h vector
old_int2f	dd	?			; old INT2fh vector

xms_num_handles	dw	32			; number of available handles
	
flags		db	0
		
hma_used	db	0			; set if HMA is used
hma_min		dw	0			; minimal space in HMA that
						; has to be requested
a20_locks	dw	0			; internal A20 lock count

IFDEF XXMS
include "xxms_variables.inc"
ENDIF ; XXMS
	
IFDEF TRACE_CODE
trace		dd	0
ENDIF ; TRACE_CODE

IFDEF DEBUG_CODE
debug		db	0
ENDIF ; DEBUG_CODE
	
;------------------------------------------------------------------------------
; strategy routine. is called by DOS to initialize the driver once.
; only thing to be done here is to store the address of the device driver
; request block.
; In:	ES:BX - address of request header
; Out:	nothing

proc	strategy	far
	mov	[word cs:request_ptr+2],es	; store segment addr
	mov	[word cs:request_ptr],bx	; store offset addr
	ret					; far return here!
endp	strategy

;------------------------------------------------------------------------------
; interrupt routine. called by DOS right after the strategy routine to
; process the incoming job. also used to initialize the driver.

proc	interrupt	far
	push	di es

	mov	di,cs
	mov	ds,di

	les	di,[request_ptr]		; load address of request header

	cmp	[es:di+request_hdr.cmd],CMD_INIT; do we have to initialize?
	jne	@@done
	test	[ flags ], FLAG_INITIALISED	; Have we initialized
						; already?
	jnz	@@done
	call	initialize			; No, do it now!
@@done:

	lds	si,[request_ptr]		; return this to DOS

	pop	es di
	ret					; far return here!
endp	interrupt

	
P386				; Turn on 386 instructions.
	
;------------------------------------------------------------------------------
; just delays a bit

proc delay
@@delay_start:
     in al, 64h
     jmp @@delay_check
@@delay_check:
     and al, 2
     jnz @@delay_start
     ret
endp delay

;------------------------------------------------------------------------------
; enables the A20 address line

proc enable_a20
     mov al,0d1h
     out 64h,al
     call delay
     mov al,0dfh
     out 60h,al
     call delay
     mov al,0ffh
     out 64h,al
     call delay
     ret
endp enable_a20

;------------------------------------------------------------------------------
; disables the A20 address line

proc disable_a20
     mov al,0d1h
     out 64h,al
     call delay
     mov al,0ddh
     out 60h,al
     call delay
     mov al,0ffh
     out 64h,al
     call delay
     ret
endp disable_a20

;------------------------------------------------------------------------------
; tests if the A20 address line is enabled.
; compares 256 bytes at 0:0 with ffffh:10h
; Out:	ZF=0 - A20 enabled
;	ZF=1 - A20 disabled

proc	test_a20
	push	cx si di ds es

	xor	si,si
	mov	ds,si

	mov	di,0ffffh
	mov	es,di
	mov	di,10h

	mov	cx,100h/4
	rep	cmpsd

	pop	es ds di si cx
	ret
endp	test_a20

;------------------------------------------------------------------------------
; checks if VDISK is already installed
; note: HMA check is skipped because of speed and some other (weird) reasons.
; In:	nothing
; Out:	ZF=0 -> VDISK is installed
;	ZF=1 -> VDISK not installed

vdisk_id	db	VDISK_IDSTR

proc	check_vdisk
	push	ax bx cx si di es

	xor	ax,ax			; get interrupt vector 19h
	mov	es,ax
	les	bx,[es:19h*4]

	mov	di,VDISK_IDOFS
	mov	si,offset vdisk_id
	mov	cx,VDISK_IDLEN
	rep	cmpsb			; is VDISK here?

	pop	es di si cx bx ax
	ret
endp	check_vdisk

;------------------------------------------------------------------------------
; Interrupt handlers
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; new INT15h handler

a20state	db	?		; keeps A20 state across INT15h call
	
proc	int15_handler
	cmp	ah,87h				; is it a block move request?
	je	@@do_move
	cmp	ah,88h				; is it a ext. mem size req.?
	je	@@ext_mem_size
	cmp	ax,0e801h			; Is it a ext. mem size req.?
	je	@@ax0xe801
	jmp	[cs:old_int15]			; jump to old handler
@@do_move:
	call	test_a20			; check if A20 is on or off
	jz	@@a20disabled
	mov	[cs:a20state],1			; preserve state
	jmp	@@call_old_mover
@@a20disabled:
	mov	[cs:a20state],0
@@call_old_mover:
	pushf					; simulate INT call
	call	[cs:old_int15]
	pushf					; save flags for return
	push	ax
	cmp	[cs:a20state],0			; see if A20 has to be switched
	jz	@@disable_it
	call	enable_a20
	jmp	@@move_done
@@disable_it:
	call	disable_a20
@@move_done:
	pop	ax
	popf
	iret

@@ax0xe801:
;	clc - Not necessary as cmp before je cleared carry flag.
	pushf
	call	[ cs:old_int15 ]
	jnc	@@ok
	
	xor	cx, cx
	xor	dx, dx

@@ok:	
	xor	bx, bx			; No memory above 16MiB available.
	;; Fall through.
	
@@ext_mem_size:
	mov	ax, bp				; Save BP.
	mov	bp, sp
	and	[ word ptr bp+4 ], 0fffeh	; No error: clc.
	mov	bp, ax				; Restore BP.
	xor	ax,ax				; No memory available.
	iret
endp	int15_handler

;------------------------------------------------------------------------------
; new INT2Fh handler. Catches Func. 4300h+4310h

proc	int2f_handler
	pushf
	cmp	ax,4300h			; is it "Installation Check"?
	jne	@@driver_address
	mov	al,80h				; yes, we are installed ;)
	popf
	iret
@@driver_address:
	cmp	ax,4310h			; is it "Get Driver Address"?
	jne	@@call_old2f
	mov	bx,cs
	mov	es,bx
	mov	bx,offset xms_dispatcher
	popf
	iret
@@call_old2f:
	popf
	jmp	[cs:old_int2f]			; jump to old handler
endp	int2f_handler

;------------------------------------------------------------------------------
; XMS functions
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; returns XMS version number
; In:	AH=0
; Out:	AX=XMS version number
;	BX=internal revision number
;	DX=1 if HMA exists, 0 if not

proc	xms_get_version
	mov	ax, INTERFACE_VER
	mov	bx, DRIVER_VER
	mov	dx,1				; HMA is always available
	popf
	retf
endp	xms_get_version

;------------------------------------------------------------------------------
; requests HMA
; In:	AH=1
;	DX=space needed in HMA (0ffffh if application tries to request HMA)
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=80h -> function not implemented (implemented here ;) )
;	  BL=81h -> VDISK is detected
;	  BL=90h -> HMA does not exist
;	  BL=91h -> HMA already in use
;	  BL=92h -> DX less than HMA_MIN

proc	xms_request_hma
	cmp	[cs:hma_used],0			; is HMA already used?
	mov	bl,XMS_HMA_IN_USE
	jnz	xrh_err
	cmp	dx,[cs:hma_min]			; is request big enough?
	mov	bl,XMS_HMAREQ_TOO_SMALL
	jb	xrh_err
	mov	[cs:hma_used],1			; assign HMA to caller
	mov	ax,1
	xor	bl,bl
	popf
	retf
xrh_err:
	xor	ax,ax
	popf
	retf
endp	xms_request_hma

;------------------------------------------------------------------------------
; releases HMA
; In:	AH=2
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=80h -> function not implemented
;	  BL=81h -> VDISK is detected
;	  BL=90h -> HMA doesn't exist
;	  BL=93h -> HMA wasn't allocated

proc	xms_release_hma
	cmp	[cs:hma_used],0			; is HMA used?
	mov	bl,XMS_HMA_NOT_USED
	jz	xrh_err
	mov	[cs:hma_used],0			; now release it
	mov	ax,1
	xor	bl,bl
	popf
	retf
endp	xms_release_hma

;------------------------------------------------------------------------------
; global A20 address line enable
; In:	AH=3
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=80h -> function is not implemented
;	  BL=81h -> VDISK is detected
;	  BL=82h -> A20 failure

proc	xms_global_enable_a20
	call	enable_a20			; enable A20
	call	test_a20			; is it really enabled?
	jz	xge_a20_err
	mov	ax,1
	xor	bl,bl
	popf
	retf
xge_a20_err:
	xor	ax,ax
	mov	bl,XMS_A20_FAILURE
	popf
	retf
endp	xms_global_enable_a20

;------------------------------------------------------------------------------
; global A20 address line disable
; In:	AH=4
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=80h -> function is not implemented
;	  BL=81h -> VDISK is detected
;	  BL=82h -> A20 failure
;	  BL=84h -> A20 still enabled

proc	xms_global_disable_a20
	call	disable_a20			; disable A20
	call	test_a20			; is it really disabled?
	jnz	xge_a20_err
	mov	ax,1
	xor	bl,bl
	popf
	retf
endp	xms_global_disable_a20

;------------------------------------------------------------------------------
; enables A20 locally
; In:	AH=5
; Out:	AX=1 if A20 is enabled, 0 otherwise
;	BL=80h -> function not implemented
;	BL=81h -> VDISK is detected
;	BL=82h -> A20 failure

proc	xms_local_enable_a20
	inc	[cs:a20_locks]			; increase lock counter
	call	enable_a20			; enable it
	call	test_a20			; test if it's really enabled
	jz	xge_a20_err
	mov	ax,1
	xor	bl,bl
	popf
	retf
endp	xms_local_enable_a20

;------------------------------------------------------------------------------
; disables A20 locally
; In:	AH=6
; Out:	AX=1 if A20 is disabled, 0 otherwise
;	BL=80h -> function not implemented
;	BL=81h -> VDISK is detected
;	BL=82h -> A20 failure

proc	xms_local_disable_a20
	dec	[cs:a20_locks]			; decrease lock counter
	jnz	@@xld_dont_disable		; disable only if needed
	call	disable_a20			; disable it
	call	test_a20			; test if it's really disabled
	jnz	xge_a20_err
@@xld_dont_disable:
	mov	ax,1
	xor	bl,bl
	popf
	retf
endp	xms_local_disable_a20

;------------------------------------------------------------------------------
; returns the state of A20
; In:	AH=7
; Out:	AX=1 if A20 is physically enabled, AX=0 if not
;	BL=00h -> function was successful
;	BL=80h -> function is not implemented
;	BL=81h -> VDISK is detected

proc	xms_query_a20
	xor	ax,ax			; suppose A20 is disabled
	call	test_a20
	jz	@@xqa_a20dis
	mov	ax,1
@@xqa_a20dis:
	xor	bl,bl
	popf
	retf
endp	xms_query_a20

;------------------------------------------------------------------------------
; searches a/next free XMS memory block
; In:	DS=CS
;	BX - offset of start handle (if search is continued)
;	CX - remaining handles (if search is continued)
; Out:	CY=1 - no free block
;	  BX - offset of end of handle table
;	CY=0 - free block found
;	  BX - offset of free handle
;	  CX - number of remaining handles

proc	xms_find_free_block
	mov	bx,offset driver_end	; start at the beginning of the table
	mov	cx,[xms_num_handles]	; check all handles
@@find_free_block:
	cmp	[bx+xms_handle.used],0	; is it used?
	jnz	xms_find_next_free_block; yes, go on
	cmp	[bx+xms_handle.xbase],0	; assigned memory block or just blank?
	jnz	@@found_block		; assigned, return it
xms_find_next_free_block:
	add	bx,size xms_handle	; skip to next handle
	loop	@@find_free_block	; check next handle
	stc				; no free block found, error
	ret
@@found_block:
	clc				; no error, return
	ret
endp	xms_find_free_block

;------------------------------------------------------------------------------
; searches a/next free XMS memory handle
; In:	DS=CS
;	BX - offset of start handle (if search is continued)
;	CX - remaining handles (if search is continued)
; Out:	CY=1 - no free handle
;	  BX - offset of end of handle table
;	CY=0 - free handle found
;	  BX - offset of free handle
;	  CX - number of remaining handles

proc	xms_find_free_handle
	mov	bx,offset driver_end	; start at the beginning of the table
	mov	cx,[xms_num_handles]	; check all handles
@@find_free_handle:
	cmp	[bx+xms_handle.used],0		; is it used?
	jnz	xms_find_next_free_handle	; yes, go on
	cmp	[bx+xms_handle.xbase],0		; really blank handle?
	jz	@@found_handle			; found a blank handle
xms_find_next_free_handle:
	add	bx,size xms_handle	; skip to next handle
	loop	@@find_free_handle	; check next handle
	stc				; no free block found, error
	ret
@@found_handle:
	clc				; no error, return
	ret
endp	xms_find_free_handle

;------------------------------------------------------------------------------
; Verifies that a handle is valid and in use.
; In:	DX - handle to verify
; Out:	CY=1 - not a valid handle
;	  BL=0xa2 - XMS_INVALID_HANDLE
;	  AX=0 - Error return.
;	CY=0 - valid handle
;	  AX=modified
	
proc	xms_handle_valid
	mov	ax, dx	
	sub	ax, offset driver_end		; Is the handle below start of handles?
	jb	@@not_valid
	
	push	dx
	push	bx
	xor	dx, dx
	mov	bx, size xms_handle
	div	bx
	or	dx, dx				; Is the handle aligned on a handle boundary?
	pop	bx
	pop	dx
	jnz	@@not_valid
	
	cmp	ax, [ cs:xms_num_handles ]	; Is the handle number less than the number of handles?
	jae	@@not_valid
	
	; If we come here, the handle is a valid handle
	mov	ax, bx
	mov	bx, dx
	cmp	[ cs:bx+xms_handle.used ], 1	; Is the handle in use?
	mov	bx, ax
	jb	@@not_valid

	; Handle is valid.
	clc
	ret
	
@@not_valid:
	; Handle is not valid
	mov	bl, XMS_INVALID_HANDLE
	xor	ax, ax
	stc
	ret
endp	xms_handle_valid
	
;------------------------------------------------------------------------------
; calculates the move address
; In:	BX - handle (0 if EDX should be interpreted as seg:ofs value)
;	ECX - length
;	EDX - offset
; Out:	CY=0 - valid input (handle).
;	  EBX - absolute move address
;	CY=1 - not valid handle or invalid length.
;	  BL=XMS_INVALID_DESTINATION_HANDLE - not valid handle.
;	  BL=XMS_INVALID_LENGTH - invalid length.
;	
; Modifies: EAX, EDX

proc	xms_get_move_addr
	or	bx,bx			; translate address in EDX?
	jnz	@@dont_translate
	movzx	eax,dx			; save offset
	xor	dx,dx			; clear lower word
	shr	edx,12			; convert segment to absolute address
	lea	ebx, [ edx + eax ]
	
	;; Assert length + offset < 0xffff:ffff + 1 = 0x10fff0
	;  "lea	eax, [ ebx + ecx ]" is no good in case offset = 0xfffffffe or so...
	mov	eax, ebx
	add	eax, ecx
	jc	@@invalid_length
	
	cmp	eax, 10fff0h
	ja	@@invalid_length
	
	clc
	ret
	
@@dont_translate:
	xchg	dx, bx		; Now handle is in DX and lower 16 bits of offset in BX.
	call	xms_handle_valid
	jc	@@not_valid_handle

	push	si
	mov	si, dx		; Put handle in SI.
	mov	dx, bx		; Get lower 16 bits of length back.
	
	;; Assert length + offset <= size.
	;  "lea eax, [ ecx+edx+1024-1 ]" is no good in case ECX or EDX 
	;  equal 0xffffffe or something similarly large.
	mov     eax, ecx			; Load eax with sum of ecx
	add     eax, edx			; and edx,
	jc      @@invalid_length_pop_si		; (no overflow is permissible)
	add     eax, 1024-1			; and add on 2^10-1 for ...
	jc      @@invalid_length_pop_si
	
	shr	eax, 10				; ... shift so eax is size in kiB.
	
IFNDEF XXMS
	test	eax, 0ffff0000h
	jnz	@@invalid_length_pop_si

	cmp	ax, [ cs:si+xms_handle.xsize ]
	ja	@@invalid_length_pop_si
	
	;; Now everything is ok.
	movzx	ebx, [ cs:si+xms_handle.xbase ]	; Get block base address.
ELSE ; IFDEF XXMS
	cmp	eax, [ cs:si+xms_handle.xsize ]
	ja	@@invalid_length_pop_si
	
	;; Now everything is ok.
	mov	ebx, [ cs:si+xms_handle.xbase ]	; Get block base address.
ENDIF ; XXMS	

	shl	ebx, 10				; Convert from kb to absolute.
	add	ebx,edx				; Add offset into block.
	pop	si
	ret
		
@@not_valid_handle:	
	mov	bl, XMS_INVALID_DESTINATION_HANDLE
	ret

@@invalid_length_pop_si:
	pop	si
@@invalid_length:
	mov	bl, XMS_INVALID_LENGTH
	stc
	ret	
endp	xms_get_move_addr

;------------------------------------------------------------------------------
; frees an XMS block
; In:	AH=0ah
;	DX=handle to allocated block that should be freed
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a2h -> handle is invalid
;	  BL=0abh -> handle is locked
IFNDEF XXMS
AX_EAX	= ax
DX_EDX	= dx
DI_EDI	= di
ELSE
AX_EAX	= eax
DX_EDX	= edx
DI_EDI	= edi
ENDIF ; XXMS
	
proc	xms_free_xms
	call	xms_handle_valid
	jc	@@xms_free_not_valid
	
	push	bx cx DX_EDX DI_EDI si ds
IFDEF XXMS
	push	eax
ENDIF
	mov	ax, cs
	mov	ds, ax

	mov	si, dx				; Get handle into SI.

	xor	DX_EDX, DX_EDX			; Set to zero for cmp and mov below.
	
	cmp	[ si+xms_handle.used ], 1	; Is the block locked?
	jne	@@locked			; Yes, bail out.

	cmp	[ si+xms_handle.xsize ], DX_EDX	; Is it a zero-length handle?
	je	@@zero_handle

	mov	AX_EAX, [ si+xms_handle.xbase ]	; Get base address.
	add	AX_EAX, [ si+xms_handle.xsize ]	; Calculate end-address.

	call	xms_find_free_block		; Check free blocks.
	jc	@@xms_free_loop_2		; No, was last handle.
	
@@try_concat:
	cmp	AX_EAX, [ bx+xms_handle.xbase ]	; Is it adjacent to old block?
	jne	@@not_adjacent

	mov	DI_EDI, [ bx+xms_handle.xsize ]	; Concat.
	add	AX_EAX, DI_EDI
	add	[ si+xms_handle.xsize ], DI_EDI

	mov	[ bx+xms_handle.xbase ], DX_EDX	; Blank handle.
	mov	[ bx+xms_handle.xsize ], DX_EDX

@@not_adjacent:
	call	xms_find_next_free_block	; See if there are other blks.
	jnc	@@try_concat

@@xms_free_loop_2:	
	call	xms_find_free_block		; Check free blocks.
	jc	@@xms_free_done			; No, was last handle.

@@try_concat_2:	
	mov	AX_EAX, [ bx+xms_handle.xbase ]	; Is it adjacent to old block?
	add	AX_EAX, [ bx+xms_handle.xsize ]
	cmp	AX_EAX, [ si+xms_handle.xbase ]
	jne	@@not_adjacent_2

	mov	AX_EAX, [ si+xms_handle.xsize ]	; Concat.
	add	[ bx+xms_handle.xsize ], AX_EAX
	mov	[ si+xms_handle.xbase ], DX_EDX	; Blank handle.
	mov	[ si+xms_handle.xsize ], DX_EDX
	mov	[ si+xms_handle.used ], dl	; Not in use anymore.
	mov	si, bx
	jmp	@@xms_free_loop_2

@@not_adjacent_2:
	call	xms_find_next_free_block	; See if there are other blks.
	jnc	@@try_concat_2

@@xms_free_done:
	mov	[ si+xms_handle.used ], dl	; Handle isn't used anymore.
IFDEF XXMS
	pop	eax
ENDIF
	pop	ds si DI_EDI DX_EDX cx bx
	mov	ax, 1
	xor	bl, bl
	popf
	retf
	
@@zero_handle:
	mov	[ si+xms_handle.xbase ], DX_EDX	; Blank handle.
	jmp	@@xms_free_done

@@locked:
IFDEF XXMS
	pop	eax
ENDIF
	pop	ds si DI_EDI DX_EDX cx bx
	xor	ax, ax
	mov	bl, XMS_BLOCK_LOCKED

@@xms_free_not_valid:
	popf
	retf
endp	xms_free_xms

;------------------------------------------------------------------------------
; moves an XMS block
; In:	AH=0bh
;	DS:SI=pointer to XMS move structure
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=082h -> A20 failure
;	  BL=0a3h -> source handle is invalid
;	  BL=0a4h -> source offset is invalid
;	  BL=0a5h -> destination handle is invalid
;	  BL=0a6h -> destination offset is invalid
;	  BL=0a7h -> length is invalid
;	  BL=0a8h -> move has invalid overlap
;	  BL=0a9h -> parity error

proc	xms_move_xms
	push	eax 
	push	ebx ecx edx edi esi
	push	ds es
	cli					; no interrupts

	call	test_a20			; get A20 state
	pushf					; save it for later
	jnz	@@a20_already_enabled	; We don't need to enable it if it's already is enabled.
	call	enable_a20			; now enable it!

@@a20_already_enabled:	
	mov	ecx,[si+xms_move_strc.len]	; get length
	test	cl,1				; Is the length even?
	jnz	@@invalid_length

	mov	bx,[si+xms_move_strc.dest_handle]
	mov	edx,[si+xms_move_strc.dest_offset]
	call	xms_get_move_addr		; get move address
	jc	@@dest_not_valid
	mov	edi,ebx				; store in destination index

	mov	bx,[si+xms_move_strc.src_handle]
	mov	edx,[si+xms_move_strc.src_offset]
	call	xms_get_move_addr		; get move address
	jc	@@src_not_valid
	mov	esi,ebx				; store in source index

	or	ecx, ecx	; Nothing to copy?
	jz	@@xms_move_ok

	cmp	esi, edi	; src == dest?
	je	@@xms_move_ok	; Yes, nothing to do.

	;; Check if we're in protected mode already.
	mov	eax, cr0
	test	al, 1
	jnz	@@do_bios_move	; Already in PM.

	test	[ cs:flags ], FLAG_USE_BIOS	; Do we want to always use the BIOS?
	jnz	@@do_bios_move		; Yes.
	
	lgdt	[fword cs:gdt32]		; load GDTR
;	mov	eax, cr0 - Not neccessary as we already did this (some lines above).
	or	al,1				; set PE bit
	mov	cr0,eax				; shazamm!
	db	0eah				; JMP FAR
	dw	offset to_pm,code16idx		; flush IPQ and load CS sel.
to_pm:

	mov	ax,core32idx
	mov	ds,ax
	mov	es,ax

	cmp	esi, edi	; src <= dest?
	ja	@@do_move	; No, go ahead.

	;; src <= dest so we must copy from higher addresses to lower in case the ranges overlap.
	std				; High to low.
	lea	esi, [ esi + ecx - 4 ]	; Adjust addresses for dword size.
	lea	edi, [ edi + ecx - 4 ]
	shr	ecx,2			; Get number of DWORDS to move.
	jnc	@@dword_boundary

	mov	ax, [ esi+2 ]
	mov	[ edi+2 ], ax
	sub	esi, 2		; We just moved a word so adjust addresses
	sub	edi, 2		; for that.
	jmp	@@dword_boundary

@@do_move:	
	shr	ecx,2				; get number of DWORDS to move
	jnc	@@dword_boundary		; is length a DWORD multiple?
	
	movs	[word esi],[word edi]		; no, move first word to adjust
	
@@dword_boundary:
	rep	movs [dword esi],[dword edi]	; now move the main block

	mov	eax,cr0
	and	al,not 1			; clear PE bit
	mov	cr0,eax				; shazomm!

	db	0eah				; JMP FAR
	dw	offset to_rm
code_seg dw	?				; flush IPQ and load CS sel.

to_rm:
	
@@xms_move_ok:	
	popf				; get A20 state
	jnz	@@a20_was_enabled	; if it was enabled, don't disable
	call	disable_a20		; it was disabled, so restore state
	
@@a20_was_enabled:
	pop	es ds
	pop	esi edi edx ecx ebx eax	; restore everything
	popf
	mov	ax,1				; success
	retf

@@do_bios_move:
	cmp	esi, edi	
	ja	@@bios_ok
	
	mov	eax, esi
	add	eax, ecx
	cmp	eax, edi
	ja	@@illegal_overlap

@@bios_ok:	
	mov	edx, ecx
	shr	edx, 1		; Length in words.
	
@@bios_move_loop:
	mov	ecx, edx
	cmp	ecx, 1000h	; 0x1000 is a good value according to Tom Ehlert.
	jbe	@@length_ok

	mov	ecx, 1000h

@@length_ok:
	cli			; Protect bios_gdt for reentrancy.

	;; Fill in source entries.
	mov	eax, esi
	mov	[ cs:bios_src_low ], ax
	shr	eax, 10h
	mov	[ cs:bios_src_middle ], al
	shr	eax, 8
	mov	[ cs:bios_src_high ], al

	;; Fill in destination entries.
	mov	eax, edi
	mov	[ cs:bios_dst_low ], ax
	shr	eax, 10h
	mov	[ cs:bios_dst_middle ], al
	shr	eax, 8
	mov	[ cs:bios_dst_high ], al

	push	ecx
	push	esi
	mov	ax, cs
	mov	es, ax

	lea	si, [ bios_gdt ]

	clc
	mov	ah, 87h
	int	15h
	sti

	pop	esi
	pop	ecx

	jc	@@a20_failure

	lea	edi, [ edi + 2*ecx ]	; dest += copied length.
	lea	esi, [ esi + 2*ecx ]	; src += copied length.
	sub	edx, ecx		; length -= copied length.
	jnz	@@bios_move_loop

	jmp	@@xms_move_ok

@@src_not_valid:
	cmp	bl, XMS_INVALID_DESTINATION_HANDLE	; bl set by xms_get_move_addr.
	jne	@@invalid_length
	mov	al, XMS_INVALID_SOURCE_HANDLE
	jmp	@@xms_move_failure_end
		
@@dest_not_valid:
	mov	al, bl		; bl set by xms_get_move_addr.
	jmp	@@xms_move_failure_end
	
@@a20_failure:
	mov	al, XMS_A20_FAILURE 
	jmp	@@xms_move_failure_end

@@illegal_overlap:
	mov	al, XMS_OVERLAP
	jmp	@@xms_move_failure_end

@@invalid_length:
	mov	al, XMS_INVALID_LENGTH
	;  Fall through.
	
@@xms_move_failure_end:	
	popf			; Saved a20 state.
	jnz	@@failure_a20_was_enabled	; if it was enabled, don't disable
	mov	bl, al
	call	disable_a20		; it was disabled, so restore state
	mov	al, bl
	
@@failure_a20_was_enabled:	
	pop	es ds
	pop	esi edi edx ecx ebx
	mov	bl, al
	pop	eax
	xor	ax,ax
	popf
	retf
endp	xms_move_xms

;------------------------------------------------------------------------------
; locks an XMS block
; In:	AH=0ch
;	DX=XMS handle to be locked
; Out:	AX=1 if block is locked
;	  DX:BX=32-bit linear address of block
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a2h -> handle is invalid
;	  BL=0ach -> lock count overflow
;	  BL=0adh -> lock fails

proc	xms_lock_xms
	call	xms_handle_valid
	jc	@@xms_lock_not_valid
	
	mov	bx,dx
	inc	[ cs:bx+xms_handle.used ]	; Increase lock count.
	jnz	@@locked_successful		; go on if no overflow
	dec	[ cs:bx+xms_handle.used ]	; Decrease lock count.
	xor	ax, ax
	mov	bl, XMS_LOCK_COUNT_OVERFLOW	; overflow, return with error
	
@@xms_lock_not_valid:	
	popf
	retf
	
@@locked_successful:
	push	eax				; Save EAX.
	
IFNDEF XXMS
	movzx	eax,[cs:bx+xms_handle.xbase]	; Get block base address
ELSE ; IFDEF XXMS
	mov	eax, [ cs:bx+xms_handle.xbase ]	; Get block base address.
ENDIF ; XXMS
	
	shl	eax,10				; Calculate linear address.
	mov	bx,ax				; Store LSW.
	shr	eax,16
	mov	dx,ax				; Store MSW.
	pop	eax				; Restore EAX.
	mov	ax,1
	popf
	retf
endp	xms_lock_xms

;------------------------------------------------------------------------------
; unlocks an XMS block
; In:	AH=0dh
;	DX=XMS handle to unlock
; Out:	AX=1 if block is unlocked
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a2h -> handle is invalid
;	  BL=0aah -> block is not locked

proc	xms_unlock_xms
	call	xms_handle_valid
	jc	@@xms_unlock_not_valid
	
	push	bx
	mov	bx, dx
	cmp	[ cs:bx+xms_handle.used ], 1	; Check if block is locked.
	ja	@@is_locked			; Go on if true.
	pop	bx
	xor	ax, ax
	mov	bl, XMS_BLOCK_NOT_LOCKED
	popf
	retf
@@is_locked:
	dec	[ cs:bx+xms_handle.used ]	; Decrease lock count.
	pop	bx
	mov	ax, 1
	xor	bl, bl
@@xms_unlock_not_valid:	
	popf
	retf

endp	xms_unlock_xms

;------------------------------------------------------------------------------
; reallocates an UMB
; In:	AH=12h
;	BX=new size for UMB in paragraphs
;	DX=segment of UMB to reallocate
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=0b0h -> no UMB large enough to satisfy request
;	    DX=size of largest UMB in paragraphs
;	  BL=0b2h -> UMB segment is invalid
	
xms_realloc_umb:		; Not pretty but saves memory.
	;; Fall through.

;------------------------------------------------------------------------------
; requests an UMB block
; In:	AH=10h
;	DX=size of requested memory block in paragraphs
; Out:	AX=1 if successful
;	  BX=segment number of UMB
;	  DX=actual size of the allocated block in paragraphs
;	AX=0 if not successful
;	  DX=size of largest available UMB in paragraphs
;	  BL=080h -> function not implemented
;	  BL=0b0h -> only a smaller UMB are available
;	  BL=0b1h -> no UMBs are available

xms_request_umb:		; Not pretty but saves memory.
	xor	dx,dx		; No UMB memory.
	;; Fall through.
	
;------------------------------------------------------------------------------
; releases an UMB block
; In:	AH=11h
;	DX=segment of UMB
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=0b2h -> UMB segment number is invalid

proc	xms_release_umb
	xor	ax,ax
	mov	bl,XMS_NOT_IMPLEMENTED
	popf
	retf
endp	xms_release_umb

IFNDEF XXMS
include "xms_resident.inc"
ELSE ; IFDEF XXMS
include "xxms_resident.inc"
ENDIF ; XXMS
	
IFDEF TRACE_CODE
trace_get_version		db	'get_version',13,10,0
trace_request_hma		db	'request_hma',13,10,0
trace_release_hma		db	'release_hma',13,10,0
trace_global_enable_a20		db	'global_enable_a20',13,10,0
trace_global_disable_a20	db	'global_disable_a20',13,10,0
trace_local_enable_a20		db	'local_enable_a20',13,10,0
trace_local_disable_a20		db	'local_disable_a20',13,10,0
trace_query_a20			db	'query_a20',13,10,0
trace_query_free_xms		db	'query_free_xms',13,10,0
trace_alloc_xms			db	'alloc_xms',13,10,0
trace_free_xms			db	'free_xms',13,10,0
trace_move_xms			db	'move_xms',13,10,0
trace_lock_xms			db	'lock_xms',13,10,0
trace_unlock_xms		db	'unlock_xms',13,10,0
trace_get_handle_info		db	'get_handle_info',13,10,0
trace_realloc_xms		db	'realloc_xms',13,10,0
trace_request_umb		db	'request_umb',13,10,0
trace_release_umb		db	'release_umb',13,10,0
trace_realloc_umb		db	'realloc_umb',13,10,0
IFDEF XXMS
trace_query_any_free_xms	db	'query_any_free_xms',13,10,0
trace_alloc_any_xms		db	'alloc_any_xms',13,10,0
trace_get_extended_handle_info	db	'get_extended_handle_info',13,10,0
trace_realloc_any_xms		db	'realloc_any_xms',13,10,0
ENDIF ; XXMS
ENDIF ; TRACE_CODE

;------------------------------------------------------------------------------
; XMS dispatcher
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; XMS dispatcher
; In:	AH - function number
; Out:	AX=0 -> function not supported
;	else see appr. routine

xms_table	dw	xms_get_version, xms_request_hma
		dw	xms_release_hma, xms_global_enable_a20
		dw	xms_global_disable_a20, xms_local_enable_a20
		dw	xms_local_disable_a20, xms_query_a20
		dw	xms_query_free_xms, xms_alloc_xms
		dw	xms_free_xms, xms_move_xms
		dw	xms_lock_xms, xms_unlock_xms
		dw	xms_get_handle_info, xms_realloc_xms
		dw	xms_request_umb, xms_release_umb
		dw	xms_realloc_umb
IFDEF XXMS
		dw	xms_query_any_free_xms, xms_alloc_any_xms
		dw	xms_get_extended_handle_info, xms_realloc_any_xms
ENDIF ; XXMS
	
IFDEF TRACE_CODE
trace_table	dw	trace_get_version, trace_request_hma
		dw	trace_release_hma, trace_global_enable_a20
		dw	trace_global_disable_a20, trace_local_enable_a20
		dw	trace_local_disable_a20, trace_query_a20
		dw	trace_query_free_xms, trace_alloc_xms
		dw	trace_free_xms, trace_move_xms
		dw	trace_lock_xms, trace_unlock_xms
		dw	trace_get_handle_info, trace_realloc_xms
		dw	trace_request_umb, trace_release_umb
		dw	trace_realloc_umb
IFDEF XXMS
		dw	trace_query_any_free_xms, trace_alloc_any_xms
		dw	trace_get_extended_handle_info, trace_realloc_any_xms
ENDIF ; XXMS
ENDIF ; TRACE_CODE
	
proc	xms_dispatcher
	jmp	short @@dispatcher_entry
	nop					;
	nop					; guarantee hookability
	nop					;
@@dispatcher_entry:
	pushf					; save flags
	cld

	cmp	ah,12h				; is it a supported function?
	jbe	@@supported
	
IFNDEF XXMS
	jmp	@@not_supported
ELSE ; IFDEF XXMS
	;; Convoluted method of mapping (0x88, 0x89, 0x8e, 0x8f) -> 
	;; (0x13, 0x14, 0x15, 0x16) and failing anything else.
	mov	al, ah
	test	ah, 88h
	jz	@@not_supported	; Bits 3 and 7 must be set.

	and	al, 70h
	jnz	@@not_supported	; Bits 4, 5, and 6 must not be set.

	and	ah, 7		; Only interested in bits 0, 1 and 2.
	test	ah, 6
	jz	@@ah88_or_ah89

	test	ah, 2		; Now bit 1 must be one (just like bit 2).
	jz	@@not_supported

	;; So now AH == 6 or 7. Map that to AH == 0x15 or 0x16.
	add	ah, 15h - 6
	jmp	@@supported

@@ah88_or_ah89:
	;; Now, AH == 0 or 1. Map that to AH == 0x13 or 0x14.
	add	ah, 13h
;	jmp	@@supported
ENDIF ; XXMS	

@@supported:	
	call	check_vdisk			; is VDISK installed?
	jz	@@vdisk_installed
hook_patch:
	jmp	short @@hook_ints
@@hook_return:
	push	bx
	movzx	bx, ah
	
IFDEF TRACE_CODE
	push	cx
	mov	cl, bl		;  Put function number in cl for trace shift check.
	shl	bx, 1
	
	;; Check if a trace is wanted.
	push	eax
	mov	eax, 1
	shl	eax, cl
	test	[ cs:trace ], eax
	jz	@@no_trace	;  No trace wanted.
	mov	ax, [ cs:trace_table + bx ]
	call	print_string
	
@@no_trace:	
	pop	eax
	pop	cx
ELSE
	shl	bx, 1	
ENDIF ; TRACE_CODE
	
	mov	ax, [ cs:xms_table + bx ]
	pop	bx
	jmp	ax
@@not_supported:
	xor	ax,ax				; everything else fails
	mov	bl,XMS_NOT_IMPLEMENTED
	popf					; and Yury strikes again...
	retf
@@vdisk_installed:
	xor	ax,ax
	mov	bl,XMS_VDISK_DETECTED
	popf					; flags should be forbidden
	retf

@@hook_ints:
	or	ah,ah				; non-version call?
	jz	@@hook_return			; no, don't patch

IFNDEF XXMS
AX_EAX = ax
ELSE
AX_EAX = eax
ENDIF	
	
	push	AX_EAX bx cx dx ds es		; Save registers.
	mov	ax,cs
	mov	ds,ax
	xor	ax,ax				; get INT15h vector
	mov	es,ax
	les	bx,[es:15h*4]
	mov	[word old_int15+2],es
	mov	[word old_int15],bx

	mov	ax,2515h			; install own INT15h
	mov	dx,offset int15_handler
	int	21h

	mov	cx, [ xms_num_handles ]		; Get number of handles.
	mov	bx, offset driver_end - size xms_handle	; Get start of handle table decreased by one handle.
	inc	cx		; Adjust cx accordingly.
	xor	AX_EAX, AX_EAX	; Zero out (E)AX so it can be use for cmp and mov below.
	
@@find_end_of_handles:
	add	bx, size xms_handle
	cmp	[ bx+xms_handle.xsize ], AX_EAX
	loopne	@@find_end_of_handles
;	jcxz	@@end - Not necessary as we know the last entry ends with xsize == 0.

@@clear_table:
	mov	[ bx+xms_handle.xbase ], AX_EAX	; Blank handle.
	mov	[ bx+xms_handle.xsize ], AX_EAX	; Block doesn't occupy any space.
	mov	[ bx+xms_handle.used ], al	; Handle not used or locked.
	add	bx,size xms_handle
	loop	@@clear_table

@@end:	
	pop	es ds dx cx bx AX_EAX	; Restore registers.

	mov	[word cs:hook_patch],9090h	; insert two NOPs
	jmp	@@hook_return			; and finish it
endp	xms_dispatcher

	
IFDEF TRACE_CODE
include "print_string.inc"
ENDIF ; TRACE_CODE
	
;------------------------------------------------------------------------------
; mark for the driver end. above has to be the resident part, below the
; transient.

label	driver_end	near

;------------------------------------------------------------------------------
; 16-bit transient code and data. only used once.
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; checks if CPU is a 386
; In:	nothing
; Out:	CY=0 - processor is a 386 or higher
;	CY=1 - processor lower than 386

P8086				; This part must be able to run on 8086.
proc	check_cpu
	pushf
	xor	ax,ax
	push	ax
	popf
	pushf
	pop	ax
	and	ah,0fh
	cmp	ah,0fh
	je	@@not386
	mov	ah,7
	push	ax
	popf
	pushf
	pop	ax
	and	ah,7
	je	@@not386
	popf
	clc
	ret
@@not386:
	popf
	stc
	ret
endp	check_cpu
P386				; Back to 386.

;------------------------------------------------------------------------------
; checks if A20 can be enabled and disabled
; Out:	CF=0 - A20 switching works
;	CF=1 - A20 failure

proc	check_a20
	call	enable_a20
	call	test_a20			; TEST_A20 should return ZF=0
	jz	@@a20failed
	call	disable_a20
	call	test_a20			; TEST_A20 should return ZF=1
	jnz	@@a20failed
	clc
	ret
@@a20failed:
	stc
	ret
endp	check_a20


;------------------------------------------------------------------------------
; Saves a string pointed to by fs:ax into command_line. 
; command_line is truncated to a length of 255 bytes and upper cased.
; In:	FS:AX - pointer to string.
; Out:	AL - destroyed.

proc	copy_command_line
	push	di si cx

	mov	cx, XMS_COMMAND_LINE_LENGTH_MAX
	mov	si, ax
	mov	di, offset command_line
@@loop:
	mov	al, [ fs:si ]
	cmp	al, 'a'
	jb	@@do_move
	cmp	al, 'z'
	ja	@@do_move
	;; Must be a lower case letter,
	add	al, 'A'-'a'	; which now is uppercase.
@@do_move:	
	mov	[di], al
	dec	cx
	jcxz	@@too_long
	inc	di
	inc	si
	cmp	al, 0ah
	je	@@too_long	; Stop if we copied 0xa - DOZE have some interesting ideas of ending the command line.
	or	al, al
	jnz	@@loop		; Stop if we did copy nul, else continue.

@@the_end:
	pop	cx si di
	ret

@@too_long:
	mov	[byte di], 0		; Terminate command line.
	jmp	@@the_end
		
endp	copy_command_line

	
;------------------------------------------------------------------------------
; Analyses the contents of command_line and sets variables accordingly.
; In:	Nothing.
; Out:	AX - destroyed.
	
proc	interpret_command_line
	push	es
	push	di si cx bx

	mov	ax, ds
	mov	es, ax
	;; First we must step over FDXMS.SYS, which we do by scaning for the first space, tab or ^M character.
	mov	si, offset command_line		; Remember where search started.
	mov	cx, XMS_COMMAND_LINE_LENGTH_MAX
	add	cx, si				; cx is guard.
	mov	bx, 0920h	; BH=tab, BL=space.
	mov	ah, 0dh		; AH=^M.
	
@@look_again_for_white_space:	
	mov	al, [ si ]
	cmp	al, bl
	je	@@found_white_space
	cmp	al, bh
	je	@@found_white_space
	cmp	al, ah
	je	@@found_white_space
	inc	si
	cmp	si, cx
	jae	@@done
	jmp	@@look_again_for_white_space
			
@@found_white_space:
@@next_arg:
	mov	bx, 0d0ah	; BH=
, BL=^J
	mov	ax, 0920h	; AH=tab, AL=space
	call	eat_characters
	
	cmp	[byte si], 0
	je	@@done		; End of string?
	
	cmp	si, cx
	jae	@@done		; Is this necessary?
	
	mov	bx, si		; Remember current position
	
IFDEF TRACE_CODE
	;; TRACE argument.
	mov	di, offset trace_str
	call	strcmp
	jc	@@try_bios

	push	eax
	cmp	[ byte si ], '='
	je	@@trace_equal_sign
	mov	eax, 0ffffffffh
	jmp	@@trace_no_value

@@trace_equal_sign:	
	inc	si
	call	read_integer
	jc	@@trace_expects_integer
	
@@trace_no_value:
	mov	[ trace ], eax
	pop	eax
	jmp	@@next_arg

@@trace_expects_integer:
	pop	eax
	mov	ax, offset trace_no_int_str
	call	print_string
	jmp	@@next_arg

	;; BIOS argument.
@@try_bios:
	mov	si, bx
	mov	di, offset bios_str
	call	strcmp
	
IFDEF DEBUG_CODE
	jc	@@try_debug
ELSE
	jc	@@try_numhandles
ENDIF ; DEBUG_CODE
	
	or	[ flags ], FLAG_USE_BIOS
	jmp	@@next_arg
ENDIF ; TRACE_CODE

IFDEF DEBUG_CODE
	;; DEBUG argument.
@@try_debug:	
	mov	si, bx
	
	mov	di, offset debug_str
	call	strcmp
	jc	@@try_numhandles

	mov	[ debug ], 1
	jmp	@@next_arg	
ENDIF ; DEBUG_CODE

	;; NUMHANDLES argument.
@@try_numhandles:
	mov	si, bx
	mov	di, offset numhandles_str
	call	strcmp
	jc	@@bad_arg

	push	eax
	cmp	[ byte si ], '='
	jne	@@numhandles_no_value_given

	inc	si
	call	read_integer
	jc	@@numhandles_no_value_given

	cmp	eax, 2
	jae	@@numhandles_at_least_two

	mov	ax, offset numhandles_too_small_str
	call	print_string
	mov	eax, 2

@@numhandles_at_least_two:
	cmp	eax, 1024
	jbe	@@numhandles_value_ok
	
	mov	ax, offset numhandles_too_big_str
	call	print_string
	mov	eax, 1024

@@numhandles_value_ok:
	mov	[ xms_num_handles ], ax
	pop	eax
	jmp	@@next_arg	

@@numhandles_no_value_given:	
	mov	ax, offset numhandles_no_value_str
	call	print_string
	pop	eax
	jmp	@@next_arg

	;; Bad argument.
@@bad_arg:
	mov	ax, offset bad_arg_pre
	call	print_string
	mov	si, bx
	mov	di, offset bad_arg_arg
	
@@bad_arg_loop:	
	mov	al, [si]
	mov	[di], al
	or	al, al
	jz	@@bad_arg_end
	cmp	al, ' '
	je	@@bad_arg_end
	cmp	al, 9		; tab
	je	@@bad_arg_end
	cmp	al, 0ah		; lf
	je	@@bad_arg_end
	cmp	al, 0dh		; cr
	je	@@bad_arg_end
	inc	si
	inc	di
	jmp	@@bad_arg_loop
	
@@bad_arg_end:
	mov	[byte di], 0
	mov	ax, offset bad_arg_arg
	call	print_string
	mov	ax, offset bad_arg_post
	call	print_string	

	mov	al, [si]
	or	al, al
	jz	@@done
	
	inc	si
	jmp	@@next_arg

	;; The end.
@@done:	
	pop	bx cx si di
	pop	es	
	ret	
endp	interpret_command_line
	
IFDEF TRACE_CODE
trace_str		db	'TRACE',0
trace_no_int_str	db	'TRACE expects an integer (e.g. TRACE=0xff)',13,10,0
ENDIF ; TRACE_CODE
	
numhandles_str			db	'NUMHANDLES',0
numhandles_too_small_str	db	'Too small argument to NUMHANDLES increased to 2',13,10,0
numhandles_too_big_str		db	'Too big argument to NUMHANDLES clamped down to 0x400',13,10,0
numhandles_no_value_str		db	'NUMHANDLES expects an argument (e.g. NUMHANDLES=32)',13,10,0
bios_str			db	'BIOS',0
debug_str			db	'DEBUG',0
bad_arg_pre			db	'Ignoring invalid option: ',0
bad_arg_post			db	13,10,0
bad_arg_arg			db	256 dup (?)

;------------------------------------------------------------------------------
; Reads an integer from a string pointed ot by SI.
; In:	SI - pointers to string.
; Out:	CY=0 - successful conversion.
;	  SI - updated.
;	  EAX - integer read.
;	CY=1 - First character wasn't a digit.
;	  EAX - destroyed.
		
proc	read_integer
	push	edx
	xor	eax, eax
	mov	al, [si]
	cmp	al, 0
	je	@@failure
	cmp	al, '0'
	jb	@@failure
	cmp	al, '9'
	ja	@@failure
	sub	al, '0'
	movzx	edx, al
	inc	si
	mov	al, [si]
	cmp	al, 'X'
	je	@@perhaps_hex_number
	
@@decimal_loop:	
	cmp	al, '0'
	jb	@@done
	cmp	al, '9'
	ja	@@done
	lea	edx, [edx+edx*4-'0'/2]	; Luckily zero and '0' are even numbers!
	shl	edx, 1
	add	edx, eax
	inc	si
	mov	al, [si]
	cmp	al, 0
	je	@@done
	jmp	@@decimal_loop
	
@@done:
	mov	eax, edx
	pop	edx
	clc
	ret

@@perhaps_hex_number:
	cmp	dx, 0
	jne	@@done

	inc	si
	mov	al, [si]
	cmp	al, 0
	je	@@looked_like_hex_but_was_not
@@hex_number:	
	cmp	al, '0'
	jb	@@looked_like_hex_but_was_not
	cmp	al, 'F'
	ja	@@looked_like_hex_but_was_not
	cmp	al, '9'
	jbe	@@digit
	cmp	al, 'A'
	jb	@@looked_like_hex_but_was_not
	add	al, '0'-'A'+10	; Sets al to the ASCII code for the characters
				; after '9'.
@@digit:
	sub	al, '0'
	movzx	edx, al
	inc	si
	mov	al, [si]
@@hex_loop:	
	cmp	al, '0'
	jb	@@done
	cmp	al, 'F'
	ja	@@done
	cmp	al, '9'
	jbe	@@another_digit
	cmp	al, 'A'
	jb	@@done
	add	al, '0'-'A'+10	; Sets al to the ASCII code for the characters
				; after '9'.
@@another_digit:
	sub	al, '0'
	shl	edx, 4
	add	edx, eax
	inc	si
	mov	al, [si]
	cmp	al, 0
	je	@@done
	jmp	@@hex_loop
	
@@looked_like_hex_but_was_not:
	dec	si
	jmp	@@done

@@failure:
	pop	edx
	stc
	ret
	
endp	read_integer
;------------------------------------------------------------------------------
; Increases SI until a character that don't match contents of AH, AL, BH or BL is found.
; In:	SI - pointer to string.
;	AL - character to step over
;	AH - character to step over, 0x0 to ignore and to ignore BX contents.
;	BL - character to step over, 0x0 to ignore and to ignore BH contents.
;	BH - character to step over, 0x0 to ignore.
; Out:	SI - updated
	
proc eat_characters
@@loop:	
	cmp	al, [si]
	jne	@@try_ah
	inc	si
	jmp	@@loop
@@try_ah:
	or	ah, ah
	jz	@@done
	cmp	ah, [si]
	jne	@@try_bl
	inc	si
	jmp	@@loop
@@try_bl:
	or	bl, bl
	jz	@@done
	cmp	bl, [si]
	jne	@@try_bh
	inc	si
	jmp	@@loop
@@try_bh:
	or	bh, bh
	jz	@@done
	cmp	bh, [si]
	jne	@@done
	inc	si
	jmp	@@loop

@@done:	
	ret
endp	eat_characters
	
;------------------------------------------------------------------------------
; Compares two strings up to DI points at nul.
; In:	DI, SI - pointers to strings to compare.
; Out:	CY=0 - strings equal
;	  DI - points at null
;	  SI - points at character where DI is null.
;	  AX - destroyed
;	CY=1 - strings not equal
;	  DI - points at character which diffs
;	  SI - points at character which diffs
;	  AX - destroyed
		
proc	strcmp
@@loop:	
	mov	al, [di]
	cmp	al, 0		; Is it nul?
	je	@@done		; Yes, we have a match!
	
	cmp	al, [si]
	jne	@@failure	; Not equal so no match.
	inc	si
	inc	di
	jmp	@@loop

@@failure:	
	stc
	ret
	
@@done:
	clc
	ret		
endp	strcmp

;------------------------------------------------------------------------------
; initializes the driver. called only once!
; may modify DI
; In:	ES:DI - pointer to init structure
;	DS - initialized with code segment

init_message		db	13,10,80 dup (''),'FreeDOS XMS-Driver',13,10
			db	'Copyright (c) 1995, Till Gerken',13,10
			db	'Copyright 2001, Martin Strmberg',13,10,13,10
			db	'Driver Version: ',DRIVER_VERSION,' ',9
			db	'Interface Version: ',INTERFACE_VERSION,13,10
			db	'Information: ',INFO_STR,13,10,'$'

old_dos			db	'XMS needs at least DOS version 3.00.$'
xms_twice		db	'XMS is already installed.$'
vdisk_detected		db	'VDISK has been detected.$'
no_386			db	'At least a 80386 is required.$'
a20_error		db	'Unable to switch A20 address line.$'
xms_memory_hole		db	'Memory hole below 16MiB detected. This release of FDXMS does not support', 13, 10
			db	'this. Using only memory below 16MiB. Please contact the maintainer!', 13, 10, '$'
xms_sizeerr		db	'Unable to determine size of extended memory.$'
xms_toosmall		db	'Extended memory is too small or not available.$'
	
error_msg		db	' Driver won''t be installed.',7,13,10,'$'
	
command_line		db	XMS_COMMAND_LINE_LENGTH_MAX dup (0), 0
	
proc	initialize
P8086				; This part must be able to run on 8086.
	pushf
	push	si
	push	ax bx cx dx

	cld

	mov	ah,9				; first, welcome the user!
	mov	dx,offset init_message
	int	21h

	mov	ax,3000h			; get DOS version number
	int	21h
	xchg	ah,al				; convert to BSD
	cmp	ax,300h				; we need at least 3.00
	mov	dx,offset old_dos
	jb	@@error_exit

	mov	ax,4300h			; check if XMS is already
	int	2fh				; installed
	cmp	al,80h
	mov	dx,offset xms_twice
	je	@@error_exit

	call	check_cpu			; do we have at least a 386?
	mov	dx,offset no_386
	jc	@@error_exit

P386				; Now we know we have at least a 386.
	pop	dx cx bx ax	; Restore all saved registers except si,
	push	eax ebx ecx edx	; and save the 32 bit variants instead
	push	edi		; plus EDI.
	
	call	check_a20			; check A20
	mov	dx,offset a20_error
	jc	@@error_exit_386

	call	check_vdisk			; is VDISK installed?
	mov	dx,offset vdisk_detected
	jz	@@error_exit_386
	
	;; Check command line.
	push	fs
	lfs	ax, [es:di+init_strc.cmd_line]
	call	copy_command_line
	pop	fs

	call	interpret_command_line
		
	;; SI points to start of XMS handle table.
	mov	si, offset driver_end

	;; Find memory.
	
IFDEF XXMS
	call	int15ax0xe820	; Try INT15 EAX=0xe820 first.
	
	cmp	eax, 0			; If INT15 EAX=0xe820 did find any 
	jne	@@print_found_memory	; memory, there's no need to try the
					; other functions.	
ENDIF ; XXMS

@@try_int15_axe801:	
	clc
	mov	ax, 0e801h
	int	15h
	jc	@@try_int15_ah88

	mov	di, ax
	or	di, bx		; If both AX and BX aren't zero use the values.
	jnz	@@use_ax_bx

	;; AX and BX was zero, use CX and DX instead.
	mov	bx, dx
	mov	ax, cx
	
@@use_ax_bx:
	cmp	ax, 3c00h
	je	@@no_memory_hole

	or	bx, bx
	jz	@@no_memory_hole
	
	;; We have a hole in memory, just below 16MiB. 
	mov	bx, ax		; Save AX.
	mov	ah, 9
	mov	dx, offset xms_memory_hole
	int	21h
	mov	ax, bx		; Restore AX.
	xor	bx, bx		; Use only memory below 16MiB.
	
@@no_memory_hole:
	and	eax, 0ffffh
	and	ebx, 0ffffh
	shl	ebx, 6
	add	eax, ebx
	
	jmp	@@found_size

@@try_int15_ah88:	
	clc
	mov	ah,88h				; extended memory size
	int	15h
	mov	dx, offset xms_sizeerr
	jc	@@error_exit_386
	and	eax, 0ffffh
	
@@found_size:	
	mov	dx, offset xms_toosmall
	sub	eax, 64				; Save HIMEM area.
	jc	@@error_exit_386		; If there aren't 64ki,
						; there's nothing to do.
	
IFNDEF XXMS
	cmp	eax, 0fbc0h
	jbe	@@fill_first_entry

	mov	ah, 9
	mov	dx, offset xms_size_clamped
	int	21h
	mov	eax, 0fbc0h
		
@@fill_first_entry:	
	mov	[ si+xms_handle.xbase] ,XMS_START	; Init first block and
	mov	[ si+xms_handle.xsize ], ax		; give it all 
							; available memory.
	and	eax, 0ffffh			; Zero out bits 16-31.
ELSE ; IFDEF XXMS
	lea	ecx, [ eax + XMS_START ]	; Calculate ending address
	shl	ecx, 10
	dec	ecx
	mov	[ xms_highest_address ], ecx	; and store it.
	
	mov	[ si+xms_handle.xbase ], XMS_START	; Init first block and
							; give it all
	mov	[ si+xms_handle.xsize ], eax		; available memory.

ENDIF ; XXMS
	
	mov	[ si+xms_handle.used ], 0	; Handle not used or locked.
	add	si, size xms_handle

	shl	eax, 10		; Convert to bytes.


@@print_found_memory:	
	; EAX contains the size of XMS in bytes now. Print what we've found.
	mov	ebx, eax		; Save EAX.
	call	print_hex_number
	mov	ah, 9
	mov	dx, offset xms_mem_found_middle
	int	21h
	mov	eax, ebx
	call	print_dec_number
	mov	ah, 9
	mov	dx, offset xms_mem_found_post
	int	21h

	mov	[ si+xms_handle.xsize ], 0		; Mark end of initialised handles.

	mov	ax,cs				; setup descriptors
	mov	[code_seg],ax			; eliminate relocation entry
	movzx	eax,ax
	shl	eax,4
	or	[dword code16dsc+2],eax
	add	[dword gdt32+2],eax

	push	es
	xor	ax,ax				; get INT2Fh vector
	mov	es,ax
	les	bx,[es:2fh*4]
	mov	[word old_int2f+2],es
	mov	[word old_int2f],bx

	mov	ax,252fh			; install own INT2Fh
	mov	dx,offset int2f_handler
	int	21h
	pop	es

	pop	edi		; Restore EDI.
	mov	[word es:di+2+init_strc.end_addr],cs	; set end address
	xor	dx,dx
	mov	ax,size xms_handle
	mul	[xms_num_handles]
	add	ax,offset driver_end
	mov	[word es:di+init_strc.end_addr],ax
	mov	[es:di+request_hdr.status],STATUS_OK	; we're alright
	jmp	short @@exit_386

@@error_exit_386:
	pop	edi		; Restore EDI.
	mov	[word es:di+2+init_strc.end_addr],cs	; set end address
	mov	[word es:di+init_strc.end_addr],0	; now, we don't use
							; any space
	mov	[es:di+request_hdr.status],STATUS_BAD	; waaaah!
	mov	ah,9					; print msg
	int	21h
	mov	dx,offset error_msg
	int	21h

@@exit_386:
	mov	ah,9
	mov	dx,offset init_finished
	int	21h
	or	[ flags ], FLAG_INITIALISED	; Mark that we have initialised.
	pop	edx ecx ebx eax
	pop	si
	popf
	ret
	
@@error_exit:
P8086				; This part must be able to run on 8086.
	mov	[word es:di+2+init_strc.end_addr],cs	; set end address
	mov	[word es:di+init_strc.end_addr],0	; now, we don't use
							; any space
	mov	[es:di+request_hdr.status],STATUS_BAD	; waaaah!
	mov	ah,9					; print msg
	int	21h
	mov	dx,offset error_msg
	int	21h
	mov	ah,9
	mov	dx,offset init_finished
	int	21h
	pop	dx cx bx ax si
	popf
	ret
P386				; Back to 386.
endp	initialize

;------------------------------------------------------------------------------
; Prints a hexadecimal number, using INT21, AH=0x9.
; In:	EAX - the number to be printed.
;	DS - initialized with code segment
number	db	'0x????????$'
proc	print_hex_number
	push	bx cx edx eax
	mov	bx, offset number + 2
	mov	cl, 28
	mov	edx, eax

@@loop:	
	mov	eax, edx
	shr	eax, cl
	and	al, 0fh
	cmp	al, 0ah
	jb	@@digit
	add	al, 'A'-'0'-10
@@digit:
	add	al, '0'
	mov	[ bx ], al
	inc	bx
	sub	cl, 4
	jb	@@print
	jmp	@@loop

@@print:
	mov	dx, offset number
	mov	ah, 9
	int	21h
	
	pop	eax edx cx bx
	ret
endp	print_hex_number

	
;------------------------------------------------------------------------------
; Prints a decimal number, using INT21, AH=0x9.
; In:	EAX - the number to be printed.
;	DS - initialized with code segment
dec_number	db	'429496729'
dec_num_start	db	'6$'
proc	print_dec_number
	push	bx edx ecx eax
	mov	bx, offset dec_num_start
	mov	ecx, 10

@@loop:
	xor	edx, edx
	div	ecx
	add	dl, '0'
	mov	[ bx ] , dl
	or	eax, eax	; Any digits left?
	jz	@@print
	dec	bx
	jmp	@@loop

@@print:
	mov	dx, bx
	mov	ah, 9
	int	21h
	
	pop	eax ecx edx bx
	ret
endp	print_dec_number

	
IFDEF XXMS
include	"xxms_nonresident.inc"	
ENDIF ; XXMS

	
IFNDEF TRACE_CODE
include "print_string.inc"
ENDIF ; TRACE_CODE	
	
;------------------------------------------------------------------------------
;; Some variables that must survive initial memory block fills.
	
xms_size_clamped	db	'Size clamped down to 64MB (0xfbc0 kiB to be precise).', 13, 10, '$'
	
init_finished		db	80 dup (''),13,10,'$'

xms_mem_found_middle	db	' ($'
xms_mem_found_post	db	' decimal) bytes XMS memory detected', 13, 10, '$'
	

ends	code16

;******************************************************************************

end
