;------------------------------------------------------------------------------
; returns free XMS
; In:	AH=8
; Out:	AX=size of largest free XMS block in kbytes
;	DX=total amount of free XMS in kbytes
;	BL=0 if ok
;	BL=080h -> function not implemented
;	BL=081h -> VDISK is detected
;	BL=0a0h -> all XMS is allocated

proc	xms_query_free_xms
	push	ecx eax edx

	push	cs
	push	offset @@return
	pushf
	jmp	xms_query_any_free_xms

@@return = $
	;; Limit EDX to 0xffff kiB.
	cmp	edx, 0ffffh
	jb	@@dx_ok

	mov	dx, 0ffffh
@@dx_ok:
	mov	cx, dx
	pop	edx
	mov	dx, cx

	;; Limit EAX to 0xffff kiB.
	cmp	eax, 0ffffh
	jb	@@ax_ok

	mov	ax, 0ffffh
@@ax_ok:
	mov	cx, ax
	pop	eax
	mov	ax, cx

	
	pop	ecx
	
	popf
	retf

endp	xms_query_free_xms

;------------------------------------------------------------------------------
; allocates an XMS block
; In:	AH=9
;	DX=amount of XMS being requested in kbytes
; Out:	AX=1 if successful
;	  DX=handle
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a0h -> all XMS is allocated
;	  BL=0a1h -> no free handles left

proc	xms_alloc_xms
	push	cx edx

	mov	ax, dx
	xor	edx, edx
	mov	dx, ax
	
	push	cs
	push	offset @@return
	pushf
	jmp	xms_alloc_any_xms
	
@@return:	
	mov	cx, dx
	pop	edx
	mov	dx, cx
	pop	cx
	popf
	retf

endp	xms_alloc_xms

;------------------------------------------------------------------------------
; returns XMS handle information
; In:	AH=0eh
;	DX=XMS block handle
; Out:	AX=1 if successful
;	  BH=block's lock count
;	  BL=number of free XMS handles
;	  DX=block's length in kbytes
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a2h -> handle is invalid

proc	xms_get_handle_info
	push	cx edx		; Save cx and edx.

	push	cs
	push	offset @@return
	pushf
	jmp	xms_get_extended_handle_info

@@return:	
	or	ax, ax
	jz	@@failure
	
	test	edx, 0ffff0000h	;  Is EDX to big?
	jnz	@@failure_edx
	
	mov	ax, dx
	pop	edx
	mov	dx, ax
	
	cmp	cx, 0ffh
	jbe	@@cx_not_too_big
	mov	cl, 0ffh
@@cx_not_too_big:	
	mov	bl, cl

	pop	cx
	mov	ax, 1
	popf
	retf

@@failure_edx:
	mov	bl, XMS_INVALID_HANDLE
	xor	ax, ax
	
@@failure:
	pop	edx cx
	popf
	retf

endp	xms_get_handle_info

;------------------------------------------------------------------------------
; reallocates an XMS block. only supports shrinking.
; In:	AH=0fh
;	BX=new size for the XMS block in kbytes
;	DX=unlocked XMS handle
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a0h -> all XMS is allocated
;	  BL=0a1h -> all handles are in use
;	  BL=0a2h -> invalid handle
;	  BL=0abh -> block is locked

proc	xms_realloc_xms
	push	ebx

	mov	ax, bx
	xor	ebx, ebx
	mov	bx, ax

	push	cs
	push	offset @@return
	pushf
	jmp	xms_realloc_any_xms

@@return:	
	or	ax, ax
	jz	@@failure

	pop	ebx
	popf
	retf

@@failure:
	mov	al, bl
	pop	ebx
	mov	bl, al
	xor	ax, ax
	popf
	retf
endp	xms_realloc_xms

;------------------------------------------------------------------------------
; returns free (X)XMS
; In:	AH=88h
; Out:	EAX=size of largest free XMS block in kiB
;	ECX=highest ending address of any memory block
;	EDX=total amount of free XMS in kiB
;	BL=0 if ok
;	BL=080h -> function not implemented
;	BL=081h -> VDISK is detected
;	BL=0a0h -> all XMS is allocated

proc	xms_query_any_free_xms
	push	bx ds esi

	mov	ax,cs
	mov	ds,ax

	mov	cx, [ xms_num_handles ]
	mov	bx, offset driver_end
	
	xor	esi, esi			; Contains largest free block.
	xor	edx, edx			; Contains total free XMS.
	
@@check_next:
	cmp	[ bx+xms_handle.used], 0
	jne	@@in_use

	mov	eax, [ bx + xms_handle.xsize ]	; Get size.
	add	edx, eax
	cmp	eax, esi	; Is this block bigger than what we've found so far?
	jbe	@@not_larger
	
	mov	esi, eax	; Larger, update.
	
@@in_use:
@@not_larger:
	add	bx, size xms_handle
	loop	@@check_next

	mov	eax, esi	; Move largest free size into EAX.
	
	mov	ecx, [ xms_highest_address ]
	pop	esi ds bx
	
	or	eax, eax	; Is there any free memory?
	jz	@@no_free_xms
	
	xor	bl, bl
	popf
	retf

@@no_free_xms:
	mov	bl, XMS_ALREADY_ALLOCATED
	popf
	retf
endp	xms_query_any_free_xms

;------------------------------------------------------------------------------
; allocates an (X)XMS block
; In:	AH=89h
;	EDX=amount of XMS being requested in kbytes
; Out:	AX=1 if successful
;	  DX=handle
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a0h -> all XMS is allocated
;	  BL=0a1h -> no free handles left

proc	xms_alloc_any_xms
	push	eax bx cx si ds

	mov	ax,cs
	mov	ds,ax
	
	call	xms_find_free_block	; See if there's a free block.
	jc	@@no_free_memory	; If there isn't, fail.
	jmp	@@check_size
	
@@get_next_block:
	call	xms_find_next_free_block
	jc	@@no_free_memory
@@check_size:
	mov	eax, [ bx+xms_handle.xsize ]	; Check if it's large enough.
	cmp	edx, eax
	ja	@@get_next_block		; No, get next block.

	mov	si, bx			; Save handle.
	inc	[ bx+xms_handle.used ]	; This block is used from now on.

@@find_handle:	
	call	xms_find_free_handle	; see if there's a blank handle
	jc	@@perfect_fit		; no, there isn't, alloc all mem left

	sub	eax, edx			; Calculate remaining memory.
	jz	@@perfect_fit			; if it fits perfectly, go on
	
	mov	[ bx+xms_handle.xsize ], eax	; Store sizes of new blocks.
	mov	[ si+xms_handle.xsize ], edx
	mov	eax, [ si+xms_handle.xbase ]	; Get base address of old block.
	add	eax, edx			; Calculate new base address.
	mov	[ bx+xms_handle.xbase ], eax	; Store it in new handle.

@@perfect_fit:
	mov	dx, si				; Return handle in DX.

	;; Success.
	pop	ds si cx bx eax
	mov	ax,1
	xor	bl,bl
	popf
	retf
	
@@no_free_memory:
	;; If no memory was asked for, just allocate a handle.
	or	edx, edx
	jz	@@zero_size_allocation
	
	mov	al, XMS_ALREADY_ALLOCATED
	jmp	@@failure_epilogue

@@zero_size_allocation:
	call	xms_find_free_handle	; see if there's a blank handle
	jc	@@no_handles_left	; No, there isn't, fail.

	mov	si, bx			; Save handle address.
	
	;; We have the handle. Mark it as used.
	inc	[ bx + xms_handle.used ]
	jmp	@@perfect_fit

@@no_handles_left:
	mov	al, XMS_NO_HANDLE_LEFT
	
@@failure_epilogue:	
	pop	ds si cx bx eax
	mov	bl, al
	xor	ax, ax
	popf
	retf

endp	xms_alloc_any_xms

;------------------------------------------------------------------------------
; returns (X)XMS handle information
; In:	AH=8eh
;	DX=XMS block handle
; Out:	AX=1 if successful
;	  BH=block's lock count
;	  CX=number of free XMS handles
;	  EDX=block's length in kiB
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a2h -> handle is invalid

proc	xms_get_extended_handle_info
	call 	xms_handle_valid
	jc	@@not_valid

	push 	ds bx		; Save ds and bx
	mov	ax, cs		; and setup ds to this segment.
	mov	ds, ax

	push	dx		; Save handle for later.

	xor	ax, ax		; ax is number of free handles.

	;; Setup for loop.
	mov	bx, offset driver_end
	mov	cx, [ xms_num_handles ]
	
@@look_again:
	cmp	[ bx + xms_handle.used ], 0	; In use?
	jne	@@add_some			; Yes, go to next.
	inc	ax				; No, one more free handle.

@@add_some:
	add	bx, size xms_handle
	loop	@@look_again

	;;  Now ax contains number of free handles.
	mov	cx, ax				; Store number of free handles.
	
	pop 	bx 				; Get handle saved earlier.
	mov	edx, [ bx+xms_handle.xsize ]	; Store block size.
	mov	al, [ bx+xms_handle.used ]	; Store lock count.
	dec	al
	
	pop	bx		; Restore BL and
	mov	bh, al		; fill in BH with lock count.

	pop	ds		; Restore ds.
	mov	ax, 1		; Success.

@@not_valid:	
	popf
	retf

endp	xms_get_extended_handle_info


;------------------------------------------------------------------------------
; reallocates an XMS block. only supports shrinking.
; In:	AH=8fh
;	EBX=new size for the XMS block in kbytes
;	DX=unlocked XMS handle
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a0h -> all XMS is allocated
;	  BL=0a1h -> all handles are in use
;	  BL=0a2h -> invalid handle
;	  BL=0abh -> block is locked

proc	xms_realloc_any_xms
	call	xms_handle_valid
	jc	@@xms_realloc_not_valid
	
	push	eax ebx ecx edx si ds
	mov	ax,cs
	mov	ds,ax

	xchg	ebx, edx
	mov	eax, [ bx+xms_handle.xsize ] 
	cmp	edx, eax
	jbe	@@shrink_it

@@no_xms_handles_left:
	pop	ds si edx ecx ebx eax
	xor	ax,ax
	mov	bl,XMS_NO_HANDLE_LEFT		; simulate a "no handle" error
	popf
	retf

@@shrink_it:
	mov	si, bx
	call	xms_find_free_handle		; Get a blank handle.
	jc	@@no_xms_handles_left		; Return if there's an error.
	
	mov	[ si+xms_handle.xsize ], edx
	
	sub	eax, edx			; Calculate what's left over.
	jz	@@dont_need_handle		; Skip if we don't need it.
	
	add	edx, [ si+xms_handle.xbase ]	; Calculate new base address.
	mov	[ bx+xms_handle.xbase ], edx	; Store it.
	mov	[ bx+xms_handle.xsize ], eax	; Store size.
	mov	[ bx+xms_handle.used ], 0	; Block is not locked or used.
	
@@dont_need_handle:
	pop	ds si edx ecx ebx eax
	mov	ax, 1
	xor	bl, bl
	
@@xms_realloc_not_valid:	
	popf
	retf
endp	xms_realloc_any_xms

