; Mouse Protocol Analyzer for serial mice
; Copyright (c) 1997-2002 Nagy Daniel <nagyd@users.sourceforge.net>
;
; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;
; History:
;
; 1.6 - by Arkady V.Belousov <ark@mos.ru>
;	Heavy optimizations
;	Source synchronized with CTMOUSE source
;	Added external assembler library
;	Keystroke now break also PNP input loop
;
; 1.5 - by Arkady V.Belousov <ark@mos.ru>
;	Small bugfixes and optimizations
;	Mouse events bytes printing moved out from IRQ handler
;
; 1.4 - by Arkady V.Belousov <ark@mos.ru>
;	Only first argument (COM port number) now is required
;
; 1.3 - by Arkady V.Belousov <ark@mos.ru>
;	Added parsing and showing PNP data
;
; 1.2 - by Arkady V.Belousov <ark@mos.ru>
;	Source synchronized with CTMOUSE source
;	Added command line option for COM LCR value
;	Added dumping of reset sequence, generated by mouse
;
; 1.1 - Added command line option for COM port selection
;
; 1.0 - First public release
;

WARN
%NOINCL
LOCALS
.model tiny

INCLUDE asm.mac
INCLUDE macro.mac
INCLUDE BIOS/area0.def
INCLUDE convert/digit.mac
INCLUDE DOS/io.mac
INCLUDE DOS/mem.mac
INCLUDE hard/PIC8259A.def
INCLUDE hard/UART.def


; DATA SEGMENT 

.data?

oldIRQaddr	dd	?		; old IRQ handler address

.data

S_COMinfo	db	'Dump data stream from working mouse.'
		db	' Press ESC to exit...',0dh,0ah
		db	'1200 bps, '
databits	db	'5 data bits, '
stopbits	db	'1 stop bit.'
		db	0dh,0ah,'Reset:$'
S_spaces	db	0dh,0ah,' '
S_bitbyte	db		 '     $'
S_byte		db	'   $'

.const

S_wrongPNP	db	'Wrong PNP data...'
CRLF2		db	0dh,0ah
CRLF		db	0dh,0ah,'$'
PNP_OEM		db	'Manufacturer: $'
PNP_header	db	'  Product ID: $'
PNP_hdnext	db	'    Serial #: $'
		db	'       Class: $'
		db	'   Driver ID: $'
		db	'   User name: $'
PNP_hdend	label

S_notCOM	db	'COM port not found!',0dh,0ah,'$'
Syntax		db	'Syntax: protocol <COM (1-4)> [<bytes in event (3-5)>'
		db	' [<COM LCR value (2-3)>]]',0dh,0ah,'$'


; CODE SEGMENT 

.code
		org	100h
start:		cld
		mov	si,80h
		lodsb
		cbw				; OPTIMIZE: instead MOV AH,0
		movadd	di,si,ax

		call	skipwhite
		jc	HELP
		mov	dx,offset DGROUP:S_notCOM
		dec	ax			; OPTIMIZE: AX instead AL
		cmp	al,3
		ja	EXITERRMSG
		call	setCOMport
		;mov	dx,offset DGROUP:S_notCOM
		jc	EXITERRMSG

		call	skipwhite
		jc	processmouse
		mov	[limit],al

		call	skipwhite
		jc	processmouse
		mov	[LCRset],al

		call	skipwhite
		jc	processmouse

HELP:		mov	dx,offset DGROUP:Syntax
EXITERRMSG:	DOSWriteS
		int	20h

;

skipwhite	proc
		cmp	di,si
		jb	@@skipret		; JB mean CF=1
		lodsb
		cmp	al,' '
		jbe	skipwhite
		sub	al,'0'
		jb	HELP
		;clc
@@skipret:	ret
skipwhite	endp

;

processmouse	proc
		mov	ax,1Fh			; disable mouse
		call	mousedrv

;---------- initialize mouse and dump PNP info
		mov	si,[IO_address]
		call	disableUART
		call	resetmouse

;---------- install IRQ handler and enable interrupt
		mov	al,[IRQintnum]
		DOSGetIntr
		saveFAR [oldIRQaddr],es,bx	; save old IRQ handler
		;mov	al,[IRQintnum]
		DOSSetIntr ,,,DGROUP:IRQhandler
		call	enableUART

;========== process mouse data until keystroke
		MOVSEG	es,ds,,DGROUP
		mov	bx,offset DGROUP:buffer

@@loop:		hlt
		mov	ah,1
		int	16h			; check for keystroke
		jnz	@@finish
		cmp	bx,[buffer@]
		je	@@loop

		cmp	bx,offset DGROUP:buffer+BUFFERSZ
		jb	@@getbyte
		mov	bx,offset DGROUP:buffer
@@getbyte:	mov	ah,[bx]
		inc	bx

		mov	di,offset DGROUP:S_bitbyte+1
		mov	cx,8			; 8 bits
@@nextbit:	mov	al,'0' SHR 1
		rol	ax,1
		stosb
		loop	@@nextbit
		DOSWriteS ,,DGROUP:S_bitbyte

		db	0B0h			; MOV AL,byte
IOdone		db	0			; processed bytes counter
		inc	ax			; OPTIMIZE: AX instead AL
		db	03Ch			; CMP AL,byte
limit		db	3
		jb	@@nextdone
		DOSWriteS ,,DGROUP:CRLF
		mov	al,0			; restart counter of bytes
@@nextdone:	mov	[IOdone],al
		j	@@loop

;========== final: flush keystroke, deinstall handler and exit
@@finish:	mov	ah,0
		int	16h			; get keystroke

		call	disableUART
		db	0B8h			; MOV AX,word
IRQintnum	db	?,25h			; INT number of selected IRQ
		lds	dx,[oldIRQaddr]
		assume	ds:nothing
		int	21h			; set INT in DS:DX

;---------- reset mouse and exit
		xor	ax,ax			; reset mouse
		call	mousedrv
		int	20h
processmouse	endp
		assume	ds:DGROUP

;

mousedrv	proc
		push	ax bx es
		DOSGetIntr 33h
		mov	ax,es
		test	ax,ax
		pop	es bx ax
		jz	@@mouseret
		int	33h
@@mouseret:	ret
mousedrv	endp

;

setCOMport	proc
		MOVSEG	es,0,bx,BIOS
		mov	bl,al
		shl	bx,1
		mov	cx,COM_base[bx]
		stc
		jcxz	@@setcomret
		mov	[IO_address],cx

		inc	ax			; OPTIMIZE: AX instead AL
		and	al,1			; 1=COM1/3, 0=COM2/4
		add	al,3			; IRQ4 for COM1/3
		mov	cl,al			; IRQ3 for COM2/4
		add	al,8			; INT=IRQ+8
		mov	[IRQintnum],al
		mov	al,1
		shl	al,cl			; convert IRQ into bit mask
		mov	[PIC1state],al		; PIC interrupt disabler
		not	al
		mov	[notPIC1state],al	; PIC interrupt enabler
		;clc
@@setcomret:	ret
setCOMport	endp


; COMM ROUTINES 

disableUART	proc
		in	al,PIC1_IMR		; {21h} get IMR
		db	0Ch			; OR AL,byte
PIC1state	db	?			; set bit to disable interrupt
		out	PIC1_IMR,al		; {21h} disable serial interrupts
;----------
		movidx	dx,LCR_index,si		; {3FBh} LCR: DLAB off
		 out_	dx,%LCR<>,%MCR<>	; {3FCh} MCR: DTR/RTS/OUT2 off
		movidx	dx,IER_index,si,LCR_index
		 ;mov	ax,(FCR<> shl 8)+IER<>	; {3F9h} IER: interrupts off
		 out	dx,ax			; {3FAh} FCR: disable FIFO
		ret
disableUART	endp

;

enableUART	proc
		movidx	dx,MCR_index,si
		 out_	dx,%MCR<,,,1,1,1,1>	; {3FCh} MCR: DTR/RTS/OUTx on
		movidx	dx,IER_index,si,MCR_index
		 out_	dx,%IER{IER_DR=1}	; {3F9h} IER: enable DR intr
;----------
		in	al,PIC1_IMR		; {21h} get IMR
		db	24h			; AND AL,byte
notPIC1state	db	?			; clear bit to enable interrupt
		out	PIC1_IMR,al		; {21h} enable serial interrupts
		ret
enableUART	endp

;

resetmouse	proc
		mov	al,[LCRset]
		maskflag al,mask LCR_stop+mask LCR_wordlen
		mov	cl,8-LCR_stop
		shl	ax,cl			; LCR_stop
		shr	al,cl			;  > LCR_wordlen
		add	[stopbits],ah
		add	[databits],al
		DOSWriteS ,,DGROUP:S_COMinfo

;---------- set communication parameters
		movidx	dx,LCR_index,si
		 out_	dx,%LCR{LCR_DLAB=1}	; {3FBh} LCR: DLAB on
		xchg	dx,si			; 1200 baud rate
		 outw	dx,96			; {3F8h},{3F9h} divisor latch
		xchg	dx,si
		 db	0B0h			; MOV AL,byte
LCRset		 LCR	<0,,LCR_noparity,0,2>
		 out	dx,al			; {3FBh} LCR: DLAB off, 7/8N1

;---------- wait current+next timer tick and then raise RTS line
		MOVSEG	es,0,ax,BIOS
@@RTSloop:	mov	ah,byte ptr [BIOS_timer]
@@RTSwait:	cmp	ah,byte ptr [BIOS_timer]
		 je	@@RTSwait		; loop until next timer tick
		xor	al,1
		 jnz	@@RTSloop		; loop until end of 2nd tick

		movidx	dx,MCR_index,si,LCR_index
		 out_	dx,%MCR<,,,0,,1,1>	; {3FCh} MCR: DTR/RTS on, OUT2 off

;---------- read and show reset sequence, generated by mouse
		mov	bx,20+1			; output counter
		mov	di,offset DGROUP:PNPdata
@@inloop:	mov	cx,2+1			; length of silence in ticks
						; (include rest of curr tick)
@@readloop:	mov	ah,byte ptr [BIOS_timer]
@@readwait:	movidx	dx,LSR_index,si
		 in	al,dx			; {3FDh} LSR (line status reg)
		testflag al,mask LSR_RBF
		 jnz	@@newbyte		; jump if data ready
		cmp	ah,byte ptr [BIOS_timer]
		 je	@@readwait		; loop until next timer tick
		loop	@@readloop		; loop until end of 2nd tick
		j	@@parsePNP		; stream terminated by silence

;---------- save and show next byte
@@newbyte:	dec	bx
		jnz	@@savebyte
		 DOSWriteS ,,DGROUP:S_spaces
		 mov	bl,20

@@savebyte:	movidx	dx,RBR_index,si
		 in	al,dx			; {3F8h} receive byte

		mov	cx,sp
		sub	cx,di
		test	ch,ch			; ZF=1 if CX<256
		jz	@@showbyte
		 mov	[di],al
		 inc	di

@@showbyte:	call	byte2hexa
		mov	word ptr S_byte[1],ax
		DOSWriteS ,,DGROUP:S_byte

		mov	ah,1
		int	16h
		jz	@@inloop		; loop until keystroke
		j	@@resetdone		; then exit

;---------- parse and show PNP data
@@parsePNP:	mov	cx,di
		mov	di,offset DGROUP:PNPdata

@@findstart:	cmp	di,cx			; find PNP data start '('
		jae	@@resetdone
		inc	di
		cmp	byte ptr [di-1],'('-20h
		jne	@@findstart

		DOSWriteS ,,DGROUP:CRLF2
		mov	dx,offset DGROUP:S_wrongPNP

		movadd	di,,2-1			; skip "(!D"
		mov	bx,di
		inc	bx			; BX=start of PNP data
		mov	al,')'-20h+3*20h	; count checksum in AL
@@checksum:	inc	di
		cmp	di,cx
		jae	@@resetexit
		add	al,[di-3]
		sub	al,20h
		add	byte ptr [di],20h	; ...and decode PNP data
		cmp	byte ptr [di],')'	; ...until PNP data end ')'
		jne	@@checksum

		movsub	di,,2			; verify checksum
		call	byte2hexa		; convert checksum to ASCII
		cmp	ax,[di]
		jne	@@resetexit

		mov	cx,3
		DOSWriteS ,,DGROUP:PNP_OEM	; show "Manufacturer" field
@@print1loop:	mov	dl,[bx]
		inc	bx
		DosWriteC
		loop	@@print1loop

		mov	cx,offset DGROUP:PNP_header
@@print2:	DOSWriteS ,,DGROUP:CRLF		; show all other PNP fields
		mov	dx,cx
		;mov	ah,9
		int	21h			; write string in DS:DX
@@print2loop:	cmp	bx,di
		jae	@@print2next
		mov	dl,[bx]
		inc	bx
		cmp	dl,'\'
		je	@@print2next
		DosWriteC
		j	@@print2loop
@@print2next:	add	cx,PNP_hdnext-PNP_header
		cmp	cx,offset DGROUP:PNP_hdend
		jb	@@print2

@@resetdone:	mov	dx,offset DGROUP:CRLF2
@@resetexit:	DOSWriteS
		ret
resetmouse	endp

INCLUDE convert/byte2xa.inc


; IRQ HANDLER 

IRQhandler	proc
		assume	ds:nothing,es:nothing,ss:nothing
		push	ax dx bx
		db	0BAh			; MOV DX,word
IO_address	dw	?			; UART IO address
		push	dx
		movidx	dx,LSR_index
		 in	al,dx			; {3FDh} LSR: clear error bits
		xchg	bx,ax			; OPTIMIZE: instead MOV BL,AL
		pop	dx
		movidx	dx,RBR_index
		 in	al,dx			; {3F8h} flush receive buffer

		shr	bl,LSR_RBF+1
		jnc	@@exitIRQ		; jump if data not ready
		db	0BBh			; MOV BX,word
buffer@		dw	offset DGROUP:buffer
		cmp	bx,offset DGROUP:buffer+BUFFERSZ
		jb	@@putbyte
		mov	bx,offset DGROUP:buffer
@@putbyte:	mov	cs:[bx],al
		inc	bx
		mov	cs:[buffer@],bx

@@exitIRQ:	out_	PIC1_OCW2,%OCW2<OCW2_EOI> ; {20h} end of interrupt
		pop	bx dx ax
		iret
IRQhandler	endp
		assume	ds:DGROUP


;

.data?		; NOTE: this segment placed after other segments,
		; so PNPdata and buffer points right after program end

PNPdata		label	byte
buffer		label	byte
BUFFERSZ	equ	32
		end	start
