/*  MEM

    Copyright (c) Express Software 1997.
    All Rights Reserved.
*/

#include <dos.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "defines.h"
#include "systypes.h"
#include "memtypes.h"
#include "mem.h"


#define MSG_no_mem	": out of memory"
#define MSG_chain_bad	" chain is corrupted"
#define MSG_XMS_call	"XMS call "
#define MSG_EMM_call	"EMM call "


/*----------------------------------------------------------------------*/
static SYSINFO	SYS;
static MEMINFO	MEM;
static EXTINFO	EXT;
static XMSINFO	XMS;
static EMSINFO	EMS;

#ifdef ASM86
#  define callint(no) (geninterrupt(no), AX)
#  pragma warn -eff /* disable "code has no effect" warning */
#  define AX _AX
#  define AH _AH
#  define AL _AL
#  define CX _CX
#  define DX _DX
#  define DH _DH
#  define DL _DL
#  define BX _BX
#  define BH _BH
#  define BL _BL
#  define DI _DI
#  define ES _ES
#  define CFLAG (_FLAGS & 1)
#else
   static union REGS r;
   static struct SREGS segr;
   static word FAST_ callint(word no) { return int86x(no, &r, &r, &segr); }
#  define AX r.x.ax
#  define AH r.h.ah
#  define AL r.h.al
#  define CX r.x.cx
#  define DX r.x.dx
#  define DH r.h.dh
#  define DL r.h.dl
#  define BX r.x.bx
#  define BH r.h.bh
#  define BL r.h.bl
#  define DI r.x.di
#  define ES segr.es
#  define CFLAG (r.x.cflag)
#endif

static char name[16];	/* temporary storage */


/*======================================================================*/
#include "mem_sys.inc"
#include "mem_mem.inc"
#include "mem_dev.inc"
#include "mem_ext.inc"
#include "mem_xms.inc"
#include "mem_ems.inc"


/*----------------------------------------------------------------------*/
#define round(x, y) (((x)+(y)/2)/(y))

static void FAST_ total_list(void){
	header(2, 1,
		"     Memory type     Total     Used     Free (largest)\n"
		"----------------  --------  -------  -----------------\n");
    {	count free_K = round(MEM.BASE.free, 1024u/16u);
	printf("    Conventional%9uK%8uK%8uK (%6uK)",
		MEM.BASE.total_K, MEM.BASE.total_K - free_K,
		free_K, round(MEM.BASE.largest, 1024u/16u));
    }
	if(MEM.UMB.state == DONE){
		count total_K, free_K;
		prefix("     Upper (DOS)");
		total_K = round(MEM.UMB.total, 1024u/16u),
		free_K = round(MEM.UMB.free, 1024u/16u);
		printf("%9uK%8uK%8uK (%6uK)",
			total_K, total_K - free_K,
			free_K, round(MEM.UMB.largest, 1024u/16u));
	}
	if(XMS.state == DONE && XMS.UMB.free){
		new_line();
		printf("     Upper (XMS)%19c%8uK (%6uK)",
			' ', round(XMS.UMB.free, 1024u/16u),
				round(XMS.UMB.largest, 1024u/16u));
	}
	if(SYS.CPU >= 2){
		new_line();
		printf("Extended (INT15)%9luK%17uK",
			round(EXT.total, 1024u), EXT.free_K);
	}
	if(XMS.state == DONE){
		prefix("Extended (XMS16");
		printf(")%19c%8uK (%6uK)",
			' ', XMS.free_K, XMS.largest_K);
		if(XMS.XMS32.state == DONE){
			new_line();
			printf("Extended (XMS32)%19c%8luK (%6luK)",
				' ', XMS.XMS32.free_K, XMS.XMS32.largest_K);
	}	}
	if(EMS.state == DONE){
		ulong total_K, free_K;
		new_line();
		total_K = EMS.pages_total * 16ul,
		free_K = EMS.pages_free * 16ul;
		printf("  Expanded (EMS)%9luK%8luK%8luK",
			total_K, total_K - free_K, free_K);
	}
	header(3, 0, "----------------  --------  -------  -----------------\n");
    {	static MAP DOS_type[] = {
		{0x00, "PC-DOS"},
		{0x5E, "RxDOS"},
		{0x66, "PTS-DOS"},
		{0xEE, "DR-DOS"},
		{0xEF, "Novell DOS"},
		{0xFD, "FreeDOS"},
		{0xFF, "MS-DOS"}
	};
	static MAP loadtype[] = {
		{0x00, "low memory"},
		{0x08, "ROM"},
		{0x10, "high memory area (HMA)"},
		{0x18, "ROM and HMA"}
	};
	printf("\n%s %u.%02u is resident in the %s.\n",
		MAP_GET(DOS_type, SYS.OS.OEM),
		SYS.OS.ver.h.major, SYS.OS.ver.h.minor,
		MAP_GET(loadtype, SYS.OS.flags & 0x18u));
}   }


/*----------------------------------------------------------------------*/
static byte	MEM_BUFFER[32*1024u];
static size_t	mem_last = 0,
		mem_left = sizeof(MEM_BUFFER);

static void *FAST_ mem_tmp(size_t sz, count n){
/*!*/	sz = sz;
	return n > mem_left/sz ? NULL : &MEM_BUFFER[mem_last];
}

static void *FAST_ mem_add(CStr errmsg, size_t sz){
	void *p = mem_tmp(sz, 1);
	if(p == NULL){
		say_error(errmsg, MSG_no_mem);
		abort(); p = NULL;
	}
	mem_left -= sz, mem_last += sz;
	return p;
}


/*----------------------------------------------------------------------*/
static void FAST_ canonize_name(PStr dst, CBFP src, size_t sz){
	for(; sz; sz--){
		byte ch = *src++;
		if(ch < (byte)' '){
			if(ch == 0) break;
			ch = ' ';
		}
		*dst++ = ch;
	}
	*dst = 0;
}

static char FAST_ fstrdiff(CBFP s1, PCStr s2){
	char ch;
	do{ ch = *s2, s2++; }while(ch && *s1++ == ch);
	return ch;
}

static PCStr FAST_ MAP_get(const MAP *m, byte ID, count sz){
/*!*/	while(--(int)sz && m->ID != ID) m++;
	return m->str;
}


/*----------------------------------------------------------------------*/
static void FAST_ say(PCStr s) { printf("%s", s); }

static void FAST_ sayv(CStr format, count v) { printf(format, v); }

static void FAST_ say_error(CStr s1, CStr s2){
	printf("\nMEM: error: %s%s\n", s1, s2);
}

static void FAST_ pause(count next, count reserve){
	if(SYS.screen.pause){
		count last = SYS.screen.last_row,
			filled = SYS.screen.fill + next + 1;
		if(filled < last && reserve < last - filled){
			SYS.screen.fill = filled;
			say("\n"); return;
		}
		next += 2, SYS.screen.fill = next;
		say("\nPress <Enter> to continue or <Esc> to exit...");
		if((byte)inkey() == 27) exit(EXIT_FAILURE);
		say("\r\t\t\t\t\t\t\r");
	}else	say("\n");
}

static void FAST_ header(count next, count reserve, PCStr s){
/*!*/	pause(next = next, reserve); say(s);
}

static void FAST_ new_line(void) { pause(0, 0); }

static void FAST_ HELP(void){
    say("Displays the amount of used and free memory in your system.\n\n"
	"Syntax: MEM [/A] [/D] [/X] [/E] [/T] [/?]\n"
	"Options:\n"
	"  /A  List all memory blocks\n"
	"  /D  List loaded device drivers\n"
	"  /X  Reports Extended memory details\n"
	"  /E  Reports Expanded memory details\n"
	"  /T  Reports size of all memory types\n"
	"      (used by default if no options given)\n"
	"  /P  Pause after each screen with information\n"
	"  /?  Displays this help\n");
}


/*======================================================================*/
#ifdef RELEASE
#  define CHECKSIZE(rec, sz)
#else
#  define CHECKSIZE(rec, sz)				\
		if(sizeof(rec) != (sz)){		\
			say_error(#rec, " size");	\
			abort();			\
		}
#endif

int STDCALL_ main(int argc, PCStr argv[]){
	enum { A_ALL    = 0x01, A_DEV = 0x02,
		A_XMS   = 0x04, A_EMS = 0x08, A_TOTAL = 0x10,
		A_PAUSE = 0x20 };
	count act, opt;

#pragma warn -ccc /* disable "condition is always true/false" warning	*/
#pragma warn -rch /* disable "unreachable code" warning			*/

#ifndef __TURBOC__
CHECKSIZE(byte,	   1); CHECKSIZE(word,      2); CHECKSIZE(dword,    4);
CHECKSIZE(OFFSET,  2); CHECKSIZE(SEGMENT,   2);
CHECKSIZE(CVFP,    4); CHECKSIZE(CBFP,      4);
CHECKSIZE(MCB,    16); CHECKSIZE(UMB_MCB,  16); CHECKSIZE(HMA_MCB, 16);
CHECKSIZE(SYSDATA_MCB, 16);

CHECKSIZE(PSP,		256);	CHECKSIZE(STACKS,	18);
CHECKSIZE(DEVICE_ATTR,    2);	CHECKSIZE(DEVICE,	18);
CHECKSIZE(DPB,		 33);	CHECKSIZE(DOSLoL,	0x47-(-2));
CHECKSIZE(XMS_BLOCK,	 10);	CHECKSIZE(XMS_TABLE,	8);
CHECKSIZE(EMS_HANDLE,	  4);	CHECKSIZE(EMS_MPA,	4);
#endif

CHECKSIZE(((DPB*)0)->u.DOS2, 13);
CHECKSIZE(((DPB*)0)->u.DOS4, 18);
CHECKSIZE(((DOSLoL*)0)->u.DOS2,   0x17+18-0x10);
CHECKSIZE(((DOSLoL*)0)->u.DOS30,  0x28+18-0x10);
CHECKSIZE(((DOSLoL*)0)->u.x.DOS3, 0x35-0x10);
CHECKSIZE(((DOSLoL*)0)->u.x.DOS4, 0x47-0x35);
CHECKSIZE(((DOSLoL*)0)->u.x,	  0x47-0x10);
CHECKSIZE(((DOSLoL*)0)->u.NT,	  0x34-0x10);

#pragma warn .ccc /* return "condition is always true/false" warning	*/
#pragma warn .rch /* return "unreachable code" warning			*/

	for(act = 0;;){
	    char ch;
	    argv++; if(--argc == 0) break; ch = (*argv)[0];
	    if((ch == '-' || ch == '/') && (*argv)[2] == 0){
		ch = (*argv)[1];
		if(ch == '?') { HELP(); return EXIT_SUCCESS; }
		ch &= ~0x20; /* fast conversion to upper case	*/
		opt = A_ALL;   if(ch == 'A') { act |= opt; continue; }
		opt = A_DEV;   if(ch == 'D') { act |= opt; continue; }
		opt = A_XMS;   if(ch == 'X') { act |= opt; continue; }
		opt = A_EMS;   if(ch == 'E') { act |= opt; continue; }
		opt = A_TOTAL; if(ch == 'T') { act |= opt; continue; }
		opt = A_PAUSE; if(ch == 'P') { act |= opt; continue; }
	    }
	    printf("MEM: unknown option: %s\n"
		   "Use /? for help\n", *argv);
	    return EXIT_FAILURE;
	}

	/*--- initialize ---*/
	SYS.screen.pause = act & A_PAUSE,
	SYS.screen.fill = 0;
	SYS_get(); MEM_get(); EXT_get();
	XMS.state = XMS_get();
	EMS.state = EMS_get();

	/*--- process ---*/
	if(act & A_ALL)   memory_list();
	if(act & A_DEV)   device_list();
	if(act & A_XMS)   XMS_list();
	if(act & A_EMS)   EMS_list();
	if(act & A_TOTAL
	   || (act & ~A_PAUSE) == 0)
			  total_list();

	if(act & A_PAUSE) pause(0, UINT_MAX);
	return EXIT_SUCCESS;
}
