/*  memory management for HTML Help Browser

    Copyright (c) Express Software 2002
    All Rights Reserved.

    This file created by: Robert Platt.
*/

#include <string.h>
#include <malloc.h>
#include "pes.h"
#include "help_gui.h"
#include "readfile.h"

struct eventState **thePesList; /* list of pointers to pes structs */
long pesListSize = 0; /* size of the list */
long pesListCurr = 0; /* current position in the list */

int pesListInitialise(void)
{
  pesListSize = 0;
  pesListCurr = -1;

  thePesList = (struct eventState **)malloc(sizeof(struct eventState *));
  if (thePesList == NULL)
  {
    show_error("Cannot allocate enough memory");
    return FAIL;
  }

  return SUCCESS;
}

struct eventState * pesListAdd(char *filename)
{
    struct eventState * newpes;
    struct eventState ** thePesListResized;

    pesListDeleteForwards();

    /* Create a new process state structure */
    /* delete old ones if need be */
    while (
     (newpes = (struct eventState *)malloc (sizeof (struct eventState)))
	     == NULL)
       if (pesListDeleteOldest())
       {
	 show_error ("Cannot allocate enough memory.");
	 return NULL;
       }

    /* Create a buffer filled with the text */
    /* delete old pes structures if necessary */
    if ((newpes->text_buf = readFile (filename)) == NULL)
    {
       free (newpes);
       return NULL;
    }

    /* Create a buffer for screen information */
    /* delete old pes structures if necessary */
    while ((newpes->screen_buf = (char *)malloc (H*W*4 + 3*LEN + 3 + 16)) == NULL)
      if (pesListDeleteOldest())
      {
	 show_error ("Cannot allocate enough memory.");
	 free (newpes->text_buf);
	 free (newpes);
	 return NULL;
      }


    while ((newpes->filename = (char *)malloc (strlen(filename)+1)) == NULL)
      if (pesListDeleteOldest())
      {
	 show_error ("Cannot allocate enough memory.");
         free (newpes->screen_buf);
	 free (newpes->text_buf);
	 free (newpes);
	 return NULL;
      }
    strcpy(newpes->filename, filename);

    while ((thePesListResized = (struct eventState **)realloc(thePesList,
				sizeof(struct eventState *) * (pesListSize+1)))
				== NULL)
      if (pesListDeleteOldest())
      {
	 pesDelete(newpes);
	 return NULL;
      }

    thePesList = thePesListResized;

    /* update pes list information */
    ++pesListCurr;
    ++pesListSize;
    return (thePesList[pesListCurr] = newpes);
}



struct eventState * pesListBackward (void)
{
  --pesListCurr;

  if (pesListCurr < 0)
  {
    pesListCurr = -1;
    return NULL;
  }

  return thePesList[pesListCurr];
}


struct eventState * pesListForward (void)
{
  if (pesListCurr == pesListSize - 1)
    return NULL;

  return thePesList[++pesListCurr];
}

struct eventState * pesListCurrent (void)
{
  if (pesListCurr < 0 || pesListSize <= 0)
    return NULL;

  return thePesList[pesListCurr];
}

struct eventState * pesResizeCurrentTextbuf(long newSize)
{
    char *newbuf;

    if (newSize == 0 || pesListCurr == -1)
       return NULL;

    while ((newbuf = (char *)realloc
                         (thePesList[pesListCurr]->text_buf, newSize))
                   == NULL)
       if (pesListDeleteOldest())
       {
	 show_error ("Cannot allocate enough memory.");
	 return NULL;
       }

   thePesList[pesListCurr]->text_buf = newbuf;
   return thePesList[pesListCurr];
}

void pesListDestroy (void)
{
  int i;

  for (i = 0; i < pesListSize; i++)
     pesDelete (thePesList[i]);

  free (thePesList);
}

int pesListDeleteOldest (void)
{
   struct eventState ** thePesListResized;
   int i;

   /* Don't delete the current pes: this might
      be in use. Don't delete the oldest if it
      is too small */
   if (pesListSize <= 1 || pesListCurr <= 0)
     return FAIL;

   pesDelete (thePesList[0]);

   /* Shift the remaining pes's back so that
      the array begins at thePesList[0] */
   for (i = 1; i < pesListSize; i++)
    thePesList[i-1] = thePesList[i];

   /* there is one less pes */
   --pesListSize;
   /* and the current pes is now less one in the list */
   --pesListCurr;

   /* Resize the list of pointers to the pes's to clear up the unused space */
    if ((thePesListResized = (struct eventState **)realloc(thePesList,
				sizeof(struct eventState *) * (pesListSize)))
				!= NULL)
      thePesList = thePesListResized;

   return SUCCESS;
}

void pesListDeleteForwards (void)
{
   long i;
   struct eventState ** thePesListResized;
   long newPesListSize = pesListSize;

   /* Delete any pes's that are after the current pes */
   for (i = pesListCurr + 1; i < pesListSize; i++)
   {
     pesDelete (thePesList[i]);
     --newPesListSize;
   }

   if (pesListSize != newPesListSize)
   {
     /* Can't reallocate to zero, so just reinitialise
        This can happen if pesListDeleteForwards is
        called with pesListCurr = -1 */
     if (newPesListSize == 0)
     {
        pesListSize = 0;
	pesListDestroy();
	pesListInitialise();
	return;
     }

     if ((thePesListResized = (struct eventState **)realloc(thePesList,
				sizeof(struct eventState *) * (newPesListSize)))
				!= NULL)
     {
       thePesList = thePesListResized;
       pesListSize = newPesListSize;
     }
   }
}

void pesDelete (struct eventState *pes)
{
  free(pes->filename);
  free(pes->screen_buf);
  free(pes->text_buf);
  free(pes);
}
