/*  HTML Parsing Stuff

    Copyright (c) Express Software 1998-2002
    All Rights Reserved.

    Created by: Joe Cosentino, Brian E. Reifsnyder,
                Paul Hsieh, and Robert Platt
*/

#include <stdio.h>
#include <string.h>
#include "pes.h"
#include "help_gui.h"
/* The following define tells parse.h to include stuff relevent only
   to parse.c itself - to eliminate the declared but never used warning
   for tab0, tab128 and tagSubsTable */
#define _PARSE_C_
#include "parse.h"
#undef _PARSE_C_

/* tags2lwr

   Much of the parsing code uses strstr to search for tags. strstr is
   case sensitive, but in html tags are case insensitive!

   This code converts all tag strings to lower case

   created by Robert Platt
*/
void tags2lwr(char *text)
{
   char *leftarrow = text;
   char *rightarrow;

   while ((leftarrow = strchr(leftarrow, '<')) != 0)
   {
      ++leftarrow; /* point to text just after left arrow */
      rightarrow = strchr(leftarrow, '>');
      if (rightarrow)
      {
         *rightarrow = 0;
         strlwr(leftarrow);
         *rightarrow = '>';
      }
      else
      {
         /* a tag has been begun but there is no closing '>'. So make all
            remaining characters lower case and exit */
         strlwr(leftarrow);
         return;
      }
  }
}

/* tag substitution.
   Adapted from Brian E. Reifsnyder's changes by Paul Hsieh,
   and subsequently generalised by Robert Platt to allow the .before
   and .after text to be of different sizes */

void simpleTagSubstitutions (struct eventState *pes,
                             tagSubsEntry *tagSubsTable) {
int i;

    for (i=0; tagSubsTable[i].before != 0; i++)
    {
       long lenafter = strlen(tagSubsTable[i].after);
       long lenbefore = strlen(tagSubsTable[i].before);
       long diff = lenafter - lenbefore;
       char *tptr;
       long searchpos = 0;

       while (1) {
         /* Find occurence to be substituted */
         tptr = strstr (pes->text_buf+searchpos, tagSubsTable[i].before);
         if (tptr == 0) break;
         searchpos = tptr - pes->text_buf;

         if (diff == 0) /* don't need to resize pes->text_buf */
            strncpy (tptr, tagSubsTable[i].after, lenafter);
         else if (diff < 0) /* must shrink the text_buf */
         {
            /* make the substitution */
            long newlen = strlen(pes->text_buf) + diff + 1;
            unsigned int j;
            unsigned int copyupto = strlen(tptr)-lenbefore;
            char *tptra, *tptrb;
            strncpy (tptr, tagSubsTable[i].after, lenafter);

            /* Copy the text back into the part of the text_buf that can
               be clipped out */
            tptra = tptr + lenafter;
            tptrb = tptr + lenbefore;
            for (j = 0; j <= copyupto; j++)
               tptra[j] = tptrb[j];

            pesResizeCurrentTextbuf(newlen);
         }
         else /* must expand the text_buf to accomodate the substituted text */
         {
            long newlen = strlen(pes->text_buf) + diff + 1;

            if (pesResizeCurrentTextbuf(newlen))
            {
               char *p;

               /* copy across the existing text to make a gap for the new
                  text without the current text being overwritten */
               tptr = pes->text_buf;
               p = tptr + newlen - diff;
               tptr += searchpos;
               while (--p >= tptr)
                  p[diff] = *p;

               strncpy (tptr, tagSubsTable[i].after, lenafter);
            }
         }

         /* Skip completed occurence */
         searchpos += lenafter;
      }
   }
} /* end simpleTagSubstitutions. */


/* The following creates a 3D-style border around text enlcosed in <h1>
   and </h1>

   by Robert Platt
*/
void headerTagSubstitution (struct eventState *pes)
{
   long searchpos = 0, newlen, headerlength;
   char *opentag, *afteropentag, *closetag;

   while (1)
   {
      char *p; /* temporary-use pointer */

      opentag = strstr (pes->text_buf+searchpos, "<h1");
      if (opentag == 0) break;
      searchpos = opentag - pes->text_buf;

      afteropentag = strchr (pes->text_buf+searchpos, '>');
      if (afteropentag == 0) break;
      searchpos = afteropentag - pes->text_buf;

      closetag = strstr(pes->text_buf+searchpos, "</h1");
      if (closetag == 0) break;
      searchpos = closetag - pes->text_buf;

      if (afteropentag - opentag >= 3)
      {
        /* go back far enough before the '>' to write a line break: */
        p = afteropentag - 3;
        strncpy(p, "<br", 3);
        /* then put blank spaces before that */
        while (p-- > opentag)
           *p = ' ';
      }

      /* How many characters wide is the contents of the header? */
      headerlength = closetag - afteropentag;
      newlen = strlen(pes->text_buf) + headerlength + 10;

      if (pesResizeCurrentTextbuf(newlen))
      {
           long i;

           /* the address of pes->text_buf may have changed. So use searchpos
              to calculate the position of the closing </h1> tag, and update
              closetag.

              Relocate the text that is located after the header, so as
              not to overwrite it with the border that is put around the
              header. */
           p = pes->text_buf + newlen - headerlength - 1;
           closetag = pes->text_buf + searchpos;
           while (--p >= closetag)
              p[headerlength+7] = *p;

           /* Now create the border */
           strncpy (pes->text_buf+searchpos, " <br> ", 7);
           searchpos += 7;
           for (i = 1; i <= headerlength; i++)
                pes->text_buf[searchpos++] = '';
      }
   }
}


/*	preformatTrim removes any unwanted whitespace from HTML.
        wordwrap should then be called to ensure that no line of text
        exceeds the screen width.

	by Robert Platt
*/
void preformatTrim (char *text) {
char *tptr, *tptrtemp;
char previous_char;

    tptr = text;
    previous_char = 0;

    tptr = strstr(tptr, "<body");
    if (tptr == 0)
      return;

    while (*tptr != 0) {

       /* Leave preformatted text in the <pre> tag */
       if (strncmp(tptr, "<pre", 4) == 0)
       {
          tptr = strstr(tptr, "</pre");
          if (tptr == 0)
            return;
       }

       switch (*tptr) {
         case '\t':
         case '\n':
         *tptr = ' ';

         case ' ':
         /* If there is already one space then copy
            the rest of the text over rather than
            have another space */
            if (previous_char == ' ')
            {
              tptrtemp = tptr;
              while (*tptrtemp != 0) {
                tptrtemp[0] = tptrtemp[1];
                ++tptrtemp;
              }
              --tptr; /* Easy way of cancelling out the ++tptr which is not
                        needed since we've copied the next character into
                        this one */
           }
       }
       previous_char = *tptr;
       ++tptr;


    }
}

void wordwrap (char *text)
{
   /* This wraps any text that may not fit on one line. Hence preformatted
      text is not required anymore.

      It works by recording the last space found whilst checking a particular
      line of text. The length of line is stored in i. i is not incremented
      whilst reading through tags, and it is reset whenever any preformatted
      line breaks are found.

      Format tags such as <br> and <p>, and any newline characters outside
      of <pre> preformatted text, will already have been dealt with by
      simpleTagSubstitutions() and preformatTrim() - RP
   */
   int i, intag;
   char *tptr, *lastspace;

   tptr = text;
   intag = 0;

   while (1) {
     i = 0;
     lastspace = 0;

     while (i <= LEN + 2)
     {
       switch (*tptr) {
        case 0:
         return;

        case '<':
         intag = 1;
        break;

        case '>':
         --i; /* do not count this character as part of the visible text*/
         intag = 0;
        break;

        case '\n':
         lastspace = 0;
         i = 0;
        break;

        case ' ':
         if (!intag)
           lastspace = tptr;
        break;
       }

       ++tptr;
       if (!intag)
         ++i;     
     }

     if (lastspace != 0)
     {
       *lastspace = '\n';
       tptr = lastspace + 1;
     }
   }
}


/* Translate &#___;

   this code translates & escape codes in standard HTML

   by Paul Hsieh */

#ifdef AMP_HASH_MATCH

int ampHashMatch (struct eventState * pes) {
int i;
    for (i=0; i < 32; i++) {
	if (0 == strncmp (pes->p+2, tab0[i].before+2, tab0[i].len)) {
	    pes->p += tab0[i].len + 1;
	    return i;
	}
    }
    for (i=0; i < 128; i++) {
	if (0 == strncmp (pes->p+2, tab128[i].before+2, tab128[i].len)) {
	    pes->p += tab128[i].len + 1;
	    return i + 128;
	}
    }

    for (i=2; pes->p[i] !=0; i++) {
	if (pes->p[i] == ';') {
	    int v, d;
            pes->p[i] = 0;
	    d = i;
	    if (1 != sscanf (&pes->p[2], "%d", &v)) {
		d = 0;
		v = -1;
	    }
	    if (v >= 128 || v <= 31) v = -1;
	    pes->p[i] = ';';
	    pes->p += d;

	    return v;
	}
    }

    return -1;
}  */

#endif

/* Substitute some &-escape codes. */

int ampSubs (struct eventState * pes) {
int ch;

    ch = '?';

    if (strncmp (pes->p, "&gt;", 4) == 0 && pes->body_end - pes->p > 4) {
	ch = '>';
	pes->p += 3;
    } else if (strncmp (pes->p, "&lt;", 4) == 0 && pes->body_end - pes->p > 4) {
	ch = '<';
	pes->p += 3;
    } else if (strncmp (pes->p, "&amp;", 5) == 0 && pes->body_end - pes->p > 5) {
	ch = '&';
	pes->p += 4;
    } else if (strncmp (pes->p, "&quot;", 6) == 0 && pes->body_end - pes->p > 6) {
	ch = '\"';
	pes->p += 5;
    } else if (strncmp (pes->p, "&nbsp;", 6) == 0 && pes->body_end - pes->p > 6) {
	ch = ' ';
	pes->p += 5;
    } else if (strncmp (pes->p, "&copy;", 6) == 0 && pes->body_end - pes->p > 6) {
        /* copyright symbol. added by RP
           Note the dummy tag <!> which is there so that these three
           characters will be skipped next time displayProcess comes across it
        */
        strncpy(pes->p, "<!>(c)", 6); 
        ch = '(';
        pes->p += 3;
    }

#ifdef AMP_HASH_MATCH
    if (pes->p[1] == '#') {
	int ret = ampHashMatch (pes);
	if (ret >= 0) return ret;
    }
#endif

    return ch;
}

/* link_length: Determines the length of the link

   Adapted from Joe Cosentino's code by Robert Platt
*/

int link_length (char *link_text) {
char *search;
int l;

    if ((search = strchr (link_text, '\"')) != 0)
	l = search - link_text;
    else if ((search = strchr (link_text, ' ')) != 0)
	l = search - link_text;
    else
	l = strlen (link_text);

    if (l < 0 || l > LEN - 12)
       l = LEN - 12;

    return l;
}
