/*			FC: FreeDOS File compare

This program implements Paul Heckel's algorithm from the Communications
of the Association for Computing Machinery, April 1978 for detecting the
differences between two files.
This algorithm has the advantage over more commonly used compare algorithms
that it is fast and can detect differences of an arbitrary number of lines.
For most applications the algorithm isolates differences similar to those
isolated by the longest common subsequence.
********************************************************************************
A design limitation of this program is that only the first
MAX_LINES lines are compared; the remaining lines are ignored.
********************************************************************************
Version 2.01: Fixed a conceptual bug in MatchNames
	      Added the /S switch and relevant routines
Version 2.10: Fixed a stupid bug in ScanArguments in case of
	      a comparison of just two files
	      Added cats for easier internationalisation
	      Added io95 for long filenames
Version 2.20: Added the /LBn and /nnn switches
	      Used CRC15 as hashing function
******************************************************************************** */

#define USE_CATS	/* Define this to use cats */
//#define USE_IO95	/* Define this to enable the use of long filenames */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <dir.h>
#include <dos.h>
#include <sys\stat.h>
#ifdef USE_CATS
  #include "catgets.h"
#endif
#ifdef USE_IO95
  #include "io95.h"
  #include "find95.h"
#else
  #define findstop(FindData)   /* This function is not needed in plain DOS */
#endif

/* ------------------------------------------------------------------------ */
#define MAX_LINES			32765
#define DEFAULT_LINES_FOR_RESYNC	2
#define BINARY_MAX_DIFF			20  /* Bytes */

/* Must be signed and able to count at least up to MAX_LINES */
typedef signed int line_counter;
/* ------------------------------------------------------------------------ */
/* Values for ERRORLEVEL */
#define EL_NO_DIFF		0
#define EL_DIFFERENT		1
#define EL_PARAM_ERROR		2
#define EL_NO_FILE_ERROR	3
#define EL_OPEN_ERROR		4
/* ------------------------------------------------------------------------ */
typedef enum { FALSE, TRUE } bool;

#define TAB_SIZE	8
#define END_OF_STRING	'\0'
#define BITS_PER_BYTE	8
#define BitMask(Bit)	(1 << (Bit))
/* ------------------------------------------------------------------------ */
/* Option flags */
#define BRIEF		BitMask(7)
#define NO_TAB_EXPAND	BitMask(6)
#define PACK_SPACE	BitMask(5)
#define IGNORE_CASE	BitMask(4)
#define LINE_NUMBER	BitMask(3)
#define SCAN_SUBDIRS	BitMask(2)
unsigned char OptFlags = 0;
/* ------------------------------------------------------------------------ */
#define FILE_SPECS	2	/* Two files to be compared each time */
#define FIRST_FILE	0
#define SECOND_FILE	1
/* ------------------------------------------------------------------------ */
#define STANDARD_CONTEXT		1   /* Lines */
#define NO_CONTEXT			0   /* Lines */
/* ------------------------------------------------------------------------ */
FILE *File1, *File2;
char FileSpecs[FILE_SPECS][MAXPATH];	/* Full pathnames */
int MaxBinDifferences = BINARY_MAX_DIFF;
int MaxASCIIDiffLines = MAX_LINES;	/* Virtually unlimited */
int LinesForResync = DEFAULT_LINES_FOR_RESYNC;
int ContextLines = STANDARD_CONTEXT;	/* 0 = don't show context lines */
enum { DEFAULT, ASCII, BINARY } CompareMode = DEFAULT;
enum { MatchingNames, CommonSource, CommonTarget } NameMatching;
char SourceNamePattern[MAXFILE + MAXEXT];
char TargetNamePattern[MAXFILE + MAXEXT];
unsigned FileCount;
bool FoundDifferences;
#ifdef USE_CATS
  nl_catd CatalogDescriptor;
#endif
/* ------------------------------------------------------------------------ */
const char AsciiDiffStart[] = "***** %s\n";
const char AsciiDiffEnd[] = "*****\n\n";
const char Omissis[] = "...\n";
const char AllFiles[] = "*.*";
const char DummyFileSpec[] = "\\.";

#define NewLine puts("")
/* ------------------------------------------------------------------------ */
/* CATS section names */
#define HELP	1
#define MESSAGE 2
/* ------------------------------------------------------------------------ */
#ifdef USE_CATS
  #define Format(setnum, msgnum, message)	\
	    catgets(CatalogDescriptor, setnum, msgnum, message)
#else
  #define Format(dummy1, dummy2, message)  message
#endif
/* ************************************************************************ */
void HelpMessage(void)
{
  puts("FreeDOS FC version 2.20");
  printf(Format(HELP, 0,"Compares two files or sets of files and displays the differences between them")); NewLine;
  NewLine;
  printf(Format(HELP, 1,"FC [switches] [drive1:][path1]filename1 [drive2][path2]filename2 [switches]")); NewLine;
  printf(Format(HELP, 2," /A    Display only first and last lines for each set of differences")); NewLine;
  printf(Format(HELP, 3," /B    Perform a binary comparison")); NewLine;
  printf(Format(HELP, 4," /C    Disregard the case of letters")); NewLine;
  printf(Format(HELP, 5," /L    Compare files as ASCII text")); NewLine;
  printf(Format(HELP,13," /LBn  Set the maximum number of consecutive different ASCII lines to n")); NewLine;
  printf(Format(HELP, 6," /Mn   Set the maximum differences in binary comparison to n bytes")); NewLine;
  printf(Format(HELP, 7,"       (default = %d, 0 = unlimited)"),BINARY_MAX_DIFF); NewLine;
  printf(Format(HELP, 8," /N    Display the line numbers on a text comparison")); NewLine;
  printf(Format(HELP, 9," /S    Extend the scan to the files in subdirectories")); NewLine;
  printf(Format(HELP,10," /T    Do not expand tabs to spaces")); NewLine;
  printf(Format(HELP,11," /W    Pack white space (tabs and spaces) for text comparison")); NewLine;
  printf(Format(HELP,12," /X    Do not show context lines in text comparison")); NewLine;
  printf(Format(HELP,14," /nnn  Set to nnn (>= 2) lines the minimum number of consecutive matching")); NewLine;
  printf(Format(HELP,15,"       lines for comparison resynchronization")); NewLine;
}
/* ************************************************************************ */
bool IsADirectory(char Filename[])
{
  struct stat statbuf;

  if (stat(Filename, &statbuf) != -1)
    if (statbuf.st_mode & S_IFDIR)
      return TRUE;
  return FALSE;
}
/* ------------------------------------------------------------------------ */
bool HasWildcards(char* Filename)
{
  while (*Filename != END_OF_STRING)
  {
    if ((*Filename == '*') || (*Filename == '?')) return TRUE;
    Filename++;
  }
  return FALSE;
}
/* ------------------------------------------------------------------------ */
/* Try to find a match between the filename pattern in Pattern and the
   filename in Name. If the match is found, the result is in Target and
   the function returns TRUE */
bool MatchNames(char Pattern[], char Name[], char Target[])
{
  /* Parse the name part */
  while ((*Pattern != END_OF_STRING) && (*Name != END_OF_STRING) &&
	 (*Name != '.') && (*Pattern != '.') && (*Pattern != '*'))
  {
    if (*Pattern == '?')
      *Target = *Name;
    else
      *Target = *Pattern;
    Pattern++; Name++; Target++;
  }
  if ((*Name == END_OF_STRING) || (*Name == '.'))
    while ((*Pattern != END_OF_STRING) && (*Pattern != '.') &&
	   (*Pattern != '?') && (*Pattern != '*'))
    {
      *Target = *Pattern;
      Pattern++; Target++;
    }
  if (*Pattern == '*')
  {
    while ((*Pattern != END_OF_STRING) && (*Pattern != '.')) Pattern++;
    while ((*Name != END_OF_STRING) && (*Name != '.'))
    {
      *Target = *Name;
      Name++; Target++;
    }
  }
  *Target = END_OF_STRING;
  /* Here to have a match *Pattern must be a dot or the end of string */
  if (*Pattern == '.')
    Pattern++;
  else
    if (*Pattern != END_OF_STRING) return FALSE;

  if (*Name == '.')
    Name++;
  else
    if (*Name != END_OF_STRING) return FALSE;
  *Target = '.'; Target++;

  /* Parse the ext part */
  while ((*Pattern != END_OF_STRING) && (*Name != END_OF_STRING) &&
	 (*Pattern != '*'))
  {
    if (*Pattern == '?')
      *Target = *Name;
    else
      *Target = *Pattern;
    Pattern++; Name++; Target++;
  }
  if (*Name == END_OF_STRING)
    while ((*Pattern != END_OF_STRING) &&
	   (*Pattern != '?') && (*Pattern != '*'))
    {
      *Target = *Pattern;
      Pattern++; Target++;
    }
  if (*Pattern == '*')
  {
    Pattern++;
    *Pattern = END_OF_STRING;
    while (*Name != END_OF_STRING)
    {
      *Target = *Name;
      Name++; Target++;
    }
  }
  *Target = END_OF_STRING;
  return ((*Name == END_OF_STRING) && (*Pattern == END_OF_STRING));
}
/* ------------------------------------------------------------------------ */
/* Check if the file extension indicates a binary file */
bool BinaryFile(const char Filename[])
{
  const char* BinExt[] = { ".EXE", ".COM", ".SYS", ".OBJ", ".LIB", ".BIN" };
  char Ext[MAXEXT];
  int i;

  if (fnsplit(Filename, NULL, NULL, NULL, Ext) & EXTENSION)
    for (i = 0; i < sizeof (BinExt)/sizeof (BinExt[0]); i++)
      if (strcmpi(BinExt[i], Ext) == 0) return TRUE;
  return FALSE;
}
/* ------------------------------------------------------------------------ */
/* Pointer to the start of the file name with path stripped off */
char* FileNameStart(char Filename[])
{
  char* CurrPos = Filename;
  while (*CurrPos != END_OF_STRING)
  {
    if ((*CurrPos == ':') || (*CurrPos == '\\')) Filename = CurrPos + 1;
    CurrPos++;
  }
  return Filename;
}

/* ************************************************************************ */
/* Scan arguments for switches and files */
bool ScanArguments(int argc, char* argv[])
{
  int FileCounter = 0;
  int i;

  for (i = 1; i < argc; i++)
  {
    int j = 0;
    while ((argv[i][j] == '/') || (argv[i][j] == '-')) /* Switch chars */
      switch (toupper(argv[i][j + 1]))
      {
	case 'A':       /* The following two switches are incompatibles */
	  OptFlags |= BRIEF;
	  ContextLines = STANDARD_CONTEXT;
	  j += 2;
	  break;

	case 'X':
	  OptFlags &= ~BRIEF;
	  ContextLines = NO_CONTEXT;
	  j += 2;
	  break;

	case 'B':
	  CompareMode = BINARY;
	  j += 2;
	  break;

	case 'L':
	  CompareMode = ASCII;
	  j += 2;
	  if (toupper(argv[i][j]) == 'B')
	  {
	    MaxASCIIDiffLines = atoi(&argv[i][j + 1]);
	    j += strlen(&argv[i][j]) - 2;
	  }
	  break;

	case 'C':
	  OptFlags |= IGNORE_CASE;
	  j += 2;
	  break;

	case 'N':
	  OptFlags |= LINE_NUMBER;
	  j += 2;
	  break;

	case 'S':
	  OptFlags |= SCAN_SUBDIRS;
	  j += 2;
	  break;

	case 'T':
	  OptFlags |= NO_TAB_EXPAND;
	  j += 2;
	  break;

	case 'W':
	  OptFlags |= PACK_SPACE;
	  j += 2;
	  break;

	case 'M':
	  MaxBinDifferences = atoi(&argv[i][j + 2]);
	  j += strlen(&argv[i][j]);
	  break;

	case '?':
	case 'H':
	  HelpMessage();
	  return FALSE;

	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  LinesForResync = atoi(&argv[i][j + 1]);
	  if (LinesForResync < 1) LinesForResync = 1;
	  j += strlen(&argv[i][j]);
	  break;

	default:
	  printf(Format(MESSAGE,0,"Invalid switch: %s"), argv[i][j]); NewLine;
	  return FALSE;
      }

    if (j == 0) /* Not a switch: it's a filename */
    {
      if (FileCounter >= FILE_SPECS)
      {
	printf(Format(MESSAGE,1,"Too many filespecs")); NewLine;
	return FALSE;
      }
      if (_fullpath(FileSpecs[FileCounter], argv[i],
		    sizeof(FileSpecs[0]) - 1) == NULL)
      {
	printf(Format(MESSAGE,2,"Invalid filename")); NewLine;
	return FALSE;
      }
      FileCounter++;
    }
  }

  switch (FileCounter)
  {
    case 0:
      printf(Format(MESSAGE,3,"No file specified")); NewLine;
      return FALSE;

    case 1:		/* Default: the current directory */
      strcpy(FileSpecs[SECOND_FILE], AllFiles);
  }

  for (i = FIRST_FILE; i < FILE_SPECS; i++)
  {
    int LastChar = strlen(FileSpecs[i]) - 1;
    if ((FileSpecs[i][LastChar] == '\\') ||  /* Path only */
	(FileSpecs[i][LastChar] == ':'))     /* Disk drive only */
					     /* Assume "All files" */
      strcpy(&FileSpecs[i][LastChar + 1], AllFiles);
    else
      if (!HasWildcards(FileSpecs[i]) &&
	  IsADirectory(FileSpecs[i]))	     /* Directory */
					     /* Assume "All files" */
	sprintf(&FileSpecs[i][LastChar + 1], "\\%s", AllFiles);
   /* else regular file (possibly with wildcards) */
  }

  {
    /* Pointers to the name part of the files */
    char* FirstFileName = FileNameStart(FileSpecs[FIRST_FILE]);
    char* SecondFileName = FileNameStart(FileSpecs[SECOND_FILE]);

    NameMatching = MatchingNames;
    if (!HasWildcards(SecondFileName))
      NameMatching = CommonTarget;
    if (!HasWildcards(FirstFileName) &&
	(strcmp(SecondFileName, AllFiles) != 0))
      NameMatching = CommonSource;

    strncpy(SourceNamePattern, FirstFileName,
	    sizeof(SourceNamePattern) - 1);
    SourceNamePattern[sizeof(SourceNamePattern) - 1] = END_OF_STRING;

    strncpy(TargetNamePattern, SecondFileName,
	    sizeof(TargetNamePattern) - 1);
    TargetNamePattern[sizeof(TargetNamePattern) - 1] = END_OF_STRING;
  }
  return TRUE;
}

/* ************************************************************************ */
bool BinaryCompare(void)
{
  unsigned char a, b;
  unsigned long Offset, FileSize1, FileSize2;
  int Differences = 0;

  FileSize1 = filelength(fileno(File1));
  FileSize2 = filelength(fileno(File2));
  if (FileSize1 != FileSize2)
  {
    printf(Format(MESSAGE,4,"Warning: the files are of different size!")); NewLine;
    if (FileSize1 < FileSize2) FileSize1 = FileSize2;
  }
  for (Offset = 0; Offset < FileSize1; Offset++)
  {
    a = fgetc(File1); b = fgetc(File2);
    if (a != b)
    {
      Differences++;
      if ((MaxBinDifferences > 0) && (Differences > MaxBinDifferences))
      {
	printf(Format(MESSAGE,5,"Comparison stopped after %d mismatches"),
	       MaxBinDifferences); NewLine;
	NewLine;
	return FALSE;
      }
      printf("%08lX:     %02X  %c      %02X  %c\n", Offset,
	     a, (iscntrl(a) ? ' ' : a), b, (iscntrl(b) ? ' ' : b));
    }
  }
  if (Differences == 0)
  {
    printf(Format(MESSAGE,6,"No differences")); NewLine;
    NewLine;
    return TRUE;
  }
  return FALSE;
}

/* ************************************************************************ */
/* Skip to the next line in the file File */
void SkipLine(FILE* File)
{
  int c;

  do
    c = fgetc(File);
  while ((c != '\n') && (c != EOF));
}
/* ------------------------------------------------------------------------ */
/* Display a line from the file File */
void OutputLine(FILE* File, line_counter LineNumber)
{
  unsigned long LineLength = 0;
  int c;

  if (OptFlags & LINE_NUMBER) printf("%5u: ", LineNumber);

  while (TRUE)
  {
    c = fgetc(File);
    if ((c == '\n') || (c == EOF)) /* End of line */
    {
      putchar('\n'); break;
    }
    if ((c == '\t') && !(OptFlags & NO_TAB_EXPAND)) /* Tab */
      do
      {
	putchar(' '); LineLength++;
      } while ((LineLength % TAB_SIZE) != (TAB_SIZE - 1));
    else
      if (c != '\r')    /* Ignore the carriage return */
      {
	putchar(c); LineLength++;
      }
  }
}
/* ************************************************************************ */
		/* Hashing procedures */

#define CRC_BITS	15	/* 15 bits CRC */
#define CRC_POLY	0xC599	/* X^15+X^14+X^10+X^8+X^7+X^4+X^3+X^0 */
#define CRC_ARRAY_ITEMS (1 << BITS_PER_BYTE)
#define CRC_ARRAY_SIZE	(sizeof(*CRCTable)*CRC_ARRAY_ITEMS)
#define HASH_BUCKETS	(1 << CRC_BITS)
#define HASH_ARRAY_SIZE (sizeof(hashing_values)*MAX_LINES)

typedef signed int hashing_values; /* Must be signed */
hashing_values HashValue;
typedef unsigned int crc_values;   /* Max value = (1 << CRC_BITS) - 1 */
crc_values* CRCTable = NULL;
/* ------------------------------------------------------------------------ */
/* Initialise the CRC register and create (only once) the CRC table
   used to speed-up the hashing function */
void InitCRC(void)
{
  int i, j;
  crc_values CRC;

  HashValue = (hashing_values)(-1);	/* Initialise the CRC register */
  if (CRCTable != NULL) return;		/* Table already created */

  CRCTable = (crc_values*)malloc(CRC_ARRAY_SIZE);
  if (CRCTable == NULL) return;		/* Insufficient memory */

  for (i = 0; i < CRC_ARRAY_ITEMS; i++)
  {
    CRC = i << (CRC_BITS - BITS_PER_BYTE);
    for (j = 0; j < BITS_PER_BYTE; j++)
    {
      CRC <<= 1;
      if ((CRC & (1 << CRC_BITS)) != 0) CRC ^= CRC_POLY;
    }
    CRCTable[i] = CRC;
  }
}
/* ------------------------------------------------------------------------ */
/* Hash a character */
void Hash(char c)
{
  /* Update the CRC */
  HashValue = (HashValue << (CRC_BITS - BITS_PER_BYTE)) ^
	      CRCTable[c ^ ((HashValue >> (CRC_BITS - BITS_PER_BYTE)) & 0xff)];
}

/* ************************************************************************ */
/* occurr indicate the number of occurrences for a line. */
typedef enum { NO_OCCURR, SINGLE_OCCURR, MULTIPLE_OCCURR } occurr;
/* Minimum number of bits to represent occurr */
#define BITS_PER_OCCUR	   2
/* The occurrence data codes are packed to save memory */
typedef unsigned char packed_occurr;
#define OCCUR_PER_BYTE	   ((BITS_PER_BYTE*sizeof(packed_occurr))/BITS_PER_OCCUR)
#define OCCUR_BIT_MASK	   ((1 << BITS_PER_OCCUR) - 1)
#define PACKED_ARRAY_ITEMS ((HASH_BUCKETS + OCCUR_PER_BYTE - 1)/OCCUR_PER_BYTE)
#define OCCURR_ARRAY_SIZE  (sizeof(packed_occurr)*PACKED_ARRAY_ITEMS)

/* ------------------------------------------------------------------------ */
/* Extract bits from the packed occurrence array
   Index must be >= 0 */
occurr GetOccurrences(const packed_occurr Array[], hashing_values Index)
{
  Index %= HASH_BUCKETS;
  return ((Array[Index/OCCUR_PER_BYTE] >>
	   (BITS_PER_OCCUR*(Index % OCCUR_PER_BYTE))) & OCCUR_BIT_MASK);
}
/* ------------------------------------------------------------------------ */
/* Store bits in the packed occurrence array
   Index must be >= 0 */
void SetOccurrences(packed_occurr Array[], hashing_values Index, occurr Occurrence)
{
  unsigned Shifts;

  Index %= HASH_BUCKETS;
  Shifts = BITS_PER_OCCUR*(Index % OCCUR_PER_BYTE);
  Index /= OCCUR_PER_BYTE;
  Array[Index] &= ~(OCCUR_BIT_MASK << Shifts);
  Array[Index] |= (Occurrence << Shifts);
}

/* ************************************************************************ */
/* Read file, build hash & occurrence tables.
   Returns the number of lines in the file. */
line_counter HashFile(FILE* file, hashing_values HashVect[],
		      packed_occurr Occurrences[])
{
  int c;
  bool Skip = FALSE;
  line_counter LineCounter = 0;
  unsigned long LineLength = 0;

  HashValue = 0;
  do
  {
    c = fgetc(file);
    /* Beware of order! There are fall-throughs */
    switch (c)
    {
      case EOF:
	if (LineLength == 0) break;	/* End of file */
	/* Unterminated line: assume line feed */
      case '\n':
	/* This is needed to avoid problems with trailing
	   blanks when blanks are packed */
	if (!Skip) Hash(' ');

	/* HashValue must be > 0 and HashVect[] must be < 0 */
	if (HashValue < 0)
	  HashValue = -HashValue;
	else
	  if (HashValue == 0) HashValue = 1;
	HashVect[LineCounter] = -HashValue;
	switch (GetOccurrences(Occurrences, HashValue))
	{
	  case NO_OCCURR:
	    SetOccurrences(Occurrences, HashValue, SINGLE_OCCURR);
	    break;
	  case SINGLE_OCCURR:
	    SetOccurrences(Occurrences, HashValue, MULTIPLE_OCCURR);
	}
	HashValue = 0; LineCounter++; LineLength = 0;
	Skip = FALSE;
	break;

      case '\t':
	/* Tabs must be expanded and blanks are not packed */
	if (!(OptFlags & NO_TAB_EXPAND) && !(OptFlags & PACK_SPACE))
	{
	  do
	  {
	    Hash(' '); LineLength++;
	  } while ((LineLength % TAB_SIZE) != 0);
	  break;
	}
      case ' ':
      case '\r':
	/* Blank and tab are packed */
	if (OptFlags & PACK_SPACE)
	{
	  if (!Skip)
	  {
	    Hash(' '); LineLength++;
	    Skip = TRUE;
	  }
	  break;
	}
      default:
	if (OptFlags & IGNORE_CASE) c = toupper(c);
	Hash(c); LineLength++;
	Skip = FALSE;
    }
  } while ((c != EOF) && (LineCounter < MAX_LINES));

  if (LineCounter >= MAX_LINES)
  {
    printf(Format(MESSAGE,7,"Warning: comparison interrupted after %d lines"),
	   LineCounter); NewLine;
  }
  return LineCounter;
}

/* ************************************************************************ */
/* Show a block of differences. Returns TRUE if there really are some */
bool ShowBlockOfDiff(line_counter* Line, FILE* File,
		     line_counter* LinesRead, line_counter LinesInFile,
		     char FileName[], hashing_values Hash[], bool* Abort)
{
  line_counter i;
  line_counter ResyncLine = *Line;
  line_counter LineCounter = *LinesRead;

  if (*Abort) return FALSE;

		    /* Go to the starting line (of the context) */
  while (LineCounter + ContextLines < ResyncLine)
  {
    LineCounter++; SkipLine(File);
  }
  *LinesRead = LineCounter;

  i = 0;
  do		    /* Find the end of the mismatching block */
  {
    ResyncLine += i;

    i = 0;	    /* Skip mismatching lines */
    while ((ResyncLine + 1 < LinesInFile) &&
	   ((Hash[ResyncLine] < 0) ||
	    (Hash[ResyncLine] + 1 != Hash[ResyncLine + 1])))
    {
      i++; ResyncLine++;
      if (i > MaxASCIIDiffLines)
      {
	*Abort = TRUE;
	return FALSE;
      }
    }
    /* Found at least 2 consecutive line matching unless eof */

    i = ResyncLine + 1; /* Count matching lines */
    while ((i - ResyncLine < LinesForResync) && (i < LinesInFile) &&
	   (Hash[i] >= 0) && (Hash[i] - 1 == Hash[i - 1]))
      i++;

    i -= ResyncLine;
  } while ((i < LinesForResync) && (ResyncLine < LinesInFile));

  /* If there are no differences and the context is not required
     we are finished */
  if ((ResyncLine <= *Line) && (ContextLines <= NO_CONTEXT))
    return FALSE;

  printf(AsciiDiffStart, FileName);
		    /* Show the starting context if needed */
  while (LineCounter < *Line)
  {
    LineCounter++; OutputLine(File, LineCounter);
  }
		    /* Show the differing lines */
  if (OptFlags & BRIEF)
  {
    printf(Omissis);
    while (LineCounter < ResyncLine)
    {
      LineCounter++; SkipLine(File);
    }
  }
  else
    while (LineCounter < ResyncLine)
    {
      LineCounter++; OutputLine(File, LineCounter);
    }
		    /* Show the final context */
  while ((LineCounter < LinesInFile) &&
	 (LineCounter < ResyncLine + ContextLines))
  {
    LineCounter++; OutputLine(File, LineCounter);
  }

  *LinesRead = LineCounter;
  *Line = ResyncLine;
  if (LineCounter >= LinesInFile) (*Line)++;
  return TRUE;
}
/* ------------------------------------------------------------------------ */
bool AsciiCompare(void)
{
  bool Linked;
  bool Different = FALSE;
  line_counter i;
  line_counter LinesInFile1, LinesInFile2, LinesRead1, LinesRead2;
  line_counter Line1, Line2;
  hashing_values* Hash1, *Hash2;
  /* Arrays of packed occurrences */
  packed_occurr* Occurr1, *Occurr2;

  InitCRC();	/* Initialise the CRC for the hashing */

  /* Allocate the big structures on the heap */
  Occurr1 = (packed_occurr*)malloc(OCCURR_ARRAY_SIZE);
  Occurr2 = (packed_occurr*)malloc(OCCURR_ARRAY_SIZE);
  Hash1 = (hashing_values*)malloc(HASH_ARRAY_SIZE);
  Hash2 = (hashing_values*)malloc(HASH_ARRAY_SIZE);
  if ((CRCTable == NULL) ||
      (Hash1 == NULL) || (Hash2 == NULL) ||
      (Occurr1 == NULL) || (Occurr2 == NULL))
  {
    printf(Format(MESSAGE,8,"Insufficient memory")); NewLine;
    return FALSE;
  }
  memset(Hash1, 0, HASH_ARRAY_SIZE);
  memset(Hash2, 0, HASH_ARRAY_SIZE);
  memset(Occurr1, 0, OCCURR_ARRAY_SIZE); /* NO_OCCURR */
  memset(Occurr2, 0, OCCURR_ARRAY_SIZE); /* NO_OCCURR */

  /* step 1: read files and hash them */
  LinesInFile1 = HashFile(File1, Hash1, Occurr1);
  LinesInFile2 = HashFile(File2, Hash2, Occurr2);

  /* step 2: identify lines that are unique in both files */
  for (i = 0; i < PACKED_ARRAY_ITEMS; i++) Occurr1[i] &= Occurr2[i];

  /* step 3: link together the nearest matching unique lines
     N.B. Hashx is originally < 0 and is set to the matching
	  line number (>= 0) if match found */
  for (i = 0; i < LinesInFile1; i++)
    if (GetOccurrences(Occurr1, -Hash1[i]) == SINGLE_OCCURR)
    {
      line_counter Forward = i, Backward = i;
      while ((Forward < LinesInFile2) || (Backward >= 0))
      {
	if (Forward < LinesInFile2)
	{
	  if (Hash2[Forward] == Hash1[i])
	  {
	    Hash1[i] = Forward; Hash2[Forward] = i;
	    break;
	  }
	  Forward++;
	}

	if (Backward >= 0)
	{
	  if (Hash2[Backward] == Hash1[i])
	  {
	    Hash1[i] = Backward; Hash2[Backward] = i;
	    break;
	  }
	  Backward--;
	}
      }
    }
  free (Occurr1); free (Occurr2);	/* No more needed */

  /* step 4: link the first and last lines, if possible */
  if ((Hash1[0] < 0) && (Hash1[0] == Hash2[0])) Hash1[0] = Hash2[0] = 0;
  if ((Hash1[LinesInFile1 - 1] < 0) &&
      (Hash1[LinesInFile1 - 1] == Hash2[LinesInFile2 - 1]))
  {
    Hash1[LinesInFile1 - 1] = LinesInFile2 - 1;
    Hash2[LinesInFile2 - 1] = LinesInFile1 - 1;
  }

  /* step 5: starting from linked lines, link following lines that match
     N.B. Hashx was set to matching line number (>= 0) if match found */
  Linked = FALSE;
  for (i = 0; i < LinesInFile1; i++)
    if (Hash1[i] >= 0)
      Linked = TRUE;
    else
      if (Linked)
      {
	Line2 = Hash1[i - 1] + 1;
	if (Hash1[i] == Hash2[Line2])
	{
	  Hash1[i] = Line2;
	  Hash2[Line2] = i;
	}
	else
	  Linked = FALSE;
      }

  /* step 6: link matching lines that precede linked lines */
  Linked = FALSE;
  for (i = LinesInFile1 - 1; i >= 0; i--)
    if (Hash1[i] >= 0)
      Linked = TRUE;
    else
      if (Linked)
      {
	Line2 = Hash1[i + 1] - 1;
	if (Hash1[i] == Hash2[Line2])
	{
	  Hash1[i] = Line2;
	  Hash2[Line2] = i;
	}
	else
	  Linked = FALSE;
      }

  /* step 7: display the results */
  rewind(File1); rewind(File2);
  LinesRead1 = LinesRead2 = 0;
  Line1 = Line2 = 0;
  while (TRUE)
  {
    bool Abort;
    bool DiffDetected;
			      /* Skip matching lines */
    while ((Line1 < LinesInFile1) && (Line2 < LinesInFile2) &&
	   (Hash1[Line1] == Line2) /* &&
	   (Hash2[Line2] == Line1) (pleonastic) */)
    {
      Line1++; Line2++;
    }
			      /* File terminated */
    if ((Line1 >= LinesInFile1) || (Line2 >= LinesInFile2)) break;

    Abort = FALSE;
    DiffDetected = ShowBlockOfDiff(&Line1, File1, &LinesRead1, LinesInFile1,
				   FileSpecs[FIRST_FILE], Hash1, &Abort);
    if (ShowBlockOfDiff(&Line2, File2, &LinesRead2, LinesInFile2,
			FileSpecs[SECOND_FILE], Hash2, &Abort))
      DiffDetected = TRUE;

    if (Abort)
    {
      printf(Format(MESSAGE,12,"Resync failed: files too different")); NewLine;
      NewLine;
      Different = TRUE;
      break;
    }

    if (DiffDetected)
    {
      printf(AsciiDiffEnd);
      Different = TRUE;
    }
  }
  free (Hash1); free (Hash2);

  if (!Different)
  {
    printf(Format(MESSAGE,6,"No differences")); NewLine;
    NewLine;
    return TRUE;
  }
  return FALSE;
}

/* ************************************************************************ */
/* Warning: this procedure alters the name part of FileSpecs
	    but leaves the path part unchanged */
bool CompareSetOfFiles(void)
{
  struct ffblk FindData1, FindData2;
  bool Done1, Done2;
  /* Pointers to the name part of the files */
  char* FirstFileName = FileNameStart(FileSpecs[FIRST_FILE]);
  char* SecondFileName = FileNameStart(FileSpecs[SECOND_FILE]);

  /* Avoid path length overflow */
  if (FirstFileName - FileSpecs[FIRST_FILE] + strlen (SourceNamePattern) >=
      sizeof(FileSpecs[FIRST_FILE]))
    return FALSE;
  strcpy(FirstFileName, SourceNamePattern);

  Done1 = (findfirst(FileSpecs[FIRST_FILE], &FindData1, FA_RDONLY) != 0);
  while (!Done1)
  {
    /* Avoid path length overflow */
    if (FirstFileName - FileSpecs[FIRST_FILE] + strlen (FindData1.ff_name) >=
	sizeof(FileSpecs[FIRST_FILE]))
    {
      findstop(&FindData1);
      return FALSE;
    }
    strcpy(FirstFileName, FindData1.ff_name);
    Done2 = FALSE;
    switch (NameMatching)
    {
      case MatchingNames:
	Done2 = !MatchNames(TargetNamePattern, FirstFileName, SecondFileName);
	break;

      case CommonSource:
	Done2 = (findfirst(FileSpecs[SECOND_FILE], &FindData2, FA_RDONLY) != 0);
	if (!Done2)
	  /* Avoid path length overflow */
	  if (SecondFileName - FileSpecs[SECOND_FILE] +
	      strlen (FindData2.ff_name) >= sizeof(FileSpecs[SECOND_FILE]))
	    Done2 = TRUE;
	  else
	    strcpy(SecondFileName, FindData2.ff_name);
	break;
    }

    while (!Done2)
    {
      File2 = fopen(FileSpecs[SECOND_FILE], "rb");
      if (File2 != NULL)
      {
	File1 = fopen(FileSpecs[FIRST_FILE], "rb");
	if (File1 == NULL)
	{
	  fcloseall();
	  printf(Format(MESSAGE,9,"Error opening file %s"),
		 FileSpecs[FIRST_FILE]); NewLine;
	  findstop(&FindData1);
	  if (NameMatching == CommonSource) findstop(&FindData2);
	  return FALSE;
	}
	FileCount++;
	printf(Format(MESSAGE,10,"Comparing %s and %s"),
	       FileSpecs[FIRST_FILE], FileSpecs[SECOND_FILE]); NewLine;
	switch (CompareMode)
	{
	  case ASCII:
	    if (!AsciiCompare()) FoundDifferences = TRUE;
	    break;

	  case BINARY:
	    if (!BinaryCompare()) FoundDifferences = TRUE;
	    break;

	  default:
	    if (BinaryFile(FileSpecs[FIRST_FILE]) ||
		BinaryFile(FileSpecs[SECOND_FILE]))
	    {
	      if (!BinaryCompare()) FoundDifferences = TRUE;
	    }
	    else
	      if (!AsciiCompare()) FoundDifferences = TRUE;
	}
	fcloseall();
      }

      Done2 = TRUE;
      if (NameMatching == CommonSource)
      {
	Done2 = (findnext(&FindData2) != 0);
	if (!Done2)
	  /* Avoid path length overflow */
	  if (SecondFileName - FileSpecs[SECOND_FILE] +
	      strlen (FindData2.ff_name) >= sizeof(FileSpecs[SECOND_FILE]))
	    Done2 = TRUE;
	  else
	    strcpy(SecondFileName, FindData2.ff_name);
      }
    }
    Done1 = (findnext(&FindData1) != 0);
  }
  return TRUE;
}
/* ------------------------------------------------------------------------ */
/* Warning: this procedure appends data to FileSpecs */
bool ScanSubdirs(void)
{
  struct ffblk FindData1;
  char* FirstSubdirName;  /* Pointers to the subdirectory names */
  char* SecondSubdirName;

  FirstSubdirName = FileNameStart(FileSpecs[FIRST_FILE]);
  SecondSubdirName = FileNameStart(FileSpecs[SECOND_FILE]);

  /* Avoid path length overflow */
  if (FirstSubdirName - FileSpecs[FIRST_FILE] + strlen (AllFiles) >=
      sizeof(FileSpecs[FIRST_FILE]))
    return FALSE;
  strcpy(FirstSubdirName, AllFiles); /* All subdirectories */

  if (findfirst(FileSpecs[FIRST_FILE], &FindData1, FA_DIREC) != 0)
    return FALSE;

  do
    if (*FindData1.ff_name != '.')
    {
      struct ffblk FindData2;
      /* Avoid path length overflow */
      if (SecondSubdirName - FileSpecs[SECOND_FILE] +
	  strlen (FindData1.ff_name) >
	  sizeof(FileSpecs[SECOND_FILE]) - sizeof(DummyFileSpec))
      {
	findstop(&FindData1);
	return FALSE;
      }
      /* If the same subdirectory exists in the second path */
      strcpy(SecondSubdirName, FindData1.ff_name);
      FileSpecs[SECOND_FILE][sizeof(FileSpecs[SECOND_FILE]) - 1] = END_OF_STRING;
      if (findfirst(FileSpecs[SECOND_FILE], &FindData2, FA_DIREC) == 0)
      {
	findstop(&FindData2);
	/* Avoid path length overflow */
	if (FirstSubdirName - FileSpecs[FIRST_FILE] +
	    strlen (FindData1.ff_name) >
	    sizeof(FileSpecs[FIRST_FILE]) - sizeof(DummyFileSpec))
	{
	  findstop(&FindData1);
	  return FALSE;
	}
	strcpy(FirstSubdirName, FindData1.ff_name);
	FileSpecs[FIRST_FILE][sizeof(FileSpecs[FIRST_FILE]) - 1] = END_OF_STRING;

	/* Do compare the files in the subdirs */
	strcat(FileSpecs[FIRST_FILE], DummyFileSpec);
	strcat(FileSpecs[SECOND_FILE], DummyFileSpec);
	if (!CompareSetOfFiles() || !ScanSubdirs())
	{
	  findstop(&FindData1);
	  return FALSE;
	}
      }
  } while (findnext(&FindData1) == 0);
  return TRUE;
}
/* ------------------------------------------------------------------------ */
void OnExit(void)
{
  /* If created destroy the CRC speed-up table */
  if (CRCTable != NULL) free(CRCTable);
#ifdef USE_CATS
  catclose(CatalogDescriptor); /* Close the NLS catalog */
#endif
}
/* ------------------------------------------------------------------------ */
int main(int argc, char* argv[])
{
#ifdef USE_CATS
  /* Try opening NLS catalog */
  CatalogDescriptor = catopen("FC", 1);
#endif

  /* Install the clean-up procedure */
  atexit (OnExit);

  if (!ScanArguments(argc, argv)) return EL_PARAM_ERROR;

  FileCount = 0; FoundDifferences = FALSE;

  if (!CompareSetOfFiles()) return EL_OPEN_ERROR;

  if (OptFlags & SCAN_SUBDIRS)
    if (!ScanSubdirs()) return EL_OPEN_ERROR;

  if (FileCount == 0)
  {
    printf(Format(MESSAGE,11,"No such file or directory")); NewLine;
    return EL_NO_FILE_ERROR;
  }

  if (FoundDifferences) return EL_DIFFERENT;
  return EL_NO_DIFF;
}
