/*
   Fileloop.c - file loops checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdio.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"
#include "..\errmsgs\PrintBuf.h"

static BOOL LoopChecker(RDWRHandle handle, struct DirectoryPosition* pos,
                        struct DirectoryEntry* entry, void** structure);
static BOOL LoopTruncator(RDWRHandle handle, struct DirectoryPosition* pos,
                          struct DirectoryEntry* entry, void** structure);
static BOOL ScanForLoops(RDWRHandle handle, CLUSTER firstcluster,
                         CLUSTER* loopingcluster);

/*================================ CHECKING ==============================*/
/*************************************************************************
**                           CheckFilesForLoops
**************************************************************************
** Goes through all the files of a volume and sees wether there is one with
** a loop.
**************************************************************************/
RETVAL CheckFilesForLoops(RDWRHandle handle)
{
    BOOL invalid = FALSE, *pinvalid = &invalid;

    if (!FastWalkDirectoryTree(handle, LoopChecker, (void**) &pinvalid))
       return ERROR;

    return (invalid) ? FAILED : SUCCESS;
}

/*************************************************************************
**                             LoopChecker
**************************************************************************
** Goes through all the clusters of a file and sees wether there is a loop.
*************************************************************************/

static BOOL LoopChecker(RDWRHandle handle, struct DirectoryPosition* pos,
                        struct DirectoryEntry* entry, void** structure)
{
    CLUSTER firstcluster, dummy;
    BOOL *invalid = *((BOOL**) structure);

    pos = pos;

    firstcluster = GetFirstCluster(entry);
    if (firstcluster)
    {
       switch (ScanForLoops(handle, firstcluster, &dummy))
       {
          case FALSE:                  
               ShowDirectoryViolation(handle, pos, entry, 
                                      "%s contains a loop");
               *invalid = TRUE;
               break;
          case TRUE:
               break;
          case FAIL:
               return FAIL;
       }
    }
    
    return TRUE;
}

/*================================ FIXING ==============================*/
/*************************************************************************
**                           TruncateLoopingFiles
**************************************************************************
** Goes through all the files of a volume and sees wether there is one with
** a loop. If there is one then the loop is broken.
**************************************************************************/

RETVAL TruncateLoopingFiles(RDWRHandle handle)
{
    return (FastWalkDirectoryTree(handle, LoopTruncator, NULL)) ?
           SUCCESS : ERROR;
}

/*************************************************************************
**                             LoopTruncator
**************************************************************************
** Goes through all the clusters of a file and sees wether there is a loop.
** If there is the file is truncated.
*************************************************************************/

static BOOL LoopTruncator(RDWRHandle handle, struct DirectoryPosition* pos,
                          struct DirectoryEntry* entry, void** structure)
{
    CLUSTER firstcluster, offending;

    structure = structure, pos = pos;

    firstcluster = GetFirstCluster(entry);
    if (firstcluster)
    {
       switch (ScanForLoops(handle, firstcluster, &offending))
       {
          case FALSE:
               ShowDirectoryViolation(handle, pos, entry, 
                                      "%s contains a loop");
               
               if (!WriteFatLabel(handle, offending, FAT_LAST_LABEL))
                  return FAIL;
               break;
          case TRUE:
               break;
          case FAIL:
               return FAIL;
       }
    }
    
    return TRUE;
}

/*================================== Common ============================*/

/*************************************************************************
**                               ScanForLoops
**************************************************************************
** Goes through all the clusters of a file and sees wether there is a loop.
** If there is it returns the cluster just before the junction, so that
** the longest file chain is retained.
**
**   Returns:
**       FALSE if there is a loop or if there is an invalid cluster value
**            then returns the second reference to the duplicated cluster
**            in loopingcluster
**       
**       TRUE if there is no loop
**
**       FAIL if there was an error accessing the volume.
***************************************************************************/

static BOOL ScanForLoops(RDWRHandle handle, CLUSTER firstcluster,
                         CLUSTER* loopingcluster)
{
    char* bitfield;
    unsigned long labelsinfat;
    CLUSTER current, label;

    labelsinfat = GetLabelsInFat(handle);
    if (!labelsinfat) return FAIL;

    /* Check the given first cluster on validity */
    if (!FAT_NORMAL(firstcluster))
       return TRUE;

    /* Take the first cluster of the file */
    if (!GetNthCluster(handle, firstcluster, &label))
    {
       return FAIL;
    }
       
    if (!FAT_NORMAL(label)) /* One cluster cannot contain a loop */
    {
       return TRUE;
    }  
    
    if (!IsLabelValid(handle, label))
    {
       *loopingcluster = firstcluster;
       return FALSE;
    }    
    
    bitfield = CreateBitField(labelsinfat);
    if (!bitfield) return FAIL;  
    
    SetBitfieldBit(bitfield, firstcluster);

    if (GetBitfieldBit(bitfield, label)) /* First cluster points to itself */
    {
       *loopingcluster = firstcluster;
       DestroyBitfield(bitfield);
       return FALSE;
    }
    SetBitfieldBit(bitfield, label);

    for (;;)
    {
        current = label;

        if (!GetNthCluster(handle, current, &label))
        {
           DestroyBitfield(bitfield);
           return FAIL;
        }
        
        if (!FAT_NORMAL(label)) break;
        
        if (!IsLabelValid(handle, label))
        {
           *loopingcluster = current;
           DestroyBitfield(bitfield);
           return FALSE;
        }

        if (GetBitfieldBit(bitfield, label))
        {
           *loopingcluster = current;
           DestroyBitfield(bitfield);
           return FALSE;
        }
        SetBitfieldBit(bitfield, label);
    }

    DestroyBitfield(bitfield);
    return TRUE;
}