/*    
   FTEMem.c - heap memory management for the FAT transformation engine.

   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdlib.h>

#include "FTE.h"
#include "FTEMEM.h"
#include "backmem.h"
#include "suremem.h"

//#define DEBUG_FTEMEM
#ifdef DEBUG_FTEMEM
 long FTEMEM_Debug_Counter=0;
#endif

ALLOCRES AllocateFTEMemory(unsigned guaranteed, unsigned char guaranteedblocks,
                           unsigned backupbytes)
{
   if (guaranteed && guaranteedblocks)
   {
      if (!AllocateGuaranteedMemory(guaranteed, guaranteedblocks))
      {
         return TOTAL_FAIL;
      }
   }

   if (backupbytes)
   {
      if (!AllocateBackupMemory(backupbytes))
      {
         return BACKUP_FAIL;
      }
   }

   return ALLOC_SUCCESS;
}

void DeallocateFTEMemory(void)
{
   FreeGuaranteedMemory();
   FreeBackupMemory();
}

/* Generic allocation */
void* FTEAlloc(size_t bytes)
{
   void* retval;

#ifdef DEBUG_FTEMEM
   printf("Alloc: %ld\n", FTEMEM_Debug_Counter++);
#endif

   retval = malloc(bytes);
   if (!retval)
   {
      retval = BackupAlloc(bytes);
      if (retval)
      {
         return retval;
      }
      SetFTEerror(FTE_MEM_INSUFFICIENT);
   }

#ifdef DEBUG_FTEMEM
   printf("Allocated at: %lu\n", (unsigned long) retval);
#endif

   return retval;
}

void  FTEFree(void* tofree)
{
#ifdef DEBUG_FTEMEM
   printf("Free: %ld\n", --FTEMEM_Debug_Counter);
   printf("Freeing: %lu\n", (unsigned long) tofree);
#endif

   if (InBackupMemoryRange(tofree))
   {
      BackupFree(tofree);
   }
   else
   {
      free(tofree);
   }
}

/* Sectors -- Assumes that bytespersector field is filled in at handle
              creation (InitReadWriteSectors,...)                     */
SECTOR* AllocateSector(RDWRHandle handle)
{
   return (SECTOR*) FTEAlloc(handle->BytesPerSector);
}

SECTOR* AllocateSectors(RDWRHandle handle, int count)
{
   return (SECTOR*) FTEAlloc(handle->BytesPerSector * count);
}

void FreeSectors(SECTOR* sector)
{
   FTEFree((void*)sector);
}

/* Boot */
struct BootSectorStruct* AllocateBootSector(void)
{
   return (struct BootSectorStruct*)
          FTEAlloc(sizeof(struct BootSectorStruct));
}

void FreeBootSector(struct BootSectorStruct* boot)
{
   FTEFree((void*) boot);
}

/* Directories */
struct DirectoryEntry* AllocateDirectoryEntry(void)
{
   return (struct DirectoryEntry*) FTEAlloc(sizeof(struct DirectoryEntry));
}

void FreeDirectoryEntry(struct DirectoryEntry* entry)
{
   FTEFree(entry);
}

struct FSInfoStruct* AllocateFSInfo(void)
{
   return (struct FSInfoStruct*) FTEAlloc(sizeof(struct FSInfoStruct));
}

void FreeFSInfo(struct FSInfoStruct* info)
{
   FTEFree(info);
}
