/*
   ChkEntrs.c - directory entry checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"

#define INVALIDSFNCHARS "\0x22\0x2a\0x2b\0x2c\0x2e\0x3a\0x3b\0x3c\0x3d" \
                        "\0x3e\0x3f\0x5b\0x5c\0x5d\0x7c"

static BOOL EntryChecker(RDWRHandle handle, struct DirectoryPosition* pos,
			 struct DirectoryEntry* entry, void** structure);
static BOOL EntryFixer(RDWRHandle handle, struct DirectoryPosition* pos,
		       struct DirectoryEntry* entry, void** structure);
static BOOL LookAtEntry(RDWRHandle handle, struct DirectoryPosition* pos,
			BOOL fixit);

/* Checking */

RETVAL CheckDirectoryEntries(RDWRHandle handle)
{
    BOOL invalid = FALSE, *pinvalid = &invalid;

    if (!FastWalkDirectoryTree(handle, EntryChecker, (void**) &pinvalid))
       return ERROR;

    return (invalid) ? FAILED : SUCCESS;
}

static BOOL EntryChecker(RDWRHandle handle, struct DirectoryPosition* pos,
			 struct DirectoryEntry* entry, void** structure)
{
    BOOL *invalid = *((BOOL**) structure);

    entry = entry;

    switch (LookAtEntry(handle, pos, FALSE))
    {
      case FALSE:
           *invalid = TRUE;
      case TRUE:
           return TRUE;
      case FAIL:
           return ERROR;
    }

    return FAIL;                         /* Should never be executed. */
}

/* Fixing */

RETVAL FixDirectoryEntries(RDWRHandle handle)
{
    return (FastWalkDirectoryTree(handle, EntryFixer, NULL)) ?
           SUCCESS : ERROR;
}

static BOOL EntryFixer(RDWRHandle handle, struct DirectoryPosition* pos,
		       struct DirectoryEntry* entry, void** structure)
{
    structure = structure, entry = entry;

    if (LookAtEntry(handle, pos, TRUE) == FAIL)
       return FAIL;

    return TRUE;
}

/* Common */

static BOOL LookAtEntry(RDWRHandle handle, struct DirectoryPosition* pos,
			BOOL fixit)
{
    int i;
    BOOL invalidname = FALSE;
    BOOL retval = TRUE;
    struct DirectoryEntry entry;
    struct LongFileNameEntry* lfnentry = (struct LongFileNameEntry*) &entry;

    if (!GetDirectory(handle, pos, &entry))
       return FAIL;

    /* Do not check deleted entries */
    if (IsDeletedLabel(entry))
       return TRUE;

    if (IsLFNEntry(&entry))
    {
       if (lfnentry->firstclust)
       { 
          printf("Found an invalid long file name entry\n");
          if (fixit)
          {
             lfnentry->firstclust = 0;
          }
          
          retval = FALSE;
       }
    }
    else
    {
       /* Check the file name */
       if (entry.filename[0] == ' ')
       {
          invalidname = TRUE;
          if (fixit)
             entry.filename[0] = 'A';
       }

       /* file name */
       for (i = 0; i < 8; i++)
       {
           if ((strchr(INVALIDSFNCHARS, entry.filename[i]))                ||
               ((entry.filename[i] < 0x20) && (entry.filename[i] != 0x05)) ||
               (islower(entry.filename[i])))
               
           {
              if (fixit)
                 entry.filename[i] = 'A';
              invalidname = TRUE;
           }
       }

       /* extension */
       for (i = 0; i < 3; i++)
       {
           if ((strchr(INVALIDSFNCHARS, entry.extension[i]))               ||
               ((entry.extension[i] < 0x20) &&
                                             (entry.extension[i] != 0x05)) ||
               (islower(entry.extension[i])))
               
           {
              if (fixit)
                 entry.extension[i] = 'A';
              invalidname = TRUE;
           }
       }

       if (invalidname)
       {
          printf("Found invalid char(s) in a name\n");
          retval = FALSE;
       }
       
       /* Check the attribute */
       if (entry.attribute & 0xC0)
       {
          printf("found a file with an invalid attribute\n");

          if (entry.attribute & FA_DIREC)
          {
             entry.attribute = FA_DIREC;
          }
          retval = FALSE;
       }

       /*
          Check the time stamps:
             Notice that year is valid over it's entire range.
       */

       /* Creation time */

       if ((memcmp(&entry.timestamp, "\0\0", 2) != 0) &&  /* Optional field */
           ((entry.timestamp.second > 29)  ||
            (entry.timestamp.minute > 59)  ||
            (entry.timestamp.hours  > 23)))
       {
          printf("Found an entry with an invalid creation time\n");
          if (fixit)
          {
             memset(&entry.timestamp, 0, sizeof(struct PackedTime));
          }
       }

       /* Creation date */
       
       if ((memcmp(&entry.datestamp, "\0\0", 2) != 0) &&  /* Optional field */
           (((entry.datestamp.day   < 1) || (entry.datestamp.day > 31))  ||
            ((entry.datestamp.month < 1) || (entry.datestamp.month > 12))))
       {
	  printf("Found an entry with an invalid creation date\n");
          if (fixit)
          {
             memset(&entry.datestamp, 0, sizeof(struct PackedDate));
          }
       }

       /* Last access date */
       
       if ((memcmp(&entry.LastAccessDate, "\0\0", 2) != 0) &&
           (((entry.LastAccessDate.day   < 1) ||
                                         (entry.LastAccessDate.day > 31))  ||
            ((entry.LastAccessDate.month < 1) ||
                                         (entry.LastAccessDate.month > 12))))
       {
	  printf("Found an entry with an invalid access date\n");
          if (fixit)
          {
             memset(&entry.LastAccessDate, 0, sizeof(struct PackedDate));
          }
       }
       
       /* Last write time (mandatory) */
       if ((entry.LastWriteTime.second > 29)  ||
           (entry.LastWriteTime.minute > 59)  ||
           (entry.LastWriteTime.hours  > 23))
       {
          printf("Found an entry with an invalid last write time\n");
          if (fixit)
          {
             memset(&entry.LastWriteTime, 0, sizeof(struct PackedTime));
          }
       }

       /* Last write date (mandatory) */

       if (((entry.LastWriteDate.day   < 1) || (entry.LastWriteDate.day > 31))  ||
           ((entry.LastWriteDate.month < 1) || (entry.LastWriteDate.month > 12)))
       {
	  printf("Found an entry with an invalid last write date\n");
          if (fixit)
          {
             memset(&entry.LastWriteDate, 0, sizeof(struct PackedDate));
          }
       }
    }

    if (fixit && !retval)
    {
       if (!WriteDirectory(handle, pos, &entry))
          return FAIL;
    }

    return retval;
}
