/*
   ChkLFNs.c - LFN checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <string.h>
#include <stdio.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"

#define SCANNING_SFN_STATE 1
#define SCANNING_LFN_STATE 2

static int LFNState;
static int LFNIndex;
static unsigned char LFNChkSum;
static struct DirectoryPosition LFNStart;

struct Pipe
{
   BOOL    invalid;
   BOOL    fixit;
   CLUSTER firstcluster;
};

struct RemovePipe
{
   struct DirectoryPosition* begin;
   struct DirectoryPosition* end;

   BOOL hasstarted;
};

static BOOL LFNChecker(RDWRHandle handle, struct DirectoryPosition* pos,
		       struct DirectoryEntry* entry, void** structure);
static BOOL LFNFixer(RDWRHandle handle, struct DirectoryPosition* pos,
		     struct DirectoryEntry* entry, void** structure);
static BOOL DirChecker(RDWRHandle handle, struct DirectoryPosition* pos,
		       void** structure);
static BOOL CheckDirectory(RDWRHandle handle, CLUSTER firstcluster,
                           BOOL fixit);
static BOOL RemoveInvalidLFNs(RDWRHandle handle, CLUSTER firstcluster,
                              struct DirectoryPosition* pos1,
                              struct DirectoryPosition* pos2);
static BOOL LFNRemover(RDWRHandle handle, struct DirectoryPosition* pos,
		       void** structure);


/* Checking */

RETVAL CheckLFNs(RDWRHandle handle)
{
    BOOL invalid = FALSE, *pinvalid = &invalid;

    switch (CheckDirectory(handle, 0, FALSE))
    {
       case TRUE:
            break;
            
       case FALSE:
            invalid = TRUE;
            break;

       case FAIL:
            return ERROR;
    }

    if (!FastWalkDirectoryTree(handle, LFNChecker, (void**) &pinvalid))
       return ERROR;

    return (invalid) ? FAILED : SUCCESS;
}

static BOOL LFNChecker(RDWRHandle handle, struct DirectoryPosition* pos,
		       struct DirectoryEntry* entry, void** structure)
{
    BOOL *invalid = *((BOOL**) structure);
    CLUSTER firstcluster;

    pos = pos;

    if ((entry->attribute & FA_DIREC) &&
	(!IsDeletedLabel(*entry))     &&
	(!IsCurrentDir(*entry))       &&
	(!IsPreviousDir(*entry))      &&
	(!IsLFNEntry(entry)))
    {
       firstcluster = GetFirstCluster(entry);
       if (firstcluster)
       {
	  switch (CheckDirectory(handle, firstcluster, FALSE))
          {
            case TRUE:
                 break;

            case FALSE:
                 *invalid = TRUE;
                 break;

            case FAIL:
                 return FAIL;
          }
       }
    }
    
    return TRUE;
}

/* Fixing */

RETVAL EraseInvalidLFNs(RDWRHandle handle)
{
    switch (CheckDirectory(handle, 0, TRUE))
    {
       case TRUE:
       case FALSE:
            break;

       case FAIL:
            return ERROR;
    }

    if (!FastWalkDirectoryTree(handle, LFNFixer, NULL))
       return ERROR;

    return SUCCESS;
}

static BOOL LFNFixer(RDWRHandle handle, struct DirectoryPosition* pos,
		     struct DirectoryEntry* entry, void** structure)
{
    CLUSTER firstcluster;

    structure = structure, pos = pos;

    if ((entry->attribute & FA_DIREC) &&
        (!IsCurrentDir(*entry))        &&
        (!IsPreviousDir(*entry)))
    {
       firstcluster = GetFirstCluster(entry);
       if (firstcluster)
       {
          if (CheckDirectory(handle, 0, TRUE) == FAIL)
             return FAIL;
       }
    }
    
    return TRUE;
}

/* Common */

static BOOL CheckDirectory(RDWRHandle handle, CLUSTER firstcluster,
                           BOOL fixit)
{
    struct Pipe pipe, *ppipe = &pipe;

    pipe.invalid      = FALSE;
    pipe.fixit        = fixit;
    pipe.firstcluster = firstcluster;

    LFNState = SCANNING_SFN_STATE;
    LFNIndex = 0;

    if (!TraverseSubdir(handle, firstcluster, DirChecker,
                        (void**) &ppipe, TRUE))
       return FAIL;

    if ((LFNState != SCANNING_SFN_STATE) ||
        (LFNIndex > 0))
    {
       printf("Found a number of invalid long file name entries\n");
       
       if (fixit)
       {
          if (!RemoveInvalidLFNs(handle, firstcluster, &LFNStart, NULL))
             return FAIL;
       }
       
       return FALSE;
    }

    return !pipe.invalid;
}

static BOOL DirChecker(RDWRHandle handle, struct DirectoryPosition* pos,
		       void** structure)
{
    unsigned char ChkSum;
    struct Pipe *pipe = *((struct Pipe**) structure);
    struct DirectoryEntry entry;
    struct LongFileNameEntry* lfnentry = (struct LongFileNameEntry*) &entry;

    if (!GetDirectory(handle, pos, &entry))
       return FAIL;

    if (IsDeletedLabel(entry))
       return TRUE;

    if (!IsLFNEntry(&entry)) /* A short filename entry */
    {
       if (LFNState == SCANNING_LFN_STATE)
       {
          /* Calculate the checksum for this SFN entry */
          ChkSum = CalculateSFNCheckSum(&entry);

          if ((LFNIndex != 0) ||
              (ChkSum != LFNChkSum))
          {
	     printf("Found a number of invalid long file name entries\n");

             if (pipe->fixit)
             {
                if (!RemoveInvalidLFNs(handle, pipe->firstcluster,
                                       &LFNStart, pos))
                   return FAIL;
             }
             else
             {
		pipe->invalid = TRUE;
             }
          }
          LFNState = SCANNING_SFN_STATE;
       }
    }
    else if (IsFirstLFNEntry(lfnentry)) /* First LFN entry */
    {
       if (LFNState == SCANNING_SFN_STATE)
       {
          memcpy(&LFNStart, pos, sizeof(struct DirectoryPosition));
          LFNState  = SCANNING_LFN_STATE;
	  LFNIndex  = GetNrOfLFNEntries(lfnentry)-1;
          LFNChkSum = lfnentry->checksum;
       }
       else
       {
          printf("Found a number of invalid long file name entries\n");
          if (pipe->fixit)
          {
             if (!RemoveInvalidLFNs(handle, pipe->firstcluster,
                                    &LFNStart, pos))
                return FAIL;
          }
          else
          {
             pipe->invalid = TRUE;
          }
          
          LFNState = SCANNING_SFN_STATE;
       }
    }
    else /* LFN entry in the middle. */
    {
       if ((LFNState == SCANNING_SFN_STATE)  ||
           (lfnentry->NameIndex != LFNIndex) ||
           (LFNIndex == 0)                   ||
           (lfnentry->checksum != LFNChkSum))
       {
          if (pipe->fixit)
          {
             if (!RemoveInvalidLFNs(handle, pipe->firstcluster,
                                    &LFNStart, pos))
                return FAIL;
          }
          else
          {
             printf("Found a number of invalid long file name entries\n");
             pipe->invalid = TRUE;
          }
       }
       else
       {
	  LFNIndex--;
       }
    }
    
    return TRUE;
}

/*
   Changes all the entries in the directory pointed to by firstcluster.

   If pos2 == 0, all the long file name entries starting at pos1 are
   changed.
*/

static BOOL RemoveInvalidLFNs(RDWRHandle handle, CLUSTER firstcluster,
                              struct DirectoryPosition* pos1,
                              struct DirectoryPosition* pos2)
{
    struct RemovePipe pipe, *ppipe = &pipe;

    pipe.begin = pos1;
    pipe.end   = pos2;


    return TraverseSubdir(handle, firstcluster, LFNRemover, (void**) &ppipe,
                          TRUE);
}


static BOOL LFNRemover(RDWRHandle handle, struct DirectoryPosition* pos,
                       void** structure)
{
    struct RemovePipe* pipe = *((struct RemovePipe**) structure);
    struct DirectoryEntry entry;

    if (!GetDirectory(handle, pos, &entry))
       return FAIL;

    if (!IsLFNEntry(&entry))
       return TRUE;

    if (!pipe->hasstarted)
    {
       if (memcmp(pos, pipe->begin, sizeof(struct DirectoryPosition)) == 0)
          pipe->hasstarted = TRUE;
    }
    else
    {
       if ((pipe->end != NULL) &&
           (memcmp(pos, pipe->end, sizeof(struct DirectoryPosition)) == 0))
       {
          pipe->hasstarted = TRUE;
       }
    }
    
    if (pipe->hasstarted)
    {
       struct tm* tmp;
       time_t now;
    
       /* file name and extension */
       memcpy(entry.filename, "\0xe5" "LFNRMVD", 8);
       memset(entry.extension, ' ', 3);

       /* attribute */
       entry.attribute = 0;

       /* first cluster */
       SetFirstCluster(2, &entry);

       /* file size */
       entry.filesize = 0;

       /* NT reserved field */
       entry.NTReserved = 0;

       /* Mili second stamp */
       entry.MilisecondStamp = 0;

       /* Last access date */
       memset(&entry.LastAccessDate, 0, sizeof(struct PackedDate));
    
       /* Time last modified */
       memset(&entry.timestamp, 0, sizeof(struct PackedTime));

       /* Date last modified */
       memset(&entry.datestamp, 0, sizeof(struct PackedDate));

       /* Get the current date and time and store it in the last write
          time and date. */
       time(&now);
       tmp = localtime(&now);

       entry.LastWriteTime.second = tmp->tm_sec / 2;
       if (entry.LastWriteTime.second == 30) /* DJGPP help says range is [0..60] */
          entry.LastWriteTime.second--;
    
       entry.LastWriteTime.minute = tmp->tm_min;
       entry.LastWriteTime.hours  = tmp->tm_hour;

       entry.LastWriteDate.day   = tmp->tm_mday;
       entry.LastWriteDate.month = tmp->tm_mon + 1;

       if (tmp->tm_year < 80)
          entry.LastWriteDate.year = 0;
       else
          entry.LastWriteDate.year = (tmp->tm_year+1900)-1980;

       return TRUE;
    }

    return TRUE;
}
