/*
   dblfiles.c - double files checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdio.h>
#include <string.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"

static BOOL SearchDirForDoubles(RDWRHandle handle, CLUSTER firstcluster);
static BOOL DoubleFilenameFinder(RDWRHandle handle,
				 struct DirectoryPosition* pos,
				 struct DirectoryEntry* entry,
				 void** structure);
static BOOL RenameDoublesInDirs(RDWRHandle handle, CLUSTER firstcluster);
static BOOL RenameDoubleFile(RDWRHandle handle,
                             struct DirectoryPosition* pos,
                             struct DirectoryEntry* entry,
                             CLUSTER firstcluster);
static BOOL DoubleFilenameRenamer(RDWRHandle handle,
                                  struct DirectoryPosition* pos,
				  struct DirectoryEntry* entry,
                                  void** structure);


/* Checking */

RETVAL FindDoubleFiles(RDWRHandle handle)
{
    BOOL invalid = FALSE, *pinvalid = &invalid, inroot;

    inroot = SearchDirForDoubles(handle, 0);
    if (inroot == FAIL) return ERROR;

    if (!FastWalkDirectoryTree(handle, DoubleFilenameFinder,
                               (void**) &pinvalid))
       return ERROR;

    return (inroot || invalid) ? FAILED : SUCCESS;
}

/*
    Searches through a directory for doubles and returns:

    TRUE  if there were doubles.
    FALSE if there were no doubles.
    FAIL  if there was a medium error.
*/
static BOOL SearchDirForDoubles(RDWRHandle handle, CLUSTER firstcluster)
{
    int k;
    unsigned long i, j;
    struct DirectoryPosition pos = {1,1};
    struct DirectoryEntry entry;
    
    char filename[8];
    char extension[3];
    BOOL doublesfound = FALSE;    

    for (i = 1;; i++)
    {
        pos.sector = 0;
        pos.offset = 0;

	if (!GetNthDirectoryPosition(handle, firstcluster, i, &pos))
           return FAIL;

        if ((pos.sector == 0) && (pos.offset == 0)) break;

        if (!GetDirectory(handle, &pos, &entry))
           return FAIL;

	if ((entry.attribute & FA_LABEL) ||
	    IsLFNEntry(&entry)           ||
	    IsDeletedLabel(entry))
	{
	   continue;
	}

        memcpy(filename, entry.filename, 8);
        memcpy(extension, entry.extension, 3);

        for (j = 0; j < i; j++)
        {
	    if (!GetNthDirectoryPosition(handle, firstcluster, j, &pos))
               return FAIL;

           if (!GetDirectory(handle, &pos, &entry))
	      return FAIL;

	   if ((entry.attribute & FA_LABEL) ||
	       IsLFNEntry(&entry)           ||
	       IsDeletedLabel(entry))
	   {
	      continue;
	   }

	   if (IsLFNEntry(&entry) || IsDeletedLabel(entry))
	      continue;

	   if ((memcmp(filename, entry.filename, 8) == 0) &&
	       (memcmp(extension, entry.extension, 3) == 0))
	   {
              doublesfound = TRUE;
              printf("Found double file ");
	      for (k = 0; k < 8; k++)
		  if (filename[k] != ' ')
		     printf("%c", filename[k]);
	      printf(".");
	      for (k = 0; k < 3; k++)
		  if (filename[k] != ' ')
		     printf("%c", extension[k]);
	      puts("");
	   }
	}
    }
    
    return doublesfound;
}

static BOOL DoubleFilenameFinder(RDWRHandle handle,
                                 struct DirectoryPosition* pos,
				 struct DirectoryEntry* entry,
                                 void** structure)
{
    BOOL* invalid = *((BOOL**) structure);

    pos = pos;

    if (IsLFNEntry(entry))
       return TRUE;

    if (IsDeletedLabel(*entry))
       return TRUE;

    if (entry->attribute & FA_DIREC)
    {
       switch (SearchDirForDoubles(handle, GetFirstCluster(entry)))
       {
          case TRUE:
               *invalid = TRUE;
               return TRUE;
               
          case FAIL:
               return FAIL;
       }
    }

    return TRUE;
}


/* Fixing */

RETVAL RenameDoubleFiles(RDWRHandle handle)
{
    if (RenameDoublesInDirs(handle, 0) == FAIL)
       return ERROR;

    return (FastWalkDirectoryTree(handle, DoubleFilenameRenamer, NULL)) ?
           SUCCESS : ERROR;
}

static BOOL DoubleFilenameRenamer(RDWRHandle handle,
                                  struct DirectoryPosition* pos,
				  struct DirectoryEntry* entry,
                                  void** structure)
{
    structure = structure, pos = pos;

    if (IsLFNEntry(entry))
       return TRUE;

    if (IsDeletedLabel(*entry))
       return TRUE;

    if (entry->attribute & FA_DIREC)
    {
       return RenameDoublesInDirs(handle, GetFirstCluster(entry));
    }

    return TRUE;
}

/*
    Searches through a directory for doubles and returns:

    TRUE  if everything processed.
    FAIL  if there was a medium error.
*/

static BOOL RenameDoublesInDirs(RDWRHandle handle, CLUSTER firstcluster)
{
    int k;
    unsigned long i, j;
    struct DirectoryPosition pos = {1,1};
    struct DirectoryEntry entry;
    
    char filename[8];
    char extension[3];

    for (i = 1;; i++)
    {
        pos.sector = 0;
        pos.offset = 0;

        if (!GetNthDirectoryPosition(handle, firstcluster, i, &pos))
           return FAIL;

        if ((pos.sector == 0) && (pos.offset == 0)) break;

        if (!GetDirectory(handle, &pos, &entry))
	   return FAIL;

	if ((entry.attribute & FA_LABEL) ||
	    IsLFNEntry(&entry)           ||
	    IsDeletedLabel(entry))
	{
	   continue;
	}

        memcpy(filename, entry.filename, 8);
        memcpy(extension, entry.extension, 3);
        
        for (j = 0; j < i; j++)
        {
	    if (!GetNthDirectoryPosition(handle, firstcluster, j, &pos))
               return FAIL;

           if (!GetDirectory(handle, &pos, &entry))
              return FAIL;

	   if ((entry.attribute & FA_LABEL) ||
	       IsLFNEntry(&entry)           ||
	       IsDeletedLabel(entry))
	   {
	      continue;
	   }

	   if ((memcmp(filename, entry.filename, 8) == 0) &&
               (memcmp(extension, entry.extension, 3) == 0))
           {
	      printf("Found double file ");
	      for (k = 0; k < 8; k++)
		  if (filename[k] != ' ')
		     printf("%c", filename[k]);
	      printf(".");
	      for (k = 0; k < 3; k++)
		  if (filename[k] != ' ')
		     printf("%c", extension[k]);
	      puts("");                   
                   
              if (!RenameDoubleFile(handle, &pos, &entry, firstcluster))
		 return FAIL;
           }
        }
    }
    
    return TRUE;
}

static BOOL RenameDoubleFile(RDWRHandle handle,
                             struct DirectoryPosition* pos,
                             struct DirectoryEntry* entry,
                             CLUSTER firstcluster)
{
    BOOL retval;
    int counter=0;
    char newext[3];

    for (;;)
    {
        sprintf(newext, "%3d", counter);
        
        retval = LoFileNameExists(handle, firstcluster,
                                  entry->filename, newext);

        if (retval == FALSE) break;
        if (retval == FAIL)  return FALSE;

        counter++;
    }

    memcpy(entry->extension, newext, 3);

    return WriteDirectory(handle, pos, entry);
}

