/*
   Crosslnk.c - cross linked cluster checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdio.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"

struct Pipe
{
   char* bitfield;
   BOOL  invalid;
};

/* Checking */

static BOOL CrossLinkFinder(RDWRHandle handle, CLUSTER label,
                            SECTOR datasector, void** structure);
                            
static BOOL FirstClusterChecker(RDWRHandle handle,
                                struct DirectoryPosition* pos,
                                struct DirectoryEntry* entry,
                                void** structure);

static BOOL CrossLinkSplitter(RDWRHandle handle, CLUSTER label,
                              SECTOR datasector, void** structure);

static BOOL CopyCompleteFileChain(RDWRHandle handle,
                                  struct DirectoryPosition* pos);

static BOOL CloneFileChain(RDWRHandle handle,
                           CLUSTER clusterbeforecross);

static BOOL CopyFileChain(RDWRHandle handle, CLUSTER source, CLUSTER target);

static BOOL FirstClusterSplitter(RDWRHandle handle,
                                 struct DirectoryPosition* pos,
                                 struct DirectoryEntry* entry,
                                 void** structure);

RETVAL FindCrossLinkedFiles(RDWRHandle handle)
{
   unsigned long clustersindataarea;
   struct Pipe pipe, *ppipe = &pipe;

   clustersindataarea = GetClustersInDataArea(handle);
   if (!clustersindataarea) return ERROR;

   pipe.bitfield = CreateBitField(clustersindataarea);
   if (!pipe.bitfield) return ERROR;

   pipe.invalid = FALSE;

   if (!LinearTraverseFat(handle, CrossLinkFinder, (void**) &ppipe))
   {
      DestroyBitfield(pipe.bitfield);
      return ERROR;
   }

   if (!FastWalkDirectoryTree(handle, FirstClusterChecker, (void**) &ppipe))
   {
      DestroyBitfield(pipe.bitfield);
      return ERROR;
   }
      
   DestroyBitfield(pipe.bitfield);
   return (pipe.invalid) ? FAILED : SUCCESS;
}

static BOOL CrossLinkFinder(RDWRHandle handle, CLUSTER label,
                            SECTOR datasector, void** structure)
{
   unsigned long bit;
   struct Pipe* pipe = *((struct Pipe**) structure);

   datasector = datasector, handle = handle;

   if (FAT_NORMAL(label))
   {
      bit = label-2;
      if (GetBitfieldBit(pipe->bitfield, bit))
      {
         printf("Found a cross linked cluster\n");
         pipe->invalid = TRUE;
      }

      SetBitfieldBit(pipe->bitfield, bit);
   }

   return TRUE;
}

static BOOL FirstClusterChecker(RDWRHandle handle,
                                struct DirectoryPosition* pos,
                                struct DirectoryEntry* entry,
                                void** structure)
{
   CLUSTER firstcluster;
   unsigned long bit;
   struct Pipe* pipe = *((struct Pipe**) structure);

   pos = pos, handle = handle;

   if ((entry->attribute & FA_LABEL) ||
       IsLFNEntry(entry)             ||
       IsPreviousDir(*entry)         ||
       IsCurrentDir(*entry)          ||
       IsDeletedLabel(*entry))
   {
      return TRUE;
   }

   firstcluster = GetFirstCluster(entry);
   if (firstcluster)
   {
      bit = firstcluster-2;
      if (GetBitfieldBit(pipe->bitfield, bit))
      {
         printf("Found a cross linked cluster\n");
         pipe->invalid = TRUE;
      }

      SetBitfieldBit(pipe->bitfield, bit);
   }

   return TRUE;
}

/* Fixing */

RETVAL SplitCrossLinkedFiles(RDWRHandle handle)
{
   unsigned long clustersindataarea;
   struct Pipe pipe, *ppipe = &pipe;

   clustersindataarea = GetClustersInDataArea(handle);
   if (!clustersindataarea) return ERROR;

   pipe.bitfield = CreateBitField(clustersindataarea);
   if (!pipe.bitfield) return ERROR;

   pipe.invalid = FALSE;

   if (!LinearTraverseFat(handle, CrossLinkSplitter, (void**) &ppipe))
   {
      DestroyBitfield(pipe.bitfield);
      return ERROR;
   }

   if (!FastWalkDirectoryTree(handle, FirstClusterSplitter, (void**) &ppipe))
   {
      DestroyBitfield(pipe.bitfield);
      return ERROR;
   }

   DestroyBitfield(pipe.bitfield);
   return SUCCESS;
}

static BOOL CrossLinkSplitter(RDWRHandle handle, CLUSTER label,
                            SECTOR datasector, void** structure)
{
   unsigned long bit;
   struct Pipe* pipe = *((struct Pipe**) structure);
   CLUSTER cluster;

   datasector = datasector, handle = handle;

   if (FAT_NORMAL(label))
   {
      bit = label-2;
      if (GetBitfieldBit(pipe->bitfield, bit))
      {
         printf("Found a cross linked cluster\n");
         cluster = DataSectorToCluster(handle, datasector);
         if (CloneFileChain(handle, cluster) == FAIL)
            return FAIL;
      }

      SetBitfieldBit(pipe->bitfield, bit);
   }

   return TRUE;
}

static BOOL FirstClusterSplitter(RDWRHandle handle,
                                 struct DirectoryPosition* pos,
                                 struct DirectoryEntry* entry,
                                 void** structure)
{
   CLUSTER firstcluster;
   unsigned long bit;
   struct Pipe* pipe = *((struct Pipe**) structure);

   if ((entry->attribute & FA_LABEL) ||
       IsLFNEntry(entry)             ||
       IsPreviousDir(*entry)         ||
       IsCurrentDir(*entry)          ||
       IsDeletedLabel(*entry))
   {
      return TRUE;
   }

   firstcluster = GetFirstCluster(entry);
   if (firstcluster)
   {
      bit = firstcluster-2;
      if (GetBitfieldBit(pipe->bitfield, bit))
      {
         printf("Found a cross linked cluster\n");
	 if (CopyCompleteFileChain(handle, pos) == FAIL)
            return FAIL;
      }

      SetBitfieldBit(pipe->bitfield, bit);
   }

   return TRUE;
}

static BOOL CopyCompleteFileChain(RDWRHandle handle,
                                  struct DirectoryPosition* pos)
{
   CLUSTER newcluster, firstcluster;
   struct DirectoryEntry entry;

   if (!GetDirectory(handle, pos, &entry))
      return FALSE;

   firstcluster = GetFirstCluster(&entry);
   if (!firstcluster) return TRUE;

   switch (CreateFileChain(handle, pos, &newcluster))
   {
      case FALSE:
           printf("Insufficient disk space to undouble cross linked clusters\n");
           return TRUE;
           
      case FAIL:
           return FAIL;
   }

   return CopyFileChain(handle, firstcluster, newcluster);
}

static BOOL CloneFileChain(RDWRHandle handle,
                           CLUSTER clusterbeforecross)
{
   CLUSTER current;

   /* Get the cluster that contains the cross. */
   if (!GetNthCluster(handle, clusterbeforecross, &current))
      return FAIL;

   /* Make a new chain, by first breaking the file right before the
      intersection point. */
   if (!WriteFatLabel(handle, clusterbeforecross, FAT_LAST_LABEL))
      return FAIL;

   return CopyFileChain(handle, current, clusterbeforecross);
}

/*
   Copies the remaining of the file chain addressed by source, extending
   the target before copying each cluster.
*/

static BOOL CopyFileChain(RDWRHandle handle, CLUSTER source, CLUSTER target)
{
   BOOL pasttheend;
   CLUSTER current = source, label, newcluster;
   SECTOR srcsect, dstsect;
   unsigned char sectorspercluster;

   sectorspercluster = GetSectorsPerCluster(handle);
   if (!sectorspercluster) return FAIL;

   /* Get the next cluster in the original file chain */
   current = GetNthFileCluster(handle, current, 0, &pasttheend);
   if (!current) return FAIL;

   while (!pasttheend)
   {
      if (!GetNthCluster(handle, current, &label))
         return FAIL;

       /* Reserve a new cluster */
       switch (ExtendFileChain(handle, target, &newcluster))
       {
         case FAIL:
              return FAIL;
         case FALSE:
              printf("Insufficient disk space to undouble cross linked clusters\n");
              return FALSE;
       }

       /* Copy the contents of the source cluster */
       srcsect = ConvertToDataSector(handle, current);
       dstsect = ConvertToDataSector(handle, newcluster);

       if (!CopySectors(handle, srcsect, dstsect, sectorspercluster))
          return FAIL;

      /* Get the next cluster in the original file chain */
      current = GetNthFileCluster(handle, current, 0, &pasttheend);
      if (!current) return FAIL;
   }

   return TRUE;
}

