/*
   LostClts.c - lost cluster checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <string.h>
#include <stdio.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"

static BOOL SalvageClusters(RDWRHandle handle, char* bitfield,
                            unsigned long clustersindataarea);

static BOOL FillNewEntry(RDWRHandle handle,
                         struct DirectoryEntry* entry,
                         CLUSTER firstcluster);

static BOOL CheckFileChain(RDWRHandle handle, CLUSTER firstcluster,
                           unsigned long* filesize);

static BOOL MakeUpFileName(RDWRHandle handle,
                           char* filename, char* extension);
                           
static BOOL MarkClusters(RDWRHandle handle, char* fatmap);

static BOOL UsedMarker(RDWRHandle handle, CLUSTER label,
                       SECTOR datasector, void** structure);

static BOOL FirstClusterMarker(RDWRHandle handle,
                               struct DirectoryPosition* pos,
                               struct DirectoryEntry* entry,
                               void** structure);

static BOOL IsStartOfChain(RDWRHandle handle, CLUSTER cluster);

static BOOL ReferedMarker(RDWRHandle handle, CLUSTER label,
                          SECTOR datasector, void** structure);

/* Checking */

RETVAL FindLostClusters(RDWRHandle handle)
{
   char* fatmap;
   unsigned long clustersindataarea, i, clustercount = 0, chains=0;
   
   clustersindataarea = GetClustersInDataArea(handle);
   if (!clustersindataarea) return ERROR;

   fatmap = CreateBitField(clustersindataarea);
   if (!fatmap) return ERROR;
   
   if (!MarkClusters(handle, fatmap))
   {
      DestroyBitfield(fatmap);
      return ERROR;
   }

   for (i = 0; i < clustersindataarea; i++)
   {
       if (GetBitfieldBit(fatmap, i))
       {
          clustercount++;
          switch (IsStartOfChain(handle, i+2))
          {
             case TRUE:
                  chains++;
                  break;
             case FAIL:
                  return ERROR;
          }
       }
   }

   if (clustercount)
      printf("Found %lu lost clusters in %lu chains\n", clustercount);

   DestroyBitfield(fatmap);
   return (clustercount) ?  FAILED: SUCCESS;
}

/* Fixing */

RETVAL ConvertLostClustersToFiles(RDWRHandle handle)
{
   char* fatmap;
   unsigned long clustersindataarea, i, clustercount = 0, chains=0;
   
   clustersindataarea = GetClustersInDataArea(handle);
   if (!clustersindataarea) return ERROR;

   fatmap = CreateBitField(clustersindataarea);
   if (!fatmap) return ERROR;
   
   if (!MarkClusters(handle, fatmap))
   {
      DestroyBitfield(fatmap);
      return ERROR;
   }

   for (i = 0; i < clustersindataarea; i++)
   {
       if (GetBitfieldBit(fatmap, i))
       {
          clustercount++;
          switch (IsStartOfChain(handle, i+2))
          {
             case TRUE:
                  chains++;
                  break;
             case FAIL:
                  return ERROR;
          }
       }
   }

   if (clustercount)
   {
      printf("Found %lu lost clusters\n", clustercount);
      if (!SalvageClusters(handle, fatmap, clustersindataarea))
      {
         DestroyBitfield(fatmap);
         return ERROR;
      }
   }

   DestroyBitfield(fatmap);
   return SUCCESS;
}

static BOOL SalvageClusters(RDWRHandle handle, char* bitfield,
                            unsigned long clustersindataarea)
{
    unsigned long i;
    struct DirectoryPosition newpos;
    struct DirectoryEntry entry;

    for (i = 0; i < clustersindataarea; i++)
    {
        if (GetBitfieldBit(bitfield, i))
        {
           if (IsStartOfChain(handle, i+2))
           {
              /* We have found a lost cluster chain, convert it to a file. */
              /* Allocate a new directory entry in the root directory. */
              switch (AddDirectory(handle, 0, &newpos))
              {
                 case FALSE:
                      printf("Insufficient space in the root directory to save lost clusters\n");
                      return TRUE;
            
                 case TRUE:
                      if (!FillNewEntry(handle, &entry, i))
                         return FALSE;
                      if (!WriteDirectory(handle, &newpos, &entry))
                         return FALSE;
                      return TRUE;
       
                 case FAIL:
                      return FALSE;
              }
           }
        }
    }

    return TRUE;
}

static BOOL FillNewEntry(RDWRHandle handle,
                         struct DirectoryEntry* entry,
                         CLUSTER firstcluster)
{
    struct tm* tmp;
    time_t now;
    
    unsigned long filesize;
    char filename[8], extension[3];

    /* Check the file chain */
    if (!CheckFileChain(handle, firstcluster, &filesize))
       return FALSE;
    
    /* file name and extension */
    if (!MakeUpFileName(handle, filename, extension))
       return FALSE;

    memcpy(entry->filename, filename, 8);
    memcpy(entry->extension, extension, 3);

    /* attribute */
    entry->attribute = 0;

    /* first cluster */
    SetFirstCluster(firstcluster, entry);

    /* file size */
    entry->filesize = filesize;

    /* NT reserved field */
    entry->NTReserved = 0;

    /* Mili second stamp */
    entry->MilisecondStamp = 0;

    /* Last access date */
    memset(&entry->LastAccessDate, 0, sizeof(struct PackedDate));
    
    /* Time last modified */
    memset(&entry->timestamp, 0, sizeof(struct PackedTime));

    /* Date last modified */
    memset(&entry->datestamp, 0, sizeof(struct PackedDate));

    /* Get the current date and time and store it in the last write
       time and date. */
    time(&now);
    tmp = localtime(&now);

    entry->LastWriteTime.second = tmp->tm_sec / 2;
    if (entry->LastWriteTime.second == 30) /* DJGPP help says range is [0..60] */
       entry->LastWriteTime.second--;
    
    entry->LastWriteTime.minute = tmp->tm_min;
    entry->LastWriteTime.hours  = tmp->tm_hour;

    entry->LastWriteDate.day   = tmp->tm_mday;
    entry->LastWriteDate.month = tmp->tm_mon + 1;

    if (tmp->tm_year < 80)
       entry->LastWriteDate.year = 0;
    else
       entry->LastWriteDate.year  = (tmp->tm_year+1900)-1980;

    return TRUE;
}
    
static BOOL CheckFileChain(RDWRHandle handle, CLUSTER firstcluster,
                           unsigned long* filesize)
{
    CLUSTER current, previous1 = firstcluster, previous2;
    unsigned long numcluster = 1, clustersindataarea;
    unsigned char sectorspercluster;

    clustersindataarea = GetClustersInDataArea(handle);
    if (!clustersindataarea) return FALSE;

    if (!GetNthCluster(handle, firstcluster, &current))
       return FALSE;

    while (FAT_NORMAL(current) &&
           (current >= 2) && (current < clustersindataarea))
    {
       previous2 = previous1;
       previous1 = current;
       
       if (!GetNthCluster(handle, current, &current))
          return FALSE;

       numcluster++;
    }

    if (FAT_BAD(current))
    {
       if (!WriteFatLabel(handle, previous2, FAT_LAST_LABEL))
          return FALSE;
    }

    if (FAT_FREE(current) || (current < 2) || (current >= clustersindataarea))
    {
       if (!WriteFatLabel(handle, previous1, FAT_LAST_LABEL))
          return FALSE;
    }

    /* Return the size of the file */
    sectorspercluster = GetSectorsPerCluster(handle);
    if (!sectorspercluster) return FALSE;
    
    *filesize = numcluster * sectorspercluster * BYTESPERSECTOR;

    return TRUE;
}

static BOOL MakeUpFileName(RDWRHandle handle,
                           char* filename, char* extension)
{
    int counter=0;
    char buffer[9];
    BOOL retval;

    do {
        sprintf(buffer, "FILE%4d", counter++);
        memcpy(filename, buffer, 8);
        memcpy(extension, "CHK", 3);
        
        retval = LoFileNameExists(handle, 0, filename, extension);
        
        if (retval == FAIL)
           return FALSE;
        
    } while (!retval);

    return TRUE;
}

/* Common */

static BOOL MarkClusters(RDWRHandle handle, char* fatmap)
{
   if (!LinearTraverseFat(handle, UsedMarker, (void**) &fatmap))
      return FALSE;
      
   if (!LinearTraverseFat(handle, ReferedMarker, (void**) &fatmap))
      return FALSE;

   if (!FastWalkDirectoryTree(handle, FirstClusterMarker, (void**) &fatmap))
      return FALSE;
      
   return TRUE;
}

static BOOL UsedMarker(RDWRHandle handle, CLUSTER label,
                       SECTOR datasector, void** structure)
{
   char* fatmap = *((char**) structure);
   CLUSTER cluster;

   if (FAT_NORMAL(label) || FAT_LAST(label))
   {
      cluster = DataSectorToCluster(handle, datasector);
      if (!cluster) return FALSE;
   
      SetBitfieldBit(fatmap, cluster-2);
   }
   
   return TRUE;
}

static BOOL ReferedMarker(RDWRHandle handle, CLUSTER label,
                          SECTOR datasector, void** structure)
{
   char* fatmap = *((char**) structure);
   CLUSTER cluster;

   if (FAT_NORMAL(label))
   {
      cluster = DataSectorToCluster(handle, datasector);
      if (!cluster) return FALSE;
   
      ClearBitfieldBit(fatmap, label-2);
   }
   
   return TRUE;
}

static BOOL FirstClusterMarker(RDWRHandle handle,
                               struct DirectoryPosition* pos,
                               struct DirectoryEntry* entry,
                               void** structure)
{
   CLUSTER firstcluster;
   char* fatmap = *((char**) structure);

   pos = pos, handle = handle;

   if ((entry->attribute & FA_LABEL) ||
       (IsLFNEntry(entry))           ||
       (IsCurrentDir(*entry))        ||
       (IsPreviousDir(*entry))       ||
       (IsDeletedLabel(*entry)))
   {
      return TRUE;
   }

   firstcluster = GetFirstCluster(entry);
   if (firstcluster)
   {
      ClearBitfieldBit(fatmap, firstcluster-2);
   }

   return TRUE;
}

static BOOL IsStartOfChain(RDWRHandle handle, CLUSTER cluster)
{
    BOOL foundindir;
    CLUSTER found;
    struct DirectoryPosition dummy;
    
    /* See wether this cluster is refered to in the FAT */
    if (!FindClusterInFAT(handle, cluster, &found))
       return FAIL;

    /* If it is refered to in the FAT, go on to the next cluster. */
    if (found) return FALSE;

    /* See wether this cluster is the first cluster of a file or
       directory. */
    if (!FindClusterInDirectories(handle, cluster, &dummy, &foundindir))
       return FAIL;

    if (foundindir) return FALSE;

    return TRUE;
}

