/*
   Filessum.c - file summary gattering.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <string.h>
#include <limits.h>

#include "fte.h"
#include "..\struct\FstTrMap.h"

#include "filessum.h"

static BOOL DirSizeGetter(RDWRHandle handle, struct DirectoryPosition* pos,
                          void** structure);

static BOOL FilesSummaryGetter(RDWRHandle handle,
			       struct DirectoryPosition* position,
			       struct DirectoryEntry* entry,
                               void** structure);

BOOL GetFilesSummary(RDWRHandle handle, struct FilesSummary* info)
{
    memset(info, 0, sizeof(struct FilesSummary));
    return FastWalkDirectoryTree(handle, FilesSummaryGetter, (void**)&info);
}

static BOOL FilesSummaryGetter(RDWRHandle handle,
			       struct DirectoryPosition* position,
			       struct DirectoryEntry* entry,
                               void** structure)
{
    CLUSTER firstcluster;
    struct FilesSummary** info = (struct FilesSummary**) structure;
    unsigned long dirsize=0, *pdirsize = &dirsize;
    unsigned long rest;

    position = position;

    if (entry->attribute & FA_LABEL)
       return TRUE;

    if (IsLFNEntry(entry))
       return TRUE;
    if (IsDeletedLabel(*entry))
       return TRUE;

    if (entry->attribute & FA_DIREC)
    {
       if (IsCurrentDir(*entry))  return TRUE;
       if (IsPreviousDir(*entry)) return TRUE;

       (*info)->DirectoryCount++;

       firstcluster = GetFirstCluster(entry);
       if (!TraverseSubdir(handle, firstcluster, DirSizeGetter,
                           (void**) &pdirsize, TRUE))
       {
          return FAIL;
       }

       /* Add the found length to the total size of directories. */
       /* Notice that FAT32 supports 2Tb of data and this does not fit
          into 1 unsigned long. */
       rest = ULONG_MAX - (*info)->SizeOfDirectories[0];
       if (rest >= dirsize)
          (*info)->SizeOfDirectories[0] += dirsize;
       else
       {
          (*info)->SizeOfDirectories[0]  = dirsize - rest - 1;
          (*info)->SizeOfDirectories[1] += 1;
       }

       return TRUE;
    }

    (*info)->TotalFileCount++;

    rest = ULONG_MAX - (*info)->TotalSizeofFiles[0];
    if (rest >= entry->filesize)
       (*info)->TotalSizeofFiles[0] += entry->filesize;
    else
    {
       (*info)->TotalSizeofFiles[0]  = dirsize - rest - 1;
       (*info)->TotalSizeofFiles[1] += 1;
    }

    if (entry->attribute & FA_SYSTEM)
    {
       (*info)->SystemFileCount++;

       rest = ULONG_MAX - (*info)->SizeOfSystemFiles[0];
       if (rest >= entry->filesize)
	  (*info)->SizeOfSystemFiles[0] += entry->filesize;
       else
       {
          (*info)->SizeOfSystemFiles[0]  = dirsize - rest - 1;
          (*info)->SizeOfSystemFiles[1] += 1;
       }
    }

    if (entry->attribute & FA_HIDDEN)
    {
       (*info)->HiddenFileCount++;

       rest = ULONG_MAX - (*info)->SizeOfSystemFiles[0];
       if (rest >= entry->filesize)
	  (*info)->SizeOfHiddenFiles[0] += entry->filesize;
       else
       {
          (*info)->SizeOfHiddenFiles[0]  = dirsize - rest - 1;
          (*info)->SizeOfHiddenFiles[1] += 1;
       }
    }

    return TRUE;
}

static BOOL DirSizeGetter(RDWRHandle handle, struct DirectoryPosition* pos,
                          void** structure)
{
   unsigned long **dirsize = (unsigned long**) structure;

   pos = pos, handle = handle;
    
   **dirsize += sizeof(struct DirectoryEntry);

   return TRUE;
}

