/***
*init.c - keyword and parameter initialize/set/display
*
*this file is part of DISKED
*Copyright (c) 1991-1998, Gregg Jennings.  All rights reserved.
*   P O Box 200, Falmouth, MA 02541-0200
*
*Purpose:
*   Configuration handling.
*
*Notice:
*   This program can be distributed only in accordance with, and
*   accompanied by, the DPU Software License. See COPYING.TXT or,
*   <http://www.diskwarez.com/dpu.htm>.
*******************************************************************************/

/*
   Versions:

   1.5   06-Feb-1998    added (kludged in) TEXT_ARGs in KEYWORDS
   1.4   30-Nov-1997    enhanced KEYWORDS functionality and support for
                        NUM_ARG; displaykeys() much better
   1.3   07-Mar-1997    BIOSX i/o value; displaykeys()
   1.2   21-Feb-1997    changes to the 'kw' structure - see NOTES below;
                        Radix check for 10 or 16 only in initialize();
                        better error handling (display) in getinit()
   1.1   13-Sep-1996    slightly modified the KEYWORDS array
   1.1   04-Sep-1994    save/restore now has a structure
   1.0   18-Apr-1994

   Release Notes:

   Just a way to place module wide globals into one struct.

   Programming Notes:

   TEXT_ARG KEYWORD arguments are a last minute kludge. The 'value'
   KEYWORD member should be a void pointer. Since it is not I opted
   to use the recently added 'chk' member as the pointer for text.

   I did this so as to keep the one text argument in the 'kw' struct
   and not just as another global that would have to be handled in
   a more kludgy way during the .INI file parsing.

   This is a prime example of not designing algorithms or planning
   ahead -- you can program yourself into a corner.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <assert.h>

#include "disked.h"
#include "keys.h"
#include "init.h"
#include "console.h"

#define MAXARGLEN 100

/* global data defined here */

   /* static stuff for text arguments */
   static char temp_dir[MAXARGLEN];

KEYWORDS kw[] = {
 /*  key, value, keyword, type, chk */
   { ALTD, 1, "display", BOOL_ARG },
   { ALTG, 0, "get", BOOL_ARG},
   { ALTH, 1, "home", BOOL_ARG },
   { ALTV, 1, "verify", BOOL_ARG },
   { ALTW, 0, "write", BOOL_ARG },
   { ALTF, 0, "files", BOOL_ARG },
   { ALTZ, 0, "Debug", BOOL_ARG },
   { ALTA, 0, "ascii", BOOL_ARG },
   { ALTP, 1, "partial", BOOL_ARG },
   { ALTL, 0, "logical", BOOL_ARG },
   { ALTT, 0, "translate", BOOL_ARG },
   { ALTE, 0, "filter", NUM_ARG, "0-2" },
   { ALTR,10, "radix", NUM_ARG, "10,16" },
   {    0, 1, "signon", BOOL_ARG },
   {    0, 2, "term",   NUM_ARG, "0-3" },
   { ALTI, 0, "ignorecase", BOOL_ARG},
#ifdef HEAP_CHECKING
   { ALTK, 0, "heapcheck", BOOL_ARG},
#endif
   { ALTM, 0, "bufmask", NUM_ARG, "0-3"},
   {    0, 0, "int13", BOOL_ARG},
   {    0, 0, "ioctl", BOOL_ARG},
   {    0, 0, "nopartitions", BOOL_ARG},
   {    0, 2, "dskretries", NUM_ARG, "0-10"},
   {    0, 1, "dskretrychange", BOOL_ARG},
   {    0, 0, "tempdir", TEXT_ARG, temp_dir},
   {    0, 0, NULL, 0}
};

#define SIZEOFKW ((sizeof(kw)/sizeof(KEYWORDS))-1)

/* KEYWORD NOTES:  They grew a little after I first designed the
   structure.  They were first only for boolean values and keyboard
   input, now they are for all global operational parameters and are
   also set by the INI file.  I may enhance this structure someday.

   Only BOOL and NUM ARGTYPEs are used, the others are for future
   enhancements. TEXT_ARG type also should not have a non-zero 'key'.

*/

/* internal functions */

static int lookup(char *t, KEYWORDS *kw, int num);
static int boolarg(char *);
static int numarg(char *);
static int rangechk(const char *str, int num);
static int rangetst(const char *str, int num);
static int rangechg(const char *str, int num);
static char kname(KEY key);

/* BOOL_ARG, NUM_ARG, TEXT_ARG, LONG_ARG, KEY_ARG, BITS_ARG */
static int (*setup_func[])(char *) = {boolarg,numarg};


/***
*initialize -  read DISKED.INI and set the world
*
****/

extern int initialize(char *filename)
{
int i;
char string[MAXARGLEN], *seps = " \t\n,:=";
char *p,*arg;
FILE *file;

   strcpy(temp_dir,"%TEMP%");

   assert(NKEYWORDS != SIZEOFKW-1);

   if (_stricmp(filename,"none")==0 || filename[0]=='\0')
      return 0;

   if ((file = fopen(filename,"r"))==NULL)
      return -1;

   while (!feof(file))
   {
      if (fgets(string,MAXARGLEN,file) == NULL)
         break;

      if ((p = strtok(string,seps)) == NULL)
         continue;

      if (strchr(";/[",*p))
         continue;

      if (strcmp("Expert",p) == 0)
      {
         write_to = Write = Files = Translate = 1;
         Display = Verify = Home = Filter = Bufmask = Signon = 0;
         break;
      }
      if ((arg = strtok(NULL,seps)) != NULL)
      {
         if ((i = lookup(p,kw,SIZEOFKW)) != -1)
         {
            int v;

            if (kw[i].type == NUM_ARG || kw[i].type == BOOL_ARG)
            {
               v = setup_func[kw[i].type](arg);
               if (kw[i].type == NUM_ARG && kw[i].chk)
                  kw[i].value = rangechk(kw[i].chk,v);
               else
                  kw[i].value = v;
            }
            else if (kw[i].type == TEXT_ARG)
               strcpy(kw[i].chk,arg);
         }
      }
   }
   fclose(file);

   /* Now deal with all of the special cases */

   if (Write)
   {
      write_to = 1;
      Write = 0;        /* and make user turn it on */
   }

   return 1;
}

/***
*setkeyword - set boolean and (ranged) number keyword by key
*
****/

extern int setkeyword(unsigned int c, KEYWORDS *kw)
{
   for (; kw->keyword; kw++)
   {
      if (c == kw->key)
      {
         if (kw->type == BOOL_ARG)
         {
            if (kw->key == ALTW && !write_to)      /* special double check */
               return 1;                           /* for ALTW */
            kw->value = !kw->value;
            print("%s:%s ",kw->keyword,(kw->value) ? "ON" : "OFF");
         }
         else if (kw->type == NUM_ARG && kw->chk)
         {
            kw->value = rangechg(kw->chk,kw->value);
            print("%s:%d ",kw->keyword,kw->value);
         }
         return 1;
      }
   }
   return 0;
}

/***
*dkey - display all keywords by value
*
*  1.0   11-Nov-1997 added width checking
*
****/

extern void dkey(KEYWORDS *pkw)
{
int i,j;

   print("\nON: ");

   for (j = i = 0; i < SIZEOFKW; i++)
   {
      if (pkw[i].key && pkw[i].value && !pkw[i].chk)
      {
         if (j + strlen(pkw[i].keyword) > 79-5)
            j = print("\n    ") - 1;
         j += print(" %s",pkw[i].keyword);
      }
   }
   print("\nOFF:");

   for (j = i = 0; pkw[i].keyword; i++)
   {
      if (pkw[i].key && !pkw[i].value && !pkw[i].chk)
      {
         if (j + strlen(pkw[i].keyword) > 79-5)
            j = print("\n    ") - 1;
         j += print(" %s",pkw[i].keyword);
      }
   }
}

/***
*displaykeys   -  display all in a nice format
*
****/

extern void displaykeys(KEYWORDS *kw)
{
int i;
KEYWORDS *tkw = kw;

   for (i = 0; kw->keyword; kw++)
   {
      if (kw->type == BOOL_ARG && kw->key)
      {
         print("%-12.12s %s ",kw->keyword,kw->value?"ON ":"OFF");
         print("(Alt-%c)\t",kname(kw->key));
         if (i&1)
            output('\n');
         ++i;
      }
   }
   output('\n');
   if (i&1)
      output('\n');
   for (i = 0, kw = tkw; kw->keyword; kw++)
   {
      if (kw->type == BOOL_ARG && !kw->key)
      {
         if (strcmp(kw->keyword,"signon") == 0)
            continue;

         print("%-12.12s %s\t\t",kw->keyword,kw->value?"YES":"NO ");
         if (i&1)
            output('\n');
         ++i;
      }
   }
   output('\n');
   if (i&1)
      output('\n');
   for (i = 0, kw = tkw; kw->keyword; kw++)
   {
      if (kw->type == NUM_ARG)
      {
         print("%-12.12s %d\t\t\t",kw->keyword,kw->value);
         if (i&1)
            output('\n');
         ++i;
      }
   }
   output('\n');
   if (i&1)
      output('\n');
   for (i = 0, kw = tkw; kw->keyword; kw++)
   {
      if (kw->type == TEXT_ARG)
      {
         print("%-12.12s %s\t\t\t",kw->keyword,kw->chk);
         if (i&1)
            output('\n');
         ++i;
      }
   }
}

/***
*ckey - change boolean keyword(s) by first letter / string
*
****/

extern void ckey(char *s, KEYWORDS *kw)
{
int i;

   do
   {
      for (i = 0; kw[i].keyword; i++)     /* loop through all keywords */
      {
          if (kw[i].type == BOOL_ARG && kw[i].keyword[0] == *s)
          {
             kw[i].value++;
             if (kw[i].value > 1)
                kw[i].value = 0;
             break;
         }
      }
   } while (*++s);                        /* for every letter in string */
}

/* Internal Functions */

/* .ini file lookup */

static int lookup(char *t, KEYWORDS *kw, int num)
{
int i;

   for (i = 0; i < num; i++,kw++)
   {
      if (kw->keyword && strcmp(t,kw->keyword) == 0)
         return i;
   }
   return -1;
}

static int boolarg(char *arg)
{
   return (arg && (_stricmp("on",arg)==0 || _stricmp("yes",arg)==0));
}

static int numarg(char *arg)
{
   if (arg)
      return (int)strtol(arg,NULL,0);
   return 0;
}

#if 0    /* these are not used yet */

static unsigned long longarg(char *arg)
{
unsigned long l=0L;

   if (arg)
      sscanf(arg,"%lu",&l);
   return l;
}

/***
*  assigns a key value: "^a" - "^z"  returns to 01 - 1Ah
*                       ""           returns key
*                       "off"        returns -1
****/

static int keyarg(char *arg,int key)
{
register int i;

   if (arg==NULL)
      i=key;
   else if (_stricmp("off",arg)==0)
      i=-1;
   else if ((_stricmp("^`",arg)+_stricmp("^{",arg))==0)
      i=*(++arg)-'`';
   else
      i=*arg;
   return(i);
}

#endif   /* 0 */

/***
*rangechk   -  range check
*
****/

static int rangechk(const char *str, int num)
{
char c;
int a,b;

   if (sscanf(str,"%d%c%d",&a,&c,&b) < 3)
      return num;

   if (c == ',')                    /* two values */
   {
      if (num != a && num != b)
         return a;
   }
   if (c == '-')                    /* min/max range */
   {
      if (num < a || num > b)
         return a;
   }
   return num;
}

/***
*rangechg   -  range change
*
****/

static int rangechg(const char *str, int num)
{
char c;
int a,b;

   if (sscanf(str,"%d%c%d",&a,&c,&b) < 3)
      return 0;

   if (c == ',')                    /* two values */
   {
      if (num == a)
         return b;
      return a;
   }
   else if (c == '-')               /* min/max range */
   {
      if (++num > b)
         return a;
      return num;
   }
   return 0;
}


static struct KN {
   KEY key;
   char letter;
} kn[] = {
   ALTA,'A',ALTB,'B',ALTC,'C',ALTD,'D',ALTE,'E',ALTF,'F',ALTG,'G',
   ALTH,'H',ALTI,'I',ALTJ,'J',ALTK,'K',ALTL,'L',ALTM,'M',ALTN,'N',
   ALTO,'O',ALTP,'P',ALTQ,'Q',ALTR,'R',ALTS,'S',ALTT,'T',ALTU,'U',
   ALTV,'V',ALTW,'W',ALTX,'X',ALTY,'Y',ALTZ,'Z'
};

/***
*kname   -  match ASCII charcater to Alt-Key
*
****/

static char kname(KEY key)
{
int i;

   for (i = 0; i < sizeof(kn)/sizeof(struct KN); i++)
      if (kn[i].key == key)
         break;
   return (char)(i < sizeof(kn)/sizeof(struct KN) ? kn[i].letter : 0);
}
