/***
*disklib.c - library functions
*
*this file is part of DISKED
*Copyright (c) 1991-1998, Gregg Jennings.  All rights reserved.
*   P O Box 200, Falmouth, MA 02541-0200
*
*Purpose:
*   Main library functions.
*
*Notice:
*   This program can be distributed only in accordance with, and
*   accompanied by, the DPU Software License. See COPYING.TXT or,
*   <http://www.diskwarez.com/dpu.htm>.
*******************************************************************************/

/*
   Versions:

   3.3   03-Jul-1998    fixed yet another bug in range()
   3.2   02-Jul-1998    _winmajor, _winminor
   3.1   21-Apr-1998    added "Current Drive:" to dparams()
   3.0   04-Jan-1998    displaymask() stuff; mid. stuff; better OS version
                        detect; set_physical() detects 'h' for start sector
   2.9   21-Feb-1997    got rid of the #if 1 in prompt()
   2.8   12-Sep-1996    "%r"
   2.7   18-Mar-1995    moved find() into SEARCH.C
   2.6   15-Mar-1995    file tracking stuff
   2.7   05-Mar-1995    rewrote dparams() to show everything
   2.6   10-Sep-1994    fixed %u on num_sectors in dparams()
                         complier checks in dparams()
   2.5   02-Sep-1994    consolodated print vs printf
   2.4   18-Apr-1994    removed text to data.c
   2.3   13-Mar-1994    find() changes
   2.2   01-Feb-1994    'dir_cluster' test bug fix
   2.1   04-Jan-1994    added Translate if !Files for startup
                        added 'dir_cluster'
                        help()

   Release Notes:

   Everything that defied classification was put in here. Mostly
   there is input and output stuff.

   Programming Notes:

   More old and ugly functions. The string input stuff is kind of
   flexible though.

*/

#include <stdio.h>
#include <ctype.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include <time.h>
#include <dos.h>

#include "disked.h"
#include "diskio.h"
#include "init.h"
#include "mylib.h"
#include "files.h"
#include "keys.h"
#include "error.h"
#include "dosio.h"
#include "dirent.h"
#include "console.h"
#include "int2f.h"            /* winver() */

/* static functions */

static void filename(unsigned int);


/* three argument command 's' */

struct Msg st[] = {
   {" track ",3},
   {" head ",1},
   {" sector ",2}
};

/***
*set_physical  -  go to sector defined by physical parameters
*
****/

extern void set_physical(void)
{
int c;
unsigned int temps,tempt,temph;

   tempt = phy_track;
   temps = phy_sector;
   temph = phy_head;

   st[1].len = (max_head > 9) ? 2 : 1;

   if ((c = set(st,(int *)&tempt,(int *)&temph,(int *)&temps)) != 0)
   {
      if (c == 'f')
         fatsector();
      else if (c == 'b')
         bootsector();
      else if (c == 'r')
         rootsector();
      else if (c == 'h')
      {
         if (set_sector != -1L)
            setsector(set_sector);
         else
            rootsector();
      }
      else
      {
         if (tempt > max_track)
            tempt = phy_track;
         if (temps > max_sector || temps==0)
            temps = phy_sector;
         if (temph >= max_head)
            temph = phy_head;
         if (tempt != phy_track || temps != phy_sector || temph != phy_head)
            setphysical(tempt,temps,temph);
      }
   }
}

/***
*set  -   get track, sector, head
*
*   Passed pointer to string, number of values, and struct of type
*   Msg which holds the messages to display for each value and
*   the number of digits for each value.
*
*   Modification of get() to return immediately on an 'f' or 'b'.
*
*   Returns a string containing the values seperated by commas
*   to be converted by sscanf().
*
*   Globals: Display, Radix.
*
****/

extern int set(struct Msg *str, int *t, int *s, int *h)
{
register int c,i,n;
long l;
char ts[7];

   for (i = n = 0; n < 3; n++, i = 0, str++)
   {
      if (Display)
         print(str->msg);
      while ((c = input()) != '\r' && c != ' ' && c != ',')
      {
         if (n == 0 && i == 0 && (c == 'f' || c == 'b' || c == 'r' || c == 'h'))
         {
            output(c);                 /* special: Fat, Boot, Root */
            return(c);
         }
         if (c == '\b' && i != 0)      /* backspace, delete number */
         {
            output(c);                 /* in buffer */
            output(' ');               /* and on screen */
            output(c);
            --i;                       /* move pointer back */
         }
         else if (c == 0x1b || c == 3) /* ESC, ^C to abort */
            return 0;                  /* RETURN */

         else if (Radix == 16 && !isxdigit(c))
            continue;
         else if (Radix==10 && !isdigit(c))
            continue;
         else
         {
            output(c);
            ts[i++] = (char)c;         /* save and echo numbers */
            if (i == str->len)
               break;
         }
      }
      ts[i] = '\0';
      if (i)
      {
         l = strtol(ts,NULL,Radix);    /* strtol() for radix */
         if (n == 0)
            *t = (int)l;
         else if (n == 1)
            *s = (int)l;
         else
            *h = (int)l;
      }
      if (c == '\r')
         break;
      if (!Display && n < 2)
         output(',');
   }
   return 1;
}

/***
*get  -  get a string for conversion to numbers
*
*   Passed pointer to string, number of values, and struct of type
*   Msg which holds the messages to display for each value and
*   the number of digits for each value.
*
*   Returns a string containing the values seperated by commas
*   to be converted by sscanf().
*
*   Globals: Display, Radix.
****/

extern int get(struct Msg *str,int *one, int *two)
{
register int t,v;
char temp[7];

   for (t = 0; t < 2; t++, str++)
   {
      if (Display)
         print(str->msg);
      if (getstr(temp,str->len,(Radix == 16) ? _HEX : _DIGIT) > 0)
      {
         v = atoi(temp);
         if (t == 0)
            *one = v;
         else
            *two = v;
      }
      else
         return 0;                     /* RETURN */

      if (!Display && t < 1)
         output(',');
   }
   return 1;
}

/***
*getl -  long version for 'P' command; expects *one to equal to
*        log_sector
****/

extern int getl(struct Msg *str, long *one, long *two)
{
register int t;
int i;
char temp[7];
long l;

   for (t = 0; t < 2; t++, str++)
   {
      if (Display) print(str->msg);
      str->len = getlength(num_sectors,Radix);
      str->spec = _DIGIT;
      if (Radix == 16) str->spec |= _HEX;
      if (t == 1) str->spec |= _SIGN;

      if ((i = getstr(temp,str->len,str->spec)) > 0)
      {
         if (strcmp(temp,"-") == 0)    /* we allow '-' for second number */
            l = -1;
         else
            l = strtol(temp,NULL,Radix);
         if (t == 0)
            *one = l;
         else
            *two = l;
      }
      else if (!t && i == ERROR)       /* if first entry is just a */
         print("%lr",*one);            /* '\r' (ERROR) use default */
      else
         return 0;
      if (!Display && t < 1)
         output(',');
   }
   return(1);
}

/***
*append  -  Append bytes into the databuffer.  The bytes are from the
*           sectorbuffer or a file.
*
*           returns OK or ERROR if full
****/

extern int append(int mask, unsigned char *buffer, unsigned int nbytes)
{
int ch;
unsigned int i;
char b[5];

   for (i = 0; i < nbytes; i++)
   {
      ch = buffer[i];
      if (mask && (!isprint(ch) && !isspace(ch)))
      {
         if (mask == B_STRIP)
            continue;
         else if (mask == B_MASK)
            data_buf[byte_cnt++] = (unsigned char)(ch & 0x7F);
         else if (mask == B_CONV && byte_cnt < max_bytes - 4)
         {
            sprintf(b,"<%02x>",ch);
            memcpy(data_buf+byte_cnt,b,4);
            byte_cnt += 4;
            continue;
         }
      }
      else
         data_buf[byte_cnt++] = (unsigned char)ch;

      if (byte_cnt == max_bytes)
         return ERROR;
   }
   return 1;
}

/***
*header  -  display sector header
*
****/

extern void header(void)
{
register unsigned int i,j;

   print("\nSector %lR",log_sector);

   i = sectortocluster(log_sector);
   j = clustersector(log_sector)+1;

   if (log_sector > (UINT32)data_sector)
   {
      print(", Cluster %R:%R",i,j);
   }
   else
   {  unsigned sector = (UINT16)log_sector;
      if (sector < reserved_secs)
      {
         print("  BOOT");
         if (reserved_secs > 1)
            print(" %r:%r",sector+1,reserved_secs);
      }
      else if (sector >= reserved_secs && sector <= secs_fat)
         print("  FAT 1 %r:%r %r-BIT",
            sector-reserved_secs+1,secs_fat,fat_size);
      else if (num_fats > 1 && sector > secs_fat
         && sector <= (secs_fat*num_fats))
            print("  FAT 2 %r:%r %d-BIT",
               (sector-reserved_secs-secs_fat)+1,secs_fat,fat_size);
      else if (sector >= dir_sector && sector <= data_sector)
         print("  ROOT %r:%r",(sector-dir_sector)+1,dir_sectors);
   }
   if (!Logical)
      print(" (t%R,h%R,s%R)",phy_track,phy_head,phy_sector);

   if (diskio_error)
      print(" <BAD>");
   if (Files > 0 && n_files)
      filename(i);
}

/***
*filename   -  get name of file that owns a cluster
*
****/

static void filename(unsigned int i)
{
   if (log_sector > (UINT32)data_sector && i < num_clusters)
   {
      if (clusters[i] != 0 && (unsigned)clusters[i] <= n_files)
      {
         output(' ');
         print(gfile(clusters[i]));
         if (files[clusters[i]].dir)
            print(" <DIR>");
      }
   }
}

/***
*dparams -  display disk drive info
*
*   Serial No:               Drive Size:              Root Sector:
*   Buffer Size:             Sectors:                 Data Sector:
*   DOS Version:             Sector Size:             Reserved Sectors:
*   Number of Drives:        Clusters:                Hidden Sectors:
*   Current Drive:           Cluster Size:            Tracks:
*   Media:                   Secs/Cluster:            Heads:
*   Volume:                  Number FATs:             Sectors:
*   Serial ID:               Secs/FAT:                Sector Size:
*   Current Dir:
*
****/

extern void dparams(char *drivedir)
{
   print("\n");

   print("\n Serial No:%s",Version);
   print("\t  Drive Size: %lR",drive_size);
   if (drive_size < 1024L*1024L*1000L) print("\t");
   print("  Root Sector: %R",dir_sector);

   print("\n Buffer Size: %R",max_bytes);
   print("\t  Sectors: 0-%lR",num_sectors-1);
   print("\t  Data Sector: %R",data_sector);

   if (_winmajor != 0)
   {
      print("\n WIN Version: %d.%d",_winmajor,_winminor);
   }
   else
   {
      print("\n DOS Version: %d.%d",_osmajor,_osminor);
      if (_osminor == 0x32)
         print(" (NT)");
   }
   print("\t  Sector Size: %R",sec_size);
   print("\t  Reserved Sectors: %R",reserved_secs);

   print("\n Number of Drives: %R",max_drive);
   print("\t  Clusters: 2-%R",num_clusters);
   print("\t  Hidden Sectors: %lR",hidden_secs);

   print("\n Media: %-8.8s",dsk_format);
   print("\t  Cluster Size: %R",cluster_size);
   print("\t  Tracks: 0-%R",max_track);

   print("\n Volume: %-11.11s",mid.vollabel);
   print("\t  Sectors/Cluster: %R",secs_cluster);
   print("\t  Heads: 0-%R",max_head-1);

   print("\n Serial ID: %04X-%04X",(int)(mid.serialnum>>16),(int)(short)mid.serialnum);
   print("\t  Number FATs: %R",num_fats);
   print("\t  Sectors: 1-%R",max_sector);

   print("\n File System: %-8.8s",mid.filesys);
   print("\t  Sectors/FAT: %R",secs_fat);
   print("\t  Sector Size: %R",sec_size);

   print("\n Current Drive: %c:",cur_disk+'@');
   print("\t  Working Dir: %s",drivedir);


}

/* stolen from SNIPPETS - getspeed() */
unsigned long getclocks(void)
{
clock_t tickbase,newtick;
unsigned long i =0;        /* zero the counter */

   tickbase = clock();           /* get a tick */
   for (;;)
   {
      newtick = clock();
      if(newtick != tickbase)    /* tick just changed */
      {
         tickbase = newtick;     /* make new tick the base */
         break;
      }
   }
   for (;;)     /* start counting from fresh tick */
   {
      newtick = clock();
      if(newtick != tickbase)    /* tick just changed again */
         break;                  /* now counter has run for exactly 1 tick */
      i++;                       /* increment counter */
   }
   return(i);
}

/***
*range   -  continuous read and display sectors
*
*   Read the next sector, display, continue.  Checks console for
*   speed control and abort.
*   pass: direction, forward or backward. value, number of sectors
*
****/

extern void range(int val)
{
#ifdef __GNUC__
#define CLK  10
#else
#define CLK  55
#endif
clock_t clk;
unsigned int i;
int k,io;
static int c='4';
unsigned int speed;
unsigned long clks = getclocks();

   k = c;

   speed = (c-'0') * clks;

   for (;;)
   {
      if ((io = movesector(val)) != DISK_OK)
      {
         output('\n');
         printerror(Debug);
         if (io == DOS_ENREADY || io == BIOS_ETIMEOUT || io == BIOS_ECHANGE)
            break;
      }

      dumpsector();

      clk = clock()+CLK;
      for (i = 0; i <= speed; i++)     /* loop in wait */
      {
         while (clock() < clk)
            ;
         if (_kbhit())
            k = input();
      }

      if (isdigit(k))                  /* reset speed */
      {
         c = k;
         speed = (c-'0') * clks;
      }
      else if (k == ' ')
      {
         input();
         k = c;                        /* reset */
      }
      else
         break;
   }
}

/***
*dumpsector -  display sector buffer (front-end)
*
****/

extern void dumpsector(void)
{
int i;
UINT16 sector;

   i = sectortocluster(log_sector);
   sector = (UINT16)log_sector;

   if (Partial)
      output('\n');
   header();
   if (Translate)
   if (  (Files && clusters[i] != 0 && files[clusters[i]].dir)
      || (sector >= dir_sector && sector <= data_sector)
      || (dir_cluster && sectortocluster(log_sector) == dir_cluster)
      )
   {
      dumpdir(sec_buf,sec_size);
      return;
   }
   if (!Ascii)
   {
      if (big_screen)
         dump(sec_buf,0,sec_size,sec_size,Radix);
      else if (Partial)
         dump(sec_buf,0,sec_size,128,Radix);
      else
         dumpf(sec_buf,sec_size,Radix);
   }
   else
      view(sec_buf,0,sec_size,NOPAUSE,Radix);
   if (Partial || Ascii)
      output('\n');
}

/***
*view -  view data buffer
*
*  display databuffer, displaying all non-printable
*  characters enclosed in <>
*
*  Another complicated, horrible looking function.
*
****/

extern int view(
unsigned char *buffer,              /* array of data */
unsigned int bfptr,                 /* index into array */
unsigned int bufend,                /* size of array */
int pause,
int base)
{
register int c;
int length;
int p,n,nn,np;

   p = n = nn = np = 0;
   output('\n');
   output('\n');
   if (pause)
   {
      length = 21;
      if (ft_track)
         print("%lr:\n",(long)bfptr + filebytes());
      else
         print("%R:\n",bfptr);
   }
   else
   {
      length=sec_size;
   }

   do
   {
      c = (int)buffer[bfptr++];
      if (bfptr > bufend)
      {
         bfptr = 0;
         break;
      }
      if (p > 75)
      {
         p = 0;
         ++n;
         ++nn;
         ++np;
      }
      if (!isprint(c) && !isspace(c))
      {
         if (Bufmask == B_STRIP)
            continue;
         if (Bufmask == B_MASK)
            c&=0x7f;
         else if (Bufmask == B_CONV)
         {
            print("<%02x>",c);
            p+=4;
            continue;
         }
      }
      if (c==7 || c==255 || c==0 || c==8)
         continue;
      if (c=='\n')
      {
         n++;
         ++nn;
         p=0;
      }
      if (c!='\r')
      {
         output(c);
         p++;
      }
   } while (n<length);

   if (pause)
      print("\n%0*R:",getlen(bfptr,base),bfptr);

   output('\n');
   return(bfptr);
}

/***
*dumpboot   -  lame attempt to display BOOT parameters
*
*  Hmmm, I should probably fix this and have it check the media
*  version number...
*
****/

extern void dumpboot(BYTE *buf)
{
struct _BOOT b;

   memcpy(&b,buf,sizeof(struct _BOOT));
   output('\n');
   switch(b.jump[0])
   {
      case 0xeb:
              print("JMP SHORT %04x",b.jump[1]);
              break;
      case 0xe9:
              print("JMP %04x:%04x",b.jump[1],b.jump[2]);
              break;
      case 0xff:
      case 0xea:
              print("JMP other ");
              break;
      default:
              print("unknown");
   }
   output('\n');
   print("OEM name:\t\t\t");
   print("%.8s\n",b.name);
   print("bytes per sector:                    %R\n",b.sec_size);
   print("sectors per allocation unit:         %R\n",b.secs_cluster);
   print("reserved sectors:                    %R\n",b.reserved_secs);
   print("number of FATs:                      %R\n",b.num_fats);
   print("number of root dir entries:          %R\n",b.dir_entries);
   print("number of sectors in logical image:  %R\n",b.num_sectors);
   print("media descriptor:                    %02x\n",b.media_desc);
   print("number of FAT sectors:               %R\n",b.secs_fat);
   print("sectors per track:                   %R\n",b.secs_track);
   print("number of heads:                     %R\n",b.num_heads);
   print("number of hidden sectors:            %lR\n",b.hidden_sectors);
/*
   print("high order number of hidden sectors: %R\n",b.large_sectors);
   print("number of logical sectors:           %R\n",b.total_sectors);
*/
   print("number of huge sectors:              %lR\n",b.huge_sectors);
   print("drive number                         %02x\n",b.drive_number);
   print("signature                            %02x\n",     b.signature);
   print("volume id                            %lR\n",    b.volume_id);
   print("volume label                         %.11s\n",  b.volume_label);
   print("file system                          %.8s\n",   b.file_system);
}

/***
*help -  lame attempt at more online help
*
****/

extern void help(void)
{
int c;

   print(" Keywords Tranlations Parameters Dir ?");
   c = input();
   c = tolower(c);
   switch (c)
   {
      case 'd': disptext(dir_help); break;
      case 'k': disptext(keywords_help); break;
      case 't': disptext(translate_help); break;
      case 'p': disptext(io_help); break;
   }
}

/***
*dump -  typical hex/ascii dump
*
*  Display data in a HEX-ASCII format, starting at buf, ending at
*  bufend.  Returns the number of bytes displayed.  bufend _must_
*  be a multiple of 128
*
*  Notes:   The charout() function is used for the bytes rather
*           than the normal output() function.  charout() prints
*           ALL characters (except 0 and 255) and output() doesn't.
*
*    unsigned char *buffer;  - array of data
*    unsigned int bfptr;     - index into array
*    unsigned int bufend;    - size of array
*    unsigned int len;       - number of bytes to display
*    int base;
*
****/

extern int dump(unsigned char *buffer, unsigned int bfptr, unsigned int bufend,
         unsigned int len, int base)
{
unsigned int l,n;
register int b;
#define WARP 0

   if (bfptr > bufend)
      bfptr = 0;
   if (len > bufend)
      len = bufend;
   n = bfptr;
   for (l = 0; l < len; l+=16)
   {
      if (ft_track)
         print("\n%0*lr: ",(base==10)?8:6,bfptr+filebytes());
      else
         print("\n%0*R: ",getlen(bufend,base),bfptr);

      /* hex */

      for (b = 0; b < 16; b++)
      {
         print("%02x ",buffer[bfptr]);
         if (++bfptr == bufend)
         {
#if WRAP
            bfptr = 0;
#endif
            break;
         }
      }

      /* ascii */

      for (b = 0; b < 16; b++)
      {
          output(displaymask(buffer[n]));
          if (++n == bufend)
          {
#if WRAP
              n = 0;
              if (l < (len-16))
                 output('\n');
#endif
              break;
          }
      }
#if !WRAP
      if (n == bufend)
      {
         bfptr = 0;
         break;
      }
#endif
   }
   return bfptr;
}

/***
*dumpf   -  Display full sector contents to the screen in HEX/ASCII.
*
****/

extern void dumpf(register unsigned char *buffer, int size, int base)
{
int i,j;
int ptr;

   output('\n');
   for (ptr = i = 0; i < size; i += 24)
   {
#if 0                                        /* for this dump, file tracking */
      if (ft_track)                          /* offset display messes things */
      {                                      /* up */
         long l = filebytes();
         if (base == 10)
            print("%08ld: ",(long)i + l);
         else
            print("%06lx: ",(long)i + l);
      }
      else
#endif
         print("%0*r: ",getlen(size,base),i);

      for (j = 1; j <= 24; j++)
      {
         print("%02x",buffer[ptr]);
         if (++ptr >= size)
         {
            output(' ');
            j = 9;
            break;
         }
         if (j%8 == 0)
            output(' ');
      }
      ptr -= (j-1);
      for (j = 1; j <= 24; j++)
      {
         output(displaymask(buffer[ptr]));
         if (++ptr >= size)
            break;
      }
   }
}

/***
*prompt  -  command prompt
*
****/

extern int prompt(int n, int base)
{
KEY key;

   print("\n(%0*R)",(base==16)?4:5,n);

   if (byte_cnt == max_bytes)
      charouta(' ',15);

   if (ft_track)
      output('t');

   if (Files && n_files > 1)
      output(15);
   else
      output('*');

   _bdos(0x0c,0,0);              /* flush keyboard buffer */

   if ((key = input()) == 0)
      key = input() << 8;

#ifndef NDEBUG
   print("(%04X)",key);
#endif

   if (key < ' ')
   {
      output('^');
      output(key+'@');
   }
   else if (key < 0xff)
      output(key);

   return key;
}

/***
*displaymask   -  mask, or convert, charcaters
*
****/

extern int displaymask(int c)
{
int i = '.';

   if (Filter == D_NONE)
   {
      if ((c < 7 || c > 10) && c != 0 && c != 13 && c != 255)
         i = c;
   }
   else if (Filter == D_IBM)
   {
      if (c >= ' ' && c <= 0xfe)
         i = c;
   }
   else if (Filter == D_ASCII)
   {
      if (isprint(c))
         i = c;
   }
   return i;
}
