/*
 *  wf.c        write file to disk; assumes file created by sa.c
 *
 *  This file is part of the D_DISK library.
 *  Copyright (C) 1998, Gregg Jennings.
 *
 *  See D_DISK.HTM for the library specifications.
 *  See D_DISK.TXT for overview the implementation.
 *  See NOTES.TXT for particulars about the source files.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#ifdef _WIN32
#include <direct.h>     /* _getdrive() */
#else
#include <dos.h>        /* _dos_getdrive() */
#endif
#include <conio.h>

/*
    The sectors that make up the FAT of a typical GB drive can
    exceed the DOS/80x86 64K segment limit. Therefore, the __huge
    data C extensions must be used with 16-bit DOS compilers.
*/

#include "d_disk.h"
#include "d_lib.h"      /* clustertosector() */

#ifdef SMALL_DATA
#define lmalloc(n) halloc(n,1)
#define lfree(p)   hfree(p)
#else
#define lmalloc(n) malloc(n)
#define lfree(p)   free(p)
#endif

void usage(void);
int writefile(int dh, struct d_stat *ds, FILE *fh, long filesize);
int percent(long m, long n);


int main(int argc, char **argv)
{
FILE *fh;
int i,dh;
long fsize;
struct d_stat ds;
unsigned cdisk,sdisk,ddisk;     /* current, source, destination */


    setvbuf(stdout,NULL,_IONBF,0);      /* flush stdout every printf */

    printf("\n");

    if (argc < 3 || argv[1][0] == '-' || argv[1][0] == '/')
        usage();

    if (argv[1][1] != ':')
        usage();

#ifdef _WIN32
    cdisk = _getdrive();
#else
    _dos_getdrive(&cdisk);
#endif

    ddisk = toupper(argv[1][0])-'@';
    sdisk = (argv[2][1] == ':') ? toupper(argv[2][0])-'@' : cdisk;

#ifndef NONFLOPPY
    if (ddisk > 2) {
        printf("This version will only write to drives A: and B:\n\n");
        usage();
        /* NOT REACHED */
    }
#endif

    if (ddisk == sdisk) {
        printf("Writing the file to the same disk defeats the purpose"
               " of this program.\n\n");
        usage();
        /* NOT REACHED */
    }

    if ((fh = fopen(argv[2],"rb")) == NULL) {
        perror(argv[2]);
        return 0;
    }

    fseek(fh,0,SEEK_END);
    fsize = ftell(fh);
    rewind(fh);

    /* open drive */

    printf("opening drive...\r");

    if ((dh = d_open(argv[1],D_RDWR)) == -1) {
        d_perror("d_open");
        printf("\n");
        return 1;
    }

    /* get drive geometry */

    printf("checking drive...\r");

    if (d_hstat(dh,&ds) == -1) {
        d_perror("d_stat");
        printf("\n");
        return 1;
    }

    printf("Drive %c:, ",ddisk+'@');
    printf("%lu bytes/sector, %lu sectors/cluster.\n",
        ds.d_bsize,ds.d_usize);

    printf("File %s, %ld bytes, will be written to drive %c:.\n",
        argv[2],fsize,ddisk+'@');
    printf("\nANY PREVIOUS DATA WILL BE LOST!\n");

    printf("\nDo you want to continue? [y/n] ");

    i = getchar();
    if (i != '\n')
        while (getchar() != '\n')
            ;

    if (i == 'y') {
        printf("\n");
#ifdef NONFLOPPY
        if (ddisk == 3) {
            printf("Destination is the C: drive. Are you REALLY sure? [y/n] ");
            i = getchar();
            if (i != '\n')
                while (getchar() != '\n')
                    ;
            if (i != 'y')
                goto oops;
            printf("\n");
        }
#endif
        writefile(dh,&ds,fh,fsize);
    }

#ifdef NONFLOPPY
oops:
#endif

    fclose(fh);

    printf("\n");

    return 0;
}

/*
 * writefile - writes reserved sectors and up to `maxcluster' clusters
 *
 */

int writefile(int dh, struct d_stat *ds, FILE *fh, long filesize)
{
long t;
size_t i;
size_t bufsiz;
unsigned char *buf;

    bufsiz = (size_t)(ds->d_dsec * ds->d_bsize);

    if ((buf = malloc(bufsiz)) == NULL) {
        return 1;
    }

    t = bufsiz;
    printf("system sectors - ");
    printf(" %lu bytes so far",t);
    printf("\r");

    d_lseek(dh,0,SEEK_SET);

    if (fread(buf,1,bufsiz,fh) != bufsiz) {
        printf("\nFile Read Error!");
        return 1;
    }

    d_write(dh,buf,(size_t)ds->d_dsec);

    bufsiz = (size_t)(ds->d_usize * ds->d_bsize);

    if ((buf = realloc(buf,bufsiz)) == NULL) {
        return 1;
    }

    i = 2;
    while (fread(buf,1,bufsiz,fh) > 0) {

        if (kbhit() && getch() == '\033')
            break;

        printf("cluster %5u",i);
        printf(" - %2d%% done",percent(t,filesize));
        printf(", %lu bytes so far",t);
        printf("\r");

        d_write(dh,buf,(unsigned int)ds->d_usize);

        t += bufsiz;
        i++;
    }

    printf("cluster %5u",i-1);
    printf(" - %2d%% done",percent(t,filesize));
    printf(", %lu bytes total ",t);
    printf("\r");

    free(buf);
    return 0;
}


int percent(long m, long n)
{
float f;

    f = (float)m / n;
    f *= 100;
    return (int)f;
}

void usage(void)
{
   printf("usage: wf <d:> <file>\n\n"
          "Writes drive `d:' with `file'. `file' is assumed to have been\n"
          "ceated by the program SA.\n"
          "\n"
          "Built with the free D_DISK library: "
          "http://www.diskwarez.com/d_disk.htm\n");
   exit(1);
}
