	page	59,132
	title	UHDD -- DOS UltraDMA Disk Caching Driver.
;
; UHDD, 7-Jul-2020.   Updated from the 31-Mar-2019 UHDD --
;
;  * All 10-MB+ caches are added to help old small-memory PCs.
;
;  * For 10-MB+ caches, Read-Ahead is used for UltraDMA disks.
;
;  * Default cache size is 20-MB, if /S is omitted or invalid.
;
;  * BUG fix:  Controller-number masks are now 03Ch (not 01Ch),
;      to handle 9 controllers properly.
;
;  * BUG fix:  Init logic does not clobber disk "LBA48" flags.
; 
; UHDD, 31-Mar-2019.   Updated from the 5-Mar-2015 UHDD --
;
;  * All known errors and problems are corrected.
;
;  * 10-MB and all 20-MB+ caches are added to help old systems.
;
;  * With 20-MB+ caches, Read-Ahead is used for UltraDMA disks.
;
;  * SSDs and CompactFlash cards with an ATAPI flag are handled
;      as UltraDMA hard disks, not as slow Call-the-BIOS units.
;
; The 5-Mar-2015 UDVD2 can load AFTER this UHDD, to cache CD/DVD
; data files, and the 5-Mar-2015 CC can clear UHDD's cache, when
; required.   For a full description of UHDD, see the 5-Mar-2015
; README file.
;
; General Program Equations.
;
	.386p			;Allow 80386 commands.
s	equ	<short>		;Make a conditional jump "short".
MAXBIOS	equ	22		;Maximum BIOS disks or diskettes.
MXCACHE	equ	4093		;Maximum cache size in megabytes.
STACK	equ	448		;Caching driver stack size.
STACKSA	equ	368		;Non-cached driver stack size.
CDDVD	equ	40		;CD/DVD initial cache-unit number.
CDTYP	equ	07Ch		;CD/DVD device-type code.
CMDTO	equ	00Ah		;Disk 500-msec min. command timeout.
STARTTO	equ	07Fh		;Disk 7-second startup timeout.
HDWRFL	equ	00410h		;BIOS hardware-installed flag.
DKTSTAT	equ	00441h		;BIOS diskette status byte.
BIOSTMR equ	0046Ch		;BIOS "tick" timer address.
HDISKS	equ	00475h		;BIOS hard-disk count address.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
HDI_OFS	equ	0048Eh-BIOSTMR	;BIOS hard-disk int. flag "offset".
MCHDWFL	equ	0048Fh		;BIOS diskette media-change flags.
CR	equ	00Dh		;ASCII carriage-return & line-feed.
LF	equ	00Ah
;
; Driver Error Return Codes.
;
MCHANGE	equ	006h		;Diskette media-change.
DMAERR	equ	00Fh		;Disk DMA error.
CTLRERR equ	020h		;Disk controller not-ready.
DISKERR equ	0AAh		;Disk drive not-ready.
FAULTED	equ	0CCh		;Disk drive FAULTED.
HARDERR equ	0E0h		;Disk I-O error.
XMSERR	equ	0FFh		;XMS memory error.
;
; "Legacy IDE" Controller I-O Base Addresses.
;
NPDATA	equ	001F0h		;Normal primary   address.
NSDATA	equ	00170h		;Normal secondary address.
APDATA	equ	001E8h		;Alternate primary   address.
ASDATA	equ	00168h		;Alternate secondary address.
;
; IDE Controller Register Offsets.
;
CDATA	equ	0		;Data port offset.
CDSEL	equ	6		;Disk-select and upper LBA offset.
CCMD	equ	7		;Command-register offset.
CSTAT	equ	7		;Primary-status register offset.
;
; Controller Status and Command Definitions.
;
BSY	equ	080h		;IDE controller is busy.
RDY	equ	040h		;IDE disk is "ready".
FLT	equ	020h		;IDE disk has a "fault".
DRQ	equ	008h		;IDE data request.
ERR	equ	001h		;IDE general error flag.
DMI	equ	004h		;DMA interrupt occured.
DME	equ	002h		;DMA error occurred.
DRCMD	equ	0C8h		;LBA28 read command (write is 0CAh,
				;    LBA48 commands are 025h/035h).
LBABITS equ	0E0h		;Fixed LBA command bits.
;
; Byte, Word, and Double-Word Definitions.
;
BDF	struc
lb	db	?
hb	db	?
BDF	ends

WDF	struc
lw	dw	?
hw	dw	?
WDF	ends

DDF	struc
dwd	dd	?
DDF	ends
;
; LBA "Device Address Packet" Layout.
;
DAP	struc
DapPL	db	?		;Packet length.
	db	?		;(Reserved).
DapSC	db	?		;I-O sector count.
	db	?		;(Reserved).
DapBuf	dd	?		;I-O buffer address (seg:offs).
DapLBA	dw	?		;48-bit logical block address.
DapLBA1	dd	?
DAP	ends
;
; DOS "Request Packet" Layout.
;
RP	struc
RPHLen	db	?		;Header byte count.
RPSubU	db	?		;Subunit number.
RPOp	db	?		;Opcode.
RPStat	dw	?		;Status word.
	db	8 dup (?)	;(Unused by us).
RPUnit	db	?		;Number of units found.
RPLen	dw	?		;Resident driver 32-bit length.
RPSeg	dw	?
RPCL	dd	?		;Command-line data pointer.
RP	ends
RPERR	equ	08003h		;Packet error flags.
RPDON	equ	00100h		;Packet done flag.
;
; Segment Declarations.
;
CODE	segment	public use16 'CODE'
	assume	cs:CODE,ds:CODE
;
; DOS Driver Device Header.
;
@	dd	0FFFFFFFFh	;Link to next header block.
	dw	08000h		;Driver device attributes.
WrkBf	dw	(I_Stra-@)	;Strategy routine pointer.
				;(LRU work buffer after Init).
XMSHndl	dw	(I_DevI-@)	;Device-Interrupt routine pointer.
				;(XMS handle number after Init).
	db	'UHDD'		;Our name:  "UHDD$" if caching, or
DNam2	db	'$',0,0,0	;	    "UHDD-SA$" non-cached.
;
; General Driver Variables.
;
VLF	db	0		;VDS lock flag (001h = buffer lock).
IOF	db	0		;I-O control flags --
				;  Bit 1:  Cache must be flushed.
				;  Bit 0:  Driver is currently busy.
DMAAd	dw	0		;DMA status/command register address.
IdeDA	dw	0		;IDE data register address.
RqSec	db	0		;Caching request I-O sector count.
RqCmd	db	0		;Caching I-O command bits.
RqCSC	db	0		;Caching current-block sector count.
RqTyp	db	0		;Caching device-type flag.
;
; VDS Parameter Block and UltraDMA Command-List.
;
VDSLn	dd	LMEnd		;VDS block length.
VDSOf	dd	0		;VDS 32-bit buffer offset.
VDSSg	dd	0		;VDS 16-bit segment (hi-order zero).
IOAdr	dd	0		;VDS and DMA 32-bit buffer address.
IOLen	dd	080000000h	;DMA byte count and "end" flag.
;
; "Physical Region Descriptor" Address and IDE Parameter Area.
;
PRDAd	dd	(IOAdr-@)	;PRD 32-bit command addr. (Init set).
	db	0		;IDE upper sector count   (always 0).
LBA2	db	0,0,0		;IDE upper LBA bits 24-47.
SecCt	db	0		;IDE lower sector count.
LBA	db	0,0,0		;IDE lower LBA bits 0-23.
DSCmd	db	0		;IDE disk-select and LBA commands.
IOCmd	db	0		;IDE command byte.
Temp	equ	[LBA2+1].lw	;Ye olde "temporary bucket".
;
; BIOS "Active Unit" Table.
;
Units	db	MAXBIOS dup (0)	;BIOS unit number for each device.
;
; Caching Variables.
;
CStack	dd	0		;Caller's saved stack pointer.
RqBuf	dd	0		;Request I-O buffer address.
RqXMS	dd	128		;Request XMS buffer offset.
RqLBA	dw	0,0		;Request initial LBA (6 bytes).
RqUNo	equ	[$+2].lb	;Request "cache unit" number.
RAhdF	equ	[$+3].lb	;Request UltraDMA Read-Ahead flag.
CBLBA	equ	[$+4].lw	;Current-buffer initial LBA.
CBUNo	equ	[$+10].lb	;Current-buffer "cache unit" number.
CBSec	equ	[$+11].lb	;Current-buffer sector count.
CBLst	equ	[$+12].lw	;Current-buffer "last" LRU index.
CBNxt	equ	[$+14].lw	;Current-buffer "next" LRU index.
;
; Initialization UltraDMA "Mode" Table (digit count in low 4 bits).
;   After Init, this area is the "equated" caching variables above.
;
Modes	dw	01602h		;Mode 0, ATA-16.
	dw	02502h		;Mode 1, ATA-25.
	dw	03302h		;Mode 2, ATA-33.
	dw	04402h		;Mode 3, ATA-44.
	dw	06602h		;Mode 4, ATA-66.
	dw	01003h		;Mode 5, ATA-100.
	dw	01333h		;Mode 6, ATA-133.
	dw	01663h		;Mode 7, ATA-166 (CF cards, etc.).
;
; Cache Control Variables.
;
STLmt	dw	0		;Search-table limit index.
LUTop	dw	0FFFFh		;LRU list "top" & "end" indexes.
LUEnd	dw	0FFFFh
;
; Subroutine to issue "A20" local enable and local disable requests.
;
A20Dis:	mov	ah,006h		;"A20" local disable -- get XMS code.
A20Req:	db	09Ah		;Call XMS manager for "A20" request.
@XEntry	dd	0
	sti			;Restore all critical settings.
	cld
	push	cs
	pop	ds
	sub	al,1		;Zero AL-reg. if success, -1 if error.
	ret			;Exit.
;
; Int 13h Entry Routine.
;
Entry:	pushf			;Entry -- Save CPU flags and BP-reg.
	push	bp
	mov	bp,0		;Reset active-units table index.
@LastU	equ	[$-2].lw
NextU:	dec	bp		;More active units to check?
	js s	QuickX		   ;No, not for us -- exit quick!
	cmp	dl,cs:[bp+Units-@] ;Does request unit match table?
	jne s	NextU		   ;No, see if more units remain.
	cli			;Disable CPU interrupts.
	bts	cs:IOF,0	;Is driver currently busy?
	jc s	InvalF		;Yes?  Set "Invalid function" & exit!
	mov	cs:CStack.lw,sp	;Save caller's stack pointer.
	mov	cs:CStack.hw,ss
	push	cs		;Switch to this driver's stack.
	pop	ss
	mov	sp,0
@Stack	equ	[$-2].lw
	sti			;Re-enable CPU interrupts.
	pushad			;Save all CPU registers.
	push	ds
	push	es
	pusha			;Issue "A20" local-enable request.
	mov	ah,005h
	call	A20Req
	popa
	jc s	A20Err		;If "A20" error, use routine below.
	db	0EAh		;Go to main (HMA) request handler.
@HDMain	dd	(HDReq-@)
InvalF:	mov	ah,001h		;We're BUSY, you FOOL!  Post "Invalid
	jmp s	GetOut		;  function" code and bail out QUICK!
A20Err:	mov	al,XMSERR	;"A20" error!  Post "XMS error" code.
	jmp s	Abandn		;Abandon this request and exit QUICK!
PassRq:	call	A20Dis		;Pass request -- Disable "A20" line.
	cli			;Disable CPU interrupts.
	dec	IOF		;Reset driver "busy" flag.
	pop	es		;Reload all CPU regs. we used.
	pop	ds
	popad
	lss	sp,cs:CStack	;Switch back to caller's stack.
QuickX:	pop	bp		;Reload BP-reg. and CPU flags.
	popf
	nop			;(Unused filler).
	db	0EAh		;Go to next routine in Int 13h chain.
@Prev13	dd	0
IOExit:	call	A20Dis		;Exit:  Issue "A20" local disable.
	pop	ax		;Reload error code and error flag.
	popf
Abandn:	mov	bp,sp		;Point to saved registers on stack.
	mov	[bp+33],al	;Set error code in exiting AH-reg.
	cli			;Disable CPU interrupts.
	dec	IOF		;Reset driver "busy" flag.
	pop	es		;Reload all CPU regs. we used.
	pop	ds
	popad
	lss	sp,cs:CStack	;Switch back to caller's stack.
GetOut:	mov	bp,sp		;Set error flag in exiting carry bit.
	rcr	[bp+8].lb,1
	rol	[bp+8].lb,1
	pop	bp		;Reload BP-reg. and CPU flags.
	popf
	iret			;Exit.
;
; Miscellaneous Caching Variables.
;
LBABuf	dw	00082h,0	;"Calculated LBA" buffer (8 bytes).
PCISubC	dw	00100h		;PCI subclass/function (Init only).
PCITbl	dw	(Ctl1Tbl-@)	;PCI ctlr. table addr. (Init only).
LMDvr	equ	($+STACKSA-@)	;(Length of non-cached and cache
LMEnd	equ	($+STACK-@)	;   drivers' "low memory" areas).
;
; Main Disk and Diskette Request Handler.
;
HDReq:	cmp	bp,CDDVD	;CD/DVD request?
	jae	Cache		;Yes, set all user parameters below.
	mov	dl,0BEh		;Mask out LBA & write request bits.
	and	dl,ah
	cmp	dl,002h		   ;CHS or LBA read/write request?
	mov	dl,cs:[bp+TypeF-@] ;(Set unit "type" flag anyway).
@TypeF	equ	[$-2].lw
	mov	RqTyp,dl
	je s	ChkLBA		;Yes, check for actual LBA request.
	cmp	dl,0C0h		;Not for us -- Diskette I-O request?
	jne s	Pass		;No, "flush" cache and pass request.
	cmp	ah,005h		;Diskette track-format request?
	jne s	Pass1		;No, "pass" other diskette requests.
Pass:	or	IOF,002h	;Flush cache on next caching request.
Pass1:	push	ss		;Return to request "pass" routine.
	push	(PassRq-@)
	retf
ChkLBA:	mov	di,sp		;Point DI-reg. to current stack.
	shl	ah,1		;LBA read or write request?
	jns s	ValSC		;No, go validate CHS sector count.
	push	ds		;Reload "DAP" segment into DS-reg.
	mov	ds,[di+2]
	cmp	[si].DapBuf,-1	;Check for 64-bit "DAP" I-O buffer.
	mov	al,[si].DapSC	;(Get all "DAP" I-O parameters).
	les	dx,[si].DapLBA1
	mov	di,es
	les	cx,[si].DapBuf
	mov	si,[si].DapLBA
	pop	ds
	je s	Pass		;If 64-bit buffer, pass this request!
ValSC:	dec	al		;Is sector count zero or over 128?
	js s	Pass		;Yes?  Let BIOS have this request!
	inc	ax		;Restore sector count -- LBA request?
	xchg	ax,cx		;(Save sector count & command in CX).
	js s	ZeroBX		;Yes, go zero BX-reg. for our logic.
	mov	es,[di]		;CHS -- Reload I-O buffer segment.
	xor	di,di		;Reset upper LBA address bits.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	shr	al,6		;Set AX-reg. to starting cylinder.
	xchg	al,ah
	xchg	ax,dx		   ;Swap cylinder & head values.
	mov	al,cs:[bp+CHSec-@] ;Get disk CHS sectors/head value.
@CHSec	equ	[$-2].lw
	or	al,al		   ;Legitimate disk CHS values?
	jz s	Pass		   ;No?  Let BIOS have this request!
	push	ax		   ;Save CHS sectors/head value.
	mul	ah		   ;Convert head to sectors.
	add	si,ax		   ;Add result to starting sector.
	pop	ax		   ;Reload CHS sectors/head value.
	mul	cs:[bp+CHSHd-@].lb ;Convert cylinder to sectors.
@CHSHd	equ	[$-2].lw
	mul	dx
	add	si,ax		   ;Add to head/sector value.
	adc	dx,di
	xchg	ax,bx		   ;Get buffer offset in AX-reg.
ZeroBX:	xor	bx,bx		   ;Zero BX-reg. for our logic.
	mov	[bx+VDSOf-@].lw,ax ;Set VDS I-O buffer address.
	mov	[bx+VDSSg-@],es
	movzx	ax,cl		   ;Set VDS buffer length =
	shl	ax,1		   ;  sectors * (512 bytes/sector).
	mov	[bx+VDSLn+1-@],ax
	pusha			    ;Save all 16-bit CPU regs.
	push	ds
	or	[bx+IOAdr-@].dwd,-1 ;Invalidate VDS buffer address.
	mov	ax,08103h	    ;Execute VDS "lock" request.
	mov	dx,0000Ch
	mov	di,(VDSLn-@)
	push	ds
	pop	es
	int	04Bh
	sti			    ;Restore all critical settings.
	cld
	pop	ds		    ;Reload all 16-bit regs.
	popa
	jc	Pass		    ;Lock error?  Pass this request!
	mov	eax,[bx+IOAdr-@]    ;Get 32-bit VDS buffer address.
	cmp	eax,-1		    ;Is VDS address still all-ones?
	adc	[bx+VLF-@],bl	    ;(Use carry as VDS "lock" flag).
	jnz s	Cache		    ;No, set all caching parameters.
	movzx	eax,[bx+VDSSg-@].lw ;VDS logic is NOT present --
	shl	eax,4		    ;  set 20-bit buffer address.
	add	eax,[bx+VDSOf-@]
;
; Main Caching Routine.
;
Cache:	and	ch,006h		    ;Mask off read/write commands.
	mov	[bx+RqSec-@],cx	    ;Set sector count & I-O command.
	mov	[bx+RqBuf-@],eax    ;Set user I-O buffer address.
	mov	[bx+RqLBA-@],di	    ;Set initial request LBA.
	mov	[bx+RqLBA+2-@],dx
	mov	[bx+RqLBA+4-@],si
	mov	[bx+RqUNo-@],bp	     ;Set "cache unit" number.
	mov	[bx+RAhdF-@],bl	     ;Reset UltraDMA Read-Ahead flag.
	btr	[bx+IOF-@].lb,1	     ;Does our cache need a flush?
	jc s	Flush		     ;Yes, do flush before request.
	cmp	[bx+RqTyp-@].lb,0C0h ;Diskette request?
	jb s	NxtBlk		     ;No, begin request execution.
	mov	es,bx		     ;Point ES-reg. to low memory.
	mov	al,es:[DKTSTAT]	     ;Get BIOS diskette status code.
	and	al,01Fh
	cmp	al,MCHANGE	    ;Diskette media-change seen?
	jne s	NxtBlk		    ;No, begin request execution.
Flush:	mov	[bx+STLmt-@],bx	    ;Reset search-table limit index.
	or	[bx+LUTop-@].dwd,-1 ;Reset LRU "top"/"end" indexes.
NxtBlk:	mov	dx,ds		    ;Set ES-reg. same as DS-reg.
	mov	es,dx
	mov	dx,0		;Set initial search-table offset.
@MP1	equ	[$-2].lw
	mov	bp,dx		;Set initial search-table pointer.
	dec	bp
	mov	al,0		;Get cache "granularity".
@GRAN1	equ	[$-1].lb
	cmp	al,[bx+RqSec-@]	;Multiple cache blocks needed?
	jbe s	SetSC		;Yes, use "full block" sector count.
	mov	al,[bx+RqSec-@]	;Use remaining request sector count.
SetSC:	mov	[bx+RqCSC-@],al	   ;Set maximum I-O sector count.
	and	[bx+RqXMS-@].dwd,0 ;Reset current XMS buffer offset.
Search:	shr	dx,1		   ;Halve search-table offset.
	cmp	bp,[bx+STLmt-@]	;Is search-table address too high?
	jae s	SrchLo		;Yes, cut table address by offset.
	call	SIGet		;Get next cache entry into buffer.
	jc s	XMErr1		;If any XMS error, exit immediately!
	mov	si,(RqLBA-@)	;Compare initial request & table LBA.
	call	CComp
	jae s	ChkEnd		;Request >= table:  Check for found.
SrchLo:	sub	bp,dx		;Subtract offset from search pointer.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	Search		;No, go compare next table entry.
	jmp s	NoFind		;Handle request as "not found".
ChkEnd:	je	Found		;Request = table:  Handle as "found".
	mov	cl,[bx+CBSec-@]	;Calculate and set ending entry LBA.
	mov	si,(CBLBA-@)
	call	CalcEA
	mov	si,(RqLBA-@)	;Compare request start & entry end.
	call	CComp1
	jb s	Found		  ;Request < Entry:  Handle as found.
	ja s	SrchHi		  ;Request > Entry:  Bump search ptr.
	cmp	[bx+CBSec-@].lb,0 ;Is this cache block full?
@GRAN2	equ	[$-1].lb
	jb s	Found		;No, handle request as "found".
SrchHi:	add	bp,dx		;Add offset to search pointer.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	Search		;No, go compare next table entry.
	inc	bp		;Bump index to next table entry.
NoFind:	mov	[bx+Temp-@],bp	;Unfound:  Save search-table offset.
	mov	bp,[bx+STLmt-@]	;Get next "free" cache-table index
	call	STGet		;  and leave index in work buffer.
XMErr1:	jc s	XMErr2		      ;If any XMS error, exit now!
	cmp	[bx+RqTyp-@].lb,CDTYP ;UltraDMA disk I-O request?
	jae s	MoveUp		      ;No, go move up search table.
	test	[bx+RqCmd-@].lb,042h  ;I-O done, or output request?
	jnz s	MoveUp		      ;Yes, go move up search table.
@NoRA:	call	SetRA		   ;Set Read-Ahead flag if needed.
MoveUp:	movzx	ecx,bp		   ;Get move-up word count.
	movzx	esi,[bx+Temp-@].lw
	sub	cx,si		   ;Any search indexes to move up?
	jz s	NFind2		   ;No, go set up new cache entry.
	shl	ecx,1		;Set move-up byte count.
	shl	esi,1		;Set 32-bit move-up addresses.
	add	esi,080000000h
@STAdr1	equ	[$-4].dwd
	lea	edi,[esi+2]
	push	ecx		;Save move length & destination addr.
	push	edi
	mov	edi,0		;Get & save search-table buffer addr.
@STBuff	equ	[$-4].dwd
	push	edi
	call	MvData		;Send needed data to search buffer.
	pop	esi		;Reload search-table buffer address.
	pop	edi		;Reload move destination & length.
	pop	ecx
	jc s	XMErr2		;If any XMS error, exit now!
	call	MvData		;Bring data BACK from search buffer.
XMErr2:	jc s	XMErr3		;If any XMS error, exit now!
	mov	bp,[bx+Temp-@]	;Set up table-index "put" to XMS.
	inc	bx
	call	STPut		;Insert "free" index in search table.
	jc s	XMErr3		;If any XMS error, exit now!
NFind2:	inc	[bx+STLmt-@].lw	;Bump binary-search limit index.
	mov	si,(RqLBA-@)	;Set 48-bit LBA in new entry.
	mov	di,(CBLBA-@)
	movsd
	movsw
	movsb			  ;Set "cache unit" in new entry.
	mov	[di],bl		  ;Reset new entry's sector count.
	mov	ax,[bx+WrkBf-@]	  ;Reload "free" cache-table index.
Found:	mov	cx,[bx+RqLBA+4-@] ;Get starting I-O offset in block.
	sub	cx,[bx+CBLBA+4-@]
	mov	[bx+RqXMS-@],cl	  ;Set starting XMS sector offset.
	mov	[bx+Temp-@],cx	  ;Save starting I-O sector offset.
	cmp	[bx+CBSec-@],bl	  ;Is this a new cache-table entry?
	je s	ReLink		  ;Yes, relink entry as top-of-list.
	push	ax		  ;Unlink this entry from LRU list.
	call	UnLink
	pop	ax
ReLink:	movzx	edx,ax		   ;Get 32-bit cache block number.
	shl	edx,0		   ;Shift number to starting sector.
@GRSSC	equ	[$-1].lb
	add	[bx+RqXMS-@],edx   ;Add to preset sector offset.
	shl	[bx+RqXMS-@].dwd,9	 ;Convert sectors to bytes.
	add	[bx+RqXMS-@].dwd,020000h ;Add in XMS base address.
@XBase	equ	[$-4].dwd
	mov	cx,0FFFFh	;Make this entry "top of list".
	or	[bx+CBLst-@],cx	;Set this entry's "last" index.
	mov	dx,ax		;Swap top-of-list & entry index.
	xchg	dx,[bx+LUTop-@]
	mov	[bx+CBNxt-@],dx	;Set this entry's "next" index.
	cmp	dx,cx		;Is this the only LRU index?
	mov	cx,ax		;(Get this entry's index in CX-reg.).
	je s	ReLnk1		;Yes, make entry last on LRU list.
	push	ax		;Link entry to prior "top of list".
	call	UnLnk3
	pop	ax
	jmp s	ReLnk2		;Go deal with I-O sector count.
Update:	mov	[bx+CBSec-@],cl	;Update entry's total sectors.
	movzx	cx,[bx+RqCSC-@]	;Get current I-O sector count.
	bt	[bx+RAhdF-@],bl ;Read-Ahead request?
	jnc s	Updat1		;No, update this cache-table entry.
	mov	cl,128		;Use full count of 128 sectors.
	mov	[bx+CBSec-@],cl
Updat1:	call	CBPut		;Update this cache-table entry.
XMErr3:	jc	XMErrC		;If any XMS error, exit now!
	mov	si,(RqLBA-@)	;Calculate ending LBA for this I-O.
	call	CalcEA
	add	bp,1		;Skip to next search index.
Ovrlap:	cmp	bp,[bx+STLmt-@]	;More cache-table entries to check?
	jae s	CachIO		;No, O.K. to handle "found" entry.
	call	SIGet		;Get next cache entry into buffer.
	jc s	XMErr3		;If any XMS error, exit now!
	mov	si,(LBABuf-@)	;Compare request end & entry start.
	call	CComp
	jbe s	CachIO		;Request <= entry:  O.K. to proceed.
	push	bp		;Delete this overlapping cache block.
	call	Delete
	pop	bp
	jmp s	Ovrlap		;Go check for more entry overlap.
ReLnk1:	mov	[bx+LUEnd-@],ax	;Make entry last on LRU list, too!
ReLnk2:	mov	cx,[bx+Temp-@]	;Reload initial I-O sector offset.
	mov	ch,0		;Get entry's available sectors.
@GRAN3	equ	[$-1].lb
	sub	ch,cl
	cmp	ch,[bx+RqCSC-@]	;More I-O sectors than available?
	jae s	Larger		;No, retain maximum sector count.
	mov	[bx+RqCSC-@],ch	;Reduce current I-O sector count.
	nop			;(Unused filler).
Larger:	add	cl,[bx+RqCSC-@]	;Get ending I-O sector number.
	cmp	cl,[bx+CBSec-@]	;More sectors than entry has now?
	ja s	Update		;Yes, update entry sectors.
	inc	bx		;Update this cache-table entry.
	call	CBPut
	jc s	XMErr3		      ;If any XMS error, exit now!
	test	[bx+RqCmd-@].lb,002h  ;Read request?
	jz s	BufMov		      ;Yes, move cache data to user.
CachIO:	bts	[bx+RqCmd-@].lb,6     ;I-O done during a prior block?
	jc s	BfMore		      ;Yes, buffer more cache data.
	cmp	[bx+RqTyp-@].lb,CDTYP ;What type of unit is this?
	jae s	ExtIO		      ;External/BIOS -- go see which.
	call	UdmaIO		 ;Use "UdmaIO" for SATA/IDE disk I-O.
	jmp s	ErChk1		 ;Go check for any I-O errors.
ExtIO:	or	[bx+RqTyp-@],bl	 ;BIOS-handled unit?
	js s	BiosIO		 ;Yes, go call the BIOS below.
	call	[bx+DMAAd-@].dwd ;Use external-driver "callback" rtn.
	jmp s	ErrChk		 ;Go check for any I-O errors.
BiosIO:	lds	si,CStack	 ;BIOS I-O:  Reload caller CPU flags.
	push	[si+2].lw
	popf
	pop	es		;Reload all CPU registers.
	pop	ds
	popad
	pushf			;Call the BIOS to do this request.
	call	ss:@Prev13
	pushad			;Save all CPU regs. again.
	push	ds
	push	es
	mov	al,ah		;Move any BIOS error code to AL-reg.
ErrChk:	sti			;Restore all critical settings.
	cld
	push	ss		;Reload this driver's DS- & ES-regs.
	pop	ds
ErChk1:	push	ds
	pop	es
	mov	bx,0		;Rezero BX-reg. but save carry.
	jnc s	InCach		;If no error, see if I-O cached data.
IOErr:	or	[bx+IOF-@].lb,2	;I-O error!  Request a cache flush.
	stc			;Set carry again after "or" command.
EndIO:	pushf			;End of I-O:  Save error flag & code.
	push	ax
	shr	[bx+VLF-@].lb,1	;User's I-O buffer "locked" by VDS?
	jnc s	IOEnd		;No, go issue "A20" local disable.
	mov	ax,08104h	;Set VDS "unlock" parameters.
	mov	dx,0
	mov	di,(VDSLn-@)	;Issue VDS "unlock" request.
	push	ds
	pop	es
	int	04Bh
IOEnd:	push	ss		;Return to low-memory "end" routine.
	push	(IOExit-@)
	retf
InCach:	or	[bx+RqCmd-@],bl	;UltraDMA data cached during I-O?
	js s	ChkFul		;Yes, see if search table is full.
BfMore:	or	al,1		    ;Ensure we LOAD data into cache!
BufMov:	movzx	ecx,[bx+RqCSC-@].lb ;Set XMS move sector count.
	mov	esi,[bx+RqXMS-@]    ;Set desired XMS buffer address.
	mov	edi,[bx+RqBuf-@]    ;Set user buffer as destination.
	call	MovBuf		    ;Move data between user and XMS.
	jc s	IOErr		       ;If any XMS error, exit now!
ChkFul:	cmp	[bx+STLmt-@].lw,08000h ;Is binary-search table full?
@TE1	equ	[$-2].lw
	jb s	MoreIO		  ;No, check for more sectors to go.
	push	ax		  ;Delete least-recently-used entry.
	mov	ax,[bx+LUEnd-@]
	call	SrchD
	pop	ax
MoreIO:	xor	ax,ax		  ;Reset error code for exit above.
	movzx	cx,[bx+RqCSC-@]	  ;Get current I-O sector count.
	sub	[bx+RqSec-@],cl	  ;More data sectors left to handle?
	jz s	EndIO		  ;No, go do VDS "unlock" and exit.
	add	[bx+RqLBA+4-@],cx ;Update LBA for next cache block.
	adc	[bx+RqLBA+2-@],bx
	adc	[bx+RqLBA-@],bx
	shl	cx,1		  ;Convert sector count to bytes.
	add	[bx+RqBuf+1-@],cl ;Update user I-O buffer address.
	adc	[bx+RqBuf+2-@],bx
	jmp	NxtBlk		  ;Go handle next cache data block.
;
; Subroutine to calculate ending request or cache-table LBA values.
;
CalcEA:	mov	di,(LBABuf-@)	;Point to address-comparison buffer.
	push	[si].dwd	;Move high-order LBA into buffer.
	pop	[di].dwd
	add	cx,[si+4]	;Calculate ending LBA & exit.
	mov	[di+4],cx
	adc	[di+2],bx
	adc	[di],bx
CalcX:	ret
;
; Subroutine to do 7-byte "cache unit" and LBA comparisons.
;
CComp:	mov	di,(CBLBA-@)	;Point to current-buffer LBA value.
CComp1:	mov	cl,[bx+RqUNo-@]	;Compare our "unit" & table "unit".
	cmp	cl,[bx+CBUNo-@]
	jne s	CalcX		;If units are different, exit now.
	mov	cx,3		;Compare our LBA & cache-table LBA.
	rep	cmpsw
	ret			;Exit.
;
; Subroutine to "put" an LRU index into its cache-data table entry.
;
LRUPut:	push	ax		;Save needed 16-bit regs. & Z-flag.
	push	cx
	push	dx
	pushf
	mov	edi,(WrkBf-@)	;Set working buffer addr.
@WrkBf1	equ	[$-4].dwd
	mov	ecx,12		;Set cache-table entry size.
	mul	cx		;Multiply cache index by entry size.
	mov	cl,2		;Set LRU index size.
	jmp s	CEMov1		;Go get desired cache-entry offset.
;
; Subroutine to "get" or "put" a cache-data table entry.
;
SIGet:	call	STGet		;Get next binary-search table index.
	jc s	CalcX		;If any XMS error, exit now!
	mov	ax,[bx+WrkBf-@]	;Get desired cache-data table index.
CBGet:	xor	di,di		;Set Z-flag for "get".
CBPut:	mov	edi,(CBLBA-@)	;Set current buffer addr.
@CBAddr	equ	[$-4].dwd
	mov	esi,0		;Set cache-data table address.
@CTBas1	equ	[$-4].dwd
	push	ax		;Save needed 16-bit regs. & Z-flag.
	push	cx
	push	dx
	pushf
	mov	ecx,12		;Set cache-table entry size.
	mul	cx		;Multiply cache index by entry size.
CEMov1:	push	dx		;Get cache-entry offset in EAX-reg.
	push	ax
	pop	eax
	jmp s	CEMov2
;
; Subroutine to "get" or "put" a search-table index.
;
STGet:	xor	di,di		;Set Z-flag for "get".
STPut:	mov	edi,(WrkBf-@)	;Set working buffer addr.
@WrkBf2	equ	[$-4].dwd
	mov	esi,0		;Add starting search-table address.
@STAdr2	equ	[$-4].dwd
	push	ax		;Save needed 16-bit regs. & Z-flag.
	push	cx
	push	dx
	pushf
	xor	ecx,ecx		;Set search-table index size.
	mov	cl,2
	movzx	eax,bp		;Get 32-bit offset of requested
	shl	eax,1		;  index in binary-search table.
CEMov2:	add	esi,eax		;Set XMS data address (base + offset).
	popf			;Reload CPU flags.
	push	bp		;Save BP-reg. now (helps alignment).
	jz s	CEMov3		;Reading from XMS memory?
	xchg	esi,edi		;No, swap source & destination addrs.
CEMov3:	call	MvData		;Move all data to or from XMS memory.
	pop	bp		;Reload 16-bit regs. we used & exit.
	pop	dx
	pop	cx
	pop	ax
	ret
;
; Global Descriptor Table, used in all XMS moves.
;
GDT	dd	0,0,0,0		      ;"Null" and code descriptors.
	dd	00000FFFFh,000009300h ;Protected-mode source descr.
DstDsc	dd	00000FFFFh,000009300h ;Protected-mode dest. descr.
	dd	0,0,0,0		      ;(Reserved for BIOS use).
GDT_DS	dd	00000FFFFh,000CF9300h ;4-GB real-mode descriptor.
;
; Real-Mode Global Descriptor Table Pointer.
;
GDTP	dw	($-GDT)		;GDT length  (always 56 bytes).
GDTPAdr	dd	(GDT-@)		;GDT 32-bit address (Init set).
;
; Hard Disk and Diskette Parameter Tables.
;
TypeF	db	MAXBIOS dup (0)	;Device type and sub-unit number:
				;  00h-23h:  SATA/IDE UltraDMA disk.
				;  40h:      0 if LBA28, 1 if LBA48.
				;  80h:      BIOS-handled disk.
				;  C0h:      BIOS-handled diskette.
CHSec	db	MAXBIOS dup (0)	;CHS sectors-per-head table.
CHSHd	db	MAXBIOS dup (0)	;CHS heads-per-cylinder table.
;
; Subroutine to find a search-table index and delete its cache entry.
;
SrchD:	call	CBGet		;Get target cache-table entry.
	jc s	XMErr4		;If any XMS error, exit now!
	mov	si,(CBLBA-@)	;Save target LBA address & unit no.
	mov	di,(LBABuf-@)
	movsd
	movsd
	mov	dx,0		;Set initial search-table offset.
@MP2	equ	[$-2].lw
	mov	bp,dx		;Set initial search-table pointer.
	dec	bp
SrchD1:	shr	dx,1		;Halve search-table offset.
	cmp	bp,[bx+STLmt-@]	;Is search-table address too high?
	jae s	SrchD4		;Yes, cut table address by offset.
	call	SIGet		;Get next cache entry into buffer.
	jc s	XMErr4		;If XMS error, go bail out below!
	mov	si,(LBABuf-@)	;Set up target v.s. work comparison.
	mov	di,(CBLBA-@)
	mov	cl,[si+6]	;Compare target unit v.s. work unit.
	cmp	cl,[di+6]
	jne s	SrchD3		;If units differ, check results now.
	mov	cx,3		;Compare our LBA & cache-table LBA.
	rep	cmpsw
	je s	Delete		;Target = entry:  BP has our offset.
SrchD3:	ja s	SrchD5		;Target > entry:  Adjust offset up.
SrchD4:	sub	bp,dx		;Subtract offset from search ptr.
	jmp s	SrchD6		;Go see if we did our last compare.
SrchD5:	add	bp,dx		;Add offset to search pointer.
SrchD6:	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	SrchD1		;No, go compare next table entry.
XMErr4:	jmp s	XMErr5		;Not found!  Handle as an XMS error.
;
; Subroutine to delete an index from our search table.
;
Delete:	mov	[bx+WrkBf-@],ax	    ;Save index being deleted.
	movzx	ecx,[bx+STLmt-@].lw ;Get move-down word count.
	sub	cx,bp
	dec	cx		;Any indexes to move down?
	jz s	Delet1		;No, put our index on "free" list.
	shl	ecx,1		;Set move-down byte count.
	movzx	edi,bp		;Set 32-bit move-down addresses.
	shl	edi,1
	add	edi,080000000h
@STAdr3	equ	[$-4].dwd
	lea	esi,[edi+2]
	call	MvData		;Move down needed search-table data.
	jc s	XMErr5		;If any XMS error, exit now!
Delet1:	dec	[bx+STLmt-@].lw	;Decrement search-table limit index.
	mov	bp,[bx+STLmt-@]	;Put deleted index in "free" list.
	inc	bx
	call	STPut
	jc s	XMErr5		;If any XMS error, exit now!
;
; Subroutine to unlink a cache-table entry from the LRU list.
;
UnLink:	mov	cx,[bx+CBLst-@]	;Get entry's "last"/"next" indexes.
	mov	dx,[bx+CBNxt-@]
	cmp	cx,-1		;Is this entry top-of-list?
	je s	UnLnk1		;Yes, "promote" next entry.
	mov	[bx+WrkBf-@],dx	;Save this entry's "next" index.
	mov	ax,cx		;Get "last" entry cache-table index.
	mov	esi,10		;Get cache-table addr. + "next" ofs.
@CTBas2	equ	[$-4].dwd
	call	LRUPut		;Update last entry's "next" index.
	jnc s	UnLnk2		;If no XMS error, check end-of-list.
XMErr5:	pop	cx		;XMS error!  Discard exit address &
	pop	cx		;  AX/BP parameter saved upon entry.
XMErrC:	mov	al,XMSERR	;Return "XMS error" code in AL-reg.
	jmp	IOErr		;Go request a cache flush and exit.
UnLnk1:	mov	[bx+LUTop-@],dx	;Make next entry top-of-list.
UnLnk2:	cmp	dx,-1		;Is this entry end-of-list?
	jne s	UnLnk3		;No, link next to prior entry.
	mov	[bx+LUEnd-@],cx	;Make prior entry end-of-list.
	ret			;Exit.
	db	0,0		;(Unused filler).
UnLnk3:	mov	[bx+WrkBf-@],cx	;Save this entry's "last" index.
	mov	ax,dx		;Get "next" entry cache-table index.
	mov	esi,8		;Get cache-table addr. + "last" ofs.
@CTBas3	equ	[$-4].dwd
	call	LRUPut		;Update next entry's "last" index.
	jc s	XMErr5		;If any XMS error, exit now!
	ret			;All is well -- exit.
;
; Subroutine to do I-O buffer XMS moves.
;
MovBuf:	jz s	MovBf2		;Reading from XMS memory?
	xchg	esi,edi		;No, swap source & destination addrs.
	xchg	bx,bx		;(Unused filler).
MovBf2:	shl	ecx,9		;Convert sectors to byte count.
;
; Subroutine to do fast caching XMS moves.
;
MvData:	push	ds		;Save driver's DS-register.
	mov	edx,ecx		;Save byte count in EDX-reg.
	xor	ecx,ecx		;Get XMS move section count.
	mov	ch,(8192/256)
@MvRSC1	equ	[$-1].lb
	smsw	ax		;Get CPU machine-status word.
	shr	ax,1		;Running in protected-mode?
@MvPA:	jc s	MvProt		;Yes, go use protected-mode logic.
MvRNxt:	cmp	ecx,edx		;At least one section left?
	cli			;(Disable CPU interrupts now).
	jbe s	MvRGo		;Yes, use full section count.
	mov	cx,dx		;Use remaining byte count.
MvRGo:	db	02Eh,00Fh	;"lgdt cs:GDTP" done the hard way to
	db	001h,016h	;  stop annoying assembler warnings!
	dw	(GDTP-@)
@GDTP	equ	[$-2].lw
	mov	eax,cr0		;Set CPU protected-mode control bit.
	or	al,001h
	mov	cr0,eax
@386A:	mov	bh,0		;(Set to "jmp s $+2" for an 80386). 
	mov	bl,(GDT_DS-GDT)	;Set DS- and ES-reg. selectors.
	mov	ds,bx
	mov	es,bx
	shr	cx,2		;Move all desired 32-bit words.
	rep	movs es:[edi].dwd,ds:[esi].dwd
	adc	cx,cx		;If "odd" 16-bit word, move it also.
	rep	movs es:[edi].lw,ds:[esi].lw
@386B:	and	ax,0FFFEh	;Clear protected-mode control bit.
	mov	cr0,eax
	sti			;Allow interrupts after next command.
	mov	ch,(8192/256)	;Reload move section count.
@MvRSC2	equ	[$-1].lb
	sub	edx,ecx		;More data sections to move?
	ja s	MvRNxt		;Yes, go move next data section.
MvDone:	clc			;Success!  Reset carry flag.
MvExit:	pop	ds		;Reload driver's segment registers.
	push	ds
	pop	es
	mov	bx,0		;Rezero BX-reg. but save carry bit.
	ret			;Exit.
MvProt:	shl	ecx,3		;Protected-mode:  Get 64K section ct.
@MvPSC	equ	[$-1].lb	;(With /Z, "shl ecx,0" for 8K count).
MvPNxt:	push	ecx		;Save move section count.
	push	edx		;Save remaining move byte count.
	push	esi		;Save move source & destination addrs.
	push	edi
	cmp	ecx,edx		;At least one section left?
	jbe s	MvPGo		;Yes, use full section count.
@MvPB:	mov	ecx,edx		;Use remaining byte count instead.
MvPGo:	push	cs		;Point ES-reg. to GDT descriptors.
	pop	es
	shr	ecx,1		;Convert byte count to word count.
	push	edi		;Set up destination descriptor.
	mov	di,(DstDsc+2-@)
@MvDesc	equ	[$-2].lw
	pop	ax
	stosw
	pop	ax
	stosb
	mov	es:[di+2],ah
	push	esi		;Set up source descriptor ("sub"
	sub	di,11		;  zeros carry for our Int 15h).
	pop	ax
	stosw
	pop	ax
	stosb
	mov	es:[di+2],ah
	lea	si,[di-21]	;Point to start of descriptor table.
	mov	ah,087h		;Request protected-mode XMS move.
	int	015h
	sti			;Restore all critical settings.
	cld
	mov	al,XMSERR	;Post "XMS error" code if needed.
@MvPC:	pop	edi		;Reload all 32-bit move parameters.
	pop	esi
	pop	edx
	pop	ecx
	jc s	MvExit		;If any XMS error, exit now!
	sub	edx,ecx		;More data sections to move?
	jbe s	MvDone		;No, go reset carry flag and exit.
	add	esi,ecx		;Update source and dest. addresses.
	add	edi,ecx
	jmp s	MvPNxt		;Go move next section of data.
ESWLEN	equ	($+7-HDReq)	;(Length of all /E caching logic).
;
; Subroutine to set up and execute disk UltraDMA I-O requests.
;
UdmaIO:	mov	cx,[bx+RqSec-@]	  ;Get sector count and I-O command.
	les	di,[bx+RqLBA-@]	  ;Get 48-bit logical block address.
	mov	dx,es
	mov	si,[bx+RqLBA+4-@]
	mov	[bx+LBA-@],si	  ;Set LBA bits 0-31 in IDE commands.
	mov	[bx+LBA+2-@],dl
	mov	[bx+LBA2-@],dh
	cmp	[bx+RAhdF-@],bl   ;Read-Ahead request?
	je s	GetTyp		  ;No, get disk's "type" flags.
	mov	cl,128		  ;Input 128 sectors if Read-Ahead.
GetTyp:	mov	[bx+LBA2+1-@],di  ;Set LBA bits 32-47 in IDE cmds.
	mov	bp,[bx+RqTyp-@]	  ;Get this disk's "type" flags.
	mov	ax,bp		  ;Get disk's IDE controller number.
	and	al,03Ch
	shr	al,2
	mov	ah,6		  ;Point to disk's I-O addresses.
	mul	ah
	add	ax,offset (Ctl1Tbl-@)
@CtlTbl	equ	[$-2].lw
	xchg	ax,si
	lods	cs:[si].lw	  ;Get primary DMA controller addr.
	rcr	bp,2		  ;Primary channel I-O request?
	jnc s	AdrSet		  ;Yes, use primary DMA controller.
	add	ax,8		  ;Use secondary DMA controller.
	inc	si		  ;Point to secondary IDE address.
	inc	si
AdrSet:	mov	[bx+DMAAd-@],ax	  ;Set DMA controller address.
	lods	cs:[si].lw	  ;Set IDE status/command address.
	mov	[bx+IdeDA-@],ax
	btr	bp,4		  ;Does disk hold >= 128-GB of data?
	jnc s	LBA28		  ;No, use LBA28 read/write command.
	shl	ch,3		  ;LBA48:  Get command as 20h/30h.
	jmp s	SetCmd		  ;Go set LBA and IDE command bytes.
LBA28:	shr	dx,12		  ;LBA28:  Reload & reset bits 24-27.
	xchg	dh,[bx+LBA2-@]
	or	ch,(DRCMD+1)	  ;Get LBA28 read/write command + 5.
SetCmd:	shl	bp,1		  ;Shift master/slave into LBA cmds.
	mov	dl,(LBABITS/32)
	rcl	dl,5
	or	dl,dh		  ;"Or" in LBA28 bits 24-27 (if any).
	mov	dh,005h		  ;Get final IDE command byte --
	xor	dh,ch		  ;  LBA28 C8h/CAh, LBA48 25h/35h.
	mov	[bx+DSCmd-@],dx	  ;Set LBA and IDE command bytes.
	mov	[bx+SecCt-@],cl	  ;Set I-O sector count.
	mov	ch,0		  ;Set DMA buffer length.
	shl	cx,1
	mov	[bx+IOLen+1-@],cx
	shr	[bx+RAhdF-@].lb,1    ;Read-Ahead request?
	jc s	UseCB		     ;Yes, do cache-buffered input.
	mov	eax,[bx+RqBuf-@]     ;Get user's 32-bit buffer addr.
	test	al,003h		     ;Is user buffer 32-bit aligned?
	jnz s	OurBuf		     ;No, use "Main XMS" buffer.
	cmp	[bx+RqBuf-@].hw,009h ;Is DMA below our 640K limit?
	ja s	OurBuf		     ;No, use "Main XMS" buffer.
	mov	cx,[bx+IOLen-@]	     ;Get (byte count -1) + I-O addr.
	dec	cx
	add	cx,ax		     ;Would I-O cross a 64K boundary?
	jnc s	SetBuf		     ;No, O.K. to do user buffer I-O.
OurBuf:	mov	eax,0		     ;Get "Main XMS" buffer address.
@XBAddr	equ	[$-4].dwd
	mov	cl,[bx+RqCSC-@]	     ;Get current I-O sector count.
	cmp	cl,[bx+RqSec-@]	     ;Is all I-O in one cache block?
	jb s	SetBuf		     ;No, must use "Main XMS" buffer.
UseCB:	or	[bx+RqCmd-@].lb,080h ;Do not cache data, after I-O.
	mov	eax,[bx+RqXMS-@]     ;Use the cache buffer for DMA.
	stc			     ;Set carry for XMS buffer I-O.
SetBuf:	mov	[bx+IOAdr-@],eax  ;Set final 32-bit buffer addr.
	jnc s	DoDMA		  ;User buffer:  Start DMA & exit.
	test	dh,012h		  ;I-O using XMS:  Input request?
	jz s	UBufIn		  ;Yes, go start DMA input below.
	call	UBufMv		  ;Output:  Move user data to XMS.
	jnc s	DoDMA		  ;If no errors, start DMA & exit.
UdmaX:	ret			  ;XMS or I-O error!  Exit now!
;
; UltraDMA Controller I-O Address Table.
;
Ctl1Tbl	dw	0FFFFh		  ;Controller 1 DMA base address.
Ctl1Pri	dw	NPDATA		  ;   "Legacy" primary   address.
Ctl1Sec	dw	NSDATA		  ;   "Legacy" secondary address.
Ctl2Tbl	dw	0FFFFh		  ;Controller 2 I-O addresses.
Ctl2Pri	dw	APDATA
Ctl2Sec	dw	ASDATA
	dw 	21 dup (0FFFFh)	  ;Controller 3 thru 9 addresses.
CTLTSIZ	equ	(Ctl2Tbl-Ctl1Tbl) ;(Size of a controller's table).
CtlTEnd	equ	$		  ;(End of controller I-O tables).
;
; Continuation of the "UdmaIO" subroutine.
;
UBufIn:	call	DoDMA		    ;Buffered input:  Do all DMA.
	jc s	UdmaX		    ;If any DMA errors, exit now!
UBufMv:	movzx	ecx,[bx+RqSec-@].lb ;Get number of sectors to move.
	mov	esi,[bx+IOAdr-@]    ;Set move addresses for a read.
	mov	edi,esi
	xchg	edi,[bx+RqBuf-@]
	jmp	MovBuf		 ;Go move data to/from XMS and exit.
;
; Subroutine to initiate disk UltraDMA requests.
;
DoDMA:	mov	esi,(IOAdr-@)	 ;Move DMA command-list up to XMS.
@CLAddr	equ	[$-4].dwd
	mov	edi,[bx+PRDAd-@]
	mov	ecx,8
	call	MvData
	jc s	DoDMAE		;Command-list move ERROR?  Exit NOW!
DoDMA1:	mov	dx,[bx+DMAAd-@]	;Ensure any previous DMA is stopped!
	in	al,dx
	and	al,0FEh
	out	dx,al
	mov	di,[bx+IdeDA-@]	;Get disk's IDE base address.
	mov	al,[bx+DSCmd-@]	;Select our desired disk.
	and	al,0F0h
	lea	dx,[di+CDSEL]
	out	dx,al
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	mov	cx,((STARTTO*256)+FLT) ;Get timeout & "fault" mask.
	add	ch,es:[si]	       ;Set timeout limit in CH-reg.
	call	ChkRdy		     ;Await controller/disk ready.
DoDMAE:	jc s	DoDMAX		     ;If any errors, exit now!
	test	[bx+IOCmd-@].lb,012h ;Output request?
	jnz s	SetDMA		     ;Yes, initiate DMA transfer.
	mov	al,008h		;Get "DMA read" command bit.
SetDMA:	push	si		;Save BIOS timer pointer.
	mov	dx,[bx+DMAAd-@]	;Get DMA command-register addr.
	mov	si,(PRDAd-@)	;Get Physical-Region Descriptor addr.
	out	dx,al		;Reset DMA commands and set DMA mode.
	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Reset DMA status register.
	or	al,006h
	out	dx,al
	inc	dx		;Point to PRD address register.
	inc	dx
	outsd			;Set PRD pointer to our DMA address.
	mov	ax,001F7h	;Set IDE parameter-output flags.
NxtPar:	lea	dx,[di+CDATA+1]	;Point to IDE sector count reg. - 1.
IDEPar:	inc	dx		;Output all ten LBA48 parameter bytes.
	outsb			;(1st 4 overlayed by 2nd 4 if LBA28!).
	shr	ax,1		;More parameters to go in this group?
	jc s	IDEPar		;Yes, loop back and output next one.
	jnz s	NxtPar		;If first 4 done, go output last 6.
	pop	si		;Reload BIOS timer pointer.
	dec	dh		;"Legacy IDE" controller channel?
@DRQ:	jmp s	DMAGo		;No, forget about awaiting 1st DRQ.
	mov	dh,003h		;Get IDE alternate-status address.
	dec	dx		;(Primary-status address | 300h - 1).
ChkDRQ:	cmp	ch,es:[si]	;Too long without 1st data-request?
	je s	ErrDMA		;Yes?  Return carry and DMA error!
	in	al,dx		;Read IDE alternate status.
	and	al,DRQ		   ;Has 1st data-request arrived?
	jz s	ChkDRQ		   ;No, loop back and check again.
DMAGo:	mov	es:[si+HDI_OFS],bl ;Reset BIOS disk-interrupt flag.
	mov	dx,[bx+DMAAd-@]	   ;Point to DMA command register.
	in	al,dx		   ;Set DMA Start/Stop (starts DMA).
	inc	ax
	out	dx,al
ChkDMA:	inc	dx		;Point to DMA status register.
	inc	dx
	in	ax,dx		;Read DMA controller status.
	dec	dx		;Point back to DMA command register.
	dec	dx
	and	al,DMI+DME	;DMA interrupt or DMA error?
	jnz s	HltDMA		;Yes, halt DMA and check results.
	cmp	ch,es:[si]	   ;Has DMA transfer timed out?
	je s	HltDMA		   ;Yes?  Halt DMA & post timeout!
	cmp	es:[si+HDI_OFS],bl ;Did BIOS get a disk interrupt?
	je s	ChkDMA		   ;No, loop back & retest status.
	mov	al,DMI		;Set "simulated" interrupt flag.
HltDMA:	push	ax		;Save ending DMA status.
	in	al,dx		;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	pop	ax		;Reload ending DMA status.
	cmp	al,DMI		;Did DMA end with only an interrupt?
	jne s	ErrDMA		;No?  Return carry and DMA error!
	inc	dx		;Reread DMA controller status.
	inc	dx
	in	al,dx
	and	al,DME		;"Late" DMA error after DMA end?
	jnz s	ErrDMA		;Yes?  Return carry and DMA error!
	inc	cx		;Check "fault" and hard-error below.
ChkRdy:	lea	dx,[di+CSTAT]	;Read IDE primary status.
	in	al,dx
	cmp	ch,es:[si]	;Too long without becoming ready?
	je s	RdyErr		;Yes?  Go see what went wrong.
	test	al,BSY+RDY	;Controller or disk still busy?
	jle s	ChkRdy		;Yes, loop back and check again.
	and	al,cl		;Disk "fault" or hard-error?
	jnz s	HdwErr		;Yes?  Go see what went wrong.
DoDMAX:	ret			;All is well -- exit.
ErrDMA:	stc			;UltraDMA FAILURE!  Set carry flag.
	mov	al,DMAERR	;Set DMA error code, then bail out!
	ret
RdyErr:	test	al,BSY		;BAD News!  Did controller go ready?
	mov	ax,(256*CTLRERR)+DISKERR ;(Get not-ready error codes).
	jmp s	WhichE		;Go see which error code to return.
HdwErr:	test	al,FLT		;BAD News!  Is the disk "faulted"?
	mov	ax,(256*FAULTED)+HARDERR ;(Get hardware error codes).
WhichE:	jz s	Kaput		;If "zero", use AL-reg. return code.
	mov	al,ah		;Use AH-reg. return code of this pair.
Kaput:	stc			;Set carry flag (error!) and exit.
	ret
DDMALEN	equ	($-DoDMA1)	;(Length of common "DoDMA" logic).
NORALEN	equ	($+8-HDReq)	;("No Read Ahead" HMA logic length).
;
; Subroutine to set our disk Read-Ahead flag.
;
SetRA:	movzx	si,[bx+RqUNo-@].lb ;Get disk's LBA limit.
	shl	si,2
	mov	esi,cs:RALmt[si].dwd
@RALmt	equ	[$-2].lw
	or	esi,esi		  ;Read-Ahead O.K. on this disk?
	jz s	DoDMAX		  ;No, just exit above.
	or	[bx+RqLBA+1-@],bl ;Reading past disk's 1st 512-TB?
	jnz s	DoDMAX		  ;Yes, just exit above.
	mov	dh,[bx+RqLBA-@]	  ;Get LBA bits 8-39 for this
	mov	dl,[bx+RqLBA+3-@] ;  request as a 32-bit value.
	shl	edx,16
	mov	dh,[bx+RqLBA+2-@]
	mov	dl,[bx+RqLBA+5-@]
	cmp	edx,esi		  ;Would we read past end of disk?
	ja s	DoDMAX		  ;Yes, forget about Read-Ahead!
	inc	[bx+RAhdF-@].lb   ;Set input Read-Ahead flag.
	ret			  ;Exit.
;
; Read-Ahead Limit Table, LBA bits 8-39 for each BIOS disk.   Bits
;   40-47 are omitted to save space as 512-TB+ disks are unlikely!
;
RALmt	db  (MAXBIOS*4) dup (0)	;Disk Read-Ahead limit table.
HMALEN	equ	($-HDReq)	;(Length of all HMA caching logic).
;
; Non-cached Driver I-O Request Routine.   Driver Init overlays
;   this and all code below onto our cache logic if /B is given.
;
RLF	equ	($-HDReq)	;(Non-cached relocation factor).
SHDReq:	mov	dl,0BEh		;Mask out LBA & write request bits.
	and	dl,ah
	cmp	dl,002h		;CHS or LBA read/write request?
	jne s	SPass		;No, not for us -- pass this request.
	shl	ah,1		;LBA read/write request?
	jns s	SValSC		;No, go validate CHS sector count.
	mov	di,sp		;Point DI-reg. to current stack.
	push	ds		;Reload "DAP" segment into DS-reg.
	mov	ds,[di+2]
	cmp	[si].DapBuf,-1	;Check for 64-bit "DAP" I-O buffer.
	mov	al,[si].DapSC	;(Get all "DAP" I-O parameters).
	les	dx,[si].DapLBA1
	mov	di,es
	les	cx,[si].DapBuf
	mov	si,[si].DapLBA
	pop	ds
	jne s	SValSC		;If no 64-bit buffer, check sector ct.
SPass:	push	ss		;Return to low-memory "pass" routine.
	push	(PassRq-@)
	retf
SValSC:	dec	al		;Is sector count zero or over 128?
	js s	SPass		;Yes?  Let BIOS have this request.
	inc	ax		;Restore sector count -- LBA request?
	js s	SZroBX		;Yes, go zero driver's BX-reg.
	xchg	ax,cx		;CHS -- save request code and sectors.
	xor	di,di		;Reset upper LBA address bits.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	shr	al,6		;Set AX-reg. to starting cylinder.
	xchg	al,ah
	xchg	ax,dx			  ;Swap cylinder/head values.
	mov	al,cs:[bp+(SCHSec-@-RLF)] ;Get CHS sectors per head.
@SCHSec	equ	[$-2].lw
	or	al,al		;Legitimate disk CHS values?
	jz s	SPass		;No?  Let BIOS have this request!
	push	ax		;Save CHS sectors/head value.
	mul	ah		;Convert head to sectors.
	add	si,ax		;Add result to starting sector.
	pop	ax			  ;Reload sectors per head.
	mul	cs:[bp+(SCHSHd-@-RLF)].lb ;Convert cyl. to sectors.
@SCHSHd	equ	[$-2].lw
	mul	dx
	add	si,ax		;Add to head/sector value.
	adc	dx,di
	xchg	ax,bx		;Get buffer offset in AX-register.
	xchg	ax,cx		;Swap offset with request/sectors.
SZroBX:	xor	bx,bx		;Zero BX-reg. for relative logic.
	mov	[bx+VDSOf-@],cx ;Set VDS I-O buffer address.
	mov	[bx+VDSSg-@],es
	mov	[bx+LBA-@],si	;Set 48-bit logical block address.
	mov	[bx+LBA+2-@],dl
	mov	[bx+LBA2-@],dh
	mov	[bx+LBA2+1-@],di
	mov	bp,cs:[bp+(SCtlUn-@-RLF)] ;Get disk ctlr. & unit nos.
@SCtlUn	equ	[$-2].lw
	mov	si,offset (SCtlTbl-@-RLF) ;Point to ctlr. addresses.
@SCtlTb	equ	[$-2].lw
	mov	cx,0003Ch
	and	cx,bp
	add	si,cx
	shr	cx,1
	add	si,cx
	push	cs:[si].dwd	  ;Set controller base address and
	pop	[bx+DMAAd-@].dwd  ;  primary-channel data address.
	rcr	bp,2		  ;Primary channel I-O request?
	jnc s	SGetCm		  ;Yes, get LBA28 or LBA48 commands.
	add	[bx+DMAAd-@].lw,8 ;Use secondary DMA controller.
	mov	cx,cs:[si+4]	  ;Set secondary channel data addr.
	mov	[bx+IdeDA-@],cx
	nop			  ;(Unused filler).
SGetCm:	test	bp,010h		  ;Does disk hold > 128-GB of data?
	jnz s	SLBA48		  ;Yes, use LBA48 read/write command.
	shr	dx,12		  ;LBA28:  Reload & reset bits 24-27.
	xchg	dh,[bx+LBA2-@]
	or	ah,(DRCMD+1)	  ;Get LBA28 read/write command + 5.
	jmp s	SSetCm		  ;Go set LBA and IDE command bytes.
SLBA48:	shl	ah,3		  ;LBA48 -- get command as 020h/030h.
SSetCm:	shl	bp,1		  ;Shift master/slave into LBA cmds.
	mov	dl,(LBABITS/32)
	rcl	dl,5
	or	dl,dh		    ;Or in LBA28 bits 24-27, if any.
	mov	dh,005h		    ;Get final IDE command byte.
	xor	dh,ah		    ;(LBA28 C8h/CAh, LBA48 25h/35h).
	mov	[bx+DSCmd-@],dx	    ;Set LBA and IDE command bytes.
	mov	[bx+SecCt-@],al	    ;Set disk I-O sector count.
	mov	ah,0		    ;Set VDS & I-O buffer lengths.
	shl	ax,1
	mov	[bx+VDSLn+1-@],ax
	mov	[bx+IOLen+1-@],ax
	or	[bx+IOAdr-@].dwd,-1 ;Invalidate VDS buffer address.
	mov	ax,08103h	    ;Issue VDS "lock" request.
	mov	dx,0000Ch
	mov	di,(VDSLn-@)
	push	ds
	pop	es
	int	04Bh
	call	SXMRst		    ;Restore all critical settings.
	jc	SPass		    ;VDS error?  Pass this request!
	cmp	[bx+IOAdr-@].dwd,-1 ;Is VDS address still all-ones?
	jb s	SSetVF		    ;No, go set VDS "lock" flag.
	movzx	eax,[bx+VDSSg-@].lw ;VDS logic is NOT present --
	shl	eax,4		    ;  set 20-bit buffer address.
	add	eax,[bx+VDSOf-@]
	mov	[bx+IOAdr-@],eax
	xchg	bx,bx		     ;(Unused filler).
SSetVF:	adc	[bx+VLF-@],bl	     ;Set VDS "lock" flag from carry.
	test	[bx+IOAdr-@].lb,003h ;Is user buffer 32-bit aligned?
	jnz s	SSetBf		     ;No, use our XMS memory buffer.
	cmp	[bx+IOAdr-@].hw,009h ;Is DMA beyond our 640K limit?
	ja s	SSetBf		     ;Yes, use our XMS memory buffer.
	mov	cx,[bx+IOLen-@]	     ;Get (I-O length -1) + I-O addr.
	dec	cx
	add	cx,[bx+IOAdr-@].lw   ;Will I-O cross a 64K boundary?
	jnc s	SDrDMA		     ;No, do direct user-buffer DMA.
SSetBf:	mov	[bx+IOAdr-@].dwd,0   ;Set our 32-bit XMS buffer addr.
@SXBAdr	equ	[$-4].dwd
	test	[bx+IOCmd-@].lb,012h ;Buffered I-O:  Write request?
	jnz s	SBfOut		     ;Yes, use output routine below.
	call	SDoDMA		;Buffered input -- read data to XMS.
	jc s	SHDone		;If error, post return code and exit.
	call	SXMMov		;Move data from XMS to user buffer.
	jmp s	SHDone		;Go post any return code and exit.
SBfOut:	call	SXMMov		;Buffered output -- move data to XMS.
	jc s	SHDone		;If error, post return code and exit.
SDrDMA:	call	SDoDMA		;Do direct DMA or buffered output.
SHDone:	pushf			;I-O done:  Save error flag and code.
	push	ax
	shr	[bx+VLF-@].lb,1	;User's I-O  buffer "locked" by VDS?
	jnc s	SHDEnd		;No, return to low-memory exit rtn.
	mov	ax,08104h	;Set VDS "unlock" parameters.
	xor	dx,dx
	mov	di,(VDSLn-@)	;Issue VDS "unlock" request.
	push	ds
	pop	es
	int	04Bh
SHDEnd:	push	ss		;Return to low-memory exit routine.
	push	(IOExit-@)
	retf
;
; Non-cached Driver Controller and Disk Parameter Tables.
;
SCtlTbl	equ	$		    ;Controller addresses table.
SCtlUn	equ	SCtlTbl+(CTLTSIZ*9) ;Controller and unit table.
SCHSec	equ	SCtlUn+MAXBIOS	    ;CHS sectors/head table.
SCHSHd	equ	SCHSec+MAXBIOS	    ;CHS heads/cylinder table.
;
; Initialization Messages, Part 1.   After Init, this area becomes
;   the non-cached driver's Controller and Disk Parameter Tables.
;
PriMsg	db	'Primary-$'
SecMsg	db	'Secondary-$'
MstMsg	db	'master$'
SlvMsg	db	'slave$'
DNMsg	db	' disk is '
DName	equ	$
DNEnd	equ	DName+40
BCMsg	db	'BAD '
PCMsg1	db	'Controller at address '
PCMsg2	db	'    h I.D. '
PCMsg3	db	'        h.',CR,LF,'$'
NPMsg	db	'No PCI BIOS to use!',CR,LF,'$'
	db	0,0,0,0,0,0	;(Required filler).
;
; Command-List XMS Block for moving our DMA command-list up to XMS.
;
CLXBlk	dd	8		;Move length      (always 8 bytes).
	dw	0		;Source "handle"     (always zero).
CLXSA	dd	(IOAdr-@)	;Source address (seg. set by Init).
CLXDH	dw	0		;Destination handle  (set by Init).
CLXDA	dd	0		;Destination offset  (set by Init).
;
; Subroutine to move user I-O data to or from our XMS buffer.
;
SXMMov:	push	[bx+VDSSg-@].lw	 ;Save user-buffer address.
	push	[bx+VDSOf-@].lw
	push	[bx+VDSLn-@].dwd ;Get user-buffer length.
	mov	si,(RqBuf-@)	 ;Point to user-data XMS block.
	push	ss
	pop	ds
	pop	[si].dwd	;Set user-buffer length.
	lea	bx,[si+4]	;Set move indexes for a read.
	lea	di,[si+10]
	jz s	SXMSet		;Read request?
	xchg	bx,di		;No, swap indexes for a write.
SXMSet:	mov	[bx].lw,0	;Set XMS buffer "handle" & offset.
@XMHdl	equ	[$-2].lw
	mov	[bx+2].dwd,0
@XMOffs	equ	[$-4].dwd
	and	[di].lw,0	;Set user buffer "handle" to zero.
	pop	[di+2].dwd	;Set user-buffer address.
SXMGo:	mov	ah,00Bh		;Issue XMS "move memory" request.
	call	ss:@XEntry
	sub	al,1		;Zero AL-reg. if success, -1 if error.
SXMRst:	sti			;Restore all critical settings.
	cld
	push	ss
	pop	ds
	mov	bx,0		;Rezero BX-reg. but save carry flag.
SXMXit:	ret			;Exit.
;
; Subroutine to execute disk UltraDMA read and write requests.
;
SDoDMA:	mov	si,(CLXBlk-@-RLF) ;Move DMA command list up to XMS.
@CLXBlk	equ	[$-2].lw
	push	cs
	pop	ds
	call	SXMGo
	jc s	SXMXit		;Command-list move ERROR?  Exit NOW!
	db	090h,090h,090h	;(Unused filler).
SREQLEN	equ	($-SHDReq)	;(Length of HMA request logic).
SHMALEN	equ  ($+DDMALEN-SHDReq)	;(Length of all HMA driver logic).
;
; Initialization Flags.
;
EFlag	db	0		;"Use BIOS disk I-O" flag.
HFlag	db	0		;"Use HMA space" flag.
	db	0		;(Unused filler).
;
; Initialization Cache-Size Table.
;
CachSiz	dw	160		;  5-MB small cache (160 blocks).
	db	64		;    32K sectors per cache block
	db	6		;	 and sector-shift count.
	dw	128		;    128 binary-search midpoint.
	dw	3		;    3K cache-tables XMS memory.
	db	"   5"		;    Title message cache-size text.
LCBlks	dw	0FFFFh		; 10-MB to 4093-MB Read-Ahead cache.
	db	128,7		;    64K granularity values.
	dw	128		;    128 to 32768 search midpoint.
	dw	4		;    4K to 1024K cache-tables XMS.
	db	"  10"		;    Title message cache-size text.
;
; Initialization Variables.
;
HMASize	dw	HMALEN		;Total "free HMA" required.
StakSiz	dw	STACK		;Driver local-stack size.
BiosHD	db	0		;BIOS hard-disk count.
HDUnit	db	0		;Current BIOS unit number.
;
; Initialization Messages, Part 2.
;
TTLMsg	db	CR,LF,'UHDD, 7-07-2020.   '
TTL2	db	CR,LF,'$ -MB Cache.',CR,LF,'$ '
CPUMsg	db	'No 386+ CPU$'
NXMsg	db	'No XMS$'
CtlMsg	db	'IDE'
CtlrNo	db	'0 $'
BDMsg	db	'BIOS-handled disks:  '
BDCt	db	'0.',CR,LF,'$'
EDDMsg	db	'EDD$'
CHSMsg	db	'CHS'
UnitMsg	db	' data BAD, unit '
UnitNo	db	'A: ',CR,LF,'$'
IMEMsg	db	' I.D./mode error!',CR,LF,'$'
VEMsg	db	'VDS init error$'
NDMsg	db	'No disk to use$'
Suffix	db	'; UHDD not loaded!',CR,LF,'$'
;
; Initialization "Strategy" Routine.
;
I_Stra:	mov	cs:RqLBA.lw,bx	;Save DOS request-packet address.
	mov	cs:RqLBA.hw,es
	retf			;Exit & await "Device Interrupt".
	db	0		;(Unused filler).
;
; Initialization "Device Interrupt" Routine.
;
I_DevI:	pushf			;Entry -- save CPU flags.
	push	ds		;Save CPU segment registers.
	push	es
	push	ax		;Save needed 16-bit CPU registers.
	push	bx
	push	dx
	xor	ax,ax		;Get a zero for following logic.
	lds	bx,cs:RqLBA.dwd	;Point to DOS "Init" packet.
	cmp	[bx].RPOp,al	;Is this really an "Init" packet?
	jne s	I_Exit			;No?  Reload regs. and exit!
	mov	[bx].RPStat,RPDON+RPERR	;Set "Init" packet defaults
	mov	[bx].RPSeg,cs		;  and "null" driver length.
	and	[bx].RPLen,ax
	push	ds		;Set "Init" packet segment in ES-reg.
	pop	es
	push	cs		;NOW point DS-reg. to this driver!
	pop	ds
	push	sp		;See if CPU is an 80286 or newer.
	pop	ax
	cmp	ax,sp		;Did SP-reg. get saved "decremented"?
	jne s	I_Junk		;Yes?  CPU is an 8086/80186, TOO OLD!
	pushf			;80386 test -- save CPU flags.
	push	07000h		;Try to set NT|IOPL status flags.
	popf
	pushf			;Get resulting CPU status flags.
	pop	ax
	popf			;Reload starting CPU flags.
	test	ah,070h		;Did any NT|IOPL bits get set?
	jnz s	I_Sv32		;Yes, go save 32-bit CPU registers.
I_Junk:	mov	dx,(CPUMsg-@)	;Display "No 386+ CPU" error message.
	call	I_Msg
I_Quit:	mov	dx,(Suffix-@)	;Display "XHDD not loaded!" message.
	call	I_Msg
I_Exit:	jmp	I_Bye		;Go reload 16-bit regs. and exit.
I_VErr:	mov	VEMsg.dwd,eax	;Set prefix in "VDS init error" msg.
	mov	dx,(VEMsg-@)	;Point to "VDS init error" message.
I_Err:	push	dx		;Init ERROR!  Save message pointer.
	mov	ax,08104h	;"Unlock" this driver from memory.
	xor	dx,dx
	call	I_VDS
I_XDis:	mov	dx,Temp		;Load our XMS "handle" number.
	or	dx,dx		;Have we reserved any XMS memory?
	jz s	I_LDMP		;No, reload pointer & display msg.
	mov	ax,00D00h	;Unlock and "free" our XMS memory.
	push	dx
	call	I_XMS
	mov	ah,00Ah
	pop	dx
	call	I_XMS
I_LDMP:	pop	dx		;Reload error message pointer.
I_EMsg:	call	I_Msg		;Display desired error message.
	popad			;Reload all 32-bit CPU registers.
	jmp s	I_Quit		;Go display message suffix and exit!
EDDBuff equ	$		;(Start of EDD & search-index buffer).
I_Sv32:	pushad			;Save all 32-bit CPU registers.
	pushfd			;80486+ test:  Save CPU flags.
	push dword ptr 0240000h	;Try to set AC+ID flag bits.
	popfd
	pushfd
	pop	eax
	popfd                   ;Reload original CPU flags.
	shr	eax,19		;Was AC bit set, i.e. 80486+ CPU?
	jc s	I_CmdL		;Yes, go scan user's command line.
	mov	@386A.lb,0EBh	     ;Install 80386 "errata" code.
	mov	@386B.dwd,00F489067h
	shr	@MvRSC1,1	     ;Do XMS moves in 4K sections.
	shr	@MvRSC2,1
	inc	@MvPSC
I_CmdL:	les	si,es:[bx].RPCL	;Get command-line data pointer.
	xor	bx,bx		;Zero BX-reg. for relative logic.
I_NxtC:	cld			;Ensure FORWARD "string" commands!
	lods	es:[si].lb	;Get next command byte and bump ptr.
	or	al,al		;Command-line terminator?
	jz s	I_TrmJ		;Yes, check for an XMS manager.
	cmp	al,LF		;ASCII line-feed?
	je s	I_TrmJ		;Yes, check for an XMS manager.
	cmp	al,CR		;ASCII carriage-return?
I_TrmJ:	je	I_Term		;Yes, check for an XMS manager.
	and	al,0FDh		;Mask out bit 1 of this byte.
	cmp	al,'-'		;Dash or a slash?
	jne s	I_NxtC		;No, check next command-line byte.
	mov	al,es:[si]	;Get next command-line byte.
	and	al,0DFh		;Mask out 1st byte's lower-case bit.
	cmp	al,'A'		;Is byte an "A" or "a"?
	jne s	I_ChkB		;No, see if byte is "B" or "b".
	mov	Ctl1Pri.dwd,((ASDATA*65536)+APDATA) ;Reverse Legacy
	mov	Ctl2Pri.dwd,((NSDATA*65536)+NPDATA) ;  IDE addresses.
I_ChkB:	cmp	al,'B'		;Is byte a "B" or "b"?
	jne s	I_ChkE		;No, see if byte is "E" or "e".
	mov	EFlag.lb,bl		;Reset "Use BIOS I-O" flag.
	mov	[bx+DNam2-@].dwd,"$AS-"	;Set "UHDD-SA$" driver name.
	mov	[bx+VDSLn-@].lw,LMDvr	;Set non-cached variables.
	mov	HMASize.dwd,((STACKSA*65536)+SHMALEN)
I_ChkE:	cmp	al,'E'		  ;Is byte an "E" or "e"?
	jne s	I_ChkH		  ;No, see if byte is "H" or "h".
	cmp	[bx+DNam2+2-@],bx ;Non-cached driver requested?
	jne s	I_ChkH		  ;Yes, must ignore the /E switch!
	mov	EFlag,al	  ;Set "Use BIOS disk I-O" flag.
I_ChkH:	cmp	al,'H'		;Is byte an "H" or "h"?
	jne s	I_ChkZ		;No, see if byte is "Z" or "z".
	mov	HFlag,al	;Set "use HMA space" flag.
I_ChkZ:	cmp	al,'Z'		;Is byte a "Z" or "z"?
	jne s	I_ChkQ		;No, see if byte is "Q" or "q".
	mov	@MvPSC,bl	;Do "safe" protected-mode moves.
I_ChkQ:	cmp	al,'Q'		;Is byte a "Q" or "q"?
	jne s	I_ChkR		;No, see if byte is "R" or "r".
	mov	@DRQ.lb,075h	;Enable "data request" tests above.
I_ChkR:	cmp	al,'R'		;Is byte an "R" or "r"?
	jne s	I_ChkS		;No, see if byte is "S" or "s".
	mov	ax,es:[si+1]	;Get next 2 command-line bytes.
	mov	cx,15296	;Get 15-MB XMS memory size.
	cmp	ax,"51"		;Does user want 15-MB XMS reserved?
	je s	I_CkRA		;Yes, set memory size to reserve.
	mov	ch,(64448/256)	;Get 63-MB XMS memory size.
	cmp	ax,"36"		;Does user want 63-MB XMS reserved?
	jne s	I_NxtJ		;No, continue scan for a terminator.
I_CkRA:	mov	CLXDA.lw,cx	;Set desired XMS memory to reserve.
I_NxtJ:	jmp	I_NxtC		;Continue scanning for a terminator.
I_ChkS:	cmp	al,'S'		;Is byte an "S" or "s"?
	jne s	I_NxtJ		;No, continue scan for a terminator.
	mov	di,(LCBlks-@)	  ;Point to "large cache" block ct.
	mov	[di+8].dwd,"    " ;Reset "large cache" title bytes.
I_CkS0:	mov	[di].lw,08000h	  ;Invalidate cache-block count.
I_CkS1:	inc	si		;Bump ptr. past "S" or last digit.
	mov	al,es:[si]	;Get next command-line byte.
	cmp	al,'0'		;Less than a zero?
	jb s	I_NxtJ		;Yes, ignore it and continue scan.
	cmp	al,'9'		;More than a nine?
	ja s	I_NxtJ		;Yes, ignore it and continue scan.
	cmp	[di+8].lb,' '	;4 digits already found?
	jne s	I_CkS0		;Yes, set INVALID & keep scanning.
	shr	[di+8].dwd,8	;Shift "title" bytes 1 place left.
	mov	[di+11],al	;Insert next "title" message byte.
	and	ax,0000Fh	;Get cache digit's binary value.
	xchg	ax,[di]		;Multiply current block size by 10.
	mov	dx,10
	mul	dx
	add	[di],ax		;Add in next cache-size digit.
	jmp s	I_CkS1		;Go scan more cache-size digits.
I_Term:	mov	ax,04300h	;Inquire if we have an XMS manager.
	call	I_In2F
	mov	dx,(NXMsg-@)	;Point to "No XMS" error message.
	cmp	al,080h		;Is an XMS manager installed?
	jne	I_EMsg		;No?  Display error message and exit!
	mov	ax,04310h	;Get & save XMS "entry" address.
	call	I_In2F
	push	es
	push	bx
	pop	@XEntry.dwd
	mov	ax,03513h	;Get & save previous Int 13h vector.
	call	I_In21
	push	es
	push	bx
	pop	@Prev13.dwd
	mov	di,(LCBlks-@)	;Point to "large cache" block count.
	xor	cx,cx		;Set 5-MB cache table offset.
	cmp	[di].lw,10	;Less than 10-MB desired?
	jb s	I_MinM		;Yes, give user our 5-MB cache.
	mov	cl,12		;Set "large cache" table offset.
	cmp	[di].lw,MXCACHE	;Is cache size invalid or too big?
	jbe s	I_CSz1		;No, adjust cache-size parameters.
	mov	[di].lw,20	  ;Error!  Use 20-MB default cache.
	mov	[di+8].dwd,"02  "
I_CSz1:	mov	ax,16		  ;Set initial 16-MB cache limit.
I_CSz2:	cmp	[di],ax		;User cache size < current limit?
	jb s	I_CSz3		;Yes, set user cache-block count.
	shl	ax,1		;Double current cache-size limit.
	shl	[di+4].dwd,1	;Double variable cache parameters.
	jmp s	I_CSz2		;Go check user's cache size again.
I_CSz3:	shl	[di].lw,4	;Cache blocks = (16 * Megabytes).
I_MinM:	cmp	DNam2.lb,'-'	;Non-cached driver requested?
	je s	I_UseH		;Yes, see about loading in the HMA.
	mov	ax,ESWLEN	;Get length of /E HMA caching logic.
	cmp	EFlag,0		;Will we use BIOS disk I-O?
	jne s	I_Omit		;Yes, omit UltraDMA & Read-Ahead.
	or	cx,cx		;"Large cache" (64K blocks) wanted?
	jnz s	I_UseH		;Yes, see about loading in the HMA.
	mov	ax,NORALEN	;Get length of Read-Ahead logic.
I_Omit:	mov	HMASize,ax	;Omit UltraDMA and/or Read-Ahead.
	mov	@NoRA.lb,0B9h	;Omit setting Read-Ahead flag.
I_UseH:	shr	HFlag,1		;Will we load in the HMA?
	jz s	I_NoHM		;No, set upper/DOS memory size.
	push	cx		;Save user cache-table offset.
	mov	ax,04A01h	;Get total "free HMA" space.
	call	I_In2F
	pop	cx		;Reload user cache-table offset.
	cmp	bx,HMASize	;Enough "free HMA" for our driver?
	jae s	I_Size		;Yes, set up desired cache size.
I_NoHM:	mov	HFlag,ch	;Ensure NO use of HMA space!
	mov	ax,HMASize	;Set driver upper/DOS memory size.
	add	VDSLn.lw,ax
I_Size:	cmp	DNam2.lb,'-'	;Non-cached driver requested?
	je s	I_TTL		;Yes, go display driver "title".
	mov	si,(CachSiz-@)	;Point to desired cache-size table.
	add	si,cx
	movzx	eax,[si].lw	;Set binary-search limit index.
	mov	@TE1,ax
	mov	ax,[si+4]	;Set binary-search starting offsets.
	mov	@MP1,ax
	mov	@MP2,ax
	mov	ax,[si+2]	;Set cache "granularity" values.
	mov	@GRAN1,al
	mov	@GRAN2,al
	mov	@GRAN3,al
	xchg	ah,@GRSSC
	shr	ax,1		;Multiply number of cache blocks
	mul	[si].lw		;  times (sectors-per-block / 2).
	push	dx		;Get 32-bit cache data XMS size.
	push	ax
	pop	eax
	mov	RqBuf,eax	;Save cache XMS size for below.
	movzx	ecx,[si+6].lw	;Add in cache-tables XMS memory.
	add	eax,ecx
	add	RqXMS,eax	;Save total required XMS memory.
	mov	eax,[si+8]	;Set cache size in "title" message.
	mov	TTL2.dwd,eax
I_TTL:	mov	dx,(TTLMsg-@)	;Display driver "title" message.
	call	I_Msg
	movzx	cx,@MvPSC.lb	;"Safe" protected-mode moves desired?
	jcxz	I_HDCt		;Yes, set hard-disk count.
	or	cx,@TE1.lw	;Is cache size 2-GB or more?
	js s	I_HDCt			 ;Yes, set hard-disk count.
	mov	@MvPA.hb,(@MvPB-@MvPA-2) ;Disable protected-mode move
	db	0B8h			 ;  loops for improved speed.
	jmp s	$+(MvExit-@MvPC)
	mov	@MvPC.lw,ax
I_HDCt:	xor	eax,eax		;Zero EAX-reg. for 20-bit addressing.
	mov	es,ax		;Point ES-reg. to low memory.
	mov	al,es:[HDISKS]	;Save our BIOS hard-disk count.
	mov	BiosHD,al
	mov	ax,cs		;Set fixed driver segment values.
	mov	VDSSg.lw,ax
	mov	@HDMain.hw,ax
	mov	CLXSA.hw,ax
	shl	eax,4		;Set driver's VDS 20-bit address.
	mov	IOAdr,eax
	cli			     ;Avoid interrupts in VDS tests.
	test	es:[VDSFLAG].lb,020h ;"VDS services" active?
	jz s	I_REnI		     ;No, re-enable CPU interrupts.
	mov	ax,08103h	;VDS "lock" driver in memory forever.
	mov	dx,0000Ch
	call	I_VDS
	mov	dx,(VEMsg-@)	;Point to "VDS init error" message.
	jc	I_EMsg		;"Lock" error?  Display msg. & exit!
I_REnI:	sti			;Re-enable CPU interrupts.
	mov	eax,IOAdr	;Get final driver 32-bit address.
	add	GDTPAdr,eax	;Relocate "real mode" GDT base addr.
	add	@CLAddr,eax	;Relocate command-list source address.
	add	@WrkBf1,eax	;Relocate "working" buffer addresses.
	add	@WrkBf2,eax
	add	@CBAddr,eax	;Relocate cache-entry buffer address.
	cmp	EFlag,0		;Will we use BIOS disk I-O?
	jne s	I_XMem		;If so, omit scan for PCI-bus disks.
	xor	edi,edi		;Get PCI BIOS "I.D." code.
	mov	al,001h
	call	I_In1A
	cmp	edx,[NPMsg+3].dwd ;Is PCI BIOS V2.0C or newer?
	je s	I_ScnC		  ;Yes, scan UltraDMA controllers.
	inc	EFlag		;Avoid disk DPTE data checks below.
	mov	dx,(NPMsg-@)	;Display "No V2.0C+ PCI" message.
	call	I_Msg
	jmp s	I_XMem		;Go do setup for our XMS memory.
I_ScnC:	mov	al,LBABuf.lb	;Get next "interface bit" value.
	and	ax,00003h
	or	ax,PCISubC	;"Or" in subclass & current function.
	call	I_PCIC		;Test for specific PCI class/subclass.
	rol	LBABuf.lb,4	;Swap both "interface bit" values.
	mov	al,LBABuf.lb	;Load next "interface bit" value.
	or	al,al		;Both "interface" values tested?
	jns s	I_ScnC		;No, loop back and test 2nd one.
	add	PCISubC.lb,004h	;More PCI function codes to try?
	jnc s	I_ScnC		;Yes, loop back & try next function.
	test	al,001h		;Have we tested "Native PCI" ctlrs.?
	mov	LBABuf.lb,093h	;(Set "Native PCI" interface bits).
	jz s	I_ScnC		;No, loop back and test them, also.
I_XMem:	mov	edx,CLXDA	;Get "reserved" XMS memory size.
	or	edx,edx		;Any "reserved" XMS desired?
	jz s	I_XGet		;No, get driver's actual XMS memory.
	call	I_XMSM		;Request all "reserved" XMS memory.
	jnz s	I_XErr		;If error, display message and exit!
	mov	Temp,dx		;Save reserved-XMS "handle" number.
I_XGet: mov     edx,RqXMS       ;Request all necessary XMS memory.
	call	I_XMSM
	jz s	I_XFre		;If no errors, "free" reserved XMS.
I_XErr:	mov	eax," SMX"	;BAD News!  Get "XMS" msg. prefix.
	jmp	I_VErr		;Go display "XMS init error" & exit!
I_XFre:	mov	XMSHndl,dx	;Save our XMS "handle" numbers.
	mov	CLXDH,dx
	mov	@XMHdl,dx
	xchg	Temp,dx		;Swap with reserved "handle" no.
	or	dx,dx		;Any XMS reserved by the /R switch?
	jz s	I_XLok		;No, go "lock" our XMS memory.
	mov	ah,00Ah		;"Free" our reserved XMS memory.
	call	I_XMS
	jnz s	I_XErr		;If error, display message and exit!
I_XLok:	mov	ah,00Ch		;"Lock" our driver's XMS memory.
	mov	dx,Temp
	call	I_XMS
	jnz s	I_XErr		;If error, display message and exit!
	shl	edx,16		;Get unaligned 32-bit buffer address.
	or	dx,bx
	mov	esi,edx		;Initialize command-list XMS offset.
	mov	eax,edx		;Copy 32-bit address to EAX-reg.
	jz s	I_XBAd		;Any low-order XMS buffer "offset"?
	mov	ax,0FFFFh	;Yes, align address to an even 64K.
	inc	eax
I_XBAd:	mov	@XBAddr,eax	;Save aligned "main buffer" address.
	mov	@SXBAdr,eax
	add	@XBase,eax	;Initialize cache-data base address.
	mov	cx,ax		;Get buffer "offset" in XMS memory.
	sub	cx,dx
	mov	@XMOffs.lw,cx	;Set offset in "SXMMov" subroutine.
	mov	edx,000010000h	;Put command-list after XMS buffer.
	jcxz	I_PRDA		;Is buffer already on a 64K boundary?
	or	edx,-32		;No, put command-list before buffer.
	dec	@XBase.hw	;Decrement cache-data base by 64K.
I_PRDA:	add	eax,edx		;Set our 32-bit PRD address.
	mov	PRDAd,eax
	sub	eax,esi		;Set final command-list XMS offset.
	mov	CLXDA,eax
	cmp	DNam2.lb,'-'	;Non-cached driver requested?
	je	I_HDCh		;Yes, check BIOS hard-disk count.
	mov	eax,RqBuf	;Get needed cache XMS in 1K blocks.
	shl	eax,10		;Convert from 1K blocks to bytes.
	add	eax,@XBase	;Add in XMS cache-data base address.
	mov	@CTBas1,eax	;Set XMS cache-table base addresses.
	add	@CTBas2,eax
	add	@CTBas3,eax
	xchg	eax,edx		;Save data-table address in EDX-reg.
	movzx	eax,@TE1	;Get binary-search table size.
	shl	eax,1
	mov	ecx,eax		;Save search-table size in ECX-reg.
	shl	eax,1		;Get offset to binary-search buffer.
	add	eax,ecx
	shl	eax,1
	add	eax,edx
	mov	@STBuff,eax	;Set binary-search buffer address.
	add	eax,ecx		;Set XMS search table addresses.
	mov	@STAdr1,eax
	mov	@STAdr2,eax
	mov	@STAdr3,eax
	mov	ah,005h		;Issue "A20" local-enable request.
	call	I_XMS
	jnz	I_A20E		;If any "A20" error, bail out NOW!
	mov	dx,@TE1		;Initialize search-table count.
	xor	bp,bp		;Initialize search-table index.
I_RST1:	mov	ax,bp		;Set next 120 search-table indexes
	xor	ecx,ecx		;  in init tables & messages area.
	mov	cl,120
	mov	si,(EDDBuff-@)
I_RST2:	mov	[si],ax
	inc	ax
	inc	si
	inc	si
	loop	I_RST2
	xor	esi,esi		;Set 32-bit move source address.
	mov	si,(EDDBuff-@)	;(Offset of our indexes buffer +
	add	esi,IOAdr	;  32-bit driver "base" address).
	movzx	edi,bp		;Set 32-bit move destination addr.
	shl	edi,1		;(2 * current "reset" index +
	add	edi,@STAdr1	;  binary-search table address).
	mov	bp,ax		;Update next cache-table index.
	mov	cl,120		;Get 120-word move length.
	cmp	cx,dx		;At least 120 words left to go?
	jbe s	I_RST3		;Yes, use full 120-word count.
	mov	cx,dx		;Use remaining word count.
I_RST3:	shl	cx,1		;Convert word count to byte count.
	push	dx		;Save move count and cache index.
	push	bp
	xor	bx,bx		;Move 120 indexes into search table.
	call	MvData
	pop	bp		;Reload cache index and move count.
	pop	dx
	jnc s	I_RST4		;If no XMS errors, check move count.
	mov	ah,006h		;BAD News!  Do "A20" local disable.
	call	I_XMS
	jmp	I_XErr		;Go display "XMS init error" & exit!
I_RST4:	sub	dx,120		;More search-table indexes to set?
	ja s	I_RST1		;Yes, loop back and do next group.
	mov	ah,006h		;Issue "A20" local-disable request.
	call	I_XMS
I_FDsk:	xor	ax,ax		;Point ES-reg. to low memory.
	mov	es,ax
	mov	al,es:[HDWRFL]	;Get BIOS "hardware installed" flag.
	test	al,001h		;Any diskettes on this PC?
	jz s	I_HDCh		;No, scan for available hard-disks.
	mov	al,011h		;Use diskette media-change bits
	and	al,es:[MCHDWFL]	;  for our number of diskettes.
I_FScn:	test	al,001h		;Can next unit signal media-changes?
	jz s	I_FMor		;No?  CANNOT use this old "clunker"!
	push	ax		;Save our diskette counter.
	mov	al,0C0h		;Get "diskette" device-type flags.
	call	I_CHSD		;Get and save diskette's CHS values.
	pop	ax		;Reload our diskette counter.
	inc	@LastU		;Bump our unit-table index.
I_FMor:	inc	HDUnit		;Bump BIOS unit number.
	inc	UnitNo.lb	;Bump error-message unit number.
	shr	al,4		;Another diskette to check?
	jnz s	I_FScn		;Yes, loop back and do next one.
I_HDCh:	cmp	BiosHD,0	;Any BIOS hard-disks to use?
	je	I_AnyD		;No, test for non-cached driver.
	mov	HDUnit,080h	;Set 1st BIOS hard-disk unit.
I_Next:	mov	ah,HDUnit	;Set unit no. in error message.
	mov	cx,2
	mov	si,(UnitNo-@)
	call	I_HexA
	mov	[si].lb,'h'
	mov	ah,041h		;Get EDD "extensions" for this disk.
	mov	bx,055AAh
	call	I_In13
	jc s	I_DRdy		;If none, check if disk is "ready".
	cmp	bx,0AA55h	;Did BIOS "reverse" our entry code?
	jne s	I_DRdy		;No, check if this disk is "ready".
	test	cl,007h		;Any "EDD extensions" for this disk?
	jz s	I_DRdy		;No, check if this disk is "ready".
	push	cx		;Save disk "EDD extensions" flags.
	mov	si,(EDDBuff-@)	;Point to "EDD" input buffer.
	mov	[si].lw,30	;Set 30-byte buffer size.
	or	[si+26].dwd,-1	;Init NO "DPTE" data, for a bad BIOS!
	mov	ah,048h		;Get this disk's "EDD" parameters.
	call	I_In13
	pop	cx		;Reload disk "EDD extensions" flags.
	jc s	I_Drdy		;If error, check if disk is "ready".
	test	[si+2].lb,004h	;Is this HARD disk flagged "removable"?
	jnz	I_IgnD		;If so, we have NO logic to SUPPORT IT!
	cmp	EFlag,0		;Will we use BIOS disk I-O?
	jne s	I_DRdy		;Yes, go see if this disk is "ready".
	cmp	[si].lw,30	;Did we get at least 30 bytes?
	jb s	I_DRdy		;No, check if this disk is "ready".
	test	cl,004h		;Does this disk provide "DPTE" data?
	jz s	I_DRdy		;No, check if this disk is "ready".
	cmp	[si+26].dwd,-1	;"Null" drive parameter-table pointer?
	je s	I_DRdy		;Yes, check if this disk is "ready".
I_DPTE:	les	si,[si+26]	;Get this disk's "DPTE" pointer.
	mov	bx,15		;Calculate "DPTE" checksum.
	xor	cx,cx
I_CkSm:	add	cl,es:[bx+si]
	dec	bx
	jns s	I_CkSm
	jcxz	I_EDOK		;If checksum O.K., use parameters.
	mov	dx,(EDDMsg-@)	;Display "EDD data BAD" message.
	call	I_Msg
	mov	dx,(UnitMsg-@)
	call	I_Msg
I_DRdy:	mov	ah,010h		;Non-UltraDMA:  See if disk is ready.
	call	I_In13
	jc	I_IgnD		;If not ready, must IGNORE this disk!
	call	I_CHSB		;Get and save disk's CHS values.
	jmp	I_NoUJ		;Go bump non-UltraDMA disk count.
I_EDOK:	mov	di,(EDDBuff-@)	;Get 48-bit disk sector count.
	mov	dx,[di+20]
	mov	eax,[di+16]
	mov	di,@LastU	;Get disk's parameter-table index.
	or	dx,dx		;Anything in sector count bits 32-47?
	jnz s	I_LB48		;Yes, this is "surely" an LBA48 disk.
	or	eax,eax		;Anything in sector count bits 0-31?
	jz s	I_CHS		;No, just get disk's CHS values.
	test	eax,0F0000000h	;Anything in sector count bits 28-31?
	jz s	I_DoRA		;No, see if disk can use Read-Ahead.
I_LB48:	or	TypeF[di],040h	;Set this disk's "LBA48" flag.
I_DoRA:	cmp	@Gran1,128	;Setting a 10-MB disk cache or more?
	jb s	I_CHS		;No, just get disk's CHS values.
	mov	al,dl		;Get bits 8-39 of disk sector count.
	ror	eax,8
	or	dh,dh		;Anything in sector-count bits 40-47?
	jz s	I_RLmt		;No, go set disk's Read-Ahead limit.
	or	eax,-1		;Use max. sector count for bits 8-39.
I_RLmt:	dec	eax		;Allow 256 sectors before disk end.
	shl	di,2		;Set this disk's Read-Ahead limit.
	mov	RALmt[di].dwd,eax
I_CHS:	call	I_CHSB		;Get and save this disk's CHS values.
	mov	bx,00010h	;Initialize IDE unit number index.
	and	bl,es:[si+4]
	shr	bl,4
	mov	ax,es:[si]	;Get disk's IDE base address.
	mov	CtlrNo.lb,'0'	;Reset display controller number.
	mov	si,(Ctl1Tbl-@)	;Point to IDE address table.
I_ITbl:	cmp	[si].lw,-1	;Is this IDE table active?
	je s	I_NoUJ		;No, go bump non-UltraDMA disk count.
	cmp	ax,[si+2]	;Primary channel disk?
	je s	I_ChMS		;Yes, set disk channel and unit.
	inc	bx		;Adjust index for secondary channel.
	inc	bx
	cmp	ax,[si+4]	;Secondary channel disk?
	je s	I_ChMS		;Yes, set disk channel and unit.
	inc	bx		;Adjust values for next controller.
	inc	bx
	inc	CtlrNo.lb
	add	si,CTLTSIZ
	cmp	si,(CtlTEnd-@)	;More IDE controllers to check?
	jb s	I_ITbl		;Yes, loop back and check next one.
I_NoUJ:	jmp s	I_NoUD		;Go bump non-UltraDMA disk count.
I_ChMS:	push	bx		;Save disk's caching unit number.
	mov	IdeDA,ax	;Save disk's base IDE address.
	add	ax,CDSEL	;Point to IDE device-select register.
	xchg	ax,dx
	mov	al,bl		;Get drive-select command byte.
	shl	al,4
	or	al,LBABITS
	out	dx,al		;Select master or slave disk.
	push	ax		;Save drive-select and caching unit.
	push	bx
	mov	dx,(CtlMsg-@)	;Display IDE controller number.
	call	I_Msg
	pop	ax		;Reload caching unit number.
	mov	dx,(PriMsg-@)	;Point to "Primary" channel name.
	test	al,002h		;Primary channel disk?
	jz s	I_PRNm		;Yes, display "Primary" channel.
	mov	dx,(SecMsg-@)	;Point to "Secondary" channel name.
I_PRNm:	call	I_Msg		;Display disk's IDE channel name.
	pop	ax		;Reload drive-select byte.
	mov	dx,(MstMsg-@)	;Point to "master" disk name.
	test	al,010h		;Is this disk the master?
	jz s	I_MSNm		;Yes, display "master" name.
	mov	dx,(SlvMsg-@)	;Point to "slave" disk name.
I_MSNm:	call	I_Msg		;Display disk's master/slave name.
	call	I_ValD		;Validate disk as an UltraDMA unit.
	pop	ax		;Reload caching unit number.
	jc s	I_UDEr		;If any errors, display message.
	mov	bx,@LastU	     ;Change disk's type to "IDE"
	and	[bx+TypeF-@].lb,040h ;  and include channel/unit.
	or	[bx+TypeF-@],al
	jmp s	I_More		;Go check for any more BIOS disks.
I_UDEr:	call	I_Msg		;NOT UltraDMA -- Display error msg.
I_NoUD:	cmp	DNam2.lb,'$'	;Loading the caching driver?
	jne s	I_IgnD		;No, ignore disk and scan for more.
	inc	BDCt		;Bump number of non-UltraDMA disks.
	cmp	BDCt,'9'	;Over 9 non-UltraDMA hard disks?
	jbe s	I_More		;No, check for more disks.
	mov	BDCt.lw,"+9"	;Set 9+ non-UltraDMA count.
I_More:	inc	@LastU		;Bump our unit-table index.
I_IgnD:	inc	HDUnit		;Bump BIOS unit number.
	cmp	@LastU,MAXBIOS	;More entries in our units table?
	jae s	I_AnyN		;No, check for non-UltraDMA disks.
	dec	BiosHD		;More BIOS disks to check?
	jnz	I_Next		;Yes, loop back and do next one.
I_AnyN:	cmp	BDCt.lb,'0'	;Do we have any non-UltraDMA disks?
	je s	I_AnyD		;No, test for non-cached driver.
	mov	dx,(BDMsg-@)	;Display "BIOS-handled disks" msg.
	call	I_Msg
I_AnyD:	cmp	DNam2.lb,'$'	;Loading the caching driver?
	je s	I_HMA		;Yes, see if we will load in the HMA.
	mov	dx,(NDMsg-@)	;Point to "No disk to use" message.
	cmp	@LastU,0	;Did we find any hard-disks to use?
	jz	I_Err		;No?  Display error message and exit!
I_HMA:	shr	HFlag,1		;Will we load in the HMA?
	jz	I_SADv		;No, set non-cached dvr. if needed.
	mov	ax,04A02h	;Request needed memory in the HMA.
	mov	bx,HMASize
	call	I_In2F
	push	es		;Get 32-bit HMA segment/offset addr.
	push	di
	pop	eax
	mov	@HDMain,eax	;Set our HMA entry-routine address.
	inc	eax		;Is our HMA address = -1 (no HMA)?
	jnz s	I_HMA1		;No, do all needed HMA adjustments.
	mov	eax," AMH"	;Get "HMA" error-message prefix.
I_HMAX:	jmp	I_VErr		;Go display error message & exit!
I_HMA1:	xor	ecx,ecx		;Get 32-bit HMA segment address.
	mov	cx,es
	shl	ecx,4
	movzx	eax,di		;Adjust real-mode GDT base addr.
	add	ax,(GDT-HDReq)
	add	eax,ecx
	mov	GDTPAdr,eax
	lea	ax,[di-(HDReq-@)] ;Get caching HMA logic offset.
	add	@TypeF,ax	  ;Adjust disk "CHS tables" offsets.
	add	@CHSec,ax
	add	@CHSHd,ax
	add	@GDTP,ax	;Adjust XMS move routine offsets.
	add	@MvDesc,ax
	add	@CtlTbl,ax	;Adjust disk controller-table ptr.
	add	@RALmt,ax	;Adjust Read-Ahead limit table ptr.
	add	@SCtlTb,ax	;Adjust all non-cached variables.
	add	@SCtlUn,ax
	add	@SCHSec,ax
	add	@SCHSHd,ax
	add	@CLXBlk,ax
	call	I_SetS		;Set non-cached driver if needed.
	mov	ah,005h		;Issue "A20" local-enable request.
	call	I_XMS
I_A20E:	mov	eax," 02A"	;Get "A20" error-message prefix.
	jnz s	I_HMAX		;If any "A20" error, bail out FAST!
	mov     cx,HMASize      ;Move required logic up to the HMA.
	mov	si,(HDReq-@)
	les	di,@HDMain
	rep	movsb
	mov	ah,006h		;Issue "A20" local-disable request.
	call	I_XMS
	jmp s	I_Done		;Go post "Init" packet results.
I_SADv:	call	I_SetS		;Set non-cached driver if needed.
I_Done:	xor	ax,ax		;Done!  Load & reset driver length.
	xchg	ax,VDSLn.lw
	les	bx,RqLBA.dwd	;Set results in DOS "Init" packet.
	mov	es:[bx].RPLen,ax
	mov	es:[bx].RPStat,RPDON
	mov	@Stack,ax	;Set stack address in "Entry" logic.
	xor	bx,bx		;Clear driver's stack (helps debug).
	xchg	ax,bx
	mov	cx,StakSiz
I_CLS:	dec	bx
	mov	[bx],al
	loop	I_CLS
	mov	ax,02513h	;"Hook" this driver into Int 13h.
	mov	dx,(Entry-@)
	int	021h
	popad			;Reload all 32-bit CPU registers.
I_Bye:	pop	dx		;Reload 16-bit registers we used.
	pop	bx
	pop	ax
	pop	es		;Reload CPU segment registers.
	pop	ds
	popf			;Reload CPU flags and exit.
	retf
;
; Init subroutine to convert 4-digit hex values to ASCII for display.
;
I_Hex:	mov	cx,4		;Set 4-digit count.
I_HexA:	rol	ax,4		;Get next hex digit in low-order.
	push	ax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe s	I_HexB		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
I_HexB:	add	al,030h		;Convert digit to ASCII.
	mov	[si],al		;Store next digit in message.
	inc	si		;Bump message pointer.
	pop	ax		;Reload remaining digits.
	loop	I_HexA		;If more digits to go, loop back.
I_HexX:	ret			;Exit.
;
; Init subroutine to test for and set up PCI disk controllers.
;
I_PCIC:	mov	IdeDA,ax	;Save subclass & function codes.
	and	LBABuf.hw,0	   ;Reset PCI device index.
I_PCI1:	cmp	PCITbl,(CtlTEnd-@) ;More space in address tables?
 	jae s	I_HexX		   ;No, go exit above.
	mov	ax,IdeDA	;Test PCI class 1, subclass/function.
	mov	ecx,000010003h	;(Returns bus/device/function in BX).
	xchg	ax,cx
	mov	si,LBABuf.hw
	call	I_In1A
	jc s	I_HexX		;Controller not found -- exit above.
	inc	LBABuf.hw	;Bump device index for another test.
	xor	di,di		;Get controller Vendor & Device I.D.
	call	I_PCID
	push	ecx		;Save Vendor and Device I.D.
	mov	di,32		;Save DMA controller base address.
	call	I_PCID
	xchg	ax,cx
	and	al,0FCh
	mov	DMAAd,ax
	mov	si,(PCMsg2-@)	;Set controller address in msg.
	call	I_Hex
	mov	si,(PCMsg3-@)	;Set vendor I.D. in message.
	pop	ax
	call	I_Hex
	pop	ax		;Set Device I.D. in message.
	call	I_Hex
	mov	di,4		;Get low-order PCI command byte.
	call	I_PCID
	not	cl		;Get "Bus Master" & "I-O Space" bits.
	and	cl,005h		;Is controller using both BM and IOS?
	jz s	I_PCI2		;Yes, save this controller's data.
	mov	dx,(BCMsg-@)	;Display "BAD controller" message.
	call	I_Msg
I_PCIJ:	jmp s	I_PCI1		;Go test for more same-class ctlrs.
I_PCI2:	mov	si,PCITbl	;Get current I-O address table ptr.
	mov	ax,DMAAd	;Set controller DMA base address.
	mov	[si],ax
	test	LBABuf.lb,001h	;"Native PCI" controller?
	jz s	I_PCI3		;No, go display controller data.
	mov	di,16		;Set primary-channel base address.
	call	I_PCID
	and	cl,0FCh
	mov	[si+2],cx
	mov	di,24		;Set secondary-channel base address.
	call	I_PCID
	and	cl,0FCh
	mov	[si+4],cx
I_PCI3:	mov	dx,(CtlMsg-@)	;Display all controller data.
	call	I_Msg
	mov	dx,(PCMsg1-@)
	call	I_Msg
	inc	CtlrNo.lb	;Bump display controller number.
	add	PCITbl,CTLTSIZ	;Bump controller address-table ptr.
	jmp s	I_PCIJ		;Go test for more same-class ctlrs.
;
; Init subroutine to finalize our non-cached driver.
;
I_SetS:	cmp	DNam2.lb,'$'	;Loading our caching driver?
	je s	I_SSAX		;Yes, just exit below.
	cld			;Ensure FORWARD "string" commands!
	push	ds		;Set ES-reg. same as DS-reg.
	pop	es
	mov	cx,(CTLTSIZ*9)	;Copy controller/disk data from
	mov	si,(Ctl1Tbl-@)	;  caching to non-cached driver.
	mov	di,(SCtlTbl-@)
	rep	movsb
	mov	cl,(MAXBIOS*3)
	mov	si,(TypeF-@)
	rep	movsb
	mov	cx,SREQLEN	;Move non-cached driver logic
	mov	si,(SHDReq-@)	;  down to low memory for use.
	mov	di,(HDReq-@)
	rep	movsb
	mov	cl,DDMALEN
	mov	si,(DoDMA1-@)
	rep	movsb
I_SSAX:	ret			;Exit.
;
; Init subroutine to "validate" an UltraDMA hard-disk.
;
I_ValD:	xor	bx,bx		;Zero BX-reg. for timeout checks.
	mov	bp,[bx+IdeDA-@]	;Get drive's IDE address in BP-reg.
	lea	dx,[bp+CCMD]	;Point to IDE command reg.
	mov	al,0ECh		;Issue "Identify ATA" command.
	out	dx,al
	mov	ah,CMDTO	;Use 500-msec command timeout.
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	add	ah,es:[si]	;Set timeout limit in AH-reg.
I_VDTO:	cmp	ah,es:[si]	;"Identify" command timeout?
	je s	I_DErr		;Yes?  Exit & display error message.
	lea	dx,[bp+CSTAT]	;Point to IDE status reg.
	in	al,dx		;Read IDE primary status.
	test	al,BSY+DRQ	;Controller still busy, or no DRQ?
	jle s	I_VDTO		;If either, loop back & test again.
	test	al,ERR		;Did "Identify" command give errors?
	jz s	I_PIO		;No, read I.D. data using PIO.
I_DErr:	mov	dx,(IMEMsg-@)	;Point to "I.D./mode error" message.
	stc			;Set carry flag (error) and exit.
	ret
I_PIO:	mov	dx,bp		;Point to IDE data register.
	mov	cx,28		;Skip I.D. bytes 0-53 and
I_Skp1:	in	ax,dx		;  read I.D. bytes 54-55.
	loop	I_Skp1
	push	cs		;Point to disk-name message.
	pop	es
	mov	di,(DName-@)
	mov	cl,26		;Read & swap disk name into message.
I_RdNm:	xchg	ah,al		;(I.D. bytes 54-93.  Bytes 94-105 are
	stosw			;  also read but are ignored.   Bytes
	in	ax,dx		;  106-107 are left in the AX-reg.).
	loop	I_RdNm
	xchg	ax,bx		;Save "UltraDMA valid" flag in BL-reg.
	mov	cl,35		;Skip I.D. bytes 108-175 &
I_Skp2:	in	ax,dx		;  read I.D. bytes 176-177.
	loop	I_Skp2
	mov	bh,ah		;Save "UltraDMA mode" flags in BH-reg.
	mov	cl,167		;Skip remaining I.D. data.
I_Skp3:	in	ax,dx
	loop	I_Skp3
	shr	bl,1		;Get "UltraDMA valid" flag in bit 2.
	and	bl,002h		;UltraDMA bits valid for this disk?
	jz s	I_DErr		;No?  Exit & display error message.
	mov	di,(Modes-@)	;Point to UltraDMA mode table.
	or	bh,bh		;Can disk do UltraDMA mode 0?
	jz s	I_DErr		;No?  Exit & display message!
I_NxtM:	cmp	bh,bl		;Can disk do next UltraDMA mode?
	jb s	I_GotM		;No, use current mode.
	inc	di		;Point to next mode table value.
	inc	di
	shl	bl,1		;More UltraDMA modes to test?
	jnz s	I_NxtM		;Yes, loop back.
I_GotM:	mov	si,(DNEnd-@)	;Point to end of disk name.
I_NxtN:	cmp	si,(DName-@)	;Are we at the disk-name start?
	je s	I_Name		;Yes, disk name is all spaces!
	dec	si		;Decrement disk name pointer.
	cmp	[si].lb,' '	;Is this name byte a space?
	je s	I_NxtN		;No, continue scan for non-space.
	inc	si		;Skip non-space character.
	mov	[si].lw," ,"	;End disk name with comma & space.
	inc	si
	inc	si
I_Name:	mov	[si].dwd,"-ATA"	;Set "ATA-" after drive name.
	add	si,4
	mov	ax,[di]		;Set UltraDMA "mode" in message.
	mov	cl,00Fh
	and	cl,al
	call	I_HexA
	mov	[si].dwd,0240A0D2Eh ;Set message terminators.
	mov	dx,(DNMsg-@)	    ;Display mfr. name/"mode" & exit.
	jmp s	I_Msg
;
; Init subroutine to set BIOS CHS data for a hard-disk or diskette.
;
I_CHSB:	mov	al,080h		;Get "BIOS disk" device-type.
I_CHSD:	push	ax		;Save unit's device-type flag.
	mov	ah,008h		;Get BIOS CHS data for this unit.
	call	I_In13
	jc s	I_CHSE		;If BIOS error, zero CHS sectors.
	and	cl,03Fh		;Get sectors/head value (low 6 bits).
	jz s	I_CHSE		;If zero, ERROR!  Zero CHS sectors.
	inc	dh		;Get heads/cylinder (BIOS value + 1).
	jnz s	I_CHST		;If non-zero, save data in our table.
I_CHSE:	xor	cl,cl		;Error!  Zero unit's CHS sectors.
I_CHST:	mov	di,@LastU	;Point to "active units" table.
	mov	al,HDUnit	;Set BIOS unit number in our table.
	mov	[di+Units-@],al
	mov	[di+CHSec-@],cl ;Set unit's CHS data in our table.
	mov	[di+CHSHd-@],dh
	pop	ax		;Reload and set device-type flag.
	or	[di+TypeF-@],al
	mov	dx,(CHSMsg-@)	;Point to "CHS data BAD" message.
	or	cl,cl		;Valid CHS values for this unit?
	jz s	I_Msg		;No?  Display error msg. and exit!
	ret			;All is well -- exit.
;
; Init subroutines to issue "external" calls.
;
I_XMSM:	mov	ah,009h		;Get XMS V2.0 "allocate" command.
	push    edx		;Get high-order XMS request size.
	pop	cx
	pop	cx		;Requesting 64-MB+ of XMS memory?
	jcxz	I_XMS		;No, request desired XMS now.
	mov	ah,089h		;Use XMS V3.0 "allocate" command.
I_XMS:	call	@XEntry.dwd	;XMS -- issue desired request.
	dec	ax		;Zero AX-reg. if success, -1 if error.
	jmp s	I_IntX		;Restore critical settings, then exit.
I_VDS:	mov	di,(VDSLn-@)	;VDS -- Point to parameter block.
	push	cs
	pop	es
	int	04Bh		;Execute VDS "lock" or "unlock".
	jmp s	I_IntX		;Restore critical settings, then exit.
I_In13:	mov	dl,HDUnit	;BIOS data -- set BIOS unit in DL-reg.
	int	013h		;Issue BIOS data interrupt.
	jmp s	I_IntX		;Restore critical settings, then exit.
I_PCID:	push	bx		;Save PCI bus/device/function codes.
	push	si		;Save IDE address-table pointer.
	mov	al,00Ah		;Set "PCI doubleword" request code.
	call	I_In1A		;Get desired 32-bit word from PCI.
	pop	si		;Reload IDE address-table pointer.
	pop	bx		;Reload PCI bus/device/function.
	ret			;Exit.
I_In1A:	mov	ah,0B1h		;Issue PCI BIOS interrupt.
	int	01Ah
	jmp s	I_IntX		;Restore critical settings, then exit.
I_Msg:	mov	ah,009h		;Message -- Get "display string" code.
I_In21:	int	021h		;DOS -- Issue desired DOS request.
	jmp s	I_IntX		;Restore critical settings, then exit.
I_In2F:	int	02Fh		;"Multiplex" -- issue XMS/HMA request.
I_IntX:	sti			;Restore all critical settings & exit.
	cld
	push	cs
	pop	ds
	ret
CODE	ends
	end
