/*
// Program:  Free FDISK
// Written By:  Brian E. Reifsnyder
// Version:  0.5b
// Copyright:  1998 under the terms of the GNU GPL
*/

/* Phil Brutsche - November 20, 1998
  Fixed a number of bugs:
  * In several places, I found the use of gets (bad in itself...) in
    conjunction with uninitialized pointers.  Specifically, this:

	  char *input_buffer;

	  long partition_size;

	     Clear_Screen();
	     printf("\n\n\nEnter Primary Partition Size: ");
	     gets(input_buffer);

    That's a no-no; you don't know where input_buffer points to - it could
    point to, say, 0x00000010, and royally screw up your interrupt table
    when you use gets.  Plus, `gets` doesn't check the length of the string
    you are reading - automatic buffer overflow.  You may get your input,
    but it'll royally screw up anything that it physically next to the
    buffer in memory - maybe even part of your program!
  * Created some preprocessor directives to simplify the help screens.
*/



#define NAME "Free FDISK"
#define VERSION "0.5"

/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <bios.h>
#include <dos.h>
#include <mem.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bootcode.h"

/*
/////////////////////////////////////////////////////////////////////////////
//  DEFINES
/////////////////////////////////////////////////////////////////////////////
*/

#define READ 2

#define TRUE 1
#define FALSE 0

#define PRIMARY 1
#define EXTENDED 2
#define LOGICAL 3

#define LAST 99

#define NULL 0

#define PERCENTAGE 1
/*
/////////////////////////////////////////////////////////////////////////////
//  GLOBAL VARIABLES
/////////////////////////////////////////////////////////////////////////////
*/

unsigned char sector_buffer[512];

unsigned long g_cylinder;
unsigned long g_sector;

/* Hard Drive Parameters */
unsigned long total_cylinders;
unsigned long total_heads;
unsigned long total_sectors;

/* Ending Mapping Variables */
unsigned long computed_ending_cylinder;
unsigned long computed_ending_head;
unsigned long computed_ending_sector;
unsigned long computed_partition_size;

/* Partition Table Variables */
int extended_partition_exists=FALSE;

int active_flag[4];
char partition_type[24] [14];
int numeric_partition_type[24];

unsigned long starting_cylinder[24];
unsigned long starting_head[24];
unsigned long starting_sector[24];

unsigned long ending_cylinder[24];
unsigned long ending_head[24];
unsigned long ending_sector[24];

long partition_size[24];
int number_of_partitions;

unsigned long extended_partition_cylinder[24];
unsigned long extended_partition_head[24];
unsigned long extended_partition_sector[24];

/* Integers converted from long numbers */
int integer1;
int integer2;

/*
/////////////////////////////////////////////////////////////////////////////
//  PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/

char Input();

unsigned long Combine_Cylinder_and_Sector(unsigned long cylinder, unsigned long sector);

long Decimal_Number(long hex1, long hex2, long hex3);

void Calculate_Partition_Ending_Cylinder(long start_cylinder,long size);
void Clear_Screen();
void Clear_Sector_Buffer();
void Clear_Partition_Table(int drive);
void Compute_Partition_End(long scylinder,int shead,int ssector,long size);
void Convert_Long_To_Integer(long number);
void Create_MBR(int drive);
void Create_New_Partition(int drive,int create_partition_type,unsigned long size,int percentage);
void Delete_Partition(int drive, long partition_number);
void Display_Help_Screen();
void Display_L_and_C();
void Display_Partition_Table(int drive);
void Extract_Cylinder_and_Sector(unsigned long hex1, unsigned long hex2);
void Get_Hard_Drive_Parameters(int drive);
void Initialize_Variables();
void List_Available_Drives();
void Menu_Routine();
void Pause();
void Read_Partition_Table(int drive);
void Read_Sector(int drive, int head, long cyl, int sector);
void Remove_MBR(int drive);
void Toggle_Active_Partition(int drive, long partition_number, int toggle);
void Write_Sector(int drive, int head, long cyl, int sector);

/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

/* Calculate the end of a Partition */
/* Replaces void Compute_Partition_End */
void Calculate_Partition_Ending_Cylinder(long start_cylinder,long size)
{
  long cylinder_size=(total_heads+1)*(total_sectors);

  computed_partition_size=0;
  computed_ending_cylinder=start_cylinder;

  do
    {
    computed_ending_cylinder++;
    computed_partition_size=computed_partition_size+cylinder_size;
    }while(computed_partition_size<size);

  computed_ending_cylinder--;
  computed_partition_size=computed_partition_size-cylinder_size;
}

/* Clear Screen */
void Clear_Screen()
{
  asm{
    mov ah,0
    mov al,3
    int 0x10
    }
}

/* Clear Sector Buffer */
void Clear_Sector_Buffer()
{
  int index=0;

  do
    {
    sector_buffer[index]=0;
    index ++;
    } while(index<512);
}

/* Clear Partition Table */
void Clear_Partition_Table(int drive)
{
  Clear_Sector_Buffer();
  Write_Sector(drive, 0, 0, 1);
}

/* Combine Cylinder and Sector Values */
unsigned long Combine_Cylinder_and_Sector(unsigned long cylinder, unsigned long sector)
{
  long value = 0;


/*THIS CODE IS NOT 8088 COMPLIENT!!! */
  asm{
    mov ax,WORD PTR cylinder
    mov bx,WORD PTR sector

    mov dl,ah
    shl dl,6

    mov dh,al

    add dx,bx

    mov WORD PTR value,dx
    }

  return(value);
}

/* Convert Long number to 2 integers */
void Convert_Long_To_Integer(long number)
{
  integer1=0;
  integer2=0;

  asm{
    mov ax,WORD PTR number

    mov BYTE PTR integer1, al
    mov BYTE PTR integer2, ah
    }
}

/* Create Master Boot Code */
void Create_MBR(int drive)
{
  long number_of_bytes=445; /* ***** This is the number of bytes for the */
			    /* ***** boot code in the partition table.   */
			    /* ***** If the boot code in the partition   */
			    /* ***** table changes, this WILL need       */
			    /* ***** changed.  (Start the count at 0.)   */
  long loop=0;

  Read_Sector(drive,0,0,1);

  do
    {
    sector_buffer[loop]=boot_code[loop];

    loop++;
    }while(loop<=number_of_bytes);

  Write_Sector(drive,0,0,1);
}

/* Create New Partition */
void Create_New_Partition(int drive,int create_partition_type,unsigned long size,int percentage)
{
  int index;
  int next_available_partition;
  int partition_created=FALSE;

  unsigned long temp;

  unsigned long extended_partition_size;
  unsigned long extended_partition_size_high;
  unsigned long extended_partition_size_low;

  unsigned long relative_sectors;
  unsigned long relative_sectors_high;
  unsigned long relative_sectors_low;

  unsigned long total_hard_disk_size;

  long partition_table_pointer;

  long cp_starting_cylinder;
  long cp_starting_head;
  long cp_starting_sector;

  unsigned long partition_size_high;
  unsigned long partition_size_low;

  Read_Partition_Table(drive);
  Get_Hard_Drive_Parameters(drive);

  total_hard_disk_size = (total_cylinders+1) * (total_heads+1) * (total_sectors);

  /* Compute the size of the partition */
  if(percentage==PERCENTAGE)
    {
    size=(total_hard_disk_size*size)/100;
    }
  else
    {
    size=size*2000;
    }
  if(size>total_hard_disk_size) size = total_hard_disk_size;

  if((create_partition_type==PRIMARY) || (create_partition_type==EXTENDED))
    {
    index=0;

    do
      {
      if( (extended_partition_exists==TRUE) && (create_partition_type==EXTENDED) )
	{
	printf("\nExtended partition already exists...operation terminated.\n");
	exit(1);
	}
      /* Find unused primary partition */
      if(0==numeric_partition_type[index])
	{
	/* Unused primary partition found! */


	/* If this is the first partition */
	if(0==index)
	  {
	  cp_starting_cylinder=0;
	  cp_starting_head=1;
	  cp_starting_sector=1;
	  Calculate_Partition_Ending_Cylinder(cp_starting_cylinder,size);
	  }

	if(0!=index)
	  {
	  cp_starting_cylinder=(ending_cylinder[index-1]+1);
	  cp_starting_head=0;
	  cp_starting_sector=1;
	  Calculate_Partition_Ending_Cylinder(cp_starting_cylinder,size);
	  }

	if(create_partition_type==PRIMARY)
	  {
	  if(size>0xffff) numeric_partition_type[index]=0x06;
	  if(size<=0xffff) numeric_partition_type[index]=0x04;
	  if(size<=4096) numeric_partition_type[index]=0x01;
	  }
	else
	  {
	  numeric_partition_type[index]=0x05;
	  }

	/* Make computations & write partition */
	Read_Sector(drive,0,0,1);

	partition_table_pointer=0x1be+(index*16);

	temp=Combine_Cylinder_and_Sector(cp_starting_cylinder,cp_starting_sector);

	Convert_Long_To_Integer(temp);

	sector_buffer[partition_table_pointer+0x02]=integer1;
	sector_buffer[partition_table_pointer+0x03]=integer2;

	sector_buffer[partition_table_pointer+0x01]=cp_starting_head;

	sector_buffer[partition_table_pointer+0x04]=numeric_partition_type[index];

	temp=Combine_Cylinder_and_Sector(computed_ending_cylinder,total_sectors);

	Convert_Long_To_Integer(temp);

	sector_buffer[partition_table_pointer+0x06]=integer1;
	sector_buffer[partition_table_pointer+0x07]=integer2;

	sector_buffer[partition_table_pointer+0x05]=total_heads;

	/* Adjust size to fit inside partition */
	if(size> ((total_heads+1)*total_sectors))
	  {
	  size=(computed_ending_cylinder-cp_starting_cylinder)*( (total_heads+1)*(total_sectors) );
	  }

	if(create_partition_type==PRIMARY) size=size-total_sectors;

	partition_size_high = size >> 16;
	temp=partition_size_high << 16;
	if(size > 0xffff)
	  {
	  partition_size_low = size - temp;
	  }
	else
	  {
	  partition_size_low = size;
	  }

	Convert_Long_To_Integer(partition_size_low);

	sector_buffer[partition_table_pointer+0x0c]=integer1;
	sector_buffer[partition_table_pointer+0x0d]=integer2;

	Convert_Long_To_Integer(partition_size_high);

	sector_buffer[partition_table_pointer+0x0e]=integer1;
	sector_buffer[partition_table_pointer+0x0f]=integer2;

	/* Set the relative sector field in the partition table. */
	if(create_partition_type==PRIMARY)
	  {
	  sector_buffer[partition_table_pointer+0x08]=total_sectors;
	  }
	else
	  {
	  int secondary_index=0;
	  int primary_partition_number=-1;

	  /* Which partition is the primary partition? */
	  do
	    {
	    if(0x01==numeric_partition_type[secondary_index]) primary_partition_number=secondary_index;
	    if(0x04==numeric_partition_type[secondary_index]) primary_partition_number=secondary_index;
	    if(0x06==numeric_partition_type[secondary_index]) primary_partition_number=secondary_index;

	    secondary_index++;
	    }while(primary_partition_number==-1);

	  relative_sectors=(cp_starting_cylinder-starting_cylinder[primary_partition_number])*( (total_heads+1) * (total_sectors) );

	  relative_sectors_high = relative_sectors >> 16;
	  temp=relative_sectors_high << 16;

	  if(relative_sectors > 0xffff)
	    {
	    relative_sectors_low = relative_sectors - temp;
	    }
	  else
	    {
	    relative_sectors_low = relative_sectors;
	    }

	  Convert_Long_To_Integer(relative_sectors_low);

	  sector_buffer[partition_table_pointer+0x08]=integer1;
	  sector_buffer[partition_table_pointer+0x09]=integer2;

	  Convert_Long_To_Integer(relative_sectors_high);

	  sector_buffer[partition_table_pointer+0x0a]=integer1;
	  sector_buffer[partition_table_pointer+0x0b]=integer2;
	  }

	sector_buffer[0x1fe]=0x55;
	sector_buffer[0x1ff]=0xaa;

	Write_Sector(drive,0,0,1);

	/* Clear boot sector of newly created partition */
	Clear_Sector_Buffer();
	Write_Sector(drive,cp_starting_head,cp_starting_cylinder,1);

	partition_created=TRUE;
	}
      index++;
      } while((index<4) && (partition_created==FALSE));
    }

  if((create_partition_type==LOGICAL) && (extended_partition_exists==TRUE))
    {

    /* If this is not the first logical drive,   */
    /* create the extended entry in the previous */
    /* logical drive table.                      */
    if(number_of_partitions>=5)
      {
      Read_Sector(drive,extended_partition_head[number_of_partitions-4],extended_partition_cylinder[number_of_partitions-4],extended_partition_sector[number_of_partitions-4]);

      partition_table_pointer=0x1be+16;

      /* Set the 2nd partition entry in this table to extended */
      sector_buffer[partition_table_pointer+0x04]=0x05;

      Extract_Cylinder_and_Sector(sector_buffer[(0x1be+0x06)],sector_buffer[(0x1be+0x07)]);

      /* Save the location of the next partition */
      extended_partition_cylinder[number_of_partitions-3]=g_cylinder+1;
      extended_partition_sector[number_of_partitions-3]=1;
      extended_partition_head[number_of_partitions-3]=0;

      if(extended_partition_cylinder[number_of_partitions-3]>total_cylinders)
	{
	printf("\nPartition cannot be created past end of drive...Operation Terminated.\n");
	exit(1);
	}

      /* Add the logical drive pointer information to the sector buffer */
      sector_buffer[partition_table_pointer+0x01]=0;

      temp=Combine_Cylinder_and_Sector((extended_partition_cylinder[number_of_partitions-3]),1);

      Convert_Long_To_Integer(temp);

      sector_buffer[partition_table_pointer+0x02]=integer1;
      sector_buffer[partition_table_pointer+0x03]=integer2;

      /* Compute the end of the partition */
      Calculate_Partition_Ending_Cylinder((extended_partition_cylinder[number_of_partitions-3]),size);

      sector_buffer[partition_table_pointer+0x05]=total_heads;

      temp=Combine_Cylinder_and_Sector(computed_ending_cylinder,total_heads);

      Convert_Long_To_Integer(temp);

      sector_buffer[partition_table_pointer+0x06]=integer1;
      sector_buffer[partition_table_pointer+0x07]=integer2;

      /* Compute the "relative sectors" */
      relative_sectors=(extended_partition_cylinder[number_of_partitions-3]-starting_cylinder[4])*((total_heads+1)*(total_sectors));

      relative_sectors_high = relative_sectors >> 16;
      temp=relative_sectors_high << 16;

      if(relative_sectors > 0xffff)
	{
	relative_sectors_low = relative_sectors - temp;
	}
      else
	{
	relative_sectors_low = relative_sectors;
	}

      Convert_Long_To_Integer(relative_sectors_low);

      sector_buffer[partition_table_pointer+0x08]=integer1;
      sector_buffer[partition_table_pointer+0x09]=integer2;

      Convert_Long_To_Integer(relative_sectors_high);

      sector_buffer[partition_table_pointer+0x0a]=integer1;
      sector_buffer[partition_table_pointer+0x0b]=integer2;

      /* Compute the partition size */
      extended_partition_size=(computed_ending_cylinder-extended_partition_cylinder[number_of_partitions-3])*( (total_heads+1)*(total_sectors)  );

      extended_partition_size_high = extended_partition_size >> 16;
      temp=extended_partition_size_high << 16;

      if(extended_partition_size > 0xffff)
	{
	extended_partition_size_low = extended_partition_size - temp;
	}
      else
	{
	extended_partition_size_low = extended_partition_size;
	}

      Convert_Long_To_Integer(extended_partition_size_low);

      sector_buffer[partition_table_pointer+0x0c]=integer1;
      sector_buffer[partition_table_pointer+0x0d]=integer2;

      Convert_Long_To_Integer(extended_partition_size_high);

      sector_buffer[partition_table_pointer+0x0e]=integer1;
      sector_buffer[partition_table_pointer+0x0f]=integer2;

      /* Write Sector */
      Write_Sector(drive,extended_partition_head[number_of_partitions-4],extended_partition_cylinder[number_of_partitions-4],extended_partition_sector[number_of_partitions-4]);
      }

    /* Get the next available partition to use */
    next_available_partition=number_of_partitions-3;

    /* Compute the beginning of the partition */
    cp_starting_cylinder=extended_partition_cylinder[next_available_partition];

    /* Compute the end of the partition */
    Calculate_Partition_Ending_Cylinder(cp_starting_cylinder,size);

    Clear_Sector_Buffer();

    partition_table_pointer=0x1be;

    /* Create partition table */
    temp=Combine_Cylinder_and_Sector(cp_starting_cylinder,1);

    Convert_Long_To_Integer(temp);

    sector_buffer[partition_table_pointer+0x02]=integer1;
    sector_buffer[partition_table_pointer+0x03]=integer2;

    sector_buffer[partition_table_pointer+0x01]=1;

    if(size>0xffff) sector_buffer[partition_table_pointer+0x04]=0x06;
    if(size<=0xffff) sector_buffer[partition_table_pointer+0x04]=0x04;
    if(size<=4096) sector_buffer[partition_table_pointer+0x04]=0x01;

    temp=Combine_Cylinder_and_Sector(computed_ending_cylinder,total_sectors);

    Convert_Long_To_Integer(temp);

    sector_buffer[partition_table_pointer+0x06]=integer1;
    sector_buffer[partition_table_pointer+0x07]=integer2;

    sector_buffer[partition_table_pointer+0x05]=total_sectors;

    /* Adjust size to fit inside partition */
    size=((computed_ending_cylinder-cp_starting_cylinder)*( (total_heads+1) * (total_sectors) )) - total_sectors;

    partition_size_high = size >> 16;
    temp=partition_size_high << 16;
    if(size > 0xffff)
      {
      partition_size_low = size - temp;
      }
    else
      {
      partition_size_low = size;
      }

    Convert_Long_To_Integer(partition_size_low);

    sector_buffer[partition_table_pointer+0x08]=total_sectors;

    sector_buffer[partition_table_pointer+0x0c]=integer1;
    sector_buffer[partition_table_pointer+0x0d]=integer2;

    Convert_Long_To_Integer(partition_size_high);

    sector_buffer[partition_table_pointer+0x0e]=integer1;
    sector_buffer[partition_table_pointer+0x0f]=integer2;

    sector_buffer[0x1fe]=0x55;
    sector_buffer[0x1ff]=0xaa;

    Write_Sector(drive,0,cp_starting_cylinder,1);

    /* Clear "boot" sector of new partition */
    Clear_Sector_Buffer();
    Write_Sector(drive,1,cp_starting_cylinder,1);
    }

  if((create_partition_type==LOGICAL) && (extended_partition_exists==FALSE))
    {
    printf("\nExtended partition non-existant...operation terminated.\n");
    exit(1);
    }
}

/* Compute Partition Ending */
void Compute_Partition_End(long scylinder,int shead,int ssector,long size)
{
  int begin=FALSE;

  unsigned long sector_counter;

  sector_counter=1;

  computed_ending_cylinder=scylinder;
  computed_ending_head=shead;
  computed_ending_sector=ssector;

  do
    {
    if(begin==TRUE)
      {
      computed_ending_head=0;
      computed_ending_cylinder++;
      }
    do
      {
      if(begin==TRUE)
	{
	computed_ending_sector=1;
	computed_ending_head++;
	}
      begin=TRUE;

      do
	{
	sector_counter++;
	computed_ending_sector++;
	}while((sector_counter<size) && (computed_ending_sector <= total_sectors));

      }while((sector_counter<size) && (computed_ending_head <= total_heads));

    }while(sector_counter<size);

  computed_ending_head--;

  if(computed_ending_cylinder>total_cylinders) computed_ending_cylinder=total_cylinders;
}

/* Convert Hexidecimal Number to a Decimal Number */
long Decimal_Number(long hex1, long hex2, long hex3)
{
  return((hex1) + (hex2*256) + (hex3*65536));
}

/* Delete Partition */
void Delete_Partition(int drive,long partition_number)
{
  int loop=0;

  long offset;

  if(partition_number==LAST)
    {
    Read_Partition_Table(drive);

    if(number_of_partitions==5)
      {
      Clear_Sector_Buffer();
      Write_Sector(drive,extended_partition_head[1],extended_partition_cylinder[1],extended_partition_sector[1]);
      }
    else
      {
      Read_Sector(drive,extended_partition_head[number_of_partitions-5],extended_partition_cylinder[number_of_partitions-5],extended_partition_sector[number_of_partitions-5]);

      offset=(0x1be+16);

      do
	{
	sector_buffer[offset+loop]=0x00;
	loop++;
	}while(loop < 16);

      Write_Sector(drive,extended_partition_head[number_of_partitions-5],extended_partition_cylinder[number_of_partitions-5],extended_partition_sector[number_of_partitions-5]);
      }
    }
  else
    {
    Read_Sector(drive,0,0,1);

    offset=((partition_number*16)-16)+0x1be;

    do
      {
      sector_buffer[offset+loop]=0x00;
      loop++;
      }while(loop < 16);

    Write_Sector(drive,0,0,1);
    }
}

/* Display Help Screen */
void Display_Help_Screen()
{
  printf("Syntax:                                               *************************\n");
  printf("                                                      *        Warning!       *\n");
  printf("                                                      * Use this program with *\n");
  printf("FDISK                                                 *  extreme care!  Type  *\n");
  printf("FDISK /??                                             *   FDISK /?? for more  *\n");
  printf("FDISK [/N drive# partition_type size [/P] ]           *       information.    *\n");
  printf("FDISK [/D drive# [partition#] || [/L]]                *************************\n");
  printf("FDISK [/I [drive#] ]\n");
  printf("FDISK [/L]\n");
  printf("FDISK [/MBR [drive#] ]\n");
  printf("FDISK [/RMBR [drive#] ]\n");
  printf("FDISK [/C drive#]\n");
  printf("FDISK [/T drive# partition# active_toggle]");
  printf("\n  /?      Displays this help screen.\n");
  printf("  /??     Displays licensing, credit, and warning screen.\n");
  printf("  /N      Creates a new partition based upon the drive# (1-4), type (P, E,\n");
  printf("            or L), and size (in megabytes).  If the /P switch is added,\n");
  printf("            the size is a percentage of the total drive space.\n");
  printf("  /D      Deletes a partition based upon the drive# (1-4) and the partition\n");
  printf("            number or the last logical partition.\n");
  printf("  /I      Displays the partition table for drive# (1-4).\n");
  printf("  /L      Lists the available hard disks and their sizes.\n");
  printf("  /MBR    Writes the Master Boot Record to drive#.\n");
  printf("  /RMBR   Removes the Master Boot Record from drive#.\n");
  printf("  /C      Clears the partition sector.  (Use with extreme caution!)\n");
  printf("  /T      Toggles the partition# as active or not active (1 or 0).\n");
}

/* Display Licensing and Credit Screen */
void Display_L_and_C()
{
  printf("\n\n%10s                                                  Version %s\n", NAME, VERSION);
  printf("\nWritten By:  Brian E. Reifsnyder\n");
  printf("\nCopyright 1998 under the terms of the GNU General Public License.\n");
  printf("\nThis program comes as-is and without warranty of any kind.  The\n");
  printf("author of this software assumes no responsibility pertaining to\n");
  printf("the use or mis-use of this software.  By using this software, the\n");
  printf("operator is understood to be agreeing to the terms of the above.\n");
  printf("\n\n\n");
}

/* Display Partition Table */
void Display_Partition_Table(int drive)
{
  int partition_number=0;
  unsigned long total_hard_disk_size;

  /* Get partition table */
  Read_Partition_Table(drive);

  /* Display partition table */
  printf("\n\nPrimary Partition:\n");
  printf("-------------------------------------------------------------------------------\n");
  printf(" #  A   Type             Starting Location   Ending Location     Partition Size\n");
  printf("                         Cyl   Head  Sect    Cyl   Head  Sect     in Megabytes\n\n");

  do
    {
    /* Print partition number */
    printf(" %d",(partition_number+1));

    /* Is partition a boot partition? */
    if(active_flag[partition_number]==TRUE)
      {
      printf("  *   ");
      }
    else
      {
      printf("      ");
      }

    /* Display Partition Type */
    printf(partition_type[partition_number]);

    /* Display Partition Mappings */
    if(0!=strcmp(partition_type[partition_number],"Unused       "))
      {
      printf("    %4d",starting_cylinder[partition_number]);
      printf("  %4d",starting_head[partition_number]);
      printf("  %4d",starting_sector[partition_number]);

      printf("    %4d",ending_cylinder[partition_number]);
      printf("  %4d",ending_head[partition_number]);
      printf("  %4d",ending_sector[partition_number]);

      printf("       %6d",partition_size[partition_number]);
      }

    /* Carriage Return */
    printf("\n");

    partition_number++;
    } while(partition_number<4);

  printf("-------------------------------------------------------------------------------\n\n");

  if(number_of_partitions>4)
    {
    partition_number=4;

    printf("Extended Partition:\n");
    printf("-------------------------------------------------------------------------------\n");
    printf(" #      Type             Starting Location   Ending Location     Partition Size\n");
    printf("                         Cyl   Head  Sect    Cyl   Head  Sect     in Megabytes\n\n");

    do{

      /* Print partition number */
      printf(" %d",(partition_number+1));

      /* Display Partition Type */
      printf("      ");
      printf(partition_type[partition_number]);

      /* Display Partition Mappings */
      if(1!=strcmp(partition_type[partition_number],"Unused       "))
	{
	printf("    %4d",starting_cylinder[partition_number]);
	printf("  %4d",starting_head[partition_number]);
	printf("  %4d",starting_sector[partition_number]);

	printf("    %4d",ending_cylinder[partition_number]);
	printf("  %4d",ending_head[partition_number]);
	printf("  %4d",ending_sector[partition_number]);

	printf("       %6d",partition_size[partition_number]);
	}

      /* Carriage Return */
      printf("\n");

      partition_number++;
      }while(partition_number<number_of_partitions);
    printf("-------------------------------------------------------------------------------\n");
    }

  Get_Hard_Drive_Parameters(drive);
  total_hard_disk_size=(total_cylinders+1)*(total_heads+1)*total_sectors;
  total_hard_disk_size=total_hard_disk_size/2000;
  printf("                                                  Total hard drive size:  %d\n",total_hard_disk_size);
}

/* Extract Cylinder & Sector */
void Extract_Cylinder_and_Sector(unsigned long hex1, unsigned long hex2)
{
  unsigned long cylinder_and_sector = ( (256*hex2) + hex1 );

  g_sector = cylinder_and_sector % 64;

  g_cylinder = ( ( (cylinder_and_sector*4) & 768) + (cylinder_and_sector /256) );
}


/* Get Hard Drive Parameters */
void Get_Hard_Drive_Parameters(int drive)
{
/* NOT 8088 COMPLIENT CODE!!!!! */

  asm{
    mov ah, 0x08
    mov dl, BYTE PTR drive
    int 0x13

    mov bl,cl
    and bl,00111111B

    mov BYTE PTR total_sectors, bl

    mov bl,cl
    mov cl,ch
    shr bl,6
    mov ch,bl

    mov WORD PTR total_cylinders, cx
    mov BYTE PTR total_heads, dh
    }
}

/* Initialize Variables */
void Initialize_Variables()
{
  int loop=0;

  do
    {
    active_flag[loop]=FALSE;

    loop++;
    }while(loop<=4);

}

/* Get input from keyboard */
char Input()
{
  char input;

  asm{
    mov ah,7
    int 0x21
    mov BYTE PTR input,al
    }

  return(input);
}

/* List Available Hard Drives */
void List_Available_Drives()
{
  char temp_sector_buffer[512];

  int loop=128;
  int result;
  unsigned long total_hard_disk_size;

  printf("\n  Drive              Size\n");
  printf("  Number          (Megabytes)\n\n");
  do
    {
    result=biosdisk(2, loop, 0, 0, 1, 1, temp_sector_buffer);

    if (result==0)
      {
      Get_Hard_Drive_Parameters(loop);
      total_hard_disk_size=(total_cylinders+1)*(total_heads+1)*total_sectors;
      total_hard_disk_size=total_hard_disk_size/2000;
      printf("    %d                %d\n",(loop-127),total_hard_disk_size);
      }

    loop++;
    }while(loop<=131);
  printf("\n\n");
}

/* Menu Routine */
void Menu_Routine()
{
  int exit=FALSE;
  int exit_sub_menu=FALSE;
  int drive=128;
  long partition_number;

  char input;

  do
    {
    Initialize_Variables();
    Clear_Screen();

    printf("                                  Version %s\n",VERSION);
    printf("                            Fixed Disk Setup Program\n");
    printf("                      Copyright 1998 by Brian E. Reifsnyder\n");
    printf("                         under the terms of the GNU GPL\n\n");
    printf("                                 FDISK Options\n\n");
    printf("Current fixed disk drive: %d\n\n",(drive-127));
    printf("Choose one of the following:\n\n");
    printf("1. Create DOS partition or Logical DOS Drive\n");
    printf("2. Set active partition\n");
    printf("3. Delete partition or Logical DOS Drive\n");
    printf("4. Display partition information\n");
    printf("5. Change current fixed disk drive\n");
    printf("6. Exit this utility\n\n");
    printf("Enter choice: ");

    /* Get input from keyboard. */
    input=Input();

    /* Create DOS partition or Logical DOS Drive */
    if(input==49)
      {
      exit_sub_menu=FALSE;

      do
	{
	Clear_Screen();
	printf("\n\n\n\n\n                   Create DOS Partition or Logical DOS Drive\n\n");
	printf("Current fixed disk drive: %d\n\n",(drive-127));
	printf("Choose on of the following:\n\n");
	printf("1. Create Primary DOS Partition\n");
	printf("2. Create Extended DOS Partition\n");
	printf("3. Create Logical DOS Drive(s) in the Extended DOS Partition\n");
	printf("4. Return to Main Menu\n\n\n");
	printf("Enter choice: ");
	input=Input();

	/* Create Primary DOS Partition */
	if(input==49)
	  {
	  int percentage_flag=NULL;
	  int true=1;

	  char input_buffer[10];

	  long partition_size;

	  Clear_Screen();
	  printf("\n\n\nEnter Primary Partition Size: ");
	  fgets(input_buffer,10,stdin);

	  partition_size=atol(input_buffer);

	  /* trap bad input */
	  if(partition_size<=0)
	    {
	    partition_size=5;
	    strcpy(input_buffer,"null");
	    }

	  if(partition_size>2048)
	    {
	    printf("\n\nEntered partition size is too large (> 2048)...Operation Terminated.\n");
	    }
	  else
	    {
	    if(partition_size<=100)
	      {
	      printf("\nIs the partition size entered a percentage (Y/N)?");
	      fgets(input_buffer,10,stdin);

	      true=1;
	      true=stricmp(input_buffer,"Y\n");
	      if(0==true)
		{
		percentage_flag=PERCENTAGE;
		}
	      else
	      /* trap bad input */
		{
		percentage_flag=NULL;
		strcpy(input_buffer,"null");
		}
	      }
	    }

	  Create_New_Partition(drive,PRIMARY,partition_size,percentage_flag);
	  }

	/* Create Extended DOS Partition */
	if(input==50)
	  {
	  int percentage_flag=NULL;
	  int true=1;

	  char input_buffer[10];

	  long partition_size;

	  Clear_Screen();
	  printf("\n\n\nEnter Extended Partition Size: ");
	  fgets(input_buffer,10,stdin);

	  partition_size=atol(input_buffer);

	  /* trap bad input */
	  if(partition_size<=0)
	    {
	    partition_size=5;
	    strcpy(input_buffer,"null");
	    }

	  if(partition_size>2048)
	    {
	    printf("\n\nEntered partition size is too large (> 2048)...Operation Terminated.\n");
	    Pause();
	    }
	  else
	    {
	    if(partition_size<=100)
	      {
	      printf("\nIs the partition size entered a percentage (Y/N)?");
	      fgets(input_buffer,10,stdin);

	      true=1;
	      true=stricmp(input_buffer,"Y\n");
	      if(0==true)
		{
		percentage_flag=PERCENTAGE;
		}
	      else
		{
		percentage_flag=NULL;
		strcpy(input_buffer,"null");
		}
	      }
	    }

	  Create_New_Partition(drive,EXTENDED,partition_size,percentage_flag);
	  }

	/* Create Logical DOS Drive(s) in the Extended DOS Partition */
	if(input==51)
	  {
	  char input_buffer[10];

	  long partition_size;

	  int percentage_flag=NULL;
	  int true=1;

	  Clear_Screen();
	  printf("\n\n\nEnter Logical DOS Drive Size: ");
	  fgets(input_buffer,10,stdin);

	  partition_size=atol(input_buffer);

	  /* trap bad input */
	  if(partition_size<=0)
	    {
	    partition_size=5;
	    strcpy(input_buffer,"null");
	    }

	  if(partition_size>2048)
	    {
	    printf("\n\nEntered partition size is too large (> 2048)...Operation Terminated.\n");
	    Pause();
	    }
	  else
	    {
	    if(partition_size<=100)
	      {
	      printf("\nIs the partition size entered a percentage (Y/N)?");
	      fgets(input_buffer,10,stdin);

	      true=1;
	      true=stricmp(input_buffer,"Y");
	      if(0==true)
		{
		percentage_flag=PERCENTAGE;
		}
	      else
		{
		percentage_flag=NULL;
		strcpy(input_buffer,"null");
		}
	      }
	    Create_New_Partition(drive,LOGICAL,partition_size,percentage_flag);
	    }
	  }

	/* Return to Main Menu */
	if(input==52)
	  {
	  exit_sub_menu=TRUE;
	  }

	}while(exit_sub_menu==FALSE);

      input=99; /* change value to prevent any other options from being executed */
      }

    /* Set active partition */
    if(input==50)
      {
      Clear_Screen();
      printf("Fixed Disk Drive: %d",drive-127);
      Display_Partition_Table(drive);
      printf("\n\nEnter partition number to set active: ");

      /* Get input from keyboard. */
      input=Input();

      partition_number=input-48;

      if((partition_number>=1) && (partition_number <=4))
	{
	Toggle_Active_Partition(drive,partition_number,1);

	if(partition_number!=1) Toggle_Active_Partition(drive,1,0);
	if(partition_number!=2) Toggle_Active_Partition(drive,2,0);
	if(partition_number!=3) Toggle_Active_Partition(drive,3,0);
	if(partition_number!=4) Toggle_Active_Partition(drive,4,0);
	}
      else
	{
	printf("\nPartition number is out of range (1-4)...Operation Terminated.\n");
	Pause();
	}

      }

    /* Delete partition or Logical DOS Drive */
    if(input==51)
      {
      exit_sub_menu=FALSE;

      do
	{
	Clear_Screen();
	printf("\n\n\n\n\n                   Delete DOS Partition or Logical DOS Drive\n\n");
	printf("Current fixed disk drive: %d\n\n",(drive-127));
	printf("Choose one of the following:\n\n");
	printf("1. Delete Primary DOS Partition\n");
	printf("2. Delete Extended DOS Partition\n");
	printf("3. Delete Logical DOS Drive(s) in the Extended DOS Partition\n");
	printf("4. Delete Non-DOS Partition\n");
	printf("5. Return to Main Menu\n\n\n");
	input=Input();

	/* Delete Primary DOS Partition */
	if(input==49)
	  {
	  input=50;
	  }

	/* Delete Extended DOS Partition */
	if(input==50)
	  {
	  input=52;
	  }

	/* Delete Logical DOS Drive(s) in the Extended DOS Partition */
	if(input==51)
	  {
	  char input_buffer[10];

	  int true=1;

	  Clear_Screen();
	  printf("\n\n\nWarning:  you are about to DELETE the last logical drive in the extended\n");
	  printf("          partition.\n\n\n");
	  printf("Continue with operation (y/n)? ");
	  fgets(input_buffer,10,stdin);

	  true=1;
	  true=stricmp(input_buffer,"Y\n");
	  if(0==true)
	    {
	    Read_Partition_Table(drive);

	    if(number_of_partitions<5)
	      {
	      printf("\nNo logical partitions exist...Operation Terminated.\n");
	      Pause();
	      }
	    else
	      {
	      Delete_Partition(drive,LAST);
	      printf("\nPartition Deleted!\n");
	      Pause();
	      }
	    }
	  else
	    {
	    printf("\n\n\nOperation Aborted.\n");
	    Pause();
	    }
	  }

	/* Delete Non-DOS Partition */
	if(input==52)
	  {
	  char *input_buffer;

	  int partition_number;

	  Clear_Screen();
	  printf("\n\n\nWarning:  You are about to DELETE a partition.  To go back, type \"0\"\n");
	  printf("          and press [ENTER].  Otherwise, enter the partition number to\n");
	  printf("          delete.\n\n\n");
	  printf("Partition number to delete: ");
	  fgets(input_buffer,10,stdin);

	  partition_number=atoi(input_buffer);

	  if((partition_number>=1) && (partition_number<=4))
	    {
	    Delete_Partition(drive,partition_number);
	    printf("\nPartition Deleted!\n");
	    Pause();
	    }
	  else
	    /* trap bad input */
	    {
	    printf("\n\n\nOperation Aborted.\n");
	    Pause();
	    partition_number=0;
	    strcpy(input_buffer,"null");
	    }
	  }

	/* Return to Main Menu */
	if(input==53)
	  {
	  exit_sub_menu=TRUE;
	  }

	}while(exit_sub_menu==FALSE);
      input=99; /* change value to prevent any other options from being executed */
      }

    /* Diplay partition information */
    if(input==52)
      {
      Clear_Screen();
      printf("Fixed Disk Drive: %d",drive-127);
      Display_Partition_Table(drive);
      Pause();
      }

    /* Change current fixed disk drive */
    if(input==53)
      {
      Clear_Screen();
      List_Available_Drives();
      printf("Enter Fixed Disk Drive Number: ");

      /* Get input from keyboard. */
      input=Input();

      input=input-47;

      /*check to make sure the drive is a legitimate number*/
      if( (input<2) || (input>6) )
	{
	printf("\nDrive number is out of range, no changes have been made.\n");
	Pause();
	}

      drive=input+126;
      }

    /* Exit */
    if(input==54) exit=TRUE;

    }while(exit==FALSE);

  Clear_Screen();
  printf("\n\n\nReboot the computer to ensure that any changes take effect.\n\n");
}

/* Pause Routine */
void Pause()
{
  printf("\nPress any key to continue.\n");

  asm{
    mov ah,7
    int 0x21
    }
}

/* Read Partition Table */
void Read_Partition_Table(int drive)
{
  long index=0x1be;

  int exiting_primary=TRUE;
  int extended=FALSE;
  int extended_pointer=1;
  int partition_designation=PRIMARY;
  int pointer=0;
  int record_extended_info_flag=FALSE;

  int done_flag=FALSE;

  unsigned long extended_cylinder;
  unsigned long extended_head;
  unsigned long extended_sector;

  Read_Sector(drive,0,0,1);

  do{
    if(pointer==4) partition_designation=EXTENDED;

    if((pointer>=4) && (extended==TRUE))
      {
      Read_Sector(drive,extended_head,extended_cylinder,extended_sector);
      extended=FALSE;
      index=0x1be;

      if(exiting_primary==FALSE)
	{
	pointer--;
	}
      else
	{
	exiting_primary=FALSE;
	}
      }

    /* Flag boot partition */
    if((sector_buffer[index]==128) && (partition_designation==PRIMARY)) active_flag[pointer]=TRUE;

    /* Determine Partition Type */
    numeric_partition_type[pointer]=sector_buffer[index+4];

    if(sector_buffer[index+4]==0x00)
      {
      strcpy(partition_type[pointer],"Unused       ");

      if(partition_designation==EXTENDED)
	{
	number_of_partitions=pointer;
	done_flag=TRUE;
	}
      }
    if(sector_buffer[index+4]==0x01) strcpy(partition_type[pointer],"DOS-12       ");
    if(sector_buffer[index+4]==0x02) strcpy(partition_type[pointer],"XENIX root   ");
    if(sector_buffer[index+4]==0x03) strcpy(partition_type[pointer],"XENIX usr    ");
    if(sector_buffer[index+4]==0x04) strcpy(partition_type[pointer],"DOS-16 <32M  ");
    if(sector_buffer[index+4]==0x05)
      {
      strcpy(partition_type[pointer],"Extended     ");
      extended=TRUE;
      extended_partition_exists=TRUE;
      record_extended_info_flag=TRUE;
      }
    if(sector_buffer[index+4]==0x06) strcpy(partition_type[pointer],"DOS-16 >=32M ");
    if(sector_buffer[index+4]==0x07) strcpy(partition_type[pointer],"OS/2 HPFS    ");
    if(sector_buffer[index+4]==0x08) strcpy(partition_type[pointer],"AIX          ");
    if(sector_buffer[index+4]==0x09) strcpy(partition_type[pointer],"AIX bootable ");
    if(sector_buffer[index+4]==0x0a) strcpy(partition_type[pointer],"OS/2 boot mgr");
    if(sector_buffer[index+4]==0x0b) strcpy(partition_type[pointer],"Win 95 FAT32 ");
    if(sector_buffer[index+4]==0x40) strcpy(partition_type[pointer],"Venix 80286  ");
    if(sector_buffer[index+4]==0x50) strcpy(partition_type[pointer],"DM           ");
    if(sector_buffer[index+4]==0x52) strcpy(partition_type[pointer],"Microport    ");
    if(sector_buffer[index+4]==0x56) strcpy(partition_type[pointer],"GB           ");
    if(sector_buffer[index+4]==0x61) strcpy(partition_type[pointer],"Speed        ");
    if(sector_buffer[index+4]==0x63) strcpy(partition_type[pointer],"386/ix       ");
    if(sector_buffer[index+4]==0x64) strcpy(partition_type[pointer],"Novell       ");
    if(sector_buffer[index+4]==0x65) strcpy(partition_type[pointer],"Novell       ");
    if(sector_buffer[index+4]==0x75) strcpy(partition_type[pointer],"PCIX         ");
    if(sector_buffer[index+4]==0x80) strcpy(partition_type[pointer],"Old MINIX    ");
    if(sector_buffer[index+4]==0x81) strcpy(partition_type[pointer],"Linux/MINIX  ");
    if(sector_buffer[index+4]==0x82) strcpy(partition_type[pointer],"Linux Swap   ");
    if(sector_buffer[index+4]==0x83) strcpy(partition_type[pointer],"Linux Native ");
    if(sector_buffer[index+4]==0x93) strcpy(partition_type[pointer],"Amoeba       ");
    if(sector_buffer[index+4]==0x94) strcpy(partition_type[pointer],"Amoeba BBT   ");
    if(sector_buffer[index+4]==0xa5) strcpy(partition_type[pointer],"BSD/386      ");
    if(sector_buffer[index+4]==0xb7) strcpy(partition_type[pointer],"BSDI fs      ");
    if(sector_buffer[index+4]==0xb8) strcpy(partition_type[pointer],"BSDI swap    ");
    if(sector_buffer[index+4]==0xc7) strcpy(partition_type[pointer],"Syrinx       ");
    if(sector_buffer[index+4]==0xdb) strcpy(partition_type[pointer],"CP/M         ");
    if(sector_buffer[index+4]==0xe1) strcpy(partition_type[pointer],"DOS access   ");
    if(sector_buffer[index+4]==0xe3) strcpy(partition_type[pointer],"DOS R/O      ");
    if(sector_buffer[index+4]==0xf2) strcpy(partition_type[pointer],"DOS secondary");
    if(sector_buffer[index+4]==0xff) strcpy(partition_type[pointer],"BBT          ");

    starting_head[pointer] = sector_buffer[index+1];
    ending_head[pointer] = sector_buffer[index+5];

    partition_size[pointer]=Decimal_Number(sector_buffer[index+12],sector_buffer[index+13],sector_buffer[index+14])/2000;

    Extract_Cylinder_and_Sector(sector_buffer[index+2],sector_buffer[index+3]);

    starting_cylinder[pointer]=g_cylinder;
    starting_sector[pointer]=g_sector;

    if((extended==TRUE) && (record_extended_info_flag==TRUE))
      {
      extended_cylinder=starting_cylinder[pointer];
      extended_head=starting_head[pointer];
      extended_sector=starting_sector[pointer];

      extended_partition_cylinder[extended_pointer]=extended_cylinder;
      extended_partition_head[extended_pointer]=extended_head;
      extended_partition_sector[extended_pointer]=extended_sector;

      extended_pointer++;

      record_extended_info_flag=FALSE;
      }

    Extract_Cylinder_and_Sector(sector_buffer[index+6],sector_buffer[index+7]);

    ending_cylinder[pointer]=g_cylinder;
    ending_sector[pointer]=g_sector;

    pointer++;
    number_of_partitions=pointer-1;

    if((extended==FALSE) && (pointer==4))
      {
      number_of_partitions=4;
      done_flag=TRUE;
      }

    index=index + 16;
    } while(done_flag==FALSE);
}

/* Read_Sector */
void Read_Sector(int drive, int head, long cyl, int sector)
{
  int result;

  result=biosdisk(2, drive, head, cyl, sector, 1, sector_buffer);

    if (result!=0)
      {
      printf("\nDisk read error...operation terminated.\n");
      exit(1);
      }

}

/* Remove MBR */
void Remove_MBR(int drive)
{
  long pointer=0;

  Read_Sector(drive,0,0,1);

  do
    {
    sector_buffer[pointer]=0x00;
    pointer++;
    }while(pointer<0x1be);

  Write_Sector(drive,0,0,1);
}

/* Toggle Active Partition */
void Toggle_Active_Partition(int drive, long partition_number, int toggle)
{
  int value;

  long offset;

  Read_Sector(drive,0,0,1);

  if(1==toggle)
    {
    value=0x80;
    }
  else
    {
    value=0x00;
    }

  offset=((partition_number*16)-16)+0x1be;

  sector_buffer[offset]=value;

  Write_Sector(drive,0,0,1);
}

/* Write Sector */
void Write_Sector(int drive, int head, long cyl, int sector)
{
  int result;

  result=biosdisk(3, drive, head, cyl, sector, 1, sector_buffer);

  if (result!=0)
    {
    printf("\nDisk write error...operation terminated.\n");
    exit(1);
    }
}

/*
/////////////////////////////////////////////////////////////////////////////
//  MAIN ROUTINE
/////////////////////////////////////////////////////////////////////////////
*/
void main(int argc, char *argv[])
{
  int true;

  int drive;

  /* if FDISK is typed without any options */
  if(argc==1)
    {
    Menu_Routine();
    exit(0);
    }

  /* if FDISK is typed with options */
  if(argc>=2)
    {

    /* if "FDISK /?" is typed */
    true=1;
    true=strcmp("/?",argv[1]);
    if(0==true)
      {
      Display_Help_Screen();
      exit(0);
      }

    /* if "FDISK /??" is typed */
    true=1;
    true=strcmp("/??",argv[1]);
    if(0==true)
      {
      Display_L_and_C();
      exit(0);
      }

    /* if "FDISK /L" is typed */
    true=1;
    true=stricmp("/L",argv[1]);
    if(0==true)
      {
      List_Available_Drives();
      exit(0);
      }

    /*if there is a 2nd option, convert the option to a drive # */
    if(argc>=3)
      {
      drive=argv[2] [0];

      drive=drive-47;

      /*check to make sure the drive is a legitimate number */
      if( (drive<2) || (drive>6) )
	{
	printf("\nIncorrect drive designation...Operation Terminated.\n");
	exit(1);
	}

      drive=drive+126;
      }

    /*if no drive number is entered, set the drive to the first physical drive */
    if(argc<3)
      {
      drive=128;
      }

    /*if "FDISK /C" is typed */
    true=1;
    true=stricmp("/C",argv[1]);
    if(0==true)
      {
      Clear_Partition_Table(drive);
      exit(0);
      }

    /*if "FDISK /I" is typed */
    true=1;
    true=stricmp("/I",argv[1]);

    if(0==true)
      {
      Clear_Screen();
      Display_Partition_Table(drive);
      exit(0);
      }

    /* if "FDISK /N" is typed*/

    /*   Syntax: FDISK /N drive# partition_type size [/P] */

    true=1;
    true=stricmp("/N",argv[1]);

    if(0==true)
      {
      int partition_type=NULL;
      int percentage_flag=NULL;

      unsigned long size;

      if(argc<5)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      /* Determine type */
      true=1;
      true=stricmp("P",argv[3]);
      if(0==true) partition_type = PRIMARY;

      true=1;
      true=stricmp("E",argv[3]);
      if(0==true) partition_type = EXTENDED;

      true=1;
      true=stricmp("L",argv[3]);
      if(0==true) partition_type = LOGICAL;

      if(partition_type==NULL)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      /* Set partition size */

      size=atol(argv[4]);

      if(size>2048)
	{
	printf("\nEntered partition size is too large (> 2048)...Operation Terminated.\n");
	exit(1);
	}

      /* Set percentage_flag, if necessary. */
      if(argc==6)
	{
	true=1;
	true=stricmp("/P",argv[5]);
	if(0==true) percentage_flag = PERCENTAGE;

	if(percentage_flag==NULL)
	  {
	  printf("\nSyntax Error...Operation Terminated.\n");
	  exit(1);
	  }
	}
      Create_New_Partition(drive,partition_type,size,percentage_flag);
      exit(0);
      }

    /* if "FDISK /T" is typed */

    /* Syntax:  FDISK [/T drive# partition# active_toggle] */
    true=1;
    true=stricmp("/T",argv[1]);
    if(0==true)
      {
      if(argc<5)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}
      int partition_number=atoi(argv[3]);

      if((partition_number<1) || (partition_number>4))
	{
	printf("\nPartition number is out of range (1-4)...Operation Terminated.\n");
	exit(1);
	}

      int toggle=atoi(argv[4]);

      if((toggle<0) || (toggle>1))
	{
	printf("\nInvalid toggle setting...Operation Terminated.\n");
	exit(1);
	}

      Toggle_Active_Partition(drive,partition_number,toggle);
      exit(0);
      }

    /* if "FDISK /D" is typed */

    /* Syntax:  FDISK [/D drive# partition#] */
    true=1;
    true=stricmp("/D",argv[1]);
    if(0==true)
      {
      int partition_number;

      if(argc<4)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      true=1;
      true=stricmp("/L",argv[3]);
      if(0==true)
	{
	Read_Partition_Table(drive);

	if(number_of_partitions<5)
	  {
	  printf("\nNo logical partitions exist...Operation Terminated.\n");
	  exit(1);
	  }
	partition_number=LAST;
	}
      else
	{
	partition_number=atoi(argv[3]);

	if((partition_number<1) || (partition_number>4))
	  {
	  printf("\nPartition number is out of range (1-4)...Operation Terminated.\n");
	  exit(1);
	  }
	}
      Delete_Partition(drive,partition_number);
      exit(0);
      }

    /* if "FDISK /MBR" is typed */

    /* Syntax:  FDISK [/MBR drive#] */
    true=1;
    true=stricmp("/MBR",argv[1]);
    if(0==true)
      {
      Create_MBR(drive);
      exit(0);
      }

    /* if "FDISK /RMBR" is typed */

    /* Syntax:  FDISK [/RMBR drive#] */
    true=1;
    true=stricmp("/RMBR",argv[1]);
    if(0==true)
      {
      Remove_MBR(drive);
      exit(0);
      }
    }
}
